"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PoetryPyproject = exports.Poetry = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const TOML = require("@iarna/toml");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const task_runtime_1 = require("../task-runtime");
const toml_1 = require("../toml");
const util_1 = require("../util");
/**
 * Manage project dependencies, virtual environments, and packaging through the
 * poetry CLI tool.
 */
class Poetry extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.pythonExec = options.pythonExec ?? "python";
        this.installTask = project.addTask("install", {
            description: "Install dependencies and update lockfile",
            exec: "poetry update",
        });
        this.installCiTask = project.addTask("install:ci", {
            description: "Install dependencies with frozen lockfile",
            exec: "poetry check --lock && poetry install",
        });
        this.project.tasks.addEnvironment("VIRTUAL_ENV", "$(poetry env info -p)");
        this.project.tasks.addEnvironment("PATH", "$(echo $(poetry env info -p)/bin:$PATH)");
        project.packageTask.exec("poetry build");
        this.publishTestTask = project.addTask("publish:test", {
            description: "Uploads the package against a test PyPI endpoint.",
            exec: "poetry publish -r testpypi",
        });
        this.publishTask = project.addTask("publish", {
            description: "Uploads the package to PyPI.",
            exec: "poetry publish",
        });
        this.pyProject = new PoetryPyproject(project, {
            name: project.name,
            version: options.version,
            description: options.description ?? "",
            license: options.license,
            authors: [`${options.authorName} <${options.authorEmail}>`],
            homepage: options.homepage,
            classifiers: options.classifiers,
            ...options.poetryOptions,
            dependencies: () => this.synthDependencies(),
            devDependencies: () => this.synthDevDependencies(),
        });
        new toml_1.TomlFile(project, "poetry.toml", {
            committed: false,
            obj: {
                repositories: {
                    testpypi: {
                        url: "https://test.pypi.org/legacy/",
                    },
                },
            },
        });
    }
    synthDependencies() {
        const dependencies = {};
        let pythonDefined = false;
        for (const pkg of this.project.deps.all) {
            if (pkg.name === "python") {
                pythonDefined = true;
            }
            if (pkg.type === dependencies_1.DependencyType.RUNTIME) {
                dependencies[pkg.name] = pkg.version ?? "*";
            }
        }
        if (!pythonDefined) {
            // Python version must be defined for poetry projects. Default to ^3.8.
            dependencies.python = "^3.8";
        }
        return this.permitDependenciesWithMetadata(dependencies);
    }
    synthDevDependencies() {
        const dependencies = {};
        for (const pkg of this.project.deps.all) {
            if ([dependencies_1.DependencyType.DEVENV, dependencies_1.DependencyType.TEST].includes(pkg.type)) {
                dependencies[pkg.name] = pkg.version ?? "*";
            }
        }
        return this.permitDependenciesWithMetadata(dependencies);
    }
    /**
     * Allow for poetry dependencies to specify metadata, eg `mypackage@{ version="1.2.3", extras = ["my-package-extra"] }`
     * @param dependencies poetry dependencies object
     * @private
     */
    permitDependenciesWithMetadata(dependencies) {
        const parseVersionMetadata = (version) => {
            try {
                // Try parsing the version as toml to permit metadata
                return TOML.parse(`version = ${version}`).version;
            }
            catch (e) {
                // Invalid toml means it's not metadata, so should just be treated as the string
                return version;
            }
        };
        return Object.fromEntries(Object.entries(dependencies).map(([key, value]) => [
            key,
            parseVersionMetadata(value),
        ]));
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.DEVENV);
    }
    /**
     * Initializes the virtual environment if it doesn't exist (called during post-synthesis).
     */
    setupEnvironment() {
        const result = (0, util_1.execOrUndefined)("which poetry", {
            cwd: this.project.outdir,
        });
        if (!result) {
            this.project.logger.info("Unable to setup an environment since poetry is not installed. Please install poetry (https://python-poetry.org/docs/) or use a different component for managing environments such as 'venv'.");
        }
        let envPath = (0, util_1.execOrUndefined)("poetry env info -p", {
            cwd: this.project.outdir,
        });
        if (!envPath) {
            this.project.logger.info("Setting up a virtual environment...");
            (0, util_1.exec)(`poetry env use ${this.pythonExec}`, { cwd: this.project.outdir });
            envPath = (0, util_1.execOrUndefined)("poetry env info -p", {
                cwd: this.project.outdir,
            });
            this.project.logger.info(`Environment successfully created (located in ${envPath}}).`);
        }
    }
    /**
     * Installs dependencies (called during post-synthesis).
     */
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        // If the pyproject.toml file has changed, update the lockfile prior to installation
        if (this.pyProject.file.changed) {
            runtime.runTask(this.installTask.name);
        }
        else {
            runtime.runTask(this.installCiTask.name);
        }
    }
}
exports.Poetry = Poetry;
_a = JSII_RTTI_SYMBOL_1;
Poetry[_a] = { fqn: "projen.python.Poetry", version: "0.79.27" };
/**
 * Represents configuration of a pyproject.toml file for a Poetry project.
 *
 * @see https://python-poetry.org/docs/pyproject/
 */
class PoetryPyproject extends component_1.Component {
    constructor(project, options) {
        super(project);
        const decamelisedOptions = (0, util_1.decamelizeKeysRecursively)(options, {
            separator: "-",
        });
        this.file = new toml_1.TomlFile(project, "pyproject.toml", {
            omitEmpty: false,
            obj: {
                "build-system": {
                    requires: ["poetry_core>=1.0.0"],
                    "build-backend": "poetry.core.masonry.api",
                },
                tool: {
                    poetry: {
                        ...decamelisedOptions,
                    },
                },
            },
        });
    }
}
exports.PoetryPyproject = PoetryPyproject;
_b = JSII_RTTI_SYMBOL_1;
PoetryPyproject[_b] = { fqn: "projen.python.PoetryPyproject", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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