"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvePythonImportName = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const case_1 = require("case");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
const projenrc_1 = require("../projenrc");
/**
 * Allows writing projenrc files in python.
 *
 * This will install `projen` as a Python dependency and will add a
 * `synth` task which will run `.projenrc.py`.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.filePath = options.filename ?? ".projenrc.py";
        this.pythonExec = options.pythonExec ?? "python";
        project.deps.addDependency(`projen@${projenVersion}`, dependencies_1.DependencyType.DEVENV);
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`${this.pythonExec} .projenrc.py`);
        // if this is a new project, generate a skeleton for projenrc.py
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = (0, inventory_1.readJsiiManifest)(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const pythonFile = (0, path_1.join)(this.project.outdir, this.filePath);
        // skip if file exists
        if ((0, fs_1.existsSync)(pythonFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        // const openBlock = (line: string = '') => { emit(line + ' {'); indent++; };
        // const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = option.fqn;
            }
        }
        const { renderedOptions, imports } = renderPythonOptions(indent, optionFqns, bootstrap.args);
        const importName = resolvePythonImportName(jsiiFqn, jsiiManifest);
        emit(toPythonImport(importName));
        for (const fqn of imports) {
            emit(toPythonImport(fqn));
        }
        emit();
        emit(`project = ${jsiiType.name}(${renderedOptions})`);
        emit();
        emit("project.synth()");
        (0, fs_1.mkdirSync)((0, path_1.dirname)(pythonFile), { recursive: true });
        (0, fs_1.writeFileSync)(pythonFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${pythonFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.python.Projenrc", version: "0.79.27" };
function resolvePythonImportName(jsiiFqn, jsiiManifest) {
    const moduleName = jsiiManifest?.targets?.python?.module;
    // Module name prefix should take precedence in the event moduleName !== fqn prefix
    return moduleName
        ? [moduleName, ...jsiiFqn.split(".").slice(1)].join(".")
        : jsiiFqn;
}
exports.resolvePythonImportName = resolvePythonImportName;
function renderPythonOptions(indent, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        // no options
        return { renderedOptions: "", imports };
    }
    const lines = [""];
    for (const [name, value] of Object.entries(initOptions)) {
        const { pythonValue, importName } = toPythonValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`${toPythonProperty(name)}=${pythonValue},`);
    }
    const renderedOptions = lines
        .join(`\n${" ".repeat((indent + 1) * 4)}`)
        .concat("\n");
    return { renderedOptions, imports };
}
function toPythonProperty(prop) {
    return (0, case_1.snake)(prop);
}
function toPythonValue(value, name, optionFqns) {
    if (typeof value === "boolean") {
        return { pythonValue: value ? "True" : "False" };
    }
    else if (typeof value === "number") {
        return { pythonValue: JSON.stringify(value) };
    }
    else if (typeof value === "string") {
        if (optionFqns[name] !== undefined) {
            const parts = optionFqns[name].split(".");
            const base = parts[parts.length - 1];
            const choice = String(value).toUpperCase().replace(/-/g, "_");
            return { pythonValue: `${base}.${choice}`, importName: optionFqns[name] };
        }
        else {
            return { pythonValue: JSON.stringify(value) };
        }
    }
    else if (value === undefined || value === null) {
        return { pythonValue: "None" };
    }
    else {
        return { pythonValue: JSON.stringify(value) };
    }
}
function toPythonImport(fqn) {
    const parts = fqn.split(".");
    if (parts.length === 1) {
        return `import ${parts[0]}`;
    }
    else {
        return `from ${parts.slice(0, -1).join(".")} import ${parts[parts.length - 1]}`;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvamVucmMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvcHl0aG9uL3Byb2plbnJjLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkJBQTBEO0FBQzFELCtCQUFxQztBQUNyQywrQkFBNkI7QUFDN0Isc0NBQTJDO0FBQzNDLGtEQUFpRDtBQUNqRCw0Q0FBZ0Q7QUFFaEQsMENBQTJDO0FBeUIzQzs7Ozs7R0FLRztBQUNILE1BQWEsUUFBUyxTQUFRLHVCQUFZO0lBV3hDLFlBQVksT0FBZ0IsRUFBRSxVQUEyQixFQUFFO1FBQ3pELEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVmLE1BQU0sYUFBYSxHQUFHLE9BQU8sQ0FBQyxhQUFhLElBQUksdUJBQWMsQ0FBQztRQUM5RCxJQUFJLENBQUMsUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLElBQUksY0FBYyxDQUFDO1FBQ25ELElBQUksQ0FBQyxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxRQUFRLENBQUM7UUFDakQsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQ3hCLFVBQVUsYUFBYSxFQUFFLEVBQ3pCLDZCQUFjLENBQUMsTUFBTSxDQUN0QixDQUFDO1FBRUYsbUdBQW1HO1FBQ25HLE9BQU8sQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLFVBQVUsZUFBZSxDQUFDLENBQUM7UUFFN0QsZ0VBQWdFO1FBQ2hFLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO0lBQzFCLENBQUM7SUFFTyxnQkFBZ0I7UUFDdEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDM0MsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNkLE9BQU87U0FDUjtRQUNELE1BQU0sT0FBTyxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUM7UUFDOUIsTUFBTSxZQUFZLEdBQUcsSUFBQSw0QkFBZ0IsRUFBQyxPQUFPLENBQUMsQ0FBQztRQUMvQyxNQUFNLFFBQVEsR0FBRyxZQUFZLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzdDLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQztRQUN2RSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ25CLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDdEIsZ0RBQWdELENBQ2pELENBQUM7WUFDRixPQUFPO1NBQ1I7UUFFRCxNQUFNLFVBQVUsR0FBRyxJQUFBLFdBQUksRUFBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFNUQsc0JBQXNCO1FBQ3RCLElBQUksSUFBQSxlQUFVLEVBQUMsVUFBVSxDQUFDLEVBQUU7WUFDMUIsT0FBTztTQUNSO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUVsQyxJQUFJLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFDZixNQUFNLElBQUksR0FBRyxDQUFDLE9BQWUsRUFBRSxFQUFFLEVBQUUsQ0FDakMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztRQUM1Qyw2RUFBNkU7UUFDN0UscURBQXFEO1FBRXJELE1BQU0sVUFBVSxHQUEyQixFQUFFLENBQUM7UUFDOUMsS0FBSyxNQUFNLE1BQU0sSUFBSSxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMzQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2QsVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDO2FBQ3RDO1NBQ0Y7UUFFRCxNQUFNLEVBQUUsZUFBZSxFQUFFLE9BQU8sRUFBRSxHQUFHLG1CQUFtQixDQUN0RCxNQUFNLEVBQ04sVUFBVSxFQUNWLFNBQVMsQ0FBQyxJQUFJLENBQ2YsQ0FBQztRQUVGLE1BQU0sVUFBVSxHQUFHLHVCQUF1QixDQUFDLE9BQU8sRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7UUFFakMsS0FBSyxNQUFNLEdBQUcsSUFBSSxPQUFPLEVBQUU7WUFDekIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1NBQzNCO1FBQ0QsSUFBSSxFQUFFLENBQUM7UUFDUCxJQUFJLENBQUMsYUFBYSxRQUFRLENBQUMsSUFBSSxJQUFJLGVBQWUsR0FBRyxDQUFDLENBQUM7UUFDdkQsSUFBSSxFQUFFLENBQUM7UUFDUCxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUV4QixJQUFBLGNBQVMsRUFBQyxJQUFBLGNBQU8sRUFBQyxVQUFVLENBQUMsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3BELElBQUEsa0JBQWEsRUFBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRTVDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDdEIsMENBQTBDLFVBQVUsRUFBRSxDQUN2RCxDQUFDO0lBQ0osQ0FBQzs7QUExRkgsNEJBMkZDOzs7QUFFRCxTQUFnQix1QkFBdUIsQ0FBQyxPQUFlLEVBQUUsWUFBaUI7SUFDeEUsTUFBTSxVQUFVLEdBQUcsWUFBWSxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsTUFBTSxDQUFDO0lBRXpELG1GQUFtRjtJQUNuRixPQUFPLFVBQVU7UUFDZixDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7UUFDeEQsQ0FBQyxDQUFDLE9BQU8sQ0FBQztBQUNkLENBQUM7QUFQRCwwREFPQztBQUVELFNBQVMsbUJBQW1CLENBQzFCLE1BQWMsRUFDZCxVQUFrQyxFQUNsQyxXQUFpQztJQUVqQyxNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO0lBQ2xDLElBQUksQ0FBQyxXQUFXLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3pELGFBQWE7UUFDYixPQUFPLEVBQUUsZUFBZSxFQUFFLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBQztLQUN6QztJQUVELE1BQU0sS0FBSyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFbkIsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQUU7UUFDdkQsTUFBTSxFQUFFLFdBQVcsRUFBRSxVQUFVLEVBQUUsR0FBRyxhQUFhLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMzRSxJQUFJLFVBQVU7WUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ3hDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxXQUFXLEdBQUcsQ0FBQyxDQUFDO0tBQ3pEO0lBRUQsTUFBTSxlQUFlLEdBQUcsS0FBSztTQUMxQixJQUFJLENBQUMsS0FBSyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDekMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2hCLE9BQU8sRUFBRSxlQUFlLEVBQUUsT0FBTyxFQUFFLENBQUM7QUFDdEMsQ0FBQztBQUVELFNBQVMsZ0JBQWdCLENBQUMsSUFBWTtJQUNwQyxPQUFPLElBQUEsWUFBSyxFQUFDLElBQUksQ0FBQyxDQUFDO0FBQ3JCLENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FDcEIsS0FBVSxFQUNWLElBQVksRUFDWixVQUFrQztJQUVsQyxJQUFJLE9BQU8sS0FBSyxLQUFLLFNBQVMsRUFBRTtRQUM5QixPQUFPLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztLQUNsRDtTQUFNLElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxFQUFFO1FBQ3BDLE9BQU8sRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO0tBQy9DO1NBQU0sSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLEVBQUU7UUFDcEMsSUFBSSxVQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFO1lBQ2xDLE1BQU0sS0FBSyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDMUMsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDckMsTUFBTSxNQUFNLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFDOUQsT0FBTyxFQUFFLFdBQVcsRUFBRSxHQUFHLElBQUksSUFBSSxNQUFNLEVBQUUsRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7U0FDM0U7YUFBTTtZQUNMLE9BQU8sRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1NBQy9DO0tBQ0Y7U0FBTSxJQUFJLEtBQUssS0FBSyxTQUFTLElBQUksS0FBSyxLQUFLLElBQUksRUFBRTtRQUNoRCxPQUFPLEVBQUUsV0FBVyxFQUFFLE1BQU0sRUFBRSxDQUFDO0tBQ2hDO1NBQU07UUFDTCxPQUFPLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztLQUMvQztBQUNILENBQUM7QUFFRCxTQUFTLGNBQWMsQ0FBQyxHQUFXO0lBQ2pDLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDN0IsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUN0QixPQUFPLFVBQVUsS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7S0FDN0I7U0FBTTtRQUNMLE9BQU8sUUFBUSxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsV0FDekMsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUN4QixFQUFFLENBQUM7S0FDSjtBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleGlzdHNTeW5jLCBta2RpclN5bmMsIHdyaXRlRmlsZVN5bmMgfSBmcm9tIFwiZnNcIjtcbmltcG9ydCB7IGRpcm5hbWUsIGpvaW4gfSBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgc25ha2UgfSBmcm9tIFwiY2FzZVwiO1xuaW1wb3J0IHsgUFJPSkVOX1ZFUlNJT04gfSBmcm9tIFwiLi4vY29tbW9uXCI7XG5pbXBvcnQgeyBEZXBlbmRlbmN5VHlwZSB9IGZyb20gXCIuLi9kZXBlbmRlbmNpZXNcIjtcbmltcG9ydCB7IHJlYWRKc2lpTWFuaWZlc3QgfSBmcm9tIFwiLi4vaW52ZW50b3J5XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcbmltcG9ydCB7IFByb2plbnJjRmlsZSB9IGZyb20gXCIuLi9wcm9qZW5yY1wiO1xuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBQcm9qZW5yY2AuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVucmNPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBwcm9qZW5yYyBmaWxlLlxuICAgKiBAZGVmYXVsdCBcIi5wcm9qZW5yYy5weVwiXG4gICAqL1xuICByZWFkb25seSBmaWxlbmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHByb2plbiB2ZXJzaW9uIHRvIHVzZVxuICAgKiBAZGVmYXVsdCAtIGN1cnJlbnQgdmVyc2lvblxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVuVmVyc2lvbj86IHN0cmluZztcblxuICAvKipcbiAgICogUGF0aCB0byB0aGUgcHl0aG9uIGV4ZWN1dGFibGUgdG8gdXNlLlxuICAgKiBAZGVmYXVsdCBcInB5dGhvblwiXG4gICAqL1xuICByZWFkb25seSBweXRob25FeGVjPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEFsbG93cyB3cml0aW5nIHByb2plbnJjIGZpbGVzIGluIHB5dGhvbi5cbiAqXG4gKiBUaGlzIHdpbGwgaW5zdGFsbCBgcHJvamVuYCBhcyBhIFB5dGhvbiBkZXBlbmRlbmN5IGFuZCB3aWxsIGFkZCBhXG4gKiBgc3ludGhgIHRhc2sgd2hpY2ggd2lsbCBydW4gYC5wcm9qZW5yYy5weWAuXG4gKi9cbmV4cG9ydCBjbGFzcyBQcm9qZW5yYyBleHRlbmRzIFByb2plbnJjRmlsZSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgcHJvamVucmMgZmlsZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBmaWxlUGF0aDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQYXRoIHRvIHRoZSBweXRob24gZXhlY3V0YWJsZSB0byB1c2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHl0aG9uRXhlYzogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHByb2plY3Q6IFByb2plY3QsIG9wdGlvbnM6IFByb2plbnJjT3B0aW9ucyA9IHt9KSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG5cbiAgICBjb25zdCBwcm9qZW5WZXJzaW9uID0gb3B0aW9ucy5wcm9qZW5WZXJzaW9uID8/IFBST0pFTl9WRVJTSU9OO1xuICAgIHRoaXMuZmlsZVBhdGggPSBvcHRpb25zLmZpbGVuYW1lID8/IFwiLnByb2plbnJjLnB5XCI7XG4gICAgdGhpcy5weXRob25FeGVjID0gb3B0aW9ucy5weXRob25FeGVjID8/IFwicHl0aG9uXCI7XG4gICAgcHJvamVjdC5kZXBzLmFkZERlcGVuZGVuY3koXG4gICAgICBgcHJvamVuQCR7cHJvamVuVmVyc2lvbn1gLFxuICAgICAgRGVwZW5kZW5jeVR5cGUuREVWRU5WXG4gICAgKTtcblxuICAgIC8vIHNldCB1cCB0aGUgXCJkZWZhdWx0XCIgdGFzayB3aGljaCBpcyB0aGUgdGFzayBleGVjdXRlZCB3aGVuIGBwcm9qZW5gIGlzIGV4ZWN1dGVkIGZvciB0aGlzIHByb2plY3QuXG4gICAgcHJvamVjdC5kZWZhdWx0VGFzaz8uZXhlYyhgJHt0aGlzLnB5dGhvbkV4ZWN9IC5wcm9qZW5yYy5weWApO1xuXG4gICAgLy8gaWYgdGhpcyBpcyBhIG5ldyBwcm9qZWN0LCBnZW5lcmF0ZSBhIHNrZWxldG9uIGZvciBwcm9qZW5yYy5weVxuICAgIHRoaXMuZ2VuZXJhdGVQcm9qZW5yYygpO1xuICB9XG5cbiAgcHJpdmF0ZSBnZW5lcmF0ZVByb2plbnJjKCkge1xuICAgIGNvbnN0IGJvb3RzdHJhcCA9IHRoaXMucHJvamVjdC5pbml0UHJvamVjdDtcbiAgICBpZiAoIWJvb3RzdHJhcCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBqc2lpRnFuID0gYm9vdHN0cmFwLmZxbjtcbiAgICBjb25zdCBqc2lpTWFuaWZlc3QgPSByZWFkSnNpaU1hbmlmZXN0KGpzaWlGcW4pO1xuICAgIGNvbnN0IGpzaWlUeXBlID0ganNpaU1hbmlmZXN0LnR5cGVzW2pzaWlGcW5dO1xuICAgIGNvbnN0IG9wdGlvbnNUeXBlRnFuID0ganNpaVR5cGUuaW5pdGlhbGl6ZXI/LnBhcmFtZXRlcnM/LlswXS50eXBlPy5mcW47XG4gICAgaWYgKCFvcHRpb25zVHlwZUZxbikge1xuICAgICAgdGhpcy5wcm9qZWN0LmxvZ2dlci53YXJuKFxuICAgICAgICBcImNhbm5vdCBkZXRlcm1pbmUganNpaSB0eXBlIGZvciBwcm9qZWN0IG9wdGlvbnNcIlxuICAgICAgKTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBweXRob25GaWxlID0gam9pbih0aGlzLnByb2plY3Qub3V0ZGlyLCB0aGlzLmZpbGVQYXRoKTtcblxuICAgIC8vIHNraXAgaWYgZmlsZSBleGlzdHNcbiAgICBpZiAoZXhpc3RzU3luYyhweXRob25GaWxlKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGxpbmVzID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICAgIGxldCBpbmRlbnQgPSAwO1xuICAgIGNvbnN0IGVtaXQgPSAobGluZTogc3RyaW5nID0gXCJcIikgPT5cbiAgICAgIGxpbmVzLnB1c2goXCIgXCIucmVwZWF0KGluZGVudCAqIDQpICsgbGluZSk7XG4gICAgLy8gY29uc3Qgb3BlbkJsb2NrID0gKGxpbmU6IHN0cmluZyA9ICcnKSA9PiB7IGVtaXQobGluZSArICcgeycpOyBpbmRlbnQrKzsgfTtcbiAgICAvLyBjb25zdCBjbG9zZUJsb2NrID0gKCkgPT4geyBpbmRlbnQtLTsgZW1pdCgnfScpOyB9O1xuXG4gICAgY29uc3Qgb3B0aW9uRnFuczogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuICAgIGZvciAoY29uc3Qgb3B0aW9uIG9mIGJvb3RzdHJhcC50eXBlLm9wdGlvbnMpIHtcbiAgICAgIGlmIChvcHRpb24uZnFuKSB7XG4gICAgICAgIG9wdGlvbkZxbnNbb3B0aW9uLm5hbWVdID0gb3B0aW9uLmZxbjtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCB7IHJlbmRlcmVkT3B0aW9ucywgaW1wb3J0cyB9ID0gcmVuZGVyUHl0aG9uT3B0aW9ucyhcbiAgICAgIGluZGVudCxcbiAgICAgIG9wdGlvbkZxbnMsXG4gICAgICBib290c3RyYXAuYXJnc1xuICAgICk7XG5cbiAgICBjb25zdCBpbXBvcnROYW1lID0gcmVzb2x2ZVB5dGhvbkltcG9ydE5hbWUoanNpaUZxbiwganNpaU1hbmlmZXN0KTtcbiAgICBlbWl0KHRvUHl0aG9uSW1wb3J0KGltcG9ydE5hbWUpKTtcblxuICAgIGZvciAoY29uc3QgZnFuIG9mIGltcG9ydHMpIHtcbiAgICAgIGVtaXQodG9QeXRob25JbXBvcnQoZnFuKSk7XG4gICAgfVxuICAgIGVtaXQoKTtcbiAgICBlbWl0KGBwcm9qZWN0ID0gJHtqc2lpVHlwZS5uYW1lfSgke3JlbmRlcmVkT3B0aW9uc30pYCk7XG4gICAgZW1pdCgpO1xuICAgIGVtaXQoXCJwcm9qZWN0LnN5bnRoKClcIik7XG5cbiAgICBta2RpclN5bmMoZGlybmFtZShweXRob25GaWxlKSwgeyByZWN1cnNpdmU6IHRydWUgfSk7XG4gICAgd3JpdGVGaWxlU3luYyhweXRob25GaWxlLCBsaW5lcy5qb2luKFwiXFxuXCIpKTtcblxuICAgIHRoaXMucHJvamVjdC5sb2dnZXIuaW5mbyhcbiAgICAgIGBQcm9qZWN0IGRlZmluaXRpb24gZmlsZSB3YXMgY3JlYXRlZCBhdCAke3B5dGhvbkZpbGV9YFxuICAgICk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmVQeXRob25JbXBvcnROYW1lKGpzaWlGcW46IHN0cmluZywganNpaU1hbmlmZXN0OiBhbnkpIHtcbiAgY29uc3QgbW9kdWxlTmFtZSA9IGpzaWlNYW5pZmVzdD8udGFyZ2V0cz8ucHl0aG9uPy5tb2R1bGU7XG5cbiAgLy8gTW9kdWxlIG5hbWUgcHJlZml4IHNob3VsZCB0YWtlIHByZWNlZGVuY2UgaW4gdGhlIGV2ZW50IG1vZHVsZU5hbWUgIT09IGZxbiBwcmVmaXhcbiAgcmV0dXJuIG1vZHVsZU5hbWVcbiAgICA/IFttb2R1bGVOYW1lLCAuLi5qc2lpRnFuLnNwbGl0KFwiLlwiKS5zbGljZSgxKV0uam9pbihcIi5cIilcbiAgICA6IGpzaWlGcW47XG59XG5cbmZ1bmN0aW9uIHJlbmRlclB5dGhvbk9wdGlvbnMoXG4gIGluZGVudDogbnVtYmVyLFxuICBvcHRpb25GcW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+LFxuICBpbml0T3B0aW9ucz86IFJlY29yZDxzdHJpbmcsIGFueT5cbikge1xuICBjb25zdCBpbXBvcnRzID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIGlmICghaW5pdE9wdGlvbnMgfHwgT2JqZWN0LmtleXMoaW5pdE9wdGlvbnMpLmxlbmd0aCA9PT0gMCkge1xuICAgIC8vIG5vIG9wdGlvbnNcbiAgICByZXR1cm4geyByZW5kZXJlZE9wdGlvbnM6IFwiXCIsIGltcG9ydHMgfTtcbiAgfVxuXG4gIGNvbnN0IGxpbmVzID0gW1wiXCJdO1xuXG4gIGZvciAoY29uc3QgW25hbWUsIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhpbml0T3B0aW9ucykpIHtcbiAgICBjb25zdCB7IHB5dGhvblZhbHVlLCBpbXBvcnROYW1lIH0gPSB0b1B5dGhvblZhbHVlKHZhbHVlLCBuYW1lLCBvcHRpb25GcW5zKTtcbiAgICBpZiAoaW1wb3J0TmFtZSkgaW1wb3J0cy5hZGQoaW1wb3J0TmFtZSk7XG4gICAgbGluZXMucHVzaChgJHt0b1B5dGhvblByb3BlcnR5KG5hbWUpfT0ke3B5dGhvblZhbHVlfSxgKTtcbiAgfVxuXG4gIGNvbnN0IHJlbmRlcmVkT3B0aW9ucyA9IGxpbmVzXG4gICAgLmpvaW4oYFxcbiR7XCIgXCIucmVwZWF0KChpbmRlbnQgKyAxKSAqIDQpfWApXG4gICAgLmNvbmNhdChcIlxcblwiKTtcbiAgcmV0dXJuIHsgcmVuZGVyZWRPcHRpb25zLCBpbXBvcnRzIH07XG59XG5cbmZ1bmN0aW9uIHRvUHl0aG9uUHJvcGVydHkocHJvcDogc3RyaW5nKSB7XG4gIHJldHVybiBzbmFrZShwcm9wKTtcbn1cblxuZnVuY3Rpb24gdG9QeXRob25WYWx1ZShcbiAgdmFsdWU6IGFueSxcbiAgbmFtZTogc3RyaW5nLFxuICBvcHRpb25GcW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+XG4pIHtcbiAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJib29sZWFuXCIpIHtcbiAgICByZXR1cm4geyBweXRob25WYWx1ZTogdmFsdWUgPyBcIlRydWVcIiA6IFwiRmFsc2VcIiB9O1xuICB9IGVsc2UgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJudW1iZXJcIikge1xuICAgIHJldHVybiB7IHB5dGhvblZhbHVlOiBKU09OLnN0cmluZ2lmeSh2YWx1ZSkgfTtcbiAgfSBlbHNlIGlmICh0eXBlb2YgdmFsdWUgPT09IFwic3RyaW5nXCIpIHtcbiAgICBpZiAob3B0aW9uRnFuc1tuYW1lXSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBjb25zdCBwYXJ0cyA9IG9wdGlvbkZxbnNbbmFtZV0uc3BsaXQoXCIuXCIpO1xuICAgICAgY29uc3QgYmFzZSA9IHBhcnRzW3BhcnRzLmxlbmd0aCAtIDFdO1xuICAgICAgY29uc3QgY2hvaWNlID0gU3RyaW5nKHZhbHVlKS50b1VwcGVyQ2FzZSgpLnJlcGxhY2UoLy0vZywgXCJfXCIpO1xuICAgICAgcmV0dXJuIHsgcHl0aG9uVmFsdWU6IGAke2Jhc2V9LiR7Y2hvaWNlfWAsIGltcG9ydE5hbWU6IG9wdGlvbkZxbnNbbmFtZV0gfTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHsgcHl0aG9uVmFsdWU6IEpTT04uc3RyaW5naWZ5KHZhbHVlKSB9O1xuICAgIH1cbiAgfSBlbHNlIGlmICh2YWx1ZSA9PT0gdW5kZWZpbmVkIHx8IHZhbHVlID09PSBudWxsKSB7XG4gICAgcmV0dXJuIHsgcHl0aG9uVmFsdWU6IFwiTm9uZVwiIH07XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIHsgcHl0aG9uVmFsdWU6IEpTT04uc3RyaW5naWZ5KHZhbHVlKSB9O1xuICB9XG59XG5cbmZ1bmN0aW9uIHRvUHl0aG9uSW1wb3J0KGZxbjogc3RyaW5nKSB7XG4gIGNvbnN0IHBhcnRzID0gZnFuLnNwbGl0KFwiLlwiKTtcbiAgaWYgKHBhcnRzLmxlbmd0aCA9PT0gMSkge1xuICAgIHJldHVybiBgaW1wb3J0ICR7cGFydHNbMF19YDtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gYGZyb20gJHtwYXJ0cy5zbGljZSgwLCAtMSkuam9pbihcIi5cIil9IGltcG9ydCAke1xuICAgICAgcGFydHNbcGFydHMubGVuZ3RoIC0gMV1cbiAgICB9YDtcbiAgfVxufVxuIl19