"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const pip_1 = require("./pip");
const poetry_1 = require("./poetry");
const projenrc_1 = require("./projenrc");
const pytest_1 = require("./pytest");
const pytest_sample_1 = require("./pytest-sample");
const python_sample_1 = require("./python-sample");
const setuptools_1 = require("./setuptools");
const venv_1 = require("./venv");
const github_1 = require("../github");
const projenrc_2 = require("../javascript/projenrc");
const project_1 = require("../project");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
/** Allowed characters in python project names */
const PYTHON_PROJECT_NAME_REGEX = /^[A-Za-z0-9-_\.]+$/;
/**
 * Python project.
 *
 * @pjid python
 */
class PythonProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        if (!PYTHON_PROJECT_NAME_REGEX.test(options.name)) {
            throw new Error("Python projects must only consist of alphanumeric characters, hyphens, and underscores.");
        }
        this.moduleName = options.moduleName;
        this.version = options.version;
        const rcFileTypeOptions = [
            options.projenrcPython,
            options.projenrcJs,
            options.projenrcJson,
            options.projenrcTs,
        ];
        if ((0, util_1.multipleSelected)(rcFileTypeOptions)) {
            throw new Error("Only one of projenrcPython, projenrcJs, projenrcTs, and projenrcJson can be selected.");
        }
        const poetry = options.poetry ?? false;
        const pip = options.pip ?? !poetry;
        const venv = options.venv ?? !poetry;
        const setuptools = options.setuptools ?? (!poetry && this.projectType === project_1.ProjectType.LIB);
        if (poetry && (pip || venv || setuptools)) {
            throw new Error("poetry is true - pip, venv, and setuptools must be undefined or false");
        }
        if (!this.parent) {
            // default to projenrc.py if no other projenrc type was elected
            if (options.projenrcPython ?? !(0, util_1.anySelected)(rcFileTypeOptions)) {
                new projenrc_1.Projenrc(this, {
                    pythonExec: options.pythonExec,
                    ...options.projenrcPythonOptions,
                });
            }
            if (options.projenrcJs ?? false) {
                new projenrc_2.Projenrc(this, options.projenrcJsOptions);
            }
            if (options.projenrcTs ?? false) {
                new typescript_1.ProjenrcTs(this, options.projenrcTsOptions);
            }
        }
        if (venv) {
            this.envManager = new venv_1.Venv(this, {
                pythonExec: options.pythonExec,
                ...options.venvOptions,
            });
        }
        if (pip) {
            this.depsManager = new pip_1.Pip(this);
        }
        if (setuptools) {
            this.packagingManager = new setuptools_1.Setuptools(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                setupConfig: options.setupConfig,
                pythonExec: options.pythonExec,
            });
        }
        if (poetry) {
            const poetryProject = new poetry_1.Poetry(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                pythonExec: options.pythonExec,
                poetryOptions: {
                    readme: options.readme?.filename ?? "README.md",
                    ...options.poetryOptions,
                },
            });
            this.depsManager = poetryProject;
            this.envManager = poetryProject;
            this.packagingManager = poetryProject;
        }
        if (!this.envManager) {
            throw new Error("At least one tool must be chosen for managing the environment (venv, conda, pipenv, or poetry).");
        }
        if (!this.depsManager) {
            throw new Error("At least one tool must be chosen for managing dependencies (pip, conda, pipenv, or poetry).");
        }
        if (!this.packagingManager && this.projectType === project_1.ProjectType.LIB) {
            throw new Error("At least one tool must be chosen for managing packaging (setuptools or poetry).");
        }
        if (Number(venv) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing the environment (venv, conda, pipenv, or poetry)");
        }
        if (Number(pip) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing dependencies (pip, conda, pipenv, or poetry)");
        }
        if (Number(setuptools) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing packaging (setuptools or poetry)");
        }
        if (options.pytest ?? true) {
            this.pytest = new pytest_1.Pytest(this, options.pytestOptions);
            if (options.sample ?? true) {
                new pytest_sample_1.PytestSample(this, {
                    moduleName: this.moduleName,
                    testdir: this.pytest.testdir,
                });
            }
        }
        if (options.sample ?? true) {
            new python_sample_1.PythonSample(this, {
                dir: this.moduleName,
            });
        }
        for (const dep of options.deps ?? []) {
            this.addDependency(dep);
        }
        for (const dep of options.devDeps ?? []) {
            this.addDevDependency(dep);
        }
        this.addDefaultGitIgnore();
    }
    /**
     * Adds default gitignore options for a Python project based on
     * https://github.com/github/gitignore/blob/master/Python.gitignore
     */
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Byte-compiled / optimized / DLL files", "__pycache__/", "*.py[cod]", "*$py.class", "", "# C extensions", "*.so", "", "# Distribution / packaging", ".Python", "build/", "develop-eggs/", "dist/", "downloads/", "eggs/", ".eggs/", "lib/", "lib64/", "parts/", "sdist/", "var/", "wheels/", "share/python-wheels/", "*.egg-info/", ".installed.cfg", "*.egg", "MANIFEST", "", "# PyInstaller", "#  Usually these files are written by a python script from a template", "#  before PyInstaller builds the exe, so as to inject date/other infos into it.", "*.manifest", "*.spec", "", "# Installer logs", "pip-log.txt", "pip-delete-this-directory.txt", "", "# Unit test / coverage reports", "htmlcov/", ".tox/", ".nox/", ".coverage", ".coverage.*", ".cache", "nosetests.xml", "coverage.xml", "*.cover", "*.py,cover", ".hypothesis/", ".pytest_cache/", "cover/", "", "# Translations", "*.mo", "*.pot", "", "# Django stuff:", "*.log", "local_settings.py", "db.sqlite3", "db.sqlite3-journal", "", "# Flask stuff:", "instance/", ".webassets-cache", "", "# Scrapy stuff:", ".scrapy", "", "# Sphinx documentation", "docs/_build/", "", "# PyBuilder", ".pybuilder/", "target/", "", "# Jupyter Notebook", ".ipynb_checkpoints", "", "# IPython", "profile_default/", "ipython_config.py", "", "# PEP 582; used by e.g. github.com/David-OConnor/pyflow", "__pypackages__/", "", "# Celery stuff", "celerybeat-schedule", "celerybeat.pid", "", "# SageMath parsed files", "*.sage.py", "", "# Environments", ".env", ".venv", "env/", "venv/", "ENV/", "env.bak/", "venv.bak/", "", "# Spyder project settings", ".spyderproject", ".spyproject", "", "# Rope project settings", ".ropeproject", "", "# mkdocs documentation", "/site", "", "# mypy", ".mypy_cache/", ".dmypy.json", "dmypy.json", "", "# Pyre type checker", ".pyre/", "", "# pytype static type analyzer", ".pytype/", "", "# Cython debug symbols", "cython_debug/");
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        return this.depsManager.addDependency(spec);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        return this.depsManager.addDevDependency(spec);
    }
    postSynthesize() {
        super.postSynthesize();
        this.envManager.setupEnvironment();
        this.depsManager.installDependencies();
    }
}
exports.PythonProject = PythonProject;
_a = JSII_RTTI_SYMBOL_1;
PythonProject[_a] = { fqn: "projen.python.PythonProject", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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