"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const semver_1 = require("semver");
const logging = require("../logging");
const util_1 = require("../util");
const version_1 = require("../version");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    const versionFile = (0, path_1.join)(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const minor = options.minorVersion;
    const minMajorVersion = options.minMajorVersion;
    const prefix = options.tagPrefix ?? "";
    const bumpFile = (0, path_1.join)(cwd, options.bumpFile);
    const changelogFile = (0, path_1.join)(cwd, options.changelog);
    const releaseTagFile = (0, path_1.join)(cwd, options.releaseTagFile);
    if (major && minMajorVersion) {
        throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
    }
    if (minor && !major) {
        throw new Error(`minorVersion and majorVersion must be used together.`);
    }
    await fs_1.promises.mkdir((0, path_1.dirname)(bumpFile), { recursive: true });
    await fs_1.promises.mkdir((0, path_1.dirname)(changelogFile), { recursive: true });
    await fs_1.promises.mkdir((0, path_1.dirname)(releaseTagFile), { recursive: true });
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({
        cwd,
        major,
        minor,
        prerelease,
        prefix,
    });
    const { contents, newline } = await tryReadVersionFile(versionFile);
    // update version
    contents.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_1.promises.writeFile(versionFile, JSON.stringify(contents, undefined, 2) + (newline ? "\n" : ""));
    // check for commits since the last release tag
    let skipBump = false;
    // First Release is never skipping bump
    if (!isFirstRelease) {
        const findCommits = (options.releasableCommits ?? version_1.ReleasableCommits.everyCommit().cmd).replace("$LATEST_TAG", latestTag);
        const commitsSinceLastTag = (0, util_1.execOrUndefined)(findCommits, { cwd })?.split("\n");
        const numCommitsSinceLastTag = commitsSinceLastTag?.length ?? 0;
        logging.info(`Number of commits since ${latestTag}: ${numCommitsSinceLastTag}`);
        // Nothing to release right now
        if (numCommitsSinceLastTag === 0) {
            logging.info("Skipping bump...");
            skipBump = true;
            // delete the existing tag (locally)
            // if we don't do this, standard-version generates an empty changelog
            (0, util_1.exec)(`git tag --delete ${latestTag}`, { cwd });
        }
    }
    // create a standard-version configuration file
    const rcfile = (0, path_1.join)(cwd, ".versionrc.json");
    await generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, options.versionrcOptions);
    const cmd = ["npx", "standard-version@^9"];
    if (isFirstRelease && !minMajorVersion) {
        cmd.push("--first-release");
    }
    if (prefix) {
        cmd.push(`--tag-prefix ${prefix}v`);
    }
    if (minMajorVersion) {
        const [majorVersion] = latestVersion.split(".");
        const majorVersionNumber = parseInt(majorVersion, 10);
        if (majorVersionNumber < minMajorVersion) {
            cmd.push(`--release-as ${minMajorVersion}.0.0`);
        }
    }
    (0, util_1.exec)(cmd.join(" "), { cwd });
    // add the tag back if it was previously removed
    if (skipBump) {
        (0, util_1.exec)(`git tag ${latestTag}`, { cwd });
    }
    await fs_1.promises.rm(rcfile, { force: true, recursive: true });
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    if (minor) {
        if (!newVersion.startsWith(`${major}.${minor}`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major}.${minor} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_1.promises.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_1.promises.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(0, fs_1.existsSync)(versionFile)) {
        return { contents: {}, newline: true };
    }
    const raw = await fs_1.promises.readFile(versionFile, "utf-8");
    const contents = JSON.parse(raw);
    return {
        contents,
        version: contents.version,
        newline: raw.endsWith("\n"),
    };
}
function generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, configOptions) {
    return fs_1.promises.writeFile(rcfile, JSON.stringify({
        ...{
            packageFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            bumpFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            commitAll: false,
            infile: changelogFile,
            prerelease: prerelease,
            header: "",
            skip: {
                commit: true,
                tag: true,
                bump: skipBump,
            },
            ...configOptions,
        },
    }, undefined, 2));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, minor, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    let prefixFilter;
    if (major !== undefined && minor !== undefined) {
        prefixFilter = `${prefix}v${major}.${minor}.*`;
    }
    else if (major !== undefined) {
        prefixFilter = `${prefix}v${major}.*`;
    }
    else {
        prefixFilter = `${prefix}v*`;
    }
    const listGitTags = [
        "git",
        '-c "versionsort.suffix=-"',
        "tag",
        '--sort="-version:refname"',
        "--list",
        `"${prefixFilter}"`,
    ].join(" ");
    const stdout = (0, util_1.execCapture)(listGitTags, { cwd }).toString("utf8");
    let tags = stdout?.split("\n");
    // if prerelease is set and there are existing prerelease tags, filter versions that end with "-PRE.ddd".
    const prereleaseTags = tags.filter((x) => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    if (prerelease && prereleaseTags.length > 0) {
        /**
         * Cover the following case specifically
         * 1 - v1.0.0
         * 2 - v1.0.1-beta.0
         * 3 - v1.0.1-beta.1
         * 4 - v1.0.1
         * 5 - now publish a new release on the prerelease branch
         *    by setting the latestTag as v1.0.1 instead of v1.0.1-beta.1
         */
        const releaseTags = tags.filter((x) => new RegExp(`^v([0-9]+)\.([0-9]+)\.([0-9]+)$`).test(x));
        if (releaseTags.length > 0 &&
            (0, semver_1.compare)(releaseTags[0], prereleaseTags[0]) === 1) {
            tags = releaseTags;
        }
        else {
            tags = prereleaseTags;
        }
    }
    tags = tags.filter((x) => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major ?? 0}.${minor ?? 0}.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith("v")) {
        latestVersion = latestVersion.substring(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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