"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * Environment variables:
 *
 * - OUTFILE: (required) the name of the JSON output file (the "version" field
 *   will be updated with the latest version)
 * - PRERELEASE: (optional) a prerelease tag to use (e.g. "beta")
 * - MAJOR: major version number NN to filter (tags are filtered by "vNN."
 *   prefix). if not specified, the last major version is selected
 * - MIN_MAJOR: minimum major version number to use
 * - CHANGELOG: name of changelog file to create
 * - RELEASE_TAG_PREFIX: (optional) a prefix to apply to the release tag
 *
 */
const bump_version_1 = require("./bump-version");
const logging = require("../logging");
const versionFile = process.env.OUTFILE;
const prerelease = process.env.PRERELEASE;
const major = process.env.MAJOR;
const minMajor = process.env.MIN_MAJOR;
const changelog = process.env.CHANGELOG;
const bumpFile = process.env.BUMPFILE;
const releaseTagFile = process.env.RELEASETAG;
const prefix = process.env.RELEASE_TAG_PREFIX;
const versionrcOptions = process.env.VERSIONRCOPTIONS;
const releasableCommits = process.env.RELEASABLE_COMMITS;
if (!versionFile) {
    throw new Error("OUTFILE is required");
}
if (!changelog) {
    throw new Error("CHANGELOG is required");
}
if (!bumpFile) {
    throw new Error("BUMPFILE is required");
}
if (!releaseTagFile) {
    throw new Error("RELEASETAG is required");
}
const majorVersion = major == null || major === "" ? undefined : parseInt(major);
if (Number.isNaN(majorVersion)) {
    throw new Error(`MAJOR must be a number: ${majorVersion}`);
}
const minMajorVersion = minMajor == null || minMajor === "" ? undefined : parseInt(minMajor);
if (Number.isNaN(minMajorVersion)) {
    throw new Error(`minMajor must be a number: ${minMajorVersion}`);
}
const opts = {
    versionFile: versionFile,
    changelog: changelog,
    majorVersion: majorVersion,
    minMajorVersion: minMajorVersion,
    prerelease: prerelease,
    bumpFile: bumpFile,
    releaseTagFile: releaseTagFile,
    tagPrefix: prefix,
    // doesn't work with long customization
    versionrcOptions: JSON.parse(versionrcOptions ?? "{}"),
    releasableCommits,
};
logging.debug(opts);
(0, bump_version_1.bump)(process.cwd(), opts).catch((e) => {
    console.log(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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