"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isAwsCodeArtifactRegistry = exports.CodeArtifactAuthProvider = exports.Publisher = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const runner_options_1 = require("../runner-options");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11.x" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.16.0" } },
    dotnet: { dotnet: { version: "3.x" } },
};
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this._gitHubPostPublishing = [];
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.workflowNodeVersion = options.workflowNodeVersion ?? "18.x";
        this.workflowContainerImage = options.workflowContainerImage;
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.publishTasks = options.publishTasks ?? false;
        this.runsOn = options.workflowRunsOn;
        this.runsOnGroup = options.workflowRunsOnGroup;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Adds post publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPostPublishingSteps(...steps) {
        this._gitHubPostPublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: '! git log --oneline -1 | grep -q "chore(release):"',
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        this.addPublishJob((_branch, branchOptions) => {
            return {
                name: "github",
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                postPublishSteps: options.postPublishSteps ?? this._gitHubPostPublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                    GITHUB_REPOSITORY: "${{ github.repository }}",
                    GITHUB_REF: "${{ github.ref }}",
                },
                run: this.githubReleaseCommand(options, branchOptions),
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const isAwsCodeArtifactWithOidc = isAwsCodeArtifact &&
            options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC;
        const npmToken = (0, node_package_1.defaultNpmToken)(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        const prePublishSteps = options.prePublishSteps ?? [];
        if (isAwsCodeArtifactWithOidc) {
            if (options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret) {
                throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
            }
            else if (!options.codeArtifactOptions?.roleToAssume) {
                throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
            }
            const regionCaptureRegex = /codeartifact\.(.+)\.amazonaws\.com/;
            const region = options.registry?.match(regionCaptureRegex)?.[1];
            prePublishSteps.push({
                name: "Configure AWS Credentials via GitHub OIDC Provider",
                uses: "aws-actions/configure-aws-credentials@v4",
                with: {
                    "role-to-assume": options.codeArtifactOptions.roleToAssume,
                    "aws-region": region,
                },
            });
        }
        this.addPublishJob((_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            const npmProvenance = options.npmProvenance ? "true" : undefined;
            const needsIdTokenWrite = isAwsCodeArtifactWithOidc || npmProvenance;
            return {
                name: "npm",
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps,
                postPublishSteps: options.postPublishSteps ?? [],
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                    NPM_CONFIG_PROVENANCE: npmProvenance,
                },
                permissions: {
                    idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "nuget",
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: secret(isGitHubPackages
                    ? "GITHUB_TOKEN"
                    : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "maven",
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "pypi",
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv: {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            },
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "golang",
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GITHUB_REPO: options.githubRepo,
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(factory) {
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            const jobname = `release_${opts.name}`;
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [name, expression] of workflowEnvEntries) {
                requiredEnv.push(name);
                jobEnv[name] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${opts.name.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v4",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR, // this must be "dist" for publib
                    },
                },
                {
                    name: "Restore build artifact permissions",
                    continueOnError: true,
                    run: [
                        `cd ${ARTIFACTS_DOWNLOAD_DIR} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                    ].join("\n"),
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
                ...opts.postPublishSteps,
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            const container = this.workflowContainerImage
                ? {
                    image: this.workflowContainerImage,
                }
                : undefined;
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        run: 'echo "VERSION=$(cat dist/version.txt)" >> $GITHUB_OUTPUT',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    tools: {
                        node: { version: this.workflowNodeVersion },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId],
                    ...(0, runner_options_1.filteredRunsOnOptions)(this.runsOn, this.runsOnGroup),
                    container,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
    githubReleaseCommand(options, branchOptions) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghReleaseCommand = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_REF",
        ];
        if (branchOptions.prerelease) {
            ghReleaseCommand.push("-p");
        }
        const ghRelease = ghReleaseCommand.join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        return idempotentRelease;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.79.27" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
//# sourceMappingURL=data:application/json;base64,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