"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const runner_options_1 = require("../runner-options");
const path_1 = require("../util/path");
const version_1 = require("../version");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const TAG_EXISTS_STEPID = "check_tag_exists";
const LATEST_COMMIT_OUTPUT = "latest_commit";
const TAG_EXISTS_OUTPUT = "tag_exists";
/**
 * Conditional (Github Workflow Job `if`) to check if a release job should be run.
 */
const DEPENDENT_JOB_CONDITIONAL = `needs.${BUILD_JOBID}.outputs.${TAG_EXISTS_OUTPUT} != 'true' && needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`;
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project) {
        const isRelease = (c) => c instanceof Release;
        return project.components.find(isRelease);
    }
    /**
     * @param scope should be part of the project the Release belongs to.
     * @param options options to configure the Release Component.
     */
    constructor(scope, options) {
        super(scope);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = github_1.GitHub.of(this.project.root);
        this.buildTask = options.task;
        this.preBuildSteps = options.releaseWorkflowSetupSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.versionFile = options.versionFile;
        this.releaseTrigger = options.releaseTrigger ?? release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        this.workflowRunsOnGroup = options.workflowRunsOnGroup;
        this.workflowPermissions = {
            contents: workflows_model_1.JobPermission.WRITE,
            ...options.workflowPermissions,
        };
        this._branchHooks = [];
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!((options.releaseEveryCommit ?? true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(this.project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
            tagPrefix: options.releaseTagPrefix,
            releasableCommits: options.releasableCommits,
        });
        this.releaseTagFilePath = path.posix.normalize(path.posix.join(
        // temporary hack to allow JsiiProject setting a different path to the release tag file
        // see JsiiProject.releaseTagFilePath for more details
        //@ts-ignore
        this.project.releaseTagFilePath ?? this.artifactsDirectory, this.version.releaseTagFileName));
        this.publisher = new publisher_1.Publisher(this.project, {
            artifactName: this.artifactsDirectory,
            condition: DEPENDENT_JOB_CONDITIONAL,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            ...(0, runner_options_1.filteredWorkflowRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
            workflowNodeVersion: options.workflowNodeVersion,
            workflowContainerImage: options.workflowContainerImage,
        });
        const githubRelease = options.githubRelease ?? true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            minMajorVersion: options.minMajorVersion,
            workflowName: options.releaseWorkflowName ??
                workflowNameForProject("release", this.project),
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries(options.releaseBranches ?? {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook) {
        for (const branch of this._branches) {
            hook(branch.name);
        }
        this._branchHooks.push(hook);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
        // run all branch hooks
        for (const hook of this._branchHooks) {
            hook(branch);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        const workflowName = branch.workflowName ??
            workflowNameForProject(`release-${branchName}`, this.project);
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.minMajorVersion !== undefined) {
            if (branch.majorVersion !== undefined) {
                throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
            }
            env.MIN_MAJOR = branch.minMajorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.RELEASE_TAG_PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec(Release.ANTI_TAMPER_CMD);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        const postBuildSteps = [...this.postBuildSteps];
        // Read the releasetag, then check if it already exists.
        // If it does, we will cancel this release
        postBuildSteps.push(github_1.WorkflowSteps.tagExists(`$(cat ${this.releaseTagFilePath})`, {
            name: "Check if version has already been tagged",
            id: TAG_EXISTS_STEPID,
        }));
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            run: [
                `echo "${LATEST_COMMIT_OUTPUT}=$(git ls-remote origin -h \${{ github.ref }} | cut -f1)" >> $GITHUB_OUTPUT`,
                "cat $GITHUB_OUTPUT",
            ].join("\n"),
        });
        const projectPathRelativeToRoot = path.relative(this.project.root.outdir, this.project.outdir);
        postBuildSteps.push({
            name: "Backup artifact permissions",
            if: noNewCommits,
            continueOnError: true,
            run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
        }, github_1.WorkflowSteps.uploadArtifact({
            if: noNewCommits,
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: projectPathRelativeToRoot.length > 0
                    ? `${projectPathRelativeToRoot}/${this.artifactsDirectory}`
                    : this.artifactsDirectory,
            },
        }));
        if (this.github && !this.releaseTrigger.isManual) {
            // Use target (possible parent) GitHub to create the workflow
            const workflow = new github_1.GithubWorkflow(this.github, workflowName);
            workflow.on({
                schedule: this.releaseTrigger.schedule
                    ? [{ cron: this.releaseTrigger.schedule }]
                    : undefined,
                push: this.releaseTrigger.isContinuous
                    ? { branches: [branchName] }
                    : undefined,
                workflowDispatch: {}, // allow manual triggering
            });
            // Create job based on child (only?) project GitHub
            const taskjob = new github_1.TaskWorkflowJob(this, releaseTask, {
                outputs: {
                    [LATEST_COMMIT_OUTPUT]: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                    [TAG_EXISTS_OUTPUT]: {
                        stepId: TAG_EXISTS_STEPID,
                        outputName: "exists",
                    },
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                },
                permissions: this.workflowPermissions,
                checkoutWith: {
                    // fetch-depth= indicates all history for all branches and tags
                    // we must use this in order to fetch all tags
                    // and to inspect the history to decide if we should release
                    fetchDepth: 0,
                },
                preBuildSteps,
                postBuildSteps,
                jobDefaults: projectPathRelativeToRoot.length > 0 // is subproject
                    ? {
                        run: {
                            workingDirectory: (0, path_1.ensureRelativePathStartsWithDot)(projectPathRelativeToRoot),
                        },
                    }
                    : undefined,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.workflowRunsOn, this.workflowRunsOnGroup),
            });
            workflow.addJob(BUILD_JOBID, taskjob);
            return workflow;
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.79.27" };
Release.ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
function workflowNameForProject(base, project) {
    // Subprojects
    if (project.parent) {
        return `${base}_${fileSafeName(project.name)}`;
    }
    // root project doesn't get a suffix
    return base;
}
function fileSafeName(name) {
    return name.replace("@", "").replace(/\//g, "-");
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVsZWFzZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9yZWxlYXNlL3JlbGVhc2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFFN0IsMkNBQXdDO0FBQ3hDLHVEQUFtRDtBQUNuRCw0Q0FBeUM7QUFDekMsc0NBS21CO0FBQ25CLG1EQUc2QjtBQUM3QiwrREFLbUM7QUFFbkMsc0RBSTJCO0FBRTNCLHVDQUErRDtBQUMvRCx3Q0FBd0Q7QUFFeEQsTUFBTSxXQUFXLEdBQUcsU0FBUyxDQUFDO0FBQzlCLE1BQU0saUJBQWlCLEdBQUcsWUFBWSxDQUFDO0FBQ3ZDLE1BQU0saUJBQWlCLEdBQUcsa0JBQWtCLENBQUM7QUFFN0MsTUFBTSxvQkFBb0IsR0FBRyxlQUFlLENBQUM7QUFDN0MsTUFBTSxpQkFBaUIsR0FBRyxZQUFZLENBQUM7QUFFdkM7O0dBRUc7QUFDSCxNQUFNLHlCQUF5QixHQUFHLFNBQVMsV0FBVyxZQUFZLGlCQUFpQix1QkFBdUIsV0FBVyxZQUFZLG9CQUFvQixnQkFBZ0IsQ0FBQztBQStQdEs7Ozs7R0FJRztBQUNILE1BQWEsT0FBUSxTQUFRLHFCQUFTO0lBSXBDOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsT0FBZ0I7UUFDL0IsTUFBTSxTQUFTLEdBQUcsQ0FBQyxDQUFZLEVBQWdCLEVBQUUsQ0FBQyxDQUFDLFlBQVksT0FBTyxDQUFDO1FBQ3ZFLE9BQU8sT0FBTyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDNUMsQ0FBQztJQTZCRDs7O09BR0c7SUFDSCxZQUFZLEtBQWlCLEVBQUUsT0FBdUI7UUFDcEQsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBZkUsY0FBUyxHQUFHLElBQUksS0FBSyxFQUFpQixDQUFDO1FBQ3ZDLFNBQUksR0FBd0IsRUFBRSxDQUFDO1FBZ0I5QyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQ2IsK0RBQStELENBQ2hFLENBQUM7U0FDSDtRQUVELElBQUksQ0FBQyxNQUFNLEdBQUcsZUFBTSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztRQUM5QixJQUFJLENBQUMsYUFBYSxHQUFHLE9BQU8sQ0FBQyx5QkFBeUIsSUFBSSxFQUFFLENBQUM7UUFDN0QsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLEVBQUUsQ0FBQztRQUNuRCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsT0FBTyxDQUFDLGtCQUFrQixJQUFJLE1BQU0sQ0FBQztRQUMvRCxJQUFJLENBQUMsV0FBVyxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDdkMsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLGdDQUFjLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDNUUsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLENBQUMsc0JBQXNCLENBQUM7UUFDckQsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxDQUFDO1FBQzdDLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxPQUFPLENBQUMsbUJBQW1CLENBQUM7UUFDdkQsSUFBSSxDQUFDLG1CQUFtQixHQUFHO1lBQ3pCLFFBQVEsRUFBRSwrQkFBYSxDQUFDLEtBQUs7WUFDN0IsR0FBRyxPQUFPLENBQUMsbUJBQW1CO1NBQy9CLENBQUM7UUFDRixJQUFJLENBQUMsWUFBWSxHQUFHLEVBQUUsQ0FBQztRQUV2Qjs7Ozs7V0FLRztRQUNILElBQ0UsQ0FBQyxDQUNDLENBQUMsT0FBTyxDQUFDLGtCQUFrQixJQUFJLElBQUksQ0FBQztZQUNwQyxPQUFPLENBQUMsZUFBZTtZQUN2QixPQUFPLENBQUMsY0FBYyxDQUN2QixFQUNEO1lBQ0EsSUFBSSxDQUFDLGNBQWMsR0FBRyxnQ0FBYyxDQUFDLE1BQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1NBQ25FO1FBRUQsSUFBSSxPQUFPLENBQUMsZUFBZSxFQUFFO1lBQzNCLElBQUksQ0FBQyxjQUFjLEdBQUcsZ0NBQWMsQ0FBQyxTQUFTLENBQUM7Z0JBQzdDLFFBQVEsRUFBRSxPQUFPLENBQUMsZUFBZTthQUNsQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxpQkFBTyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDdkMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDbEMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtZQUMzQyxnQkFBZ0IsRUFBRSxPQUFPLENBQUMsZ0JBQWdCO1lBQzFDLFNBQVMsRUFBRSxPQUFPLENBQUMsZ0JBQWdCO1lBQ25DLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUI7U0FDN0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUM1QyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUk7UUFDYix1RkFBdUY7UUFDdkYsc0RBQXNEO1FBQ3RELFlBQVk7UUFDWixJQUFJLENBQUMsT0FBTyxDQUFDLGtCQUFrQixJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFDMUQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsQ0FDaEMsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLHFCQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMzQyxZQUFZLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtZQUNyQyxTQUFTLEVBQUUseUJBQXlCO1lBQ3BDLFVBQVUsRUFBRSxXQUFXO1lBQ3ZCLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxrQkFBa0I7WUFDOUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxtQkFBbUI7WUFDekMsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLHdCQUF3QjtZQUNuRCxHQUFHLElBQUEsOENBQTZCLEVBQzlCLE9BQU8sQ0FBQyxjQUFjLEVBQ3RCLE9BQU8sQ0FBQyxtQkFBbUIsQ0FDNUI7WUFDRCxZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDbEMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQzdCLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxtQkFBbUI7WUFDaEQsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLHNCQUFzQjtTQUN2RCxDQUFDLENBQUM7UUFFSCxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQztRQUNwRCxJQUFJLGFBQWEsRUFBRTtZQUNqQixJQUFJLENBQUMsU0FBUyxDQUFDLHVCQUF1QixDQUFDO2dCQUNyQyxhQUFhLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQzVCLElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FDL0I7Z0JBQ0QsV0FBVyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUMxQixJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUM3QjtnQkFDRCxjQUFjLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQzdCLElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsQ0FDaEM7YUFDRixDQUFDLENBQUM7U0FDSjtRQUVELDJGQUEyRjtRQUMzRixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRTtZQUNuRCxVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVU7WUFDOUIsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZO1lBQ2xDLGVBQWUsRUFBRSxPQUFPLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQ1YsT0FBTyxDQUFDLG1CQUFtQjtnQkFDM0Isc0JBQXNCLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUM7WUFDakQsU0FBUyxFQUFFLE9BQU8sQ0FBQyxnQkFBZ0I7WUFDbkMsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVO1NBQy9CLENBQUMsQ0FBQztRQUVILEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLElBQUksRUFBRSxDQUFDLEVBQUU7WUFDeEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDNUI7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGNBQWMsQ0FBQyxJQUFnQjtRQUNwQyxLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbkMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNuQjtRQUNELElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxTQUFTLENBQUMsTUFBYyxFQUFFLE9BQXNCO1FBQ3JELElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRWpDLHVCQUF1QjtRQUN2QixLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDcEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ2Q7SUFDSCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ssVUFBVSxDQUNoQixNQUFjLEVBQ2QsT0FBK0I7UUFFL0IsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxNQUFNLENBQUMsRUFBRTtZQUNqRCxNQUFNLElBQUksS0FBSyxDQUFDLHNCQUFzQixNQUFNLHFCQUFxQixDQUFDLENBQUM7U0FDcEU7UUFFRCw0RUFBNEU7UUFDNUUsaUJBQWlCO1FBQ2pCLElBQ0UsSUFBSSxDQUFDLGFBQWE7WUFDbEIsT0FBTyxDQUFDLFlBQVk7WUFDcEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUM3QztZQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsOEZBQThGLENBQy9GLENBQUM7U0FDSDtRQUVELE1BQU0sYUFBYSxHQUFrQjtZQUNuQyxJQUFJLEVBQUUsTUFBTTtZQUNaLEdBQUcsT0FBTztZQUNWLFFBQVEsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUM7U0FDL0MsQ0FBQztRQUVGLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRW5DLE9BQU8sYUFBYSxDQUFDO0lBQ3ZCLENBQUM7SUFFTSxhQUFhO1FBQ2xCLEtBQUssTUFBTSxNQUFNLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNuQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRTtnQkFDcEIsU0FBUzthQUNWO1lBRUQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQ3JCLElBQUksQ0FBQyxTQUFTLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FDekQsQ0FBQztZQUNGLE1BQU0sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNwQztJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxPQUFPLENBQUMsSUFBeUI7UUFDdEMsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDOUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUM7U0FDdkI7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLFFBQVE7UUFDakIsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7T0FFRztJQUNLLGNBQWMsQ0FDcEIsVUFBa0IsRUFDbEIsTUFBOEI7UUFFOUIsTUFBTSxZQUFZLEdBQ2hCLE1BQU0sQ0FBQyxZQUFZO1lBQ25CLHNCQUFzQixDQUFDLFdBQVcsVUFBVSxFQUFFLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWhFLDBFQUEwRTtRQUMxRSx1RUFBdUU7UUFDdkUsMkRBQTJEO1FBQzNELE1BQU0sWUFBWSxHQUFHLGNBQWMsaUJBQWlCLFlBQVksb0JBQW9CLG1CQUFtQixDQUFDO1FBRXhHLGtGQUFrRjtRQUNsRixNQUFNLGFBQWEsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRTlDLE1BQU0sR0FBRyxHQUEyQjtZQUNsQyxPQUFPLEVBQUUsTUFBTTtTQUNoQixDQUFDO1FBRUYsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUNyQyxHQUFHLENBQUMsS0FBSyxHQUFHLE1BQU0sQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7U0FDNUM7UUFFRCxJQUFJLE1BQU0sQ0FBQyxlQUFlLEtBQUssU0FBUyxFQUFFO1lBQ3hDLElBQUksTUFBTSxDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7Z0JBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQ2IsMkRBQTJELENBQzVELENBQUM7YUFDSDtZQUVELEdBQUcsQ0FBQyxTQUFTLEdBQUcsTUFBTSxDQUFDLGVBQWUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUNuRDtRQUVELElBQUksTUFBTSxDQUFDLFVBQVUsRUFBRTtZQUNyQixHQUFHLENBQUMsVUFBVSxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUM7U0FDcEM7UUFFRCxJQUFJLE1BQU0sQ0FBQyxTQUFTLEVBQUU7WUFDcEIsR0FBRyxDQUFDLGtCQUFrQixHQUFHLE1BQU0sQ0FBQyxTQUFTLENBQUM7U0FDM0M7UUFFRCwyRUFBMkU7UUFDM0UsNkRBQTZEO1FBQzdELHNEQUFzRDtRQUV0RCx5RUFBeUU7UUFDekUsOEJBQThCO1FBQzlCLE1BQU0sZUFBZSxHQUNuQixVQUFVLEtBQUssTUFBTSxJQUFJLFVBQVUsS0FBSyxRQUFRO1lBQzlDLENBQUMsQ0FBQyxTQUFTO1lBQ1gsQ0FBQyxDQUFDLFdBQVcsVUFBVSxFQUFFLENBQUM7UUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFO1lBQ3hELFdBQVcsRUFBRSwyQkFBMkIsVUFBVSxVQUFVO1lBQzVELEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxXQUFXLENBQUMsSUFBSSxDQUFDLFVBQVUsSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQztRQUN0RCxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDekMsV0FBVyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDbEMsV0FBVyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRTNDLHFFQUFxRTtRQUNyRSwwRkFBMEY7UUFDMUYsV0FBVyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFMUMsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRTtZQUNoQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQztnQkFDOUMsYUFBYSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUM1QixJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLElBQUksQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQy9CO2dCQUNELFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FDMUIsSUFBSSxDQUFDLGtCQUFrQixFQUN2QixJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FDN0I7Z0JBQ0QsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUM3QixJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLElBQUksQ0FBQyxPQUFPLENBQUMsa0JBQWtCLENBQ2hDO2dCQUNELG9CQUFvQixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYTtnQkFDdkQsU0FBUyxFQUFFLFVBQVU7Z0JBQ3JCLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWM7YUFDbkQsQ0FBQyxDQUFDO1lBRUgsV0FBVyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNoQztRQUVELE1BQU0sY0FBYyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFaEQsd0RBQXdEO1FBQ3hELDBDQUEwQztRQUMxQyxjQUFjLENBQUMsSUFBSSxDQUNqQixzQkFBYSxDQUFDLFNBQVMsQ0FBQyxTQUFTLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxFQUFFO1lBQzNELElBQUksRUFBRSwwQ0FBMEM7WUFDaEQsRUFBRSxFQUFFLGlCQUFpQjtTQUN0QixDQUFDLENBQ0gsQ0FBQztRQUVGLHVFQUF1RTtRQUN2RSwrREFBK0Q7UUFDL0QsY0FBYyxDQUFDLElBQUksQ0FBQztZQUNsQixJQUFJLEVBQUUsdUJBQXVCO1lBQzdCLEVBQUUsRUFBRSxpQkFBaUI7WUFDckIsR0FBRyxFQUFFO2dCQUNILFNBQVMsb0JBQW9CLDZFQUE2RTtnQkFDMUcsb0JBQW9CO2FBQ3JCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztTQUNiLENBQUMsQ0FBQztRQUVILE1BQU0seUJBQXlCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FDN0MsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUN4QixJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FDcEIsQ0FBQztRQUNGLGNBQWMsQ0FBQyxJQUFJLENBQ2pCO1lBQ0UsSUFBSSxFQUFFLDZCQUE2QjtZQUNuQyxFQUFFLEVBQUUsWUFBWTtZQUNoQixlQUFlLEVBQUUsSUFBSTtZQUNyQixHQUFHLEVBQUUsTUFBTSxJQUFJLENBQUMsa0JBQWtCLHNCQUFzQixrQ0FBc0IsRUFBRTtTQUNqRixFQUNELHNCQUFhLENBQUMsY0FBYyxDQUFDO1lBQzNCLEVBQUUsRUFBRSxZQUFZO1lBQ2hCLElBQUksRUFBRTtnQkFDSixJQUFJLEVBQUUsK0JBQW1CO2dCQUN6QixJQUFJLEVBQ0YseUJBQXlCLENBQUMsTUFBTSxHQUFHLENBQUM7b0JBQ2xDLENBQUMsQ0FBQyxHQUFHLHlCQUF5QixJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtvQkFDM0QsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0I7YUFDOUI7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxFQUFFO1lBQ2hELDZEQUE2RDtZQUM3RCxNQUFNLFFBQVEsR0FBRyxJQUFJLHVCQUFjLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxZQUFZLENBQUMsQ0FBQztZQUMvRCxRQUFRLENBQUMsRUFBRSxDQUFDO2dCQUNWLFFBQVEsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLFFBQVE7b0JBQ3BDLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxFQUFFLENBQUM7b0JBQzFDLENBQUMsQ0FBQyxTQUFTO2dCQUNiLElBQUksRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVk7b0JBQ3BDLENBQUMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxDQUFDLFVBQVUsQ0FBQyxFQUFFO29CQUM1QixDQUFDLENBQUMsU0FBUztnQkFDYixnQkFBZ0IsRUFBRSxFQUFFLEVBQUUsMEJBQTBCO2FBQ2pELENBQUMsQ0FBQztZQUVILG1EQUFtRDtZQUNuRCxNQUFNLE9BQU8sR0FBRyxJQUFJLHdCQUFlLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtnQkFDckQsT0FBTyxFQUFFO29CQUNQLENBQUMsb0JBQW9CLENBQUMsRUFBRTt3QkFDdEIsTUFBTSxFQUFFLGlCQUFpQjt3QkFDekIsVUFBVSxFQUFFLG9CQUFvQjtxQkFDakM7b0JBQ0QsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFO3dCQUNuQixNQUFNLEVBQUUsaUJBQWlCO3dCQUN6QixVQUFVLEVBQUUsUUFBUTtxQkFDckI7aUJBQ0Y7Z0JBQ0QsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjO29CQUM1QixDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRTtvQkFDaEMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2IsR0FBRyxFQUFFO29CQUNILEVBQUUsRUFBRSxNQUFNO2lCQUNYO2dCQUNELFdBQVcsRUFBRSxJQUFJLENBQUMsbUJBQW1CO2dCQUNyQyxZQUFZLEVBQUU7b0JBQ1osK0RBQStEO29CQUMvRCw4Q0FBOEM7b0JBQzlDLDREQUE0RDtvQkFDNUQsVUFBVSxFQUFFLENBQUM7aUJBQ2Q7Z0JBQ0QsYUFBYTtnQkFDYixjQUFjO2dCQUNkLFdBQVcsRUFDVCx5QkFBeUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLGdCQUFnQjtvQkFDbkQsQ0FBQyxDQUFDO3dCQUNFLEdBQUcsRUFBRTs0QkFDSCxnQkFBZ0IsRUFBRSxJQUFBLHNDQUErQixFQUMvQyx5QkFBeUIsQ0FDMUI7eUJBQ0Y7cUJBQ0Y7b0JBQ0gsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2YsR0FBRyxJQUFBLHNDQUFxQixFQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDO2FBQ3hFLENBQUMsQ0FBQztZQUVILFFBQVEsQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBRXRDLE9BQU8sUUFBUSxDQUFDO1NBQ2pCO2FBQU07WUFDTCxPQUFPLFNBQVMsQ0FBQztTQUNsQjtJQUNILENBQUM7O0FBMWNILDBCQTJjQzs7O0FBMWN3Qix1QkFBZSxHQUNwQyw0Q0FBNEMsQUFEUixDQUNTO0FBMmNqRCxTQUFTLHNCQUFzQixDQUFDLElBQVksRUFBRSxPQUFnQjtJQUM1RCxjQUFjO0lBQ2QsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFO1FBQ2xCLE9BQU8sR0FBRyxJQUFJLElBQUksWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO0tBQ2hEO0lBRUQsb0NBQW9DO0lBQ3BDLE9BQU8sSUFBSSxDQUFDO0FBQ2QsQ0FBQztBQUVELFNBQVMsWUFBWSxDQUFDLElBQVk7SUFDaEMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0FBQ25ELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcbmltcG9ydCB7IFB1Ymxpc2hlciB9IGZyb20gXCIuL3B1Ymxpc2hlclwiO1xuaW1wb3J0IHsgUmVsZWFzZVRyaWdnZXIgfSBmcm9tIFwiLi9yZWxlYXNlLXRyaWdnZXJcIjtcbmltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gXCIuLi9jb21wb25lbnRcIjtcbmltcG9ydCB7XG4gIEdpdEh1YixcbiAgR2l0aHViV29ya2Zsb3csXG4gIFRhc2tXb3JrZmxvd0pvYixcbiAgV29ya2Zsb3dTdGVwcyxcbn0gZnJvbSBcIi4uL2dpdGh1YlwiO1xuaW1wb3J0IHtcbiAgQlVJTERfQVJUSUZBQ1RfTkFNRSxcbiAgUEVSTUlTU0lPTl9CQUNLVVBfRklMRSxcbn0gZnJvbSBcIi4uL2dpdGh1Yi9jb25zdGFudHNcIjtcbmltcG9ydCB7XG4gIEpvYixcbiAgSm9iUGVybWlzc2lvbixcbiAgSm9iUGVybWlzc2lvbnMsXG4gIEpvYlN0ZXAsXG59IGZyb20gXCIuLi9naXRodWIvd29ya2Zsb3dzLW1vZGVsXCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcbmltcG9ydCB7XG4gIEdyb3VwUnVubmVyT3B0aW9ucyxcbiAgZmlsdGVyZWRSdW5zT25PcHRpb25zLFxuICBmaWx0ZXJlZFdvcmtmbG93UnVuc09uT3B0aW9ucyxcbn0gZnJvbSBcIi4uL3J1bm5lci1vcHRpb25zXCI7XG5pbXBvcnQgeyBUYXNrIH0gZnJvbSBcIi4uL3Rhc2tcIjtcbmltcG9ydCB7IGVuc3VyZVJlbGF0aXZlUGF0aFN0YXJ0c1dpdGhEb3QgfSBmcm9tIFwiLi4vdXRpbC9wYXRoXCI7XG5pbXBvcnQgeyBSZWxlYXNhYmxlQ29tbWl0cywgVmVyc2lvbiB9IGZyb20gXCIuLi92ZXJzaW9uXCI7XG5cbmNvbnN0IEJVSUxEX0pPQklEID0gXCJyZWxlYXNlXCI7XG5jb25zdCBHSVRfUkVNT1RFX1NURVBJRCA9IFwiZ2l0X3JlbW90ZVwiO1xuY29uc3QgVEFHX0VYSVNUU19TVEVQSUQgPSBcImNoZWNrX3RhZ19leGlzdHNcIjtcblxuY29uc3QgTEFURVNUX0NPTU1JVF9PVVRQVVQgPSBcImxhdGVzdF9jb21taXRcIjtcbmNvbnN0IFRBR19FWElTVFNfT1VUUFVUID0gXCJ0YWdfZXhpc3RzXCI7XG5cbi8qKlxuICogQ29uZGl0aW9uYWwgKEdpdGh1YiBXb3JrZmxvdyBKb2IgYGlmYCkgdG8gY2hlY2sgaWYgYSByZWxlYXNlIGpvYiBzaG91bGQgYmUgcnVuLlxuICovXG5jb25zdCBERVBFTkRFTlRfSk9CX0NPTkRJVElPTkFMID0gYG5lZWRzLiR7QlVJTERfSk9CSUR9Lm91dHB1dHMuJHtUQUdfRVhJU1RTX09VVFBVVH0gIT0gJ3RydWUnICYmIG5lZWRzLiR7QlVJTERfSk9CSUR9Lm91dHB1dHMuJHtMQVRFU1RfQ09NTUlUX09VVFBVVH0gPT0gZ2l0aHViLnNoYWA7XG5cbnR5cGUgQnJhbmNoSG9vayA9IChicmFuY2g6IHN0cmluZykgPT4gdm9pZDtcblxuLyoqXG4gKiBQcm9qZWN0IG9wdGlvbnMgZm9yIHJlbGVhc2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVsZWFzZVByb2plY3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgcmVsZWFzZSBuZXcgdmVyc2lvbnMgZXZlcnkgY29tbWl0IHRvIG9uZSBvZiBicmFuY2hlcyBpbiBgcmVsZWFzZUJyYW5jaGVzYC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYHJlbGVhc2VUcmlnZ2VyOiBSZWxlYXNlVHJpZ2dlci5jb250aW51b3VzKClgIGluc3RlYWRcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VFdmVyeUNvbW1pdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIENST04gc2NoZWR1bGUgdG8gdHJpZ2dlciBuZXcgcmVsZWFzZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gc2NoZWR1bGVkIHJlbGVhc2VzXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIFVzZSBgcmVsZWFzZVRyaWdnZXI6IFJlbGVhc2VUcmlnZ2VyLnNjaGVkdWxlZCgpYCBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSByZWxlYXNlU2NoZWR1bGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZWxlYXNlIHRyaWdnZXIgdG8gdXNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENvbnRpbnVvdXMgcmVsZWFzZXMgKGBSZWxlYXNlVHJpZ2dlci5jb250aW51b3VzKClgKVxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZVRyaWdnZXI/OiBSZWxlYXNlVHJpZ2dlcjtcblxuICAvKipcbiAgICogQSBzZXQgb2Ygd29ya2Zsb3cgc3RlcHMgdG8gZXhlY3V0ZSBpbiBvcmRlciB0byBzZXR1cCB0aGUgd29ya2Zsb3dcbiAgICogY29udGFpbmVyLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZVdvcmtmbG93U2V0dXBTdGVwcz86IEpvYlN0ZXBbXTtcblxuICAvKipcbiAgICogQ29udGFpbmVyIGltYWdlIHRvIHVzZSBmb3IgR2l0SHViIHdvcmtmbG93cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IGltYWdlXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd0NvbnRhaW5lckltYWdlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWZXJzaW9uIHJlcXVpcmVtZW50IG9mIGBwdWJsaWJgIHdoaWNoIGlzIHVzZWQgdG8gcHVibGlzaCBtb2R1bGVzIHRvIG5wbS5cbiAgICogQGRlZmF1bHQgXCJsYXRlc3RcIlxuICAgKi9cbiAgcmVhZG9ubHkganNpaVJlbGVhc2VWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTdGVwcyB0byBleGVjdXRlIGFmdGVyIGJ1aWxkIGFzIHBhcnQgb2YgdGhlIHJlbGVhc2Ugd29ya2Zsb3cuXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqL1xuICByZWFkb25seSBwb3N0QnVpbGRTdGVwcz86IEpvYlN0ZXBbXTtcblxuICAvKipcbiAgICogTWFqb3IgdmVyc2lvbiB0byByZWxlYXNlIGZyb20gdGhlIGRlZmF1bHQgYnJhbmNoLlxuICAgKlxuICAgKiBJZiB0aGlzIGlzIHNwZWNpZmllZCwgd2UgYnVtcCB0aGUgbGF0ZXN0IHZlcnNpb24gb2YgdGhpcyBtYWpvciB2ZXJzaW9uIGxpbmUuXG4gICAqIElmIG5vdCBzcGVjaWZpZWQsIHdlIGJ1bXAgdGhlIGdsb2JhbCBsYXRlc3QgdmVyc2lvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBNYWpvciB2ZXJzaW9uIGlzIG5vdCBlbmZvcmNlZC5cbiAgICovXG4gIHJlYWRvbmx5IG1ham9yVmVyc2lvbj86IG51bWJlcjtcblxuICAvKipcbiAgICogTWluaW1hbCBNYWpvciB2ZXJzaW9uIHRvIHJlbGVhc2VcbiAgICpcbiAgICpcbiAgICogVGhpcyBjYW4gYmUgdXNlZnVsIHRvIHNldCB0byAxLCBhcyBicmVha2luZyBjaGFuZ2VzIGJlZm9yZSB0aGUgMS54IG1ham9yXG4gICAqIHJlbGVhc2UgYXJlIG5vdCBpbmNyZW1lbnRpbmcgdGhlIG1ham9yIHZlcnNpb24gbnVtYmVyLlxuICAgKlxuICAgKiBDYW4gbm90IGJlIHNldCB0b2dldGhlciB3aXRoIGBtYWpvclZlcnNpb25gLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIG1pbmltdW0gdmVyc2lvbiBpcyBiZWluZyBlbmZvcmNlZFxuICAgKi9cbiAgcmVhZG9ubHkgbWluTWFqb3JWZXJzaW9uPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBCdW1wIHZlcnNpb25zIGZyb20gdGhlIGRlZmF1bHQgYnJhbmNoIGFzIHByZS1yZWxlYXNlcyAoZS5nLiBcImJldGFcIixcbiAgICogXCJhbHBoYVwiLCBcInByZVwiKS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub3JtYWwgc2VtYW50aWMgdmVyc2lvbnNcbiAgICovXG4gIHJlYWRvbmx5IHByZXJlbGVhc2U/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBucG1EaXN0VGFnIHRvIHVzZSB3aGVuIHB1Ymxpc2hpbmcgZnJvbSB0aGUgZGVmYXVsdCBicmFuY2guXG4gICAqXG4gICAqIFRvIHNldCB0aGUgbnBtIGRpc3QtdGFnIGZvciByZWxlYXNlIGJyYW5jaGVzLCBzZXQgdGhlIGBucG1EaXN0VGFnYCBwcm9wZXJ0eVxuICAgKiBmb3IgZWFjaCBicmFuY2guXG4gICAqXG4gICAqIEBkZWZhdWx0IFwibGF0ZXN0XCJcbiAgICovXG4gIHJlYWRvbmx5IG5wbURpc3RUYWc/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBkZWZhdWx0IHJlbGVhc2Ugd29ya2Zsb3cuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwicmVsZWFzZVwiXG4gICAqL1xuICByZWFkb25seSByZWxlYXNlV29ya2Zsb3dOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFkZGl0aW9uYWwgcmVsZWFzZSBicmFuY2hlcy4gQSB3b3JrZmxvdyB3aWxsIGJlIGNyZWF0ZWQgZm9yIGVhY2hcbiAgICogcmVsZWFzZSBicmFuY2ggd2hpY2ggd2lsbCBwdWJsaXNoIHJlbGVhc2VzIGZyb20gY29tbWl0cyBpbiB0aGlzIGJyYW5jaC5cbiAgICogRWFjaCByZWxlYXNlIGJyYW5jaCBfbXVzdF8gYmUgYXNzaWduZWQgYSBtYWpvciB2ZXJzaW9uIG51bWJlciB3aGljaCBpcyB1c2VkXG4gICAqIHRvIGVuZm9yY2UgdGhhdCB2ZXJzaW9ucyBwdWJsaXNoZWQgZnJvbSB0aGF0IGJyYW5jaCBhbHdheXMgdXNlIHRoYXQgbWFqb3JcbiAgICogdmVyc2lvbi4gSWYgbXVsdGlwbGUgYnJhbmNoZXMgYXJlIHVzZWQsIHRoZSBgbWFqb3JWZXJzaW9uYCBmaWVsZCBtdXN0IGFsc29cbiAgICogYmUgcHJvdmlkZWQgZm9yIHRoZSBkZWZhdWx0IGJyYW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBhZGRpdGlvbmFsIGJyYW5jaGVzIGFyZSB1c2VkIGZvciByZWxlYXNlLiB5b3UgY2FuIHVzZVxuICAgKiBgYWRkQnJhbmNoKClgIHRvIGFkZCBhZGRpdGlvbmFsIGJyYW5jaGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZUJyYW5jaGVzPzogeyBbbmFtZTogc3RyaW5nXTogQnJhbmNoT3B0aW9ucyB9O1xuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBnaXRodWIgaXNzdWUgb24gZXZlcnkgZmFpbGVkIHB1Ymxpc2hpbmcgdGFzay5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VGYWlsdXJlSXNzdWU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbGFiZWwgdG8gYXBwbHkgdG8gaXNzdWVzIGluZGljYXRpbmcgcHVibGlzaCBmYWlsdXJlcy5cbiAgICogT25seSBhcHBsaWVzIGlmIGByZWxlYXNlRmFpbHVyZUlzc3VlYCBpcyB0cnVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcImZhaWxlZC1yZWxlYXNlXCJcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VGYWlsdXJlSXNzdWVMYWJlbD86IHN0cmluZztcblxuICAvKipcbiAgICogQXV0b21hdGljYWxseSBhZGQgdGhlIGdpdmVuIHByZWZpeCB0byByZWxlYXNlIHRhZ3MuXG4gICAqIFVzZWZ1bCBpZiB5b3UgYXJlIHJlbGVhc2luZyBvbiBtdWx0aXBsZSBicmFuY2hlcyB3aXRoIG92ZXJsYXBwaW5nXG4gICAqIHZlcnNpb24gbnVtYmVycy5cbiAgICpcbiAgICogTm90ZTogdGhpcyBwcmVmaXggaXMgdXNlZCB0byBkZXRlY3QgdGhlIGxhdGVzdCB0YWdnZWQgdmVyc2lvblxuICAgKiB3aGVuIGJ1bXBpbmcsIHNvIGlmIHlvdSBjaGFuZ2UgdGhpcyBvbiBhIHByb2plY3Qgd2l0aCBhbiBleGlzdGluZyB2ZXJzaW9uXG4gICAqIGhpc3RvcnksIHlvdSBtYXkgbmVlZCB0byBtYW51YWxseSB0YWcgeW91ciBsYXRlc3QgcmVsZWFzZVxuICAgKiB3aXRoIHRoZSBuZXcgcHJlZml4LlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInZcIlxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZVRhZ1ByZWZpeD86IHN0cmluZztcblxuICAvKipcbiAgICogQ3VzdG9tIGNvbmZpZ3VyYXRpb24gdXNlZCB3aGVuIGNyZWF0aW5nIGNoYW5nZWxvZyB3aXRoIHN0YW5kYXJkLXZlcnNpb24gcGFja2FnZS5cbiAgICogR2l2ZW4gdmFsdWVzIGVpdGhlciBhcHBlbmQgdG8gZGVmYXVsdCBjb25maWd1cmF0aW9uIG9yIG92ZXJ3cml0ZSB2YWx1ZXMgaW4gaXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gc3RhbmRhcmQgY29uZmlndXJhdGlvbiBhcHBsaWNhYmxlIGZvciBHaXRIdWIgcmVwb3NpdG9yaWVzXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9ucmNPcHRpb25zPzogUmVjb3JkPHN0cmluZywgYW55PjtcblxuICAvKipcbiAgICogR2l0aHViIFJ1bm5lciBzZWxlY3Rpb24gbGFiZWxzXG4gICAqIEBkZWZhdWx0IFtcInVidW50dS1sYXRlc3RcIl1cbiAgICogQGRlc2NyaXB0aW9uIERlZmluZXMgYSB0YXJnZXQgUnVubmVyIGJ5IGxhYmVsc1xuICAgKiBAdGhyb3dzIHtFcnJvcn0gaWYgYm90aCBgcnVuc09uYCBhbmQgYHJ1bnNPbkdyb3VwYCBhcmUgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd1J1bnNPbj86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBHaXRodWIgUnVubmVyIEdyb3VwIHNlbGVjdGlvbiBvcHRpb25zXG4gICAqIEBkZXNjcmlwdGlvbiBEZWZpbmVzIGEgdGFyZ2V0IFJ1bm5lciBHcm91cCBieSBuYW1lIGFuZC9vciBsYWJlbHNcbiAgICogQHRocm93cyB7RXJyb3J9IGlmIGJvdGggYHJ1bnNPbmAgYW5kIGBydW5zT25Hcm91cGAgYXJlIHNwZWNpZmllZFxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2Zsb3dSdW5zT25Hcm91cD86IEdyb3VwUnVubmVyT3B0aW9ucztcblxuICAvKipcbiAgICogRGVmaW5lIHB1Ymxpc2hpbmcgdGFza3MgdGhhdCBjYW4gYmUgZXhlY3V0ZWQgbWFudWFsbHkgYXMgd2VsbCBhcyB3b3JrZmxvd3MuXG4gICAqXG4gICAqIE5vcm1hbGx5LCBwdWJsaXNoaW5nIG9ubHkgaGFwcGVucyB3aXRoaW4gYXV0b21hdGVkIHdvcmtmbG93cy4gRW5hYmxlIHRoaXNcbiAgICogaW4gb3JkZXIgdG8gY3JlYXRlIGEgcHVibGlzaGluZyB0YXNrIGZvciBlYWNoIHB1Ymxpc2hpbmcgYWN0aXZpdHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBwdWJsaXNoVGFza3M/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJbnN0ZWFkIG9mIGFjdHVhbGx5IHB1Ymxpc2hpbmcgdG8gcGFja2FnZSBtYW5hZ2VycywganVzdCBwcmludCB0aGUgcHVibGlzaGluZyBjb21tYW5kLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHVibGlzaERyeVJ1bj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEZpbmQgY29tbWl0cyB0aGF0IHNob3VsZCBiZSBjb25zaWRlcmVkIHJlbGVhc2FibGVcbiAgICogVXNlZCB0byBkZWNpZGUgaWYgYSByZWxlYXNlIGlzIHJlcXVpcmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBSZWxlYXNhYmxlQ29tbWl0cy5ldmVyeUNvbW1pdCgpXG4gICAqL1xuICByZWFkb25seSByZWxlYXNhYmxlQ29tbWl0cz86IFJlbGVhc2FibGVDb21taXRzO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBSZWxlYXNlYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZWxlYXNlT3B0aW9ucyBleHRlbmRzIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgdGFzayB0byBleGVjdXRlIGluIG9yZGVyIHRvIGNyZWF0ZSB0aGUgcmVsZWFzZSBhcnRpZmFjdHMuIEFydGlmYWN0cyBhcmVcbiAgICogZXhwZWN0ZWQgdG8gcmVzaWRlIHVuZGVyIGBhcnRpZmFjdHNEaXJlY3RvcnlgIChkZWZhdWx0cyB0byBgZGlzdC9gKSBvbmNlXG4gICAqIGJ1aWxkIGlzIGNvbXBsZXRlLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFzazogVGFzaztcblxuICAvKipcbiAgICogQSBuYW1lIG9mIGEgLmpzb24gZmlsZSB0byBzZXQgdGhlIGB2ZXJzaW9uYCBmaWVsZCBpbiBhZnRlciBhIGJ1bXAuXG4gICAqXG4gICAqIEBleGFtcGxlIFwicGFja2FnZS5qc29uXCJcbiAgICovXG4gIHJlYWRvbmx5IHZlcnNpb25GaWxlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IGJyYW5jaCBuYW1lIHRvIHJlbGVhc2UgZnJvbS5cbiAgICpcbiAgICogVXNlIGBtYWpvclZlcnNpb25gIHRvIHJlc3RyaWN0IHRoaXMgYnJhbmNoIHRvIG9ubHkgcHVibGlzaCByZWxlYXNlcyB3aXRoIGFcbiAgICogc3BlY2lmaWMgbWFqb3IgdmVyc2lvbi5cbiAgICpcbiAgICogWW91IGNhbiBhZGQgYWRkaXRpb25hbCBicmFuY2hlcyB1c2luZyBgYWRkQnJhbmNoKClgLlxuICAgKi9cbiAgcmVhZG9ubHkgYnJhbmNoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIEdpdEh1YiByZWxlYXNlIGZvciBlYWNoIHJlbGVhc2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGdpdGh1YlJlbGVhc2U/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBIGRpcmVjdG9yeSB3aGljaCB3aWxsIGNvbnRhaW4gYnVpbGQgYXJ0aWZhY3RzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcImRpc3RcIlxuICAgKi9cbiAgcmVhZG9ubHkgYXJ0aWZhY3RzRGlyZWN0b3J5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE5vZGUgdmVyc2lvbiB0byBzZXR1cCBpbiBHaXRIdWIgd29ya2Zsb3dzIGlmIGFueSBub2RlLWJhc2VkIENMSSB1dGlsaXRpZXNcbiAgICogYXJlIG5lZWRlZC4gRm9yIGV4YW1wbGUgYHB1YmxpYmAsIHRoZSBDTEkgcHJvamVuIHVzZXMgdG8gcHVibGlzaCByZWxlYXNlcyxcbiAgICogaXMgYW4gbnBtIGxpYnJhcnkuXG4gICAqXG4gICAqIEBkZWZhdWx0IDE4LnhcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93Tm9kZVZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBlcm1pc3Npb25zIGdyYW50ZWQgdG8gdGhlIHJlbGVhc2Ugd29ya2Zsb3cgam9iXG4gICAqIEBkZWZhdWx0IGB7IGNvbnRlbnRzOiBKb2JQZXJtaXNzaW9uLldSSVRFIH1gXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd1Blcm1pc3Npb25zPzogSm9iUGVybWlzc2lvbnM7XG59XG5cbi8qKlxuICogTWFuYWdlcyByZWxlYXNlcyAoY3VycmVudGx5IHRocm91Z2ggR2l0SHViIHdvcmtmbG93cykuXG4gKlxuICogQnkgZGVmYXVsdCwgbm8gYnJhbmNoZXMgYXJlIHJlbGVhc2VkLiBUbyBhZGQgYnJhbmNoZXMsIGNhbGwgYGFkZEJyYW5jaCgpYC5cbiAqL1xuZXhwb3J0IGNsYXNzIFJlbGVhc2UgZXh0ZW5kcyBDb21wb25lbnQge1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEFOVElfVEFNUEVSX0NNRCA9XG4gICAgXCJnaXQgZGlmZiAtLWlnbm9yZS1zcGFjZS1hdC1lb2wgLS1leGl0LWNvZGVcIjtcblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgYFJlbGVhc2VgIGNvbXBvbmVudCBvZiBhIHByb2plY3Qgb3IgYHVuZGVmaW5lZGAgaWYgdGhlIHByb2plY3RcbiAgICogZG9lcyBub3QgaGF2ZSBhIFJlbGVhc2UgY29tcG9uZW50LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvZihwcm9qZWN0OiBQcm9qZWN0KTogUmVsZWFzZSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgaXNSZWxlYXNlID0gKGM6IENvbXBvbmVudCk6IGMgaXMgUmVsZWFzZSA9PiBjIGluc3RhbmNlb2YgUmVsZWFzZTtcbiAgICByZXR1cm4gcHJvamVjdC5jb21wb25lbnRzLmZpbmQoaXNSZWxlYXNlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQYWNrYWdlIHB1Ymxpc2hlci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaXNoZXI6IFB1Ymxpc2hlcjtcblxuICAvKipcbiAgICogTG9jYXRpb24gb2YgYnVpbGQgYXJ0aWZhY3RzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFydGlmYWN0c0RpcmVjdG9yeTogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgYnVpbGRUYXNrOiBUYXNrO1xuICBwcml2YXRlIHJlYWRvbmx5IHZlcnNpb246IFZlcnNpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgcG9zdEJ1aWxkU3RlcHM6IEpvYlN0ZXBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSB2ZXJzaW9uRmlsZTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGVhc2VUcmlnZ2VyOiBSZWxlYXNlVHJpZ2dlcjtcbiAgcHJpdmF0ZSByZWFkb25seSBwcmVCdWlsZFN0ZXBzOiBKb2JTdGVwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgY29udGFpbmVySW1hZ2U/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2JyYW5jaGVzID0gbmV3IEFycmF5PFJlbGVhc2VCcmFuY2g+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgam9iczogUmVjb3JkPHN0cmluZywgSm9iPiA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRCcmFuY2g6IFJlbGVhc2VCcmFuY2g7XG4gIHByaXZhdGUgcmVhZG9ubHkgZ2l0aHViPzogR2l0SHViO1xuICBwcml2YXRlIHJlYWRvbmx5IHdvcmtmbG93UnVuc09uPzogc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya2Zsb3dSdW5zT25Hcm91cD86IEdyb3VwUnVubmVyT3B0aW9ucztcbiAgcHJpdmF0ZSByZWFkb25seSB3b3JrZmxvd1Blcm1pc3Npb25zOiBKb2JQZXJtaXNzaW9ucztcbiAgcHJpdmF0ZSByZWFkb25seSByZWxlYXNlVGFnRmlsZVBhdGg6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBfYnJhbmNoSG9va3M6IEJyYW5jaEhvb2tbXTtcblxuICAvKipcbiAgICogQHBhcmFtIHNjb3BlIHNob3VsZCBiZSBwYXJ0IG9mIHRoZSBwcm9qZWN0IHRoZSBSZWxlYXNlIGJlbG9uZ3MgdG8uXG4gICAqIEBwYXJhbSBvcHRpb25zIG9wdGlvbnMgdG8gY29uZmlndXJlIHRoZSBSZWxlYXNlIENvbXBvbmVudC5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBJQ29uc3RydWN0LCBvcHRpb25zOiBSZWxlYXNlT3B0aW9ucykge1xuICAgIHN1cGVyKHNjb3BlKTtcblxuICAgIGlmIChBcnJheS5pc0FycmF5KG9wdGlvbnMucmVsZWFzZUJyYW5jaGVzKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAnXCJyZWxlYXNlQnJhbmNoZXNcIiBpcyBubyBsb25nZXIgYW4gYXJyYXkuIFNlZSB0eXBlIGFubm90YXRpb25zJ1xuICAgICAgKTtcbiAgICB9XG5cbiAgICB0aGlzLmdpdGh1YiA9IEdpdEh1Yi5vZih0aGlzLnByb2plY3Qucm9vdCk7XG4gICAgdGhpcy5idWlsZFRhc2sgPSBvcHRpb25zLnRhc2s7XG4gICAgdGhpcy5wcmVCdWlsZFN0ZXBzID0gb3B0aW9ucy5yZWxlYXNlV29ya2Zsb3dTZXR1cFN0ZXBzID8/IFtdO1xuICAgIHRoaXMucG9zdEJ1aWxkU3RlcHMgPSBvcHRpb25zLnBvc3RCdWlsZFN0ZXBzID8/IFtdO1xuICAgIHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5ID0gb3B0aW9ucy5hcnRpZmFjdHNEaXJlY3RvcnkgPz8gXCJkaXN0XCI7XG4gICAgdGhpcy52ZXJzaW9uRmlsZSA9IG9wdGlvbnMudmVyc2lvbkZpbGU7XG4gICAgdGhpcy5yZWxlYXNlVHJpZ2dlciA9IG9wdGlvbnMucmVsZWFzZVRyaWdnZXIgPz8gUmVsZWFzZVRyaWdnZXIuY29udGludW91cygpO1xuICAgIHRoaXMuY29udGFpbmVySW1hZ2UgPSBvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2U7XG4gICAgdGhpcy53b3JrZmxvd1J1bnNPbiA9IG9wdGlvbnMud29ya2Zsb3dSdW5zT247XG4gICAgdGhpcy53b3JrZmxvd1J1bnNPbkdyb3VwID0gb3B0aW9ucy53b3JrZmxvd1J1bnNPbkdyb3VwO1xuICAgIHRoaXMud29ya2Zsb3dQZXJtaXNzaW9ucyA9IHtcbiAgICAgIGNvbnRlbnRzOiBKb2JQZXJtaXNzaW9uLldSSVRFLFxuICAgICAgLi4ub3B0aW9ucy53b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgIH07XG4gICAgdGhpcy5fYnJhbmNoSG9va3MgPSBbXTtcblxuICAgIC8qKlxuICAgICAqIFVzZSBtYW51YWwgcmVsZWFzZXMgd2l0aCBubyBjaGFuZ2Vsb2cgaWYgcmVsZWFzZUV2ZXJ5Q29tbWl0IGlzIGV4cGxpY2l0bHlcbiAgICAgKiBkaXNhYmxlZCBhbmQgbm8gb3RoZXIgdHJpZ2dlciBpcyBzZXQuXG4gICAgICpcbiAgICAgKiBUT0RPOiBSZW1vdmUgdGhpcyB3aGVuIHJlbGVhc2VFdmVyeUNvbW1pdCBhbmQgcmVsZWFzZVNjaGVkdWxlIGFyZSByZW1vdmVkXG4gICAgICovXG4gICAgaWYgKFxuICAgICAgIShcbiAgICAgICAgKG9wdGlvbnMucmVsZWFzZUV2ZXJ5Q29tbWl0ID8/IHRydWUpIHx8XG4gICAgICAgIG9wdGlvbnMucmVsZWFzZVNjaGVkdWxlIHx8XG4gICAgICAgIG9wdGlvbnMucmVsZWFzZVRyaWdnZXJcbiAgICAgIClcbiAgICApIHtcbiAgICAgIHRoaXMucmVsZWFzZVRyaWdnZXIgPSBSZWxlYXNlVHJpZ2dlci5tYW51YWwoeyBjaGFuZ2Vsb2c6IGZhbHNlIH0pO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLnJlbGVhc2VTY2hlZHVsZSkge1xuICAgICAgdGhpcy5yZWxlYXNlVHJpZ2dlciA9IFJlbGVhc2VUcmlnZ2VyLnNjaGVkdWxlZCh7XG4gICAgICAgIHNjaGVkdWxlOiBvcHRpb25zLnJlbGVhc2VTY2hlZHVsZSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMudmVyc2lvbiA9IG5ldyBWZXJzaW9uKHRoaXMucHJvamVjdCwge1xuICAgICAgdmVyc2lvbklucHV0RmlsZTogdGhpcy52ZXJzaW9uRmlsZSxcbiAgICAgIGFydGlmYWN0c0RpcmVjdG9yeTogdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICB2ZXJzaW9ucmNPcHRpb25zOiBvcHRpb25zLnZlcnNpb25yY09wdGlvbnMsXG4gICAgICB0YWdQcmVmaXg6IG9wdGlvbnMucmVsZWFzZVRhZ1ByZWZpeCxcbiAgICAgIHJlbGVhc2FibGVDb21taXRzOiBvcHRpb25zLnJlbGVhc2FibGVDb21taXRzLFxuICAgIH0pO1xuXG4gICAgdGhpcy5yZWxlYXNlVGFnRmlsZVBhdGggPSBwYXRoLnBvc2l4Lm5vcm1hbGl6ZShcbiAgICAgIHBhdGgucG9zaXguam9pbihcbiAgICAgICAgLy8gdGVtcG9yYXJ5IGhhY2sgdG8gYWxsb3cgSnNpaVByb2plY3Qgc2V0dGluZyBhIGRpZmZlcmVudCBwYXRoIHRvIHRoZSByZWxlYXNlIHRhZyBmaWxlXG4gICAgICAgIC8vIHNlZSBKc2lpUHJvamVjdC5yZWxlYXNlVGFnRmlsZVBhdGggZm9yIG1vcmUgZGV0YWlsc1xuICAgICAgICAvL0B0cy1pZ25vcmVcbiAgICAgICAgdGhpcy5wcm9qZWN0LnJlbGVhc2VUYWdGaWxlUGF0aCA/PyB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgdGhpcy52ZXJzaW9uLnJlbGVhc2VUYWdGaWxlTmFtZVxuICAgICAgKVxuICAgICk7XG5cbiAgICB0aGlzLnB1Ymxpc2hlciA9IG5ldyBQdWJsaXNoZXIodGhpcy5wcm9qZWN0LCB7XG4gICAgICBhcnRpZmFjdE5hbWU6IHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LFxuICAgICAgY29uZGl0aW9uOiBERVBFTkRFTlRfSk9CX0NPTkRJVElPTkFMLFxuICAgICAgYnVpbGRKb2JJZDogQlVJTERfSk9CSUQsXG4gICAgICBqc2lpUmVsZWFzZVZlcnNpb246IG9wdGlvbnMuanNpaVJlbGVhc2VWZXJzaW9uLFxuICAgICAgZmFpbHVyZUlzc3VlOiBvcHRpb25zLnJlbGVhc2VGYWlsdXJlSXNzdWUsXG4gICAgICBmYWlsdXJlSXNzdWVMYWJlbDogb3B0aW9ucy5yZWxlYXNlRmFpbHVyZUlzc3VlTGFiZWwsXG4gICAgICAuLi5maWx0ZXJlZFdvcmtmbG93UnVuc09uT3B0aW9ucyhcbiAgICAgICAgb3B0aW9ucy53b3JrZmxvd1J1bnNPbixcbiAgICAgICAgb3B0aW9ucy53b3JrZmxvd1J1bnNPbkdyb3VwXG4gICAgICApLFxuICAgICAgcHVibGlzaFRhc2tzOiBvcHRpb25zLnB1Ymxpc2hUYXNrcyxcbiAgICAgIGRyeVJ1bjogb3B0aW9ucy5wdWJsaXNoRHJ5UnVuLFxuICAgICAgd29ya2Zsb3dOb2RlVmVyc2lvbjogb3B0aW9ucy53b3JrZmxvd05vZGVWZXJzaW9uLFxuICAgICAgd29ya2Zsb3dDb250YWluZXJJbWFnZTogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgZ2l0aHViUmVsZWFzZSA9IG9wdGlvbnMuZ2l0aHViUmVsZWFzZSA/PyB0cnVlO1xuICAgIGlmIChnaXRodWJSZWxlYXNlKSB7XG4gICAgICB0aGlzLnB1Ymxpc2hlci5wdWJsaXNoVG9HaXRIdWJSZWxlYXNlcyh7XG4gICAgICAgIGNoYW5nZWxvZ0ZpbGU6IHBhdGgucG9zaXguam9pbihcbiAgICAgICAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgICB0aGlzLnZlcnNpb24uY2hhbmdlbG9nRmlsZU5hbWVcbiAgICAgICAgKSxcbiAgICAgICAgdmVyc2lvbkZpbGU6IHBhdGgucG9zaXguam9pbihcbiAgICAgICAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgICB0aGlzLnZlcnNpb24udmVyc2lvbkZpbGVOYW1lXG4gICAgICAgICksXG4gICAgICAgIHJlbGVhc2VUYWdGaWxlOiBwYXRoLnBvc2l4LmpvaW4oXG4gICAgICAgICAgdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgICAgdGhpcy52ZXJzaW9uLnJlbGVhc2VUYWdGaWxlTmFtZVxuICAgICAgICApLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gYWRkIHRoZSBkZWZhdWx0IGJyYW5jaCAod2UgbmVlZCB0aGUgaW50ZXJuYWwgbWV0aG9kIHdoaWNoIGRvZXMgbm90IHJlcXVpcmUgbWFqb3JWZXJzaW9uKVxuICAgIHRoaXMuZGVmYXVsdEJyYW5jaCA9IHRoaXMuX2FkZEJyYW5jaChvcHRpb25zLmJyYW5jaCwge1xuICAgICAgcHJlcmVsZWFzZTogb3B0aW9ucy5wcmVyZWxlYXNlLFxuICAgICAgbWFqb3JWZXJzaW9uOiBvcHRpb25zLm1ham9yVmVyc2lvbixcbiAgICAgIG1pbk1ham9yVmVyc2lvbjogb3B0aW9ucy5taW5NYWpvclZlcnNpb24sXG4gICAgICB3b3JrZmxvd05hbWU6XG4gICAgICAgIG9wdGlvbnMucmVsZWFzZVdvcmtmbG93TmFtZSA/P1xuICAgICAgICB3b3JrZmxvd05hbWVGb3JQcm9qZWN0KFwicmVsZWFzZVwiLCB0aGlzLnByb2plY3QpLFxuICAgICAgdGFnUHJlZml4OiBvcHRpb25zLnJlbGVhc2VUYWdQcmVmaXgsXG4gICAgICBucG1EaXN0VGFnOiBvcHRpb25zLm5wbURpc3RUYWcsXG4gICAgfSk7XG5cbiAgICBmb3IgKGNvbnN0IFtuYW1lLCBvcHRzXSBvZiBPYmplY3QuZW50cmllcyhvcHRpb25zLnJlbGVhc2VCcmFuY2hlcyA/PyB7fSkpIHtcbiAgICAgIHRoaXMuYWRkQnJhbmNoKG5hbWUsIG9wdHMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBob29rIHRoYXQgc2hvdWxkIGJlIHJ1biBmb3IgZXZlcnkgYnJhbmNoIChpbmNsdWRpbmcgdGhvc2UgdGhhdCB3aWxsXG4gICAqIGJlIGFkZGVkIGJ5IGZ1dHVyZSBgYWRkQnJhbmNoYCBjYWxscykuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF9mb3JFYWNoQnJhbmNoKGhvb2s6IEJyYW5jaEhvb2spIHtcbiAgICBmb3IgKGNvbnN0IGJyYW5jaCBvZiB0aGlzLl9icmFuY2hlcykge1xuICAgICAgaG9vayhicmFuY2gubmFtZSk7XG4gICAgfVxuICAgIHRoaXMuX2JyYW5jaEhvb2tzLnB1c2goaG9vayk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlbGVhc2UgYnJhbmNoLlxuICAgKlxuICAgKiBJdCBpcyBhIGdpdCBicmFuY2ggZnJvbSB3aGljaCByZWxlYXNlcyBhcmUgcHVibGlzaGVkLiBJZiBhIHByb2plY3QgaGFzIG1vcmUgdGhhbiBvbmUgcmVsZWFzZVxuICAgKiBicmFuY2gsIHdlIHJlcXVpcmUgdGhhdCBgbWFqb3JWZXJzaW9uYCBpcyBhbHNvIHNwZWNpZmllZCBmb3IgdGhlIHByaW1hcnkgYnJhbmNoIGluIG9yZGVyIHRvXG4gICAqIGVuc3VyZSBicmFuY2hlcyBhbHdheXMgcmVsZWFzZSB0aGUgY29ycmVjdCB2ZXJzaW9uLlxuICAgKlxuICAgKiBAcGFyYW0gYnJhbmNoIFRoZSBicmFuY2ggdG8gbW9uaXRvciAoZS5nLiBgbWFpbmAsIGB2Mi54YClcbiAgICogQHBhcmFtIG9wdGlvbnMgQnJhbmNoIGRlZmluaXRpb25cbiAgICovXG4gIHB1YmxpYyBhZGRCcmFuY2goYnJhbmNoOiBzdHJpbmcsIG9wdGlvbnM6IEJyYW5jaE9wdGlvbnMpIHtcbiAgICB0aGlzLl9hZGRCcmFuY2goYnJhbmNoLCBvcHRpb25zKTtcblxuICAgIC8vIHJ1biBhbGwgYnJhbmNoIGhvb2tzXG4gICAgZm9yIChjb25zdCBob29rIG9mIHRoaXMuX2JyYW5jaEhvb2tzKSB7XG4gICAgICBob29rKGJyYW5jaCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZWxlYXNlIGJyYW5jaC5cbiAgICpcbiAgICogSXQgaXMgYSBnaXQgYnJhbmNoIGZyb20gd2hpY2ggcmVsZWFzZXMgYXJlIHB1Ymxpc2hlZC4gSWYgYSBwcm9qZWN0IGhhcyBtb3JlIHRoYW4gb25lIHJlbGVhc2VcbiAgICogYnJhbmNoLCB3ZSByZXF1aXJlIHRoYXQgYG1ham9yVmVyc2lvbmAgaXMgYWxzbyBzcGVjaWZpZWQgZm9yIHRoZSBwcmltYXJ5IGJyYW5jaCBpbiBvcmRlciB0b1xuICAgKiBlbnN1cmUgYnJhbmNoZXMgYWx3YXlzIHJlbGVhc2UgdGhlIGNvcnJlY3QgdmVyc2lvbi5cbiAgICpcbiAgICogQHBhcmFtIGJyYW5jaCBUaGUgYnJhbmNoIHRvIG1vbml0b3IgKGUuZy4gYG1haW5gLCBgdjIueGApXG4gICAqIEBwYXJhbSBvcHRpb25zIEJyYW5jaCBkZWZpbml0aW9uXG4gICAqL1xuICBwcml2YXRlIF9hZGRCcmFuY2goXG4gICAgYnJhbmNoOiBzdHJpbmcsXG4gICAgb3B0aW9uczogUGFydGlhbDxCcmFuY2hPcHRpb25zPlxuICApOiBSZWxlYXNlQnJhbmNoIHtcbiAgICBpZiAodGhpcy5fYnJhbmNoZXMuZmluZCgoYikgPT4gYi5uYW1lID09PSBicmFuY2gpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSByZWxlYXNlIGJyYW5jaCAke2JyYW5jaH0gaXMgYWxyZWFkeSBkZWZpbmVkYCk7XG4gICAgfVxuXG4gICAgLy8gaWYgd2UgYWRkIGEgYnJhbmNoLCB3ZSByZXF1aXJlIHRoYXQgdGhlIGRlZmF1bHQgYnJhbmNoIHdpbGwgYWxzbyBkZWZpbmUgYVxuICAgIC8vIG1ham9yIHZlcnNpb24uXG4gICAgaWYgKFxuICAgICAgdGhpcy5kZWZhdWx0QnJhbmNoICYmXG4gICAgICBvcHRpb25zLm1ham9yVmVyc2lvbiAmJlxuICAgICAgdGhpcy5kZWZhdWx0QnJhbmNoLm1ham9yVmVyc2lvbiA9PT0gdW5kZWZpbmVkXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICd5b3UgbXVzdCBzcGVjaWZ5IFwibWFqb3JWZXJzaW9uXCIgZm9yIHRoZSBkZWZhdWx0IGJyYW5jaCB3aGVuIGFkZGluZyBtdWx0aXBsZSByZWxlYXNlIGJyYW5jaGVzJ1xuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCByZWxlYXNlQnJhbmNoOiBSZWxlYXNlQnJhbmNoID0ge1xuICAgICAgbmFtZTogYnJhbmNoLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIHdvcmtmbG93OiB0aGlzLmNyZWF0ZVdvcmtmbG93KGJyYW5jaCwgb3B0aW9ucyksXG4gICAgfTtcblxuICAgIHRoaXMuX2JyYW5jaGVzLnB1c2gocmVsZWFzZUJyYW5jaCk7XG5cbiAgICByZXR1cm4gcmVsZWFzZUJyYW5jaDtcbiAgfVxuXG4gIHB1YmxpYyBwcmVTeW50aGVzaXplKCkge1xuICAgIGZvciAoY29uc3QgYnJhbmNoIG9mIHRoaXMuX2JyYW5jaGVzKSB7XG4gICAgICBpZiAoIWJyYW5jaC53b3JrZmxvdykge1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgYnJhbmNoLndvcmtmbG93LmFkZEpvYnMoXG4gICAgICAgIHRoaXMucHVibGlzaGVyLl9yZW5kZXJKb2JzRm9yQnJhbmNoKGJyYW5jaC5uYW1lLCBicmFuY2gpXG4gICAgICApO1xuICAgICAgYnJhbmNoLndvcmtmbG93LmFkZEpvYnModGhpcy5qb2JzKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBqb2JzIHRvIGFsbCByZWxlYXNlIHdvcmtmbG93cy5cbiAgICogQHBhcmFtIGpvYnMgVGhlIGpvYnMgdG8gYWRkIChuYW1lID0+IGpvYilcbiAgICovXG4gIHB1YmxpYyBhZGRKb2JzKGpvYnM6IFJlY29yZDxzdHJpbmcsIEpvYj4pIHtcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCBqb2JdIG9mIE9iamVjdC5lbnRyaWVzKGpvYnMpKSB7XG4gICAgICB0aGlzLmpvYnNbbmFtZV0gPSBqb2I7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGFsbCByZWxlYXNlIGJyYW5jaCBuYW1lc1xuICAgKi9cbiAgcHVibGljIGdldCBicmFuY2hlcygpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIHRoaXMuX2JyYW5jaGVzLm1hcCgoYikgPT4gYi5uYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBhIHdvcmtmbG93IG9yIGB1bmRlZmluZWRgIGlmIGdpdGh1YiBpbnRlZ3JhdGlvbiBpcyBkaXNhYmxlZC5cbiAgICovXG4gIHByaXZhdGUgY3JlYXRlV29ya2Zsb3coXG4gICAgYnJhbmNoTmFtZTogc3RyaW5nLFxuICAgIGJyYW5jaDogUGFydGlhbDxCcmFuY2hPcHRpb25zPlxuICApOiBHaXRodWJXb3JrZmxvdyB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3Qgd29ya2Zsb3dOYW1lID1cbiAgICAgIGJyYW5jaC53b3JrZmxvd05hbWUgPz9cbiAgICAgIHdvcmtmbG93TmFtZUZvclByb2plY3QoYHJlbGVhc2UtJHticmFuY2hOYW1lfWAsIHRoaXMucHJvamVjdCk7XG5cbiAgICAvLyB0byBhdm9pZCByYWNlIGNvbmRpdGlvbnMgYmV0d2VlbiB0d28gY29tbWl0cyB0cnlpbmcgdG8gcmVsZWFzZSB0aGUgc2FtZVxuICAgIC8vIHZlcnNpb24sIHdlIGNoZWNrIGlmIHRoZSBoZWFkIHNoYSBpcyBpZGVudGljYWwgdG8gdGhlIHJlbW90ZSBzaGEuIGlmXG4gICAgLy8gbm90LCB3ZSB3aWxsIHNraXAgdGhlIHJlbGVhc2UgYW5kIGp1c3QgZmluaXNoIHRoZSBidWlsZC5cbiAgICBjb25zdCBub05ld0NvbW1pdHMgPSBgXFwke3sgc3RlcHMuJHtHSVRfUkVNT1RFX1NURVBJRH0ub3V0cHV0cy4ke0xBVEVTVF9DT01NSVRfT1VUUFVUfSA9PSBnaXRodWIuc2hhIH19YDtcblxuICAgIC8vIFRoZSBhcnJheXMgYXJlIGJlaW5nIGNsb25lZCB0byBhdm9pZCBhY2N1bXVsYXRpbmcgdmFsdWVzIGZyb20gcHJldmlvdXMgYnJhbmNoZXNcbiAgICBjb25zdCBwcmVCdWlsZFN0ZXBzID0gWy4uLnRoaXMucHJlQnVpbGRTdGVwc107XG5cbiAgICBjb25zdCBlbnY6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4gPSB7XG4gICAgICBSRUxFQVNFOiBcInRydWVcIixcbiAgICB9O1xuXG4gICAgaWYgKGJyYW5jaC5tYWpvclZlcnNpb24gIT09IHVuZGVmaW5lZCkge1xuICAgICAgZW52Lk1BSk9SID0gYnJhbmNoLm1ham9yVmVyc2lvbi50b1N0cmluZygpO1xuICAgIH1cblxuICAgIGlmIChicmFuY2gubWluTWFqb3JWZXJzaW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGlmIChicmFuY2gubWFqb3JWZXJzaW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBtaW5NYWpvclZlcnNpb24gYW5kIG1ham9yVmVyc2lvbiBjYW5ub3QgYmUgdXNlZCB0b2dldGhlci5gXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIGVudi5NSU5fTUFKT1IgPSBicmFuY2gubWluTWFqb3JWZXJzaW9uLnRvU3RyaW5nKCk7XG4gICAgfVxuXG4gICAgaWYgKGJyYW5jaC5wcmVyZWxlYXNlKSB7XG4gICAgICBlbnYuUFJFUkVMRUFTRSA9IGJyYW5jaC5wcmVyZWxlYXNlO1xuICAgIH1cblxuICAgIGlmIChicmFuY2gudGFnUHJlZml4KSB7XG4gICAgICBlbnYuUkVMRUFTRV9UQUdfUFJFRklYID0gYnJhbmNoLnRhZ1ByZWZpeDtcbiAgICB9XG5cbiAgICAvLyB0aGUgXCJyZWxlYXNlXCIgdGFzayBwcmVwYXJlcyBhIHJlbGVhc2UgYnV0IGRvZXMgbm90IHB1Ymxpc2ggYW55dGhpbmcuIHRoZVxuICAgIC8vIG91dHB1dCBvZiB0aGUgcmVsZWFzZSB0YXNrIGlzOiBgZGlzdGAsIGAudmVyc2lvbi50eHRgLCBhbmRcbiAgICAvLyBgLmNoYW5nZWxvZy5tZGAuIHRoaXMgaXMgd2hhdCBwdWJsaXNoIHRhc2tzIGV4cGVjdC5cblxuICAgIC8vIGlmIHRoaXMgaXMgdGhlIHJlbGVhc2UgZm9yIFwibWFpblwiIG9yIFwibWFzdGVyXCIsIGp1c3QgY2FsbCBpdCBcInJlbGVhc2VcIi5cbiAgICAvLyBvdGhlcndpc2UsIFwicmVsZWFzZTpCUkFOQ0hcIlxuICAgIGNvbnN0IHJlbGVhc2VUYXNrTmFtZSA9XG4gICAgICBicmFuY2hOYW1lID09PSBcIm1haW5cIiB8fCBicmFuY2hOYW1lID09PSBcIm1hc3RlclwiXG4gICAgICAgID8gXCJyZWxlYXNlXCJcbiAgICAgICAgOiBgcmVsZWFzZToke2JyYW5jaE5hbWV9YDtcbiAgICBjb25zdCByZWxlYXNlVGFzayA9IHRoaXMucHJvamVjdC5hZGRUYXNrKHJlbGVhc2VUYXNrTmFtZSwge1xuICAgICAgZGVzY3JpcHRpb246IGBQcmVwYXJlIGEgcmVsZWFzZSBmcm9tIFwiJHticmFuY2hOYW1lfVwiIGJyYW5jaGAsXG4gICAgICBlbnYsXG4gICAgfSk7XG5cbiAgICByZWxlYXNlVGFzay5leGVjKGBybSAtZnIgJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gKTtcbiAgICByZWxlYXNlVGFzay5zcGF3bih0aGlzLnZlcnNpb24uYnVtcFRhc2spO1xuICAgIHJlbGVhc2VUYXNrLnNwYXduKHRoaXMuYnVpbGRUYXNrKTtcbiAgICByZWxlYXNlVGFzay5zcGF3bih0aGlzLnZlcnNpb24udW5idW1wVGFzayk7XG5cbiAgICAvLyBhbnRpLXRhbXBlciBjaGVjayAoZmFpbHMgaWYgdGhlcmUgd2VyZSBjaGFuZ2VzIHRvIGNvbW1pdHRlZCBmaWxlcylcbiAgICAvLyB0aGlzIHdpbGwgaWRlbnRpZnkgYW55IG5vbi1jb21taXR0ZWQgZmlsZXMgZ2VuZXJhdGVkIGR1cmluZyBidWlsZCAoZS5nLiB0ZXN0IHNuYXBzaG90cylcbiAgICByZWxlYXNlVGFzay5leGVjKFJlbGVhc2UuQU5USV9UQU1QRVJfQ01EKTtcblxuICAgIGlmICh0aGlzLnJlbGVhc2VUcmlnZ2VyLmlzTWFudWFsKSB7XG4gICAgICBjb25zdCBwdWJsaXNoVGFzayA9IHRoaXMucHVibGlzaGVyLnB1Ymxpc2hUb0dpdCh7XG4gICAgICAgIGNoYW5nZWxvZ0ZpbGU6IHBhdGgucG9zaXguam9pbihcbiAgICAgICAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgICB0aGlzLnZlcnNpb24uY2hhbmdlbG9nRmlsZU5hbWVcbiAgICAgICAgKSxcbiAgICAgICAgdmVyc2lvbkZpbGU6IHBhdGgucG9zaXguam9pbihcbiAgICAgICAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgICB0aGlzLnZlcnNpb24udmVyc2lvbkZpbGVOYW1lXG4gICAgICAgICksXG4gICAgICAgIHJlbGVhc2VUYWdGaWxlOiBwYXRoLnBvc2l4LmpvaW4oXG4gICAgICAgICAgdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgICAgdGhpcy52ZXJzaW9uLnJlbGVhc2VUYWdGaWxlTmFtZVxuICAgICAgICApLFxuICAgICAgICBwcm9qZWN0Q2hhbmdlbG9nRmlsZTogdGhpcy5yZWxlYXNlVHJpZ2dlci5jaGFuZ2Vsb2dQYXRoLFxuICAgICAgICBnaXRCcmFuY2g6IGJyYW5jaE5hbWUsXG4gICAgICAgIGdpdFB1c2hDb21tYW5kOiB0aGlzLnJlbGVhc2VUcmlnZ2VyLmdpdFB1c2hDb21tYW5kLFxuICAgICAgfSk7XG5cbiAgICAgIHJlbGVhc2VUYXNrLnNwYXduKHB1Ymxpc2hUYXNrKTtcbiAgICB9XG5cbiAgICBjb25zdCBwb3N0QnVpbGRTdGVwcyA9IFsuLi50aGlzLnBvc3RCdWlsZFN0ZXBzXTtcblxuICAgIC8vIFJlYWQgdGhlIHJlbGVhc2V0YWcsIHRoZW4gY2hlY2sgaWYgaXQgYWxyZWFkeSBleGlzdHMuXG4gICAgLy8gSWYgaXQgZG9lcywgd2Ugd2lsbCBjYW5jZWwgdGhpcyByZWxlYXNlXG4gICAgcG9zdEJ1aWxkU3RlcHMucHVzaChcbiAgICAgIFdvcmtmbG93U3RlcHMudGFnRXhpc3RzKGAkKGNhdCAke3RoaXMucmVsZWFzZVRhZ0ZpbGVQYXRofSlgLCB7XG4gICAgICAgIG5hbWU6IFwiQ2hlY2sgaWYgdmVyc2lvbiBoYXMgYWxyZWFkeSBiZWVuIHRhZ2dlZFwiLFxuICAgICAgICBpZDogVEFHX0VYSVNUU19TVEVQSUQsXG4gICAgICB9KVxuICAgICk7XG5cbiAgICAvLyBjaGVjayBpZiBuZXcgY29tbWl0cyB3ZXJlIHB1c2hlZCB0byB0aGUgcmVwbyB3aGlsZSB3ZSB3ZXJlIGJ1aWxkaW5nLlxuICAgIC8vIGlmIG5ldyBjb21taXRzIGhhdmUgYmVlbiBwdXNoZWQsIHdlIHdpbGwgY2FuY2VsIHRoaXMgcmVsZWFzZVxuICAgIHBvc3RCdWlsZFN0ZXBzLnB1c2goe1xuICAgICAgbmFtZTogXCJDaGVjayBmb3IgbmV3IGNvbW1pdHNcIixcbiAgICAgIGlkOiBHSVRfUkVNT1RFX1NURVBJRCxcbiAgICAgIHJ1bjogW1xuICAgICAgICBgZWNobyBcIiR7TEFURVNUX0NPTU1JVF9PVVRQVVR9PSQoZ2l0IGxzLXJlbW90ZSBvcmlnaW4gLWggXFwke3sgZ2l0aHViLnJlZiB9fSB8IGN1dCAtZjEpXCIgPj4gJEdJVEhVQl9PVVRQVVRgLFxuICAgICAgICBcImNhdCAkR0lUSFVCX09VVFBVVFwiLFxuICAgICAgXS5qb2luKFwiXFxuXCIpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcHJvamVjdFBhdGhSZWxhdGl2ZVRvUm9vdCA9IHBhdGgucmVsYXRpdmUoXG4gICAgICB0aGlzLnByb2plY3Qucm9vdC5vdXRkaXIsXG4gICAgICB0aGlzLnByb2plY3Qub3V0ZGlyXG4gICAgKTtcbiAgICBwb3N0QnVpbGRTdGVwcy5wdXNoKFxuICAgICAge1xuICAgICAgICBuYW1lOiBcIkJhY2t1cCBhcnRpZmFjdCBwZXJtaXNzaW9uc1wiLFxuICAgICAgICBpZjogbm9OZXdDb21taXRzLFxuICAgICAgICBjb250aW51ZU9uRXJyb3I6IHRydWUsXG4gICAgICAgIHJ1bjogYGNkICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICYmIGdldGZhY2wgLVIgLiA+ICR7UEVSTUlTU0lPTl9CQUNLVVBfRklMRX1gLFxuICAgICAgfSxcbiAgICAgIFdvcmtmbG93U3RlcHMudXBsb2FkQXJ0aWZhY3Qoe1xuICAgICAgICBpZjogbm9OZXdDb21taXRzLFxuICAgICAgICB3aXRoOiB7XG4gICAgICAgICAgbmFtZTogQlVJTERfQVJUSUZBQ1RfTkFNRSxcbiAgICAgICAgICBwYXRoOlxuICAgICAgICAgICAgcHJvamVjdFBhdGhSZWxhdGl2ZVRvUm9vdC5sZW5ndGggPiAwXG4gICAgICAgICAgICAgID8gYCR7cHJvamVjdFBhdGhSZWxhdGl2ZVRvUm9vdH0vJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gXG4gICAgICAgICAgICAgIDogdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgIH0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBpZiAodGhpcy5naXRodWIgJiYgIXRoaXMucmVsZWFzZVRyaWdnZXIuaXNNYW51YWwpIHtcbiAgICAgIC8vIFVzZSB0YXJnZXQgKHBvc3NpYmxlIHBhcmVudCkgR2l0SHViIHRvIGNyZWF0ZSB0aGUgd29ya2Zsb3dcbiAgICAgIGNvbnN0IHdvcmtmbG93ID0gbmV3IEdpdGh1YldvcmtmbG93KHRoaXMuZ2l0aHViLCB3b3JrZmxvd05hbWUpO1xuICAgICAgd29ya2Zsb3cub24oe1xuICAgICAgICBzY2hlZHVsZTogdGhpcy5yZWxlYXNlVHJpZ2dlci5zY2hlZHVsZVxuICAgICAgICAgID8gW3sgY3JvbjogdGhpcy5yZWxlYXNlVHJpZ2dlci5zY2hlZHVsZSB9XVxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICBwdXNoOiB0aGlzLnJlbGVhc2VUcmlnZ2VyLmlzQ29udGludW91c1xuICAgICAgICAgID8geyBicmFuY2hlczogW2JyYW5jaE5hbWVdIH1cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgd29ya2Zsb3dEaXNwYXRjaDoge30sIC8vIGFsbG93IG1hbnVhbCB0cmlnZ2VyaW5nXG4gICAgICB9KTtcblxuICAgICAgLy8gQ3JlYXRlIGpvYiBiYXNlZCBvbiBjaGlsZCAob25seT8pIHByb2plY3QgR2l0SHViXG4gICAgICBjb25zdCB0YXNram9iID0gbmV3IFRhc2tXb3JrZmxvd0pvYih0aGlzLCByZWxlYXNlVGFzaywge1xuICAgICAgICBvdXRwdXRzOiB7XG4gICAgICAgICAgW0xBVEVTVF9DT01NSVRfT1VUUFVUXToge1xuICAgICAgICAgICAgc3RlcElkOiBHSVRfUkVNT1RFX1NURVBJRCxcbiAgICAgICAgICAgIG91dHB1dE5hbWU6IExBVEVTVF9DT01NSVRfT1VUUFVULFxuICAgICAgICAgIH0sXG4gICAgICAgICAgW1RBR19FWElTVFNfT1VUUFVUXToge1xuICAgICAgICAgICAgc3RlcElkOiBUQUdfRVhJU1RTX1NURVBJRCxcbiAgICAgICAgICAgIG91dHB1dE5hbWU6IFwiZXhpc3RzXCIsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgY29udGFpbmVyOiB0aGlzLmNvbnRhaW5lckltYWdlXG4gICAgICAgICAgPyB7IGltYWdlOiB0aGlzLmNvbnRhaW5lckltYWdlIH1cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgZW52OiB7XG4gICAgICAgICAgQ0k6IFwidHJ1ZVwiLFxuICAgICAgICB9LFxuICAgICAgICBwZXJtaXNzaW9uczogdGhpcy53b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgICAgICBjaGVja291dFdpdGg6IHtcbiAgICAgICAgICAvLyBmZXRjaC1kZXB0aD0gaW5kaWNhdGVzIGFsbCBoaXN0b3J5IGZvciBhbGwgYnJhbmNoZXMgYW5kIHRhZ3NcbiAgICAgICAgICAvLyB3ZSBtdXN0IHVzZSB0aGlzIGluIG9yZGVyIHRvIGZldGNoIGFsbCB0YWdzXG4gICAgICAgICAgLy8gYW5kIHRvIGluc3BlY3QgdGhlIGhpc3RvcnkgdG8gZGVjaWRlIGlmIHdlIHNob3VsZCByZWxlYXNlXG4gICAgICAgICAgZmV0Y2hEZXB0aDogMCxcbiAgICAgICAgfSxcbiAgICAgICAgcHJlQnVpbGRTdGVwcyxcbiAgICAgICAgcG9zdEJ1aWxkU3RlcHMsXG4gICAgICAgIGpvYkRlZmF1bHRzOlxuICAgICAgICAgIHByb2plY3RQYXRoUmVsYXRpdmVUb1Jvb3QubGVuZ3RoID4gMCAvLyBpcyBzdWJwcm9qZWN0XG4gICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICBydW46IHtcbiAgICAgICAgICAgICAgICAgIHdvcmtpbmdEaXJlY3Rvcnk6IGVuc3VyZVJlbGF0aXZlUGF0aFN0YXJ0c1dpdGhEb3QoXG4gICAgICAgICAgICAgICAgICAgIHByb2plY3RQYXRoUmVsYXRpdmVUb1Jvb3RcbiAgICAgICAgICAgICAgICAgICksXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIC4uLmZpbHRlcmVkUnVuc09uT3B0aW9ucyh0aGlzLndvcmtmbG93UnVuc09uLCB0aGlzLndvcmtmbG93UnVuc09uR3JvdXApLFxuICAgICAgfSk7XG5cbiAgICAgIHdvcmtmbG93LmFkZEpvYihCVUlMRF9KT0JJRCwgdGFza2pvYik7XG5cbiAgICAgIHJldHVybiB3b3JrZmxvdztcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24gd29ya2Zsb3dOYW1lRm9yUHJvamVjdChiYXNlOiBzdHJpbmcsIHByb2plY3Q6IFByb2plY3QpOiBzdHJpbmcge1xuICAvLyBTdWJwcm9qZWN0c1xuICBpZiAocHJvamVjdC5wYXJlbnQpIHtcbiAgICByZXR1cm4gYCR7YmFzZX1fJHtmaWxlU2FmZU5hbWUocHJvamVjdC5uYW1lKX1gO1xuICB9XG5cbiAgLy8gcm9vdCBwcm9qZWN0IGRvZXNuJ3QgZ2V0IGEgc3VmZml4XG4gIHJldHVybiBiYXNlO1xufVxuXG5mdW5jdGlvbiBmaWxlU2FmZU5hbWUobmFtZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgcmV0dXJuIG5hbWUucmVwbGFjZShcIkBcIiwgXCJcIikucmVwbGFjZSgvXFwvL2csIFwiLVwiKTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBhIHJlbGVhc2UgYnJhbmNoLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJyYW5jaE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHJlbGVhc2Ugd29ya2Zsb3cuXG4gICAqIEBkZWZhdWx0IFwicmVsZWFzZS1CUkFOQ0hcIlxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2Zsb3dOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbWFqb3IgdmVyc2lvbnMgcmVsZWFzZWQgZnJvbSB0aGlzIGJyYW5jaC5cbiAgICovXG4gIHJlYWRvbmx5IG1ham9yVmVyc2lvbjogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBtYWpvciB2ZXJzaW9uIHRvIHJlbGVhc2UuXG4gICAqL1xuICByZWFkb25seSBtaW5NYWpvclZlcnNpb24/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5vciB2ZXJzaW9ucyByZWxlYXNlZCBmcm9tIHRoaXMgYnJhbmNoLlxuICAgKi9cbiAgcmVhZG9ubHkgbWlub3JWZXJzaW9uPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBCdW1wIHRoZSB2ZXJzaW9uIGFzIGEgcHJlLXJlbGVhc2UgdGFnLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vcm1hbCByZWxlYXNlc1xuICAgKi9cbiAgcmVhZG9ubHkgcHJlcmVsZWFzZT86IHN0cmluZztcblxuICAvKipcbiAgICogQXV0b21hdGljYWxseSBhZGQgdGhlIGdpdmVuIHByZWZpeCB0byByZWxlYXNlIHRhZ3MuXG4gICAqIFVzZWZ1bCBpZiB5b3UgYXJlIHJlbGVhc2luZyBvbiBtdWx0aXBsZSBicmFuY2hlcyB3aXRoIG92ZXJsYXBwaW5nXG4gICAqIHZlcnNpb24gbnVtYmVycy5cbiAgICpcbiAgICogTm90ZTogdGhpcyBwcmVmaXggaXMgdXNlZCB0byBkZXRlY3QgdGhlIGxhdGVzdCB0YWdnZWQgdmVyc2lvblxuICAgKiB3aGVuIGJ1bXBpbmcsIHNvIGlmIHlvdSBjaGFuZ2UgdGhpcyBvbiBhIHByb2plY3Qgd2l0aCBhbiBleGlzdGluZyB2ZXJzaW9uXG4gICAqIGhpc3RvcnksIHlvdSBtYXkgbmVlZCB0byBtYW51YWxseSB0YWcgeW91ciBsYXRlc3QgcmVsZWFzZVxuICAgKiB3aXRoIHRoZSBuZXcgcHJlZml4LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHByZWZpeFxuICAgKi9cbiAgcmVhZG9ubHkgdGFnUHJlZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbnBtIGRpc3RyaWJ1dGlvbiB0YWcgdG8gdXNlIGZvciB0aGlzIGJyYW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJsYXRlc3RcIlxuICAgKi9cbiAgcmVhZG9ubHkgbnBtRGlzdFRhZz86IHN0cmluZztcbn1cblxuaW50ZXJmYWNlIFJlbGVhc2VCcmFuY2ggZXh0ZW5kcyBQYXJ0aWFsPEJyYW5jaE9wdGlvbnM+IHtcbiAgcmVhZG9ubHkgd29ya2Zsb3c/OiBHaXRodWJXb3JrZmxvdztcbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xufVxuIl19