"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateChangelog = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const logging = require("../logging");
const utils = require("../util");
/**
 * Prepends input changelog entry onto project-level changelog.
 *
 * Currently assumes a headerless changelog formatted according to
 * [conventional-changelog](https://github.com/conventional-changelog/conventional-changelog)
 * rules.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function updateChangelog(cwd, options) {
    const inputChangelog = (0, path_1.join)(cwd, options.inputChangelog);
    const outputChangelog = (0, path_1.join)(cwd, options.outputChangelog);
    const versionFile = (0, path_1.join)(cwd, options.versionFile);
    let version = (await utils.tryReadFile(versionFile)).trim();
    if (!version) {
        throw new Error(`Unable to determine version from ${versionFile}. Cannot proceed with changelog update. Did you run 'bump'?`);
    }
    const inputChangelogContent = await fs_1.promises.readFile(inputChangelog, "utf-8");
    if (!inputChangelogContent.includes(version)) {
        throw new Error(`Supplied version ${version} was not found in input changelog. You may want to check it's content.`);
    }
    const outputChangelogContent = await fs_1.promises.readFile(outputChangelog, {
        encoding: "utf-8",
        flag: "a+",
    });
    const changelogVersionSearchPattern = `[${version}]`;
    if (outputChangelogContent.indexOf(changelogVersionSearchPattern) > -1) {
        logging.info(`Changelog already contains an entry for ${version}. Skipping changelog update.`);
        return;
    }
    const newChangelog = inputChangelogContent.trimEnd() +
        "\n\n" +
        outputChangelogContent.trimStart();
    await fs_1.promises.writeFile(outputChangelog, newChangelog);
    utils.exec(`git add ${outputChangelog} && git commit -m "chore(release): ${version}"`, { cwd });
}
exports.updateChangelog = updateChangelog;
//# sourceMappingURL=data:application/json;base64,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