"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SampleDir = exports.SampleFile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const glob = require("glob");
const component_1 = require("./component");
const util_1 = require("./util");
/**
 * Produces a file with the given contents but only once, if the file doesn't already exist.
 * Use this for creating example code files or other resources.
 */
class SampleFile extends component_1.Component {
    /**
     * Creates a new SampleFile object
     * @param project - the project to tie this file to.
     * @param filePath - the relative path in the project to put the file
     * @param options - the options for the file.
     */
    constructor(project, filePath, options) {
        super(project);
        if (options.contents && options.sourcePath) {
            throw new Error("Cannot specify both 'contents' and 'source' fields.");
        }
        if (!options.contents && !options.sourcePath) {
            throw new Error("Must specify at least one of 'contents' or 'source'.");
        }
        this.filePath = filePath;
        this.options = options;
    }
    synthesize() {
        let contents;
        if (this.options.contents) {
            contents = this.options.contents;
        }
        else if (this.options.sourcePath) {
            contents = fs.readFileSync(this.options.sourcePath);
        }
        this.writeOnceFileContents(this.project.outdir, this.filePath, contents ?? "");
    }
    /**
     * A helper function that will write the file once and return if it was written or not.
     * @param dir - the directory for the new file
     * @param filename - the filename for the new file
     * @param contents - the contents of the file to write
     * @return boolean - whether a new file was written or not.
     * @private
     */
    writeOnceFileContents(dir, filename, contents) {
        const fullFilename = path.join(dir, filename);
        if (fs.existsSync(fullFilename)) {
            return;
        }
        (0, util_1.writeFile)(fullFilename, contents, { readonly: false });
    }
}
exports.SampleFile = SampleFile;
_a = JSII_RTTI_SYMBOL_1;
SampleFile[_a] = { fqn: "projen.SampleFile", version: "0.79.27" };
/**
 * Renders the given files into the directory if the directory does not exist. Use this to create sample code files
 */
class SampleDir extends component_1.Component {
    /**
     * Create sample files in the given directory if the given directory does not exist
     * @param project Parent project to add files to.
     * @param dir directory to add files to. If directory already exists, nothing is added.
     * @param options options for which files to create.
     */
    constructor(project, dir, options) {
        super(project);
        if (!options.files && !options.sourceDir) {
            throw new Error("Must specify at least one of 'files' or 'source'.");
        }
        this.dir = dir;
        this.options = options;
    }
    synthesize() {
        const fullOutdir = path.join(this.project.outdir, this.dir);
        if (fs.existsSync(fullOutdir)) {
            return;
        }
        // previously creating the directory to allow empty dirs to be created
        fs.mkdirSync(fullOutdir, { recursive: true });
        if (this.options.sourceDir) {
            const basedir = this.options.sourceDir;
            const files = glob.sync("**", {
                cwd: basedir,
                nodir: true,
                dot: true,
            }); // returns relative file paths with POSIX separators
            for (const file of files) {
                const sourcePath = path.join(basedir, file);
                const targetPath = path.join(fullOutdir, file);
                fs.mkdirSync(path.dirname(targetPath), { recursive: true });
                fs.copyFileSync(sourcePath, targetPath);
                fs.chmodSync(targetPath, (0, util_1.getFilePermissions)({ readonly: false, executable: false }));
            }
        }
        for (const filename in this.options.files) {
            (0, util_1.writeFile)(path.join(fullOutdir, filename), this.options.files[filename]);
        }
    }
}
exports.SampleDir = SampleDir;
_b = JSII_RTTI_SYMBOL_1;
SampleDir[_b] = { fqn: "projen.SampleDir", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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