"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const path = require("path");
const util_1 = require("util");
const chalk_1 = require("chalk");
const common_1 = require("./common");
const logging = require("./logging");
const ENV_TRIM_LEN = 20;
const ARGS_MARKER = "$@";
/**
 * The runtime component of the tasks engine.
 */
class TaskRuntime {
    constructor(workdir) {
        this.workdir = (0, path_1.resolve)(workdir);
        const manifestPath = (0, path_1.join)(this.workdir, TaskRuntime.MANIFEST_FILE);
        this.manifest = (0, fs_1.existsSync)(manifestPath)
            ? JSON.parse((0, fs_1.readFileSync)(manifestPath, "utf-8"))
            : { tasks: {} };
    }
    /**
     * The tasks in this project.
     */
    get tasks() {
        return Object.values(this.manifest.tasks ?? {});
    }
    /**
     * Find a task by name, or `undefined` if not found.
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * Runs the task.
     * @param name The task name.
     */
    runTask(name, parents = [], args = []) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents, args);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.TaskRuntime", version: "0.79.27" };
/**
 * The project-relative path of the tasks manifest file.
 */
TaskRuntime.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, "tasks.json");
class RunTask {
    constructor(runtime, task, parents = [], args = []) {
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = task.cwd ?? this.runtime.workdir;
        this.parents = parents;
        if (!task.steps || task.steps.length === 0) {
            this.logDebug((0, chalk_1.gray)("No actions have been specified for this task."));
            return;
        }
        this.env = this.resolveEnvironment(parents);
        const envlogs = [];
        for (const [k, v] of Object.entries(this.env)) {
            const vv = v ?? "";
            const trimmed = vv.length > ENV_TRIM_LEN ? vv.substr(0, ENV_TRIM_LEN) + "..." : vv;
            envlogs.push(`${k}=${trimmed}`);
        }
        if (envlogs.length) {
            this.logDebug((0, chalk_1.gray)(`${(0, chalk_1.underline)("env")}: ${envlogs.join(" ")}`));
        }
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log("condition exited with non-zero - skipping");
            return;
        }
        // verify we required environment variables are defined
        const merged = { ...process.env, ...this.env };
        const missing = new Array();
        for (const name of task.requiredEnv ?? []) {
            if (!(name in merged)) {
                missing.push(name);
            }
        }
        if (missing.length > 0) {
            throw new Error(`missing required environment variables: ${missing.join(",")}`);
        }
        for (const step of task.steps) {
            // evaluate step condition
            if (!this.evalCondition(step)) {
                this.log("condition exited with non-zero - skipping");
                continue;
            }
            const argsList = [
                ...(step.args || []),
                ...(step.receiveArgs ? args : []),
            ].map((a) => a.toString());
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name], argsList);
            }
            const execs = step.exec ? [step.exec] : [];
            // Parse step-specific environment variables
            const env = this.evalEnvironment(step.env ?? {});
            if (step.builtin) {
                execs.push(this.renderBuiltin(step.builtin));
            }
            for (const exec of execs) {
                let command = "";
                let hasError = false;
                const cmd = exec.split(" ")[0];
                if ((0, os_1.platform)() == "win32" &&
                    ["mkdir", "mv", "rm", "cp"].includes(cmd)) {
                    command = `shx ${exec}`;
                }
                else {
                    command = exec;
                }
                if (command.includes(ARGS_MARKER)) {
                    command = command.replace(ARGS_MARKER, argsList.join(" "));
                }
                else {
                    command = [command, ...argsList].join(" ");
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                        extraEnv: env,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if (e?.message?.startsWith("non-zero exit code:")) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${(0, path_1.resolve)(cwd ?? this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(taskOrStep) {
        // no condition, carry on
        if (!taskOrStep.condition) {
            return true;
        }
        this.log((0, chalk_1.gray)(`${(0, chalk_1.underline)("condition")}: ${taskOrStep.condition}`));
        const result = this.shell({
            command: taskOrStep.condition,
            logprefix: "condition: ",
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Evaluates environment variables from shell commands (e.g. `$(xx)`)
     */
    evalEnvironment(env) {
        const output = {};
        for (const [key, value] of Object.entries(env ?? {})) {
            if (String(value).startsWith("$(") && String(value).endsWith(")")) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : result.stderr?.toString() ?? "unknown error";
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString("utf-8").trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     */
    resolveEnvironment(parents) {
        let env = this.runtime.manifest.env ?? {};
        // add env from all parent tasks one by one
        for (const parent of parents) {
            env = {
                ...env,
                ...(this.runtime.tryFindTask(parent)?.env ?? {}),
            };
        }
        // apply the task environment last
        env = {
            ...env,
            ...(this.task.env ?? {}),
        };
        return this.evalEnvironment(env ?? {});
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(" » ");
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    logDebug(...args) {
        logging.debug(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return (0, util_1.format)(`${(0, chalk_1.underline)(this.fullname)} |`, ...args);
    }
    shell(options) {
        const quiet = options.quiet ?? false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(" "));
        }
        const cwd = options.cwd ?? this.workdir;
        if (!(0, fs_1.existsSync)(cwd) || !(0, fs_1.statSync)(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return (0, child_process_1.spawnSync)(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: "inherit",
            env: {
                ...process.env,
                ...this.env,
                ...options.extraEnv,
            },
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ["inherit", "pipe", "inherit"],
            },
        });
    }
    renderBuiltin(builtin) {
        const moduleRoot = (0, path_1.dirname)(require.resolve("../package.json"));
        const program = require.resolve((0, path_1.join)(moduleRoot, "lib", `${builtin}.task.js`));
        return `'${process.execPath}' '${program}'`;
    }
}
//# sourceMappingURL=data:application/json;base64,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