import { TaskCommonOptions, TaskSpec, TaskStep, TaskStepOptions } from "./task-model";
export interface TaskOptions extends TaskCommonOptions {
    /**
     * Shell command to execute as the first command of the task.
     * @default - add steps using `task.exec(command)` or `task.spawn(subtask)`
     */
    readonly exec?: string;
    /**
     * List of task steps to run.
     */
    readonly steps?: TaskStep[];
    /**
     * Should the provided `exec` shell command receive args passed to the task.
     * @see {@link TaskStepOptions.receiveArgs}
     *
     * @default false
     */
    readonly receiveArgs?: boolean;
    /**
     * Should the provided `exec` shell command receive fixed args.
     * @see {@link TaskStepOptions.args}
     *
     * @default - no arguments are passed to the step
     */
    readonly args?: string[];
}
/**
 * A task that can be performed on the project. Modeled as a series of shell
 * commands and subtasks.
 */
export declare class Task {
    /**
     * Task name.
     */
    readonly name: string;
    private readonly _conditions;
    private readonly _steps;
    private readonly _env;
    private _cwd?;
    private readonly requiredEnv?;
    private _locked;
    private _description?;
    constructor(name: string, props?: TaskOptions);
    /**
     * Forbid additional changes to this task.
     */
    lock(): void;
    /**
     * Returns the working directory for this task.
     */
    get cwd(): string | undefined;
    /**
     * Sets the working directory for this task.
     */
    set cwd(cwd: string | undefined);
    /**
     * Returns the description of this task.
     */
    get description(): string | undefined;
    /**
     * Sets the description of this task.
     */
    set description(desc: string | undefined);
    /**
     * A command to execute which determines if the task should be skipped. If it
     * returns a zero exit code, the task will not be executed.
     */
    get condition(): string | undefined;
    /**
     * Add a command to execute which determines if the task should be skipped.
     *
     * If a condition already exists, the new condition will be appended with ` && ` delimiter.
     * @param condition The command to execute.
     * @see {@link Task.condition}
     */
    addCondition(...condition: string[]): void;
    /**
     * Reset the task so it no longer has any commands.
     * @param command the first command to add to the task after it was cleared.
     */
    reset(command?: string, options?: TaskStepOptions): void;
    /**
     * Executes a shell command
     * @param command Shell command
     * @param options Options
     */
    exec(command: string, options?: TaskStepOptions): void;
    /**
     * Execute a builtin task.
     *
     * Builtin tasks are programs bundled as part of projen itself and used as
     * helpers for various components.
     *
     * In the future we should support built-in tasks from external modules.
     *
     * @param name The name of the builtin task to execute (e.g.
     * `release/resolve-version`).
     */
    builtin(name: string): void;
    /**
     * Say something.
     * @param message Your message
     * @param options Options
     */
    say(message: string, options?: TaskStepOptions): void;
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     *
     * @deprecated use `prependExec()`
     */
    prepend(shell: string, options?: TaskStepOptions): void;
    /**
     * Spawns a sub-task.
     * @param subtask The subtask to execute.
     */
    spawn(subtask: Task, options?: TaskStepOptions): void;
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     */
    prependExec(shell: string, options?: TaskStepOptions): void;
    /**
     * Adds a spawn instruction at the beginning of the task.
     * @param subtask The subtask to execute.
     */
    prependSpawn(subtask: Task, options?: TaskStepOptions): void;
    /**
     * Says something at the beginning of the task.
     * @param message Your message
     */
    prependSay(message: string, options?: TaskStepOptions): void;
    /**
     * Adds an environment variable to this task.
     * @param name The name of the variable
     * @param value The value. If the value is surrounded by `$()`, we will
     * evaluate it within a subshell and use the result as the value of the
     * environment variable.
     */
    env(name: string, value: string): void;
    /**
     * Returns all environment variables in the task level
     */
    get envVars(): Readonly<{
        [name: string]: string;
    }>;
    /**
     * Returns an immutable copy of all the step specifications of the task.
     */
    get steps(): TaskStep[];
    /**
     *
     * @param index The index of the step to edit
     * @param step The new step to replace the old one entirely, it is not merged with the old step
     */
    updateStep(index: number, step: TaskStep): void;
    /**
     *
     * @param index The index of the step to remove
     */
    removeStep(index: number): void;
    /**
     * Renders a task spec into the manifest.
     *
     * @internal
     */
    _renderSpec(): TaskSpec;
    private assertUnlocked;
    private warnForLazyValue;
    /**
     * Ensure that environment variables are persisted as strings
     * to prevent type errors when parsing from tasks.json in future
     */
    private getEnvString;
}
