"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tasks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const component_1 = require("./component");
const json_1 = require("./json");
const logging_1 = require("./logging");
const task_1 = require("./task");
const task_runtime_1 = require("./task-runtime");
/**
 * Defines project tasks.
 *
 * Tasks extend the projen CLI by adding subcommands to it. Task definitions are
 * synthesized into `.projen/tasks.json`.
 */
class Tasks extends component_1.Component {
    constructor(project) {
        super(project);
        this._tasks = {};
        this._env = {};
        new json_1.JsonFile(project, task_runtime_1.TaskRuntime.MANIFEST_FILE, {
            omitEmpty: true,
            obj: {
                tasks: (() => this.renderTasks()),
                env: (() => this.renderEnv()),
            },
        });
    }
    /**
     * All tasks.
     */
    get all() {
        return Object.values(this._tasks);
    }
    /**
     * Adds a task to a project.
     * @param name The name of the task
     * @param options Task options.
     */
    addTask(name, options = {}) {
        const task = new task_1.Task(name, options);
        if (this._tasks[name]) {
            throw new Error(`A task with the name ${name} already exists. To override it, call removeTask first and then create the new task.`);
        }
        this._tasks[name] = task;
        return task;
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        const dependentTasks = this.all.filter((task) => task.steps.find((step) => step.spawn == name));
        if (dependentTasks.length > 0) {
            const errList = dependentTasks.map((depTask) => depTask.name).join(", ");
            throw new Error(`Unable to remove task "${name}" because the following tasks depend on it: ${errList}`);
        }
        const task = this._tasks[name];
        if (task) {
            delete this._tasks[name];
            return task;
        }
        else {
            return undefined;
        }
    }
    /**
     * Adds global environment.
     * @param name Environment variable name
     * @param value Value
     */
    addEnvironment(name, value) {
        this._env[name] = value;
    }
    /**
     * Returns a copy of the currently global environment for this project.
     */
    get env() {
        return {
            ...this._env,
        };
    }
    /**
     * Finds a task by name. Returns `undefined` if the task cannot be found.
     * @param name The name of the task
     */
    tryFind(name) {
        return this._tasks[name];
    }
    synthesize() {
        if (this.project.ejected) {
            // Insert a task-runner script so that tasks can be run after ejecting
            fs.mkdirSync(path.join(this.project.outdir, "scripts"), {
                recursive: true,
            });
            fs.copyFileSync(path.join(__dirname, "..", "lib", "run-task.js"), path.join(this.project.outdir, "scripts", "run-task"));
            fs.chmodSync(path.join(this.project.outdir, "scripts", "run-task"), "755");
        }
    }
    renderTasks() {
        const tasks = {};
        for (const task of Object.values(this._tasks).sort((x, y) => x.name.localeCompare(y.name))) {
            tasks[task.name] = task._renderSpec();
        }
        return tasks;
    }
    renderEnv() {
        return Object.keys(this._env).reduce((prev, curr) => ({
            ...prev,
            [curr]: this.getEnvString(curr, this._env[curr]),
        }), {});
    }
    /**
     * Ensure that environment variables are persisted as strings
     * to prevent type errors when parsing from tasks.json in future
     */
    getEnvString(name, value) {
        if (typeof value !== "string" && value !== undefined) {
            (0, logging_1.warn)(`Received non-string value for environment variable ${name}. Value will be stringified.`);
            return String(value);
        }
        else {
            return value;
        }
    }
}
exports.Tasks = Tasks;
_a = JSII_RTTI_SYMBOL_1;
Tasks[_a] = { fqn: "projen.Tasks", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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