"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizePersistedPath = exports.findUp = exports.isRoot = exports.multipleSelected = exports.anySelected = exports.getNodeMajorVersion = exports.isExecutable = exports.assertExecutablePermissions = exports.isWritable = exports.tryReadFileSync = exports.tryReadFile = exports.snakeCaseKeys = exports.kebabCaseKeys = exports.getGitVersion = exports.formatAsPythonModule = exports.sorted = exports.dedupArray = exports.deepMerge = exports.isObject = exports.isTruthy = exports.decamelizeKeysRecursively = exports.writeFile = exports.getFilePermissions = exports.execOrUndefined = exports.execCapture = exports.exec = void 0;
const child_process = require("child_process");
const fs_1 = require("fs");
const path = require("path");
const Case = require("case");
const logging = require("./logging");
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.debug(command);
    child_process.execSync(command, {
        stdio: ["inherit", 2, "pipe"],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
        env: options.env,
    });
}
exports.exec = exec;
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.debug(command);
    return child_process.execSync(command, {
        stdio: ["inherit", "pipe", "pipe"],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.execCapture = execCapture;
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process
            .execSync(command, {
            stdio: ["inherit", "pipe", "pipe"],
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        })
            .toString("utf-8")
            .trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch {
        return undefined;
    }
}
exports.execOrUndefined = execOrUndefined;
function getFilePermissions(options) {
    const readonly = options.readonly ?? false;
    const executable = options.executable ?? false;
    if (readonly && executable) {
        return "544";
    }
    else if (readonly) {
        return "444";
    }
    else if (executable) {
        return "755";
    }
    else {
        return "644";
    }
}
exports.getFilePermissions = getFilePermissions;
function writeFile(filePath, data, options = {}) {
    if ((0, fs_1.existsSync)(filePath)) {
        (0, fs_1.chmodSync)(filePath, "600");
    }
    (0, fs_1.mkdirSync)(path.dirname(filePath), { recursive: true });
    (0, fs_1.writeFileSync)(filePath, data);
    (0, fs_1.chmodSync)(filePath, getFilePermissions(options));
}
exports.writeFile = writeFile;
function decamelizeKeysRecursively(input, opt) {
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = opt?.shouldDecamelize ?? shouldAlwaysDecamelize;
    const separator = opt?.separator ?? "_";
    const path_ = opt?.path ?? [];
    const maxDepth = opt?.maxDepth ?? 10;
    if (path_.length > maxDepth) {
        throw new Error("Decamelled too deeply - check that the input has no circular references");
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === "object" && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined ||
        ["null", "undefined", "0", "false", ""].includes(value.toLocaleLowerCase()));
}
exports.isTruthy = isTruthy;
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return (x !== null &&
        typeof x === "object" &&
        !Array.isArray(x) &&
        x.constructor.name === "Object");
}
exports.isObject = isObject;
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                if ("__$APPEND" in value && Array.isArray(value.__$APPEND)) {
                    if (Array.isArray(target[key])) {
                        target[key].push(...value.__$APPEND);
                    }
                    else {
                        target[key] = value.__$APPEND;
                    }
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" &&
                    Object.keys(output).length === 0 &&
                    destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== "undefined") {
                target[key] = value;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
exports.dedupArray = dedupArray;
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === "object") {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
exports.sorted = sorted;
function formatAsPythonModule(name) {
    return name.replace(/-/g, "_").replace(/\./g, "_");
}
exports.formatAsPythonModule = formatAsPythonModule;
/**
 * Extract git version number from command line
 *
 * @param gitVersionOutput the output from `git version` CLI
 * @returns the version of git
 */
function getGitVersion(gitVersionOutput) {
    const match = gitVersionOutput.match(/\d+.\d+.\d+/);
    if (!match) {
        throw new Error("Unable to retrieve git version");
    }
    return match[0];
}
exports.getGitVersion = getGitVersion;
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/gm, "-")] = v;
    }
    return result;
}
exports.kebabCaseKeys = kebabCaseKeys;
function snakeCaseKeys(obj, recursive = true, exclusiveForRecordKeys = []) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => snakeCaseKeys(v, recursive, exclusiveForRecordKeys));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = snakeCaseKeys(v, recursive, exclusiveForRecordKeys);
        }
        const modifiedKey = exclusiveForRecordKeys.length == 0 || exclusiveForRecordKeys.includes(k)
            ? Case.snake(k)
            : k;
        result[modifiedKey] = v;
    }
    return result;
}
exports.snakeCaseKeys = snakeCaseKeys;
async function tryReadFile(file) {
    if (!(0, fs_1.existsSync)(file)) {
        return "";
    }
    return fs_1.promises.readFile(file, "utf-8");
}
exports.tryReadFile = tryReadFile;
function tryReadFileSync(file) {
    if (!(0, fs_1.existsSync)(file)) {
        return undefined;
    }
    return (0, fs_1.readFileSync)(file, "utf-8");
}
exports.tryReadFileSync = tryReadFileSync;
function isWritable(file) {
    try {
        (0, fs_1.accessSync)(file, fs_1.constants.W_OK);
        return true;
    }
    catch {
        return false;
    }
}
exports.isWritable = isWritable;
/**
 * Asserts that the file should be executable. Always returns true on Windows.
 *
 * In Windows, the executable attribute is stored in the system setting PATHEXT, not in each file. Then, checking for executability is equivalent to checking for existence. To bypass checking for executability, we always return true on Windows.
 *
 * @param filePath The path to the file
 * @param shouldBeExecutable Whether the file should be executable
 * @returns true if `filePath` executable attribute matches `shouldBeExecutable` or if the platform is Windows, false otherwise
 */
function assertExecutablePermissions(filePath, shouldBeExecutable) {
    const isWindows = process.platform === "win32";
    if (isWindows) {
        return true;
    }
    const prevExecutable = isExecutable(filePath);
    return prevExecutable === shouldBeExecutable;
}
exports.assertExecutablePermissions = assertExecutablePermissions;
function isExecutable(file) {
    try {
        (0, fs_1.accessSync)(file, fs_1.constants.X_OK);
        return true;
    }
    catch (e) {
        return false;
    }
}
exports.isExecutable = isExecutable;
function decamelize(s, sep = "_") {
    if (Case.of(s) === "camel") {
        return Case.lower(s, sep);
    }
    else {
        return s;
    }
}
function getNodeMajorVersion() {
    const match = process.version.match(/(\d+)\.(\d+)\.(\d+)/);
    if (match) {
        const [major] = match.slice(1).map((x) => parseInt(x));
        return major;
    }
    return undefined;
}
exports.getNodeMajorVersion = getNodeMajorVersion;
function anySelected(options) {
    return options.some((opt) => opt);
}
exports.anySelected = anySelected;
function multipleSelected(options) {
    return options.filter((opt) => opt).length > 1;
}
exports.multipleSelected = multipleSelected;
/**
 * Checks if a path is a FS root
 *
 * Optional uses a provided OS specific path implementation,
 * defaults to use the implementation for the current OS.
 *
 * @internal
 */
function isRoot(dir, osPathLib = path) {
    const parent = osPathLib.dirname(dir);
    return parent === dir;
}
exports.isRoot = isRoot;
/**
 * Run up project tree to find a file or directory
 *
 * @param lookFor the file or directory to look for
 * @param cwd current working directory, must be an absolute path
 * @returns path to the file or directory we are looking for, undefined if not found
 */
function findUp(lookFor, cwd = process.cwd()) {
    if ((0, fs_1.existsSync)(path.join(cwd, lookFor))) {
        return cwd;
    }
    if (isRoot(cwd)) {
        // This is a root
        return undefined;
    }
    return findUp(lookFor, path.dirname(cwd));
}
exports.findUp = findUp;
/**
 * Normalizes a path that is going to be persisted to have a cross platform representation.
 *
 * Normalized paths can be persisted and doesn't need to be modified when the platform changes.
 * `normalizePersistedPath` takes care of platform-specific properties like directory separator.
 * It uses `path.posix.sep` that is supported both in Windows and Unix platforms.
 *
 *
 * @param p the path to be normalized
 * @returns the normalized path
 */
function normalizePersistedPath(p) {
    return p.replace(/\\/g, path.posix.sep);
}
exports.normalizePersistedPath = normalizePersistedPath;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy91dGlsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtDQUErQztBQUMvQywyQkFTWTtBQUNaLDZCQUE2QjtBQUM3Qiw2QkFBNkI7QUFDN0IscUNBQXFDO0FBRXJDLE1BQU0sVUFBVSxHQUFHLEVBQUUsR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDO0FBRXBDOztHQUVHO0FBQ0gsU0FBZ0IsSUFBSSxDQUNsQixPQUFlLEVBQ2YsT0FBc0Q7SUFFdEQsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN2QixhQUFhLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtRQUM5QixLQUFLLEVBQUUsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxFQUFFLE1BQU0sQ0FBQztRQUM3QixTQUFTLEVBQUUsVUFBVTtRQUNyQixHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7UUFDaEIsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO0tBQ2pCLENBQUMsQ0FBQztBQUNMLENBQUM7QUFYRCxvQkFXQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsV0FBVyxDQUFDLE9BQWUsRUFBRSxPQUF3QjtJQUNuRSxPQUFPLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZCLE9BQU8sYUFBYSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7UUFDckMsS0FBSyxFQUFFLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxNQUFNLENBQUM7UUFDbEMsU0FBUyxFQUFFLFVBQVU7UUFDckIsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO0tBQ2pCLENBQUMsQ0FBQztBQUNMLENBQUM7QUFQRCxrQ0FPQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsZUFBZSxDQUM3QixPQUFlLEVBQ2YsT0FBd0I7SUFFeEIsSUFBSTtRQUNGLE1BQU0sS0FBSyxHQUFHLGFBQWE7YUFDeEIsUUFBUSxDQUFDLE9BQU8sRUFBRTtZQUNqQixLQUFLLEVBQUUsQ0FBQyxTQUFTLEVBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBQztZQUNsQyxTQUFTLEVBQUUsVUFBVTtZQUNyQixHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7U0FDakIsQ0FBQzthQUNELFFBQVEsQ0FBQyxPQUFPLENBQUM7YUFDakIsSUFBSSxFQUFFLENBQUM7UUFFVixJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1YsT0FBTyxTQUFTLENBQUM7U0FDbEIsQ0FBQywyQ0FBMkM7UUFDN0MsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUFDLE1BQU07UUFDTixPQUFPLFNBQVMsQ0FBQztLQUNsQjtBQUNILENBQUM7QUFyQkQsMENBcUJDO0FBa0JELFNBQWdCLGtCQUFrQixDQUFDLE9BQXlCO0lBQzFELE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLElBQUksS0FBSyxDQUFDO0lBQzNDLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDO0lBQy9DLElBQUksUUFBUSxJQUFJLFVBQVUsRUFBRTtRQUMxQixPQUFPLEtBQUssQ0FBQztLQUNkO1NBQU0sSUFBSSxRQUFRLEVBQUU7UUFDbkIsT0FBTyxLQUFLLENBQUM7S0FDZDtTQUFNLElBQUksVUFBVSxFQUFFO1FBQ3JCLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7U0FBTTtRQUNMLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7QUFDSCxDQUFDO0FBWkQsZ0RBWUM7QUFFRCxTQUFnQixTQUFTLENBQ3ZCLFFBQWdCLEVBQ2hCLElBQVMsRUFDVCxVQUE0QixFQUFFO0lBRTlCLElBQUksSUFBQSxlQUFVLEVBQUMsUUFBUSxDQUFDLEVBQUU7UUFDeEIsSUFBQSxjQUFTLEVBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQzVCO0lBRUQsSUFBQSxjQUFTLEVBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZELElBQUEsa0JBQWEsRUFBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFFOUIsSUFBQSxjQUFTLEVBQUMsUUFBUSxFQUFFLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7QUFDbkQsQ0FBQztBQWJELDhCQWFDO0FBZ0NELFNBQWdCLHlCQUF5QixDQUN2QyxLQUFVLEVBQ1YsR0FBa0M7SUFFbEMsTUFBTSxzQkFBc0IsR0FBRyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUM7SUFDMUMsTUFBTSxnQkFBZ0IsR0FBRyxHQUFHLEVBQUUsZ0JBQWdCLElBQUksc0JBQXNCLENBQUM7SUFDekUsTUFBTSxTQUFTLEdBQUcsR0FBRyxFQUFFLFNBQVMsSUFBSSxHQUFHLENBQUM7SUFDeEMsTUFBTSxLQUFLLEdBQUcsR0FBRyxFQUFFLElBQUksSUFBSSxFQUFFLENBQUM7SUFDOUIsTUFBTSxRQUFRLEdBQUcsR0FBRyxFQUFFLFFBQVEsSUFBSSxFQUFFLENBQUM7SUFFckMsSUFBSSxLQUFLLENBQUMsTUFBTSxHQUFHLFFBQVEsRUFBRTtRQUMzQixNQUFNLElBQUksS0FBSyxDQUNiLHlFQUF5RSxDQUMxRSxDQUFDO0tBQ0g7SUFFRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7UUFDeEIsT0FBTyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQ3hCLHlCQUF5QixDQUFDLENBQUMsRUFBRTtZQUMzQixHQUFHLEdBQUc7WUFDTixJQUFJLEVBQUUsQ0FBQyxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7U0FDL0IsQ0FBQyxDQUNILENBQUM7S0FDSDtJQUVELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxJQUFJLEtBQUssS0FBSyxJQUFJLEVBQUU7UUFDL0MsTUFBTSxZQUFZLEdBQXdCLEVBQUUsQ0FBQztRQUM3QyxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUNoRCxNQUFNLGNBQWMsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLEdBQUcsS0FBSyxFQUFFLEdBQUcsQ0FBQyxFQUFFLEtBQUssQ0FBQztnQkFDN0QsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDO2dCQUM1QixDQUFDLENBQUMsR0FBRyxDQUFDO1lBRVIsWUFBWSxDQUFDLGNBQWMsQ0FBQyxHQUFHLHlCQUF5QixDQUFDLEtBQUssRUFBRTtnQkFDOUQsR0FBRyxHQUFHO2dCQUNOLElBQUksRUFBRSxDQUFDLEdBQUcsS0FBSyxFQUFFLEdBQUcsQ0FBQzthQUN0QixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sWUFBWSxDQUFDO0tBQ3JCO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBMUNELDhEQTBDQztBQUVEOzs7R0FHRztBQUNILFNBQWdCLFFBQVEsQ0FBQyxLQUF5QjtJQUNoRCxPQUFPLENBQUMsQ0FDTixLQUFLLEtBQUssU0FBUztRQUNuQixDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FDNUUsQ0FBQztBQUNKLENBQUM7QUFMRCw0QkFLQztBQU9EOzs7OztHQUtHO0FBQ0gsU0FBZ0IsUUFBUSxDQUFDLENBQU07SUFDN0IsT0FBTyxDQUNMLENBQUMsS0FBSyxJQUFJO1FBQ1YsT0FBTyxDQUFDLEtBQUssUUFBUTtRQUNyQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ2pCLENBQUMsQ0FBQyxXQUFXLENBQUMsSUFBSSxLQUFLLFFBQVEsQ0FDaEMsQ0FBQztBQUNKLENBQUM7QUFQRCw0QkFPQztBQUVEOzs7Ozs7Ozs7R0FTRztBQUNILFNBQWdCLFNBQVMsQ0FDdkIsT0FBb0MsRUFDcEMsY0FBdUIsS0FBSztJQUU1QixTQUFTLFFBQVEsQ0FBQyxNQUFnQixFQUFFLE1BQWdCO1FBQ2xELEtBQUssTUFBTSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNyQyxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7WUFFMUIsSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ25CLG1FQUFtRTtnQkFDbkUsMENBQTBDO2dCQUMxQyxJQUFJLE9BQU8sTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFLLFFBQVEsRUFBRTtvQkFDbkMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztpQkFDckI7Z0JBRUQsSUFBSSxXQUFXLElBQUksS0FBSyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFFO29CQUMxRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUU7d0JBQzlCLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUM7cUJBQ3RDO3lCQUFNO3dCQUNMLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO3FCQUMvQjtpQkFDRjtnQkFFRCxRQUFRLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO2dCQUU3QixrRUFBa0U7Z0JBQ2xFLDhEQUE4RDtnQkFDOUQsaUVBQWlFO2dCQUNqRSxNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQzNCLElBQ0UsT0FBTyxNQUFNLEtBQUssUUFBUTtvQkFDMUIsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQztvQkFDaEMsV0FBVyxFQUNYO29CQUNBLE9BQU8sTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2lCQUNwQjthQUNGO2lCQUFNLElBQUksS0FBSyxLQUFLLFNBQVMsSUFBSSxXQUFXLEVBQUU7Z0JBQzdDLE9BQU8sTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2FBQ3BCO2lCQUFNLElBQUksT0FBTyxLQUFLLEtBQUssV0FBVyxFQUFFO2dCQUN2QyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO2FBQ3JCO1NBQ0Y7SUFDSCxDQUFDO0lBRUQsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBb0IsQ0FBQztJQUVuRSxJQUFJLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3ZCLE9BQU8sRUFBRSxDQUFDO0tBQ1g7SUFDRCxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUVwQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDakQsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBckRELDhCQXFEQztBQUVEOzs7R0FHRztBQUNILFNBQWdCLFVBQVUsQ0FBSSxLQUFVO0lBQ3RDLE9BQU8sS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsRUFBRSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssR0FBRyxDQUFDLENBQUM7QUFDaEUsQ0FBQztBQUZELGdDQUVDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixNQUFNLENBQUksQ0FBSTtJQUM1QixJQUFJLENBQUMsSUFBSSxJQUFJLEVBQUU7UUFDYixPQUFPLFNBQVMsQ0FBQztLQUNsQjtJQUNELElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtRQUNwQixJQUFJLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ2xCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBQ0QsT0FBUSxDQUFlLENBQUMsSUFBSSxFQUFFLENBQUM7S0FDaEM7U0FBTSxJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsRUFBRTtRQUNoQyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMvQixPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUNELE1BQU0sTUFBTSxHQUE0QixFQUFFLENBQUM7UUFDM0MsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FDN0QsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FDbkIsRUFBRTtZQUNELE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7U0FDckI7UUFDRCxPQUFPLE1BQVcsQ0FBQztLQUNwQjtTQUFNO1FBQ0wsT0FBTyxDQUFDLENBQUM7S0FDVjtBQUNILENBQUM7QUF2QkQsd0JBdUJDO0FBRUQsU0FBZ0Isb0JBQW9CLENBQUMsSUFBWTtJQUMvQyxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUM7QUFDckQsQ0FBQztBQUZELG9EQUVDO0FBRUQ7Ozs7O0dBS0c7QUFDSCxTQUFnQixhQUFhLENBQUMsZ0JBQXdCO0lBQ3BELE1BQU0sS0FBSyxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNwRCxJQUFJLENBQUMsS0FBSyxFQUFFO1FBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0tBQ25EO0lBRUQsT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDbEIsQ0FBQztBQVBELHNDQU9DO0FBRUQsU0FBZ0IsYUFBYSxDQUFjLEdBQU0sRUFBRSxTQUFTLEdBQUcsSUFBSTtJQUNqRSxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxHQUFHLElBQUksSUFBSSxFQUFFO1FBQzFDLE9BQU8sR0FBRyxDQUFDO0tBQ1o7SUFFRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDdEIsSUFBSSxTQUFTLEVBQUU7WUFDYixHQUFHLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBUSxDQUFDO1NBQzFEO1FBQ0QsT0FBTyxHQUFHLENBQUM7S0FDWjtJQUVELE1BQU0sTUFBTSxHQUE0QixFQUFFLENBQUM7SUFDM0MsS0FBSyxJQUFJLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDdEMsSUFBSSxTQUFTLEVBQUU7WUFDYixDQUFDLEdBQUcsYUFBYSxDQUFDLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQztTQUNqQztRQUNELE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUMvQztJQUNELE9BQU8sTUFBYSxDQUFDO0FBQ3ZCLENBQUM7QUFwQkQsc0NBb0JDO0FBRUQsU0FBZ0IsYUFBYSxDQUMzQixHQUFNLEVBQ04sU0FBUyxHQUFHLElBQUksRUFDaEIseUJBQW1DLEVBQUU7SUFFckMsSUFBSSxPQUFPLEdBQUcsS0FBSyxRQUFRLElBQUksR0FBRyxJQUFJLElBQUksRUFBRTtRQUMxQyxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3RCLElBQUksU0FBUyxFQUFFO1lBQ2IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUNsQixhQUFhLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUM3QyxDQUFDO1NBQ1Y7UUFDRCxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQsTUFBTSxNQUFNLEdBQTRCLEVBQUUsQ0FBQztJQUMzQyxLQUFLLElBQUksQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUN0QyxJQUFJLFNBQVMsRUFBRTtZQUNiLENBQUMsR0FBRyxhQUFhLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1NBQ3pEO1FBQ0QsTUFBTSxXQUFXLEdBQ2Ysc0JBQXNCLENBQUMsTUFBTSxJQUFJLENBQUMsSUFBSSxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ3RFLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUNmLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDUixNQUFNLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQ3pCO0lBQ0QsT0FBTyxNQUFhLENBQUM7QUFDdkIsQ0FBQztBQTlCRCxzQ0E4QkM7QUFFTSxLQUFLLFVBQVUsV0FBVyxDQUFDLElBQVk7SUFDNUMsSUFBSSxDQUFDLElBQUEsZUFBVSxFQUFDLElBQUksQ0FBQyxFQUFFO1FBQ3JCLE9BQU8sRUFBRSxDQUFDO0tBQ1g7SUFFRCxPQUFPLGFBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0FBQ3BDLENBQUM7QUFORCxrQ0FNQztBQUVELFNBQWdCLGVBQWUsQ0FBQyxJQUFZO0lBQzFDLElBQUksQ0FBQyxJQUFBLGVBQVUsRUFBQyxJQUFJLENBQUMsRUFBRTtRQUNyQixPQUFPLFNBQVMsQ0FBQztLQUNsQjtJQUVELE9BQU8sSUFBQSxpQkFBWSxFQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztBQUNyQyxDQUFDO0FBTkQsMENBTUM7QUFFRCxTQUFnQixVQUFVLENBQUMsSUFBWTtJQUNyQyxJQUFJO1FBQ0YsSUFBQSxlQUFVLEVBQUMsSUFBSSxFQUFFLGNBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztLQUNiO0lBQUMsTUFBTTtRQUNOLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7QUFDSCxDQUFDO0FBUEQsZ0NBT0M7QUFFRDs7Ozs7Ozs7R0FRRztBQUNILFNBQWdCLDJCQUEyQixDQUN6QyxRQUFnQixFQUNoQixrQkFBMkI7SUFFM0IsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLFFBQVEsS0FBSyxPQUFPLENBQUM7SUFDL0MsSUFBSSxTQUFTLEVBQUU7UUFDYixPQUFPLElBQUksQ0FBQztLQUNiO0lBRUQsTUFBTSxjQUFjLEdBQUcsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBRTlDLE9BQU8sY0FBYyxLQUFLLGtCQUFrQixDQUFDO0FBQy9DLENBQUM7QUFaRCxrRUFZQztBQUVELFNBQWdCLFlBQVksQ0FBQyxJQUFZO0lBQ3ZDLElBQUk7UUFDRixJQUFBLGVBQVUsRUFBQyxJQUFJLEVBQUUsY0FBWSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXBDLE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7QUFDSCxDQUFDO0FBUkQsb0NBUUM7QUFFRCxTQUFTLFVBQVUsQ0FBQyxDQUFTLEVBQUUsTUFBYyxHQUFHO0lBQzlDLElBQUksSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxPQUFPLEVBQUU7UUFDMUIsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztLQUMzQjtTQUFNO1FBQ0wsT0FBTyxDQUFDLENBQUM7S0FDVjtBQUNILENBQUM7QUFFRCxTQUFnQixtQkFBbUI7SUFDakMsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUMzRCxJQUFJLEtBQUssRUFBRTtRQUNULE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdkQsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUNELE9BQU8sU0FBUyxDQUFDO0FBQ25CLENBQUM7QUFQRCxrREFPQztBQUVELFNBQWdCLFdBQVcsQ0FBQyxPQUFnQztJQUMxRCxPQUFPLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBQ3BDLENBQUM7QUFGRCxrQ0FFQztBQUVELFNBQWdCLGdCQUFnQixDQUFDLE9BQWdDO0lBQy9ELE9BQU8sT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztBQUNqRCxDQUFDO0FBRkQsNENBRUM7QUFFRDs7Ozs7OztHQU9HO0FBQ0gsU0FBZ0IsTUFBTSxDQUFDLEdBQVcsRUFBRSxZQUF5QixJQUFJO0lBQy9ELE1BQU0sTUFBTSxHQUFHLFNBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEMsT0FBTyxNQUFNLEtBQUssR0FBRyxDQUFDO0FBQ3hCLENBQUM7QUFIRCx3QkFHQztBQUVEOzs7Ozs7R0FNRztBQUNILFNBQWdCLE1BQU0sQ0FDcEIsT0FBZSxFQUNmLE1BQWMsT0FBTyxDQUFDLEdBQUcsRUFBRTtJQUUzQixJQUFJLElBQUEsZUFBVSxFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDLEVBQUU7UUFDdkMsT0FBTyxHQUFHLENBQUM7S0FDWjtJQUVELElBQUksTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ2YsaUJBQWlCO1FBQ2pCLE9BQU8sU0FBUyxDQUFDO0tBQ2xCO0lBQ0QsT0FBTyxNQUFNLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztBQUM1QyxDQUFDO0FBYkQsd0JBYUM7QUFFRDs7Ozs7Ozs7OztHQVVHO0FBQ0gsU0FBZ0Isc0JBQXNCLENBQUMsQ0FBUztJQUM5QyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDMUMsQ0FBQztBQUZELHdEQUVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2hpbGRfcHJvY2VzcyBmcm9tIFwiY2hpbGRfcHJvY2Vzc1wiO1xuaW1wb3J0IHtcbiAgYWNjZXNzU3luYyxcbiAgY2htb2RTeW5jLFxuICBjb25zdGFudHMgYXMgZnNfY29uc3RhbnRzLFxuICBleGlzdHNTeW5jLFxuICBta2RpclN5bmMsXG4gIHByb21pc2VzIGFzIGZzLFxuICByZWFkRmlsZVN5bmMsXG4gIHdyaXRlRmlsZVN5bmMsXG59IGZyb20gXCJmc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0ICogYXMgQ2FzZSBmcm9tIFwiY2FzZVwiO1xuaW1wb3J0ICogYXMgbG9nZ2luZyBmcm9tIFwiLi9sb2dnaW5nXCI7XG5cbmNvbnN0IE1BWF9CVUZGRVIgPSAxMCAqIDEwMjQgKiAxMDI0O1xuXG4vKipcbiAqIEV4ZWN1dGVzIGEgY29tbWFuZCB3aXRoIFNURE9VVCA+IFNUREVSUi5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGV4ZWMoXG4gIGNvbW1hbmQ6IHN0cmluZyxcbiAgb3B0aW9uczogeyBjd2Q6IHN0cmluZzsgZW52PzogUmVjb3JkPHN0cmluZywgc3RyaW5nPiB9XG4pOiB2b2lkIHtcbiAgbG9nZ2luZy5kZWJ1Zyhjb21tYW5kKTtcbiAgY2hpbGRfcHJvY2Vzcy5leGVjU3luYyhjb21tYW5kLCB7XG4gICAgc3RkaW86IFtcImluaGVyaXRcIiwgMiwgXCJwaXBlXCJdLCAvLyBcInBpcGVcIiBmb3IgU1RERVJSIG1lYW5zIGl0IGFwcGVhcnMgaW4gZXhjZXB0aW9uc1xuICAgIG1heEJ1ZmZlcjogTUFYX0JVRkZFUixcbiAgICBjd2Q6IG9wdGlvbnMuY3dkLFxuICAgIGVudjogb3B0aW9ucy5lbnYsXG4gIH0pO1xufVxuXG4vKipcbiAqIEV4ZWN1dGVzIGNvbW1hbmQgYW5kIHJldHVybnMgU1RET1VULiBJZiB0aGUgY29tbWFuZCBmYWlscyAobm9uLXplcm8pLCB0aHJvd3MgYW4gZXJyb3IuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBleGVjQ2FwdHVyZShjb21tYW5kOiBzdHJpbmcsIG9wdGlvbnM6IHsgY3dkOiBzdHJpbmcgfSkge1xuICBsb2dnaW5nLmRlYnVnKGNvbW1hbmQpO1xuICByZXR1cm4gY2hpbGRfcHJvY2Vzcy5leGVjU3luYyhjb21tYW5kLCB7XG4gICAgc3RkaW86IFtcImluaGVyaXRcIiwgXCJwaXBlXCIsIFwicGlwZVwiXSwgLy8gXCJwaXBlXCIgZm9yIFNUREVSUiBtZWFucyBpdCBhcHBlYXJzIGluIGV4Y2VwdGlvbnNcbiAgICBtYXhCdWZmZXI6IE1BWF9CVUZGRVIsXG4gICAgY3dkOiBvcHRpb25zLmN3ZCxcbiAgfSk7XG59XG5cbi8qKlxuICogRXhlY3V0ZXMgYGNvbW1hbmRgIGFuZCByZXR1cm5zIGl0cyB2YWx1ZSBvciB1bmRlZmluZWQgaWYgdGhlIGNvbW1hbmQgZmFpbGVkLlxuICovXG5leHBvcnQgZnVuY3Rpb24gZXhlY09yVW5kZWZpbmVkKFxuICBjb21tYW5kOiBzdHJpbmcsXG4gIG9wdGlvbnM6IHsgY3dkOiBzdHJpbmcgfVxuKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgdHJ5IHtcbiAgICBjb25zdCB2YWx1ZSA9IGNoaWxkX3Byb2Nlc3NcbiAgICAgIC5leGVjU3luYyhjb21tYW5kLCB7XG4gICAgICAgIHN0ZGlvOiBbXCJpbmhlcml0XCIsIFwicGlwZVwiLCBcInBpcGVcIl0sIC8vIFwicGlwZVwiIGZvciBTVERFUlIgbWVhbnMgaXQgYXBwZWFycyBpbiBleGNlcHRpb25zXG4gICAgICAgIG1heEJ1ZmZlcjogTUFYX0JVRkZFUixcbiAgICAgICAgY3dkOiBvcHRpb25zLmN3ZCxcbiAgICAgIH0pXG4gICAgICAudG9TdHJpbmcoXCJ1dGYtOFwiKVxuICAgICAgLnRyaW0oKTtcblxuICAgIGlmICghdmFsdWUpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfSAvLyBhbiBlbXB0eSBzdHJpbmcgaXMgdGhlIHNhbWUgYXMgdW5kZWZpbmVkXG4gICAgcmV0dXJuIHZhbHVlO1xuICB9IGNhdGNoIHtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgV3JpdGVGaWxlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBnZW5lcmF0ZWQgZmlsZSBzaG91bGQgYmUgbWFya2VkIGFzIGV4ZWN1dGFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICBleGVjdXRhYmxlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgZ2VuZXJhdGVkIGZpbGUgc2hvdWxkIGJlIHJlYWRvbmx5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHk/OiBib29sZWFuO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0RmlsZVBlcm1pc3Npb25zKG9wdGlvbnM6IFdyaXRlRmlsZU9wdGlvbnMpOiBzdHJpbmcge1xuICBjb25zdCByZWFkb25seSA9IG9wdGlvbnMucmVhZG9ubHkgPz8gZmFsc2U7XG4gIGNvbnN0IGV4ZWN1dGFibGUgPSBvcHRpb25zLmV4ZWN1dGFibGUgPz8gZmFsc2U7XG4gIGlmIChyZWFkb25seSAmJiBleGVjdXRhYmxlKSB7XG4gICAgcmV0dXJuIFwiNTQ0XCI7XG4gIH0gZWxzZSBpZiAocmVhZG9ubHkpIHtcbiAgICByZXR1cm4gXCI0NDRcIjtcbiAgfSBlbHNlIGlmIChleGVjdXRhYmxlKSB7XG4gICAgcmV0dXJuIFwiNzU1XCI7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIFwiNjQ0XCI7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHdyaXRlRmlsZShcbiAgZmlsZVBhdGg6IHN0cmluZyxcbiAgZGF0YTogYW55LFxuICBvcHRpb25zOiBXcml0ZUZpbGVPcHRpb25zID0ge31cbikge1xuICBpZiAoZXhpc3RzU3luYyhmaWxlUGF0aCkpIHtcbiAgICBjaG1vZFN5bmMoZmlsZVBhdGgsIFwiNjAwXCIpO1xuICB9XG5cbiAgbWtkaXJTeW5jKHBhdGguZGlybmFtZShmaWxlUGF0aCksIHsgcmVjdXJzaXZlOiB0cnVlIH0pO1xuICB3cml0ZUZpbGVTeW5jKGZpbGVQYXRoLCBkYXRhKTtcblxuICBjaG1vZFN5bmMoZmlsZVBhdGgsIGdldEZpbGVQZXJtaXNzaW9ucyhvcHRpb25zKSk7XG59XG5cbi8qKlxuICogRGVjYW1lbGl6ZXMgdGhlIGtleXMgb2YgYW4gb2JqZWN0IHN0cnVjdHVyZSwgcmVjdXJzaW5nIHRocm91Z2ggY2hpbGQgb2JqZWN0cyBhbmQgYXJyYXlzLlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIERlY2FtZWxpemVSZWN1cnNpdmVseU9wdGlvbnMge1xuICAvKipcbiAgICogTWF4IGRlcHRoIHRvIHJlY3Vyc2UgYmVmb3JlIGVycm9yaW5nLlxuICAgKiBAZGVmYXVsdCAxMFxuICAgKi9cbiAgbWF4RGVwdGg/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdHJ1ZSB3aGVuIGEga2V5IHNob3VsZCBiZSBkZWNhbWVsaXplZFxuICAgKiBAZGVmYXVsdCAtIGFsbCBrZXlzIGFyZSBkZWNhbWVsaXplZFxuICAgKi9cbiAgc2hvdWxkRGVjYW1lbGl6ZT86IChwYXRoOiBzdHJpbmdbXSwgdmFsdWU6IGFueSkgPT4gYm9vbGVhbjtcblxuICAvKipcbiAgICogU2VwYXJhdG9yIGZvciBkZWNhbWVsaXppbmcuXG4gICAqIEBkZWZhdWx0IFwiX1wiXG4gICAqL1xuICBzZXBhcmF0b3I/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEN1cnJlbnQgcGF0aC5cbiAgICogQGludGVybmFsXG4gICAqL1xuICBwYXRoPzogc3RyaW5nW107XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBkZWNhbWVsaXplS2V5c1JlY3Vyc2l2ZWx5KFxuICBpbnB1dDogYW55LFxuICBvcHQ/OiBEZWNhbWVsaXplUmVjdXJzaXZlbHlPcHRpb25zXG4pOiBhbnkge1xuICBjb25zdCBzaG91bGRBbHdheXNEZWNhbWVsaXplID0gKCkgPT4gdHJ1ZTtcbiAgY29uc3Qgc2hvdWxkRGVjYW1lbGl6ZSA9IG9wdD8uc2hvdWxkRGVjYW1lbGl6ZSA/PyBzaG91bGRBbHdheXNEZWNhbWVsaXplO1xuICBjb25zdCBzZXBhcmF0b3IgPSBvcHQ/LnNlcGFyYXRvciA/PyBcIl9cIjtcbiAgY29uc3QgcGF0aF8gPSBvcHQ/LnBhdGggPz8gW107XG4gIGNvbnN0IG1heERlcHRoID0gb3B0Py5tYXhEZXB0aCA/PyAxMDtcblxuICBpZiAocGF0aF8ubGVuZ3RoID4gbWF4RGVwdGgpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBcIkRlY2FtZWxsZWQgdG9vIGRlZXBseSAtIGNoZWNrIHRoYXQgdGhlIGlucHV0IGhhcyBubyBjaXJjdWxhciByZWZlcmVuY2VzXCJcbiAgICApO1xuICB9XG5cbiAgaWYgKEFycmF5LmlzQXJyYXkoaW5wdXQpKSB7XG4gICAgcmV0dXJuIGlucHV0Lm1hcCgoaywgaSkgPT5cbiAgICAgIGRlY2FtZWxpemVLZXlzUmVjdXJzaXZlbHkoaywge1xuICAgICAgICAuLi5vcHQsXG4gICAgICAgIHBhdGg6IFsuLi5wYXRoXywgaS50b1N0cmluZygpXSxcbiAgICAgIH0pXG4gICAgKTtcbiAgfVxuXG4gIGlmICh0eXBlb2YgaW5wdXQgPT09IFwib2JqZWN0XCIgJiYgaW5wdXQgIT09IG51bGwpIHtcbiAgICBjb25zdCBtYXBwZWRPYmplY3Q6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fTtcbiAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhpbnB1dCkpIHtcbiAgICAgIGNvbnN0IHRyYW5zZm9ybWVkS2V5ID0gc2hvdWxkRGVjYW1lbGl6ZShbLi4ucGF0aF8sIGtleV0sIHZhbHVlKVxuICAgICAgICA/IGRlY2FtZWxpemUoa2V5LCBzZXBhcmF0b3IpXG4gICAgICAgIDoga2V5O1xuXG4gICAgICBtYXBwZWRPYmplY3RbdHJhbnNmb3JtZWRLZXldID0gZGVjYW1lbGl6ZUtleXNSZWN1cnNpdmVseSh2YWx1ZSwge1xuICAgICAgICAuLi5vcHQsXG4gICAgICAgIHBhdGg6IFsuLi5wYXRoXywga2V5XSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBtYXBwZWRPYmplY3Q7XG4gIH1cblxuICByZXR1cm4gaW5wdXQ7XG59XG5cbi8qKlxuICogUmV0dXJucyBmYWxzZSBpZiB2YWx1ZSBpcyB1bnNldCBvciBhIGZhbHNleSB2YWx1ZSwgYW5kIHRydWUgb3RoZXJ3aXNlLlxuICogQHBhcmFtIHZhbHVlIGFuIGVudmlyb25tZW50IHZhcmlhYmxlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc1RydXRoeSh2YWx1ZTogc3RyaW5nIHwgdW5kZWZpbmVkKTogYm9vbGVhbiB7XG4gIHJldHVybiAhKFxuICAgIHZhbHVlID09PSB1bmRlZmluZWQgfHxcbiAgICBbXCJudWxsXCIsIFwidW5kZWZpbmVkXCIsIFwiMFwiLCBcImZhbHNlXCIsIFwiXCJdLmluY2x1ZGVzKHZhbHVlLnRvTG9jYWxlTG93ZXJDYXNlKCkpXG4gICk7XG59XG5cbi8qKlxuICogVHlwZSBvZiBhIG1hcCBtYXBwaW5nIHN0cmluZ3MgdG8gc29tZSBhcmJpdHJhcnkgdHlwZVxuICovXG5leHBvcnQgdHlwZSBPYmo8VD4gPSB7IFtrZXk6IHN0cmluZ106IFQgfTtcblxuLyoqXG4gKiBSZXR1cm4gd2hldGhlciB0aGUgZ2l2ZW4gdmFsdWUgaXMgYW4gb2JqZWN0XG4gKlxuICogRXZlbiB0aG91Z2ggYXJyYXlzIGFuZCBpbnN0YW5jZXMgb2YgY2xhc3NlcyB0ZWNobmljYWxseSBhcmUgb2JqZWN0cywgd2VcbiAqIHVzdWFsbHkgd2FudCB0byB0cmVhdCB0aGVtIGRpZmZlcmVudGx5LCBzbyB3ZSByZXR1cm4gZmFsc2UgaW4gdGhvc2UgY2FzZXMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc09iamVjdCh4OiBhbnkpOiB4IGlzIE9iajxhbnk+IHtcbiAgcmV0dXJuIChcbiAgICB4ICE9PSBudWxsICYmXG4gICAgdHlwZW9mIHggPT09IFwib2JqZWN0XCIgJiZcbiAgICAhQXJyYXkuaXNBcnJheSh4KSAmJlxuICAgIHguY29uc3RydWN0b3IubmFtZSA9PT0gXCJPYmplY3RcIlxuICApO1xufVxuXG4vKipcbiAqIFJlY3Vyc2l2ZWx5IG1lcmdlIG9iamVjdHMgdG9nZXRoZXJcbiAqXG4gKiBUaGUgbGVmdG1vc3Qgb2JqZWN0IGlzIG11dGF0ZWQgYW5kIHJldHVybmVkLiBBcnJheXMgYXJlIG5vdCBtZXJnZWRcbiAqIGJ1dCBvdmVyd3JpdHRlbiBqdXN0IGxpa2Ugc2NhbGFycy5cbiAqXG4gKiBJZiBhbiBvYmplY3QgaXMgbWVyZ2VkIGludG8gYSBub24tb2JqZWN0LCB0aGUgbm9uLW9iamVjdCBpcyBsb3N0LlxuICpcbiAqIGB1bmRlZmluZWRgcyB3aWxsIGNhdXNlIGEgdmFsdWUgdG8gYmUgZGVsZXRlZCBpZiBkZXN0cnVjdGl2ZSBpcyBlbmFibGVkLlxuICovXG5leHBvcnQgZnVuY3Rpb24gZGVlcE1lcmdlKFxuICBvYmplY3RzOiBBcnJheTxPYmo8YW55PiB8IHVuZGVmaW5lZD4sXG4gIGRlc3RydWN0aXZlOiBib29sZWFuID0gZmFsc2Vcbikge1xuICBmdW5jdGlvbiBtZXJnZU9uZSh0YXJnZXQ6IE9iajxhbnk+LCBzb3VyY2U6IE9iajxhbnk+KSB7XG4gICAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXMoc291cmNlKSkge1xuICAgICAgY29uc3QgdmFsdWUgPSBzb3VyY2Vba2V5XTtcblxuICAgICAgaWYgKGlzT2JqZWN0KHZhbHVlKSkge1xuICAgICAgICAvLyBpZiB0aGUgdmFsdWUgYXQgdGhlIHRhcmdldCBpcyBub3QgYW4gb2JqZWN0LCBvdmVycmlkZSBpdCB3aXRoIGFuXG4gICAgICAgIC8vIG9iamVjdCBzbyB3ZSBjYW4gY29udGludWUgdGhlIHJlY3Vyc2lvblxuICAgICAgICBpZiAodHlwZW9mIHRhcmdldFtrZXldICE9PSBcIm9iamVjdFwiKSB7XG4gICAgICAgICAgdGFyZ2V0W2tleV0gPSB2YWx1ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChcIl9fJEFQUEVORFwiIGluIHZhbHVlICYmIEFycmF5LmlzQXJyYXkodmFsdWUuX18kQVBQRU5EKSkge1xuICAgICAgICAgIGlmIChBcnJheS5pc0FycmF5KHRhcmdldFtrZXldKSkge1xuICAgICAgICAgICAgdGFyZ2V0W2tleV0ucHVzaCguLi52YWx1ZS5fXyRBUFBFTkQpO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB0YXJnZXRba2V5XSA9IHZhbHVlLl9fJEFQUEVORDtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICBtZXJnZU9uZSh0YXJnZXRba2V5XSwgdmFsdWUpO1xuXG4gICAgICAgIC8vIGlmIHRoZSByZXN1bHQgb2YgdGhlIG1lcmdlIGlzIGFuIGVtcHR5IG9iamVjdCwgaXQncyBiZWNhdXNlIHRoZVxuICAgICAgICAvLyBldmVudHVhbCB2YWx1ZSB3ZSBhc3NpZ25lZCBpcyBgdW5kZWZpbmVkYCwgYW5kIHRoZXJlIGFyZSBub1xuICAgICAgICAvLyBzaWJsaW5nIGNvbmNyZXRlIHZhbHVlcyBhbG9uZ3NpZGUsIHNvIHdlIGNhbiBkZWxldGUgdGhpcyB0cmVlLlxuICAgICAgICBjb25zdCBvdXRwdXQgPSB0YXJnZXRba2V5XTtcbiAgICAgICAgaWYgKFxuICAgICAgICAgIHR5cGVvZiBvdXRwdXQgPT09IFwib2JqZWN0XCIgJiZcbiAgICAgICAgICBPYmplY3Qua2V5cyhvdXRwdXQpLmxlbmd0aCA9PT0gMCAmJlxuICAgICAgICAgIGRlc3RydWN0aXZlXG4gICAgICAgICkge1xuICAgICAgICAgIGRlbGV0ZSB0YXJnZXRba2V5XTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIGlmICh2YWx1ZSA9PT0gdW5kZWZpbmVkICYmIGRlc3RydWN0aXZlKSB7XG4gICAgICAgIGRlbGV0ZSB0YXJnZXRba2V5XTtcbiAgICAgIH0gZWxzZSBpZiAodHlwZW9mIHZhbHVlICE9PSBcInVuZGVmaW5lZFwiKSB7XG4gICAgICAgIHRhcmdldFtrZXldID0gdmFsdWU7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgY29uc3Qgb3RoZXJzID0gb2JqZWN0cy5maWx0ZXIoKHgpID0+IHggIT0gbnVsbCkgYXMgQXJyYXk8T2JqPGFueT4+O1xuXG4gIGlmIChvdGhlcnMubGVuZ3RoID09PSAwKSB7XG4gICAgcmV0dXJuIHt9O1xuICB9XG4gIGNvbnN0IGludG8gPSBvdGhlcnMuc3BsaWNlKDAsIDEpWzBdO1xuXG4gIG90aGVycy5mb3JFYWNoKChvdGhlcikgPT4gbWVyZ2VPbmUoaW50bywgb3RoZXIpKTtcbiAgcmV0dXJuIGludG87XG59XG5cbi8qXG4gKiBEZWR1cGxpY2F0ZSB2YWx1ZXMgaW4gYSBsaXN0LCByZXR1cm5pbmcgYSBuZXcgYXJyYXkuXG4gKiBAcGFyYW0gYXJyYXkgbGlzdCBvZiB2YWx1ZXNcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGRlZHVwQXJyYXk8VD4oYXJyYXk6IFRbXSk6IFRbXSB7XG4gIHJldHVybiBhcnJheS5maWx0ZXIoKHZhbCwgaWR4KSA9PiBhcnJheS5pbmRleE9mKHZhbCkgPT09IGlkeCk7XG59XG5cbi8qKlxuICogUmV0dXJucyBhIHNvcnRlZCB2ZXJzaW9uIG9mIGB4YCBvciBgdW5kZWZpbmVkYCBpZiBpdCBpcyBhbiBlbXB0eSBhcnJheSBvciBvYmplY3QuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzb3J0ZWQ8VD4oeDogVCkge1xuICBpZiAoeCA9PSBudWxsKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuICBpZiAoQXJyYXkuaXNBcnJheSh4KSkge1xuICAgIGlmICh4Lmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgcmV0dXJuICh4IGFzIHVua25vd25bXSkuc29ydCgpO1xuICB9IGVsc2UgaWYgKHR5cGVvZiB4ID09PSBcIm9iamVjdFwiKSB7XG4gICAgaWYgKE9iamVjdC5rZXlzKHgpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgY29uc3QgcmVzdWx0OiBSZWNvcmQ8c3RyaW5nLCB1bmtub3duPiA9IHt9O1xuICAgIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHgpLnNvcnQoKFtsXSwgW3JdKSA9PlxuICAgICAgbC5sb2NhbGVDb21wYXJlKHIpXG4gICAgKSkge1xuICAgICAgcmVzdWx0W2tleV0gPSB2YWx1ZTtcbiAgICB9XG4gICAgcmV0dXJuIHJlc3VsdCBhcyBUO1xuICB9IGVsc2Uge1xuICAgIHJldHVybiB4O1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBmb3JtYXRBc1B5dGhvbk1vZHVsZShuYW1lOiBzdHJpbmcpIHtcbiAgcmV0dXJuIG5hbWUucmVwbGFjZSgvLS9nLCBcIl9cIikucmVwbGFjZSgvXFwuL2csIFwiX1wiKTtcbn1cblxuLyoqXG4gKiBFeHRyYWN0IGdpdCB2ZXJzaW9uIG51bWJlciBmcm9tIGNvbW1hbmQgbGluZVxuICpcbiAqIEBwYXJhbSBnaXRWZXJzaW9uT3V0cHV0IHRoZSBvdXRwdXQgZnJvbSBgZ2l0IHZlcnNpb25gIENMSVxuICogQHJldHVybnMgdGhlIHZlcnNpb24gb2YgZ2l0XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZXRHaXRWZXJzaW9uKGdpdFZlcnNpb25PdXRwdXQ6IHN0cmluZykge1xuICBjb25zdCBtYXRjaCA9IGdpdFZlcnNpb25PdXRwdXQubWF0Y2goL1xcZCsuXFxkKy5cXGQrLyk7XG4gIGlmICghbWF0Y2gpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXCJVbmFibGUgdG8gcmV0cmlldmUgZ2l0IHZlcnNpb25cIik7XG4gIH1cblxuICByZXR1cm4gbWF0Y2hbMF07XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBrZWJhYkNhc2VLZXlzPFQgPSB1bmtub3duPihvYmo6IFQsIHJlY3Vyc2l2ZSA9IHRydWUpOiBUIHtcbiAgaWYgKHR5cGVvZiBvYmogIT09IFwib2JqZWN0XCIgfHwgb2JqID09IG51bGwpIHtcbiAgICByZXR1cm4gb2JqO1xuICB9XG5cbiAgaWYgKEFycmF5LmlzQXJyYXkob2JqKSkge1xuICAgIGlmIChyZWN1cnNpdmUpIHtcbiAgICAgIG9iaiA9IG9iai5tYXAoKHYpID0+IGtlYmFiQ2FzZUtleXModiwgcmVjdXJzaXZlKSkgYXMgYW55O1xuICAgIH1cbiAgICByZXR1cm4gb2JqO1xuICB9XG5cbiAgY29uc3QgcmVzdWx0OiBSZWNvcmQ8c3RyaW5nLCB1bmtub3duPiA9IHt9O1xuICBmb3IgKGxldCBbaywgdl0gb2YgT2JqZWN0LmVudHJpZXMob2JqKSkge1xuICAgIGlmIChyZWN1cnNpdmUpIHtcbiAgICAgIHYgPSBrZWJhYkNhc2VLZXlzKHYsIHJlY3Vyc2l2ZSk7XG4gICAgfVxuICAgIHJlc3VsdFtkZWNhbWVsaXplKGspLnJlcGxhY2UoL18vZ20sIFwiLVwiKV0gPSB2O1xuICB9XG4gIHJldHVybiByZXN1bHQgYXMgYW55O1xufVxuXG5leHBvcnQgZnVuY3Rpb24gc25ha2VDYXNlS2V5czxUID0gdW5rbm93bj4oXG4gIG9iajogVCxcbiAgcmVjdXJzaXZlID0gdHJ1ZSxcbiAgZXhjbHVzaXZlRm9yUmVjb3JkS2V5czogc3RyaW5nW10gPSBbXVxuKTogVCB7XG4gIGlmICh0eXBlb2Ygb2JqICE9PSBcIm9iamVjdFwiIHx8IG9iaiA9PSBudWxsKSB7XG4gICAgcmV0dXJuIG9iajtcbiAgfVxuXG4gIGlmIChBcnJheS5pc0FycmF5KG9iaikpIHtcbiAgICBpZiAocmVjdXJzaXZlKSB7XG4gICAgICBvYmogPSBvYmoubWFwKCh2KSA9PlxuICAgICAgICBzbmFrZUNhc2VLZXlzKHYsIHJlY3Vyc2l2ZSwgZXhjbHVzaXZlRm9yUmVjb3JkS2V5cylcbiAgICAgICkgYXMgYW55O1xuICAgIH1cbiAgICByZXR1cm4gb2JqO1xuICB9XG5cbiAgY29uc3QgcmVzdWx0OiBSZWNvcmQ8c3RyaW5nLCB1bmtub3duPiA9IHt9O1xuICBmb3IgKGxldCBbaywgdl0gb2YgT2JqZWN0LmVudHJpZXMob2JqKSkge1xuICAgIGlmIChyZWN1cnNpdmUpIHtcbiAgICAgIHYgPSBzbmFrZUNhc2VLZXlzKHYsIHJlY3Vyc2l2ZSwgZXhjbHVzaXZlRm9yUmVjb3JkS2V5cyk7XG4gICAgfVxuICAgIGNvbnN0IG1vZGlmaWVkS2V5ID1cbiAgICAgIGV4Y2x1c2l2ZUZvclJlY29yZEtleXMubGVuZ3RoID09IDAgfHwgZXhjbHVzaXZlRm9yUmVjb3JkS2V5cy5pbmNsdWRlcyhrKVxuICAgICAgICA/IENhc2Uuc25ha2UoaylcbiAgICAgICAgOiBrO1xuICAgIHJlc3VsdFttb2RpZmllZEtleV0gPSB2O1xuICB9XG4gIHJldHVybiByZXN1bHQgYXMgYW55O1xufVxuXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gdHJ5UmVhZEZpbGUoZmlsZTogc3RyaW5nKSB7XG4gIGlmICghZXhpc3RzU3luYyhmaWxlKSkge1xuICAgIHJldHVybiBcIlwiO1xuICB9XG5cbiAgcmV0dXJuIGZzLnJlYWRGaWxlKGZpbGUsIFwidXRmLThcIik7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiB0cnlSZWFkRmlsZVN5bmMoZmlsZTogc3RyaW5nKSB7XG4gIGlmICghZXhpc3RzU3luYyhmaWxlKSkge1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICByZXR1cm4gcmVhZEZpbGVTeW5jKGZpbGUsIFwidXRmLThcIik7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1dyaXRhYmxlKGZpbGU6IHN0cmluZykge1xuICB0cnkge1xuICAgIGFjY2Vzc1N5bmMoZmlsZSwgZnNfY29uc3RhbnRzLldfT0spO1xuICAgIHJldHVybiB0cnVlO1xuICB9IGNhdGNoIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbn1cblxuLyoqXG4gKiBBc3NlcnRzIHRoYXQgdGhlIGZpbGUgc2hvdWxkIGJlIGV4ZWN1dGFibGUuIEFsd2F5cyByZXR1cm5zIHRydWUgb24gV2luZG93cy5cbiAqXG4gKiBJbiBXaW5kb3dzLCB0aGUgZXhlY3V0YWJsZSBhdHRyaWJ1dGUgaXMgc3RvcmVkIGluIHRoZSBzeXN0ZW0gc2V0dGluZyBQQVRIRVhULCBub3QgaW4gZWFjaCBmaWxlLiBUaGVuLCBjaGVja2luZyBmb3IgZXhlY3V0YWJpbGl0eSBpcyBlcXVpdmFsZW50IHRvIGNoZWNraW5nIGZvciBleGlzdGVuY2UuIFRvIGJ5cGFzcyBjaGVja2luZyBmb3IgZXhlY3V0YWJpbGl0eSwgd2UgYWx3YXlzIHJldHVybiB0cnVlIG9uIFdpbmRvd3MuXG4gKlxuICogQHBhcmFtIGZpbGVQYXRoIFRoZSBwYXRoIHRvIHRoZSBmaWxlXG4gKiBAcGFyYW0gc2hvdWxkQmVFeGVjdXRhYmxlIFdoZXRoZXIgdGhlIGZpbGUgc2hvdWxkIGJlIGV4ZWN1dGFibGVcbiAqIEByZXR1cm5zIHRydWUgaWYgYGZpbGVQYXRoYCBleGVjdXRhYmxlIGF0dHJpYnV0ZSBtYXRjaGVzIGBzaG91bGRCZUV4ZWN1dGFibGVgIG9yIGlmIHRoZSBwbGF0Zm9ybSBpcyBXaW5kb3dzLCBmYWxzZSBvdGhlcndpc2VcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydEV4ZWN1dGFibGVQZXJtaXNzaW9ucyhcbiAgZmlsZVBhdGg6IHN0cmluZyxcbiAgc2hvdWxkQmVFeGVjdXRhYmxlOiBib29sZWFuXG4pOiBib29sZWFuIHtcbiAgY29uc3QgaXNXaW5kb3dzID0gcHJvY2Vzcy5wbGF0Zm9ybSA9PT0gXCJ3aW4zMlwiO1xuICBpZiAoaXNXaW5kb3dzKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBjb25zdCBwcmV2RXhlY3V0YWJsZSA9IGlzRXhlY3V0YWJsZShmaWxlUGF0aCk7XG5cbiAgcmV0dXJuIHByZXZFeGVjdXRhYmxlID09PSBzaG91bGRCZUV4ZWN1dGFibGU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0V4ZWN1dGFibGUoZmlsZTogc3RyaW5nKSB7XG4gIHRyeSB7XG4gICAgYWNjZXNzU3luYyhmaWxlLCBmc19jb25zdGFudHMuWF9PSyk7XG5cbiAgICByZXR1cm4gdHJ1ZTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxufVxuXG5mdW5jdGlvbiBkZWNhbWVsaXplKHM6IHN0cmluZywgc2VwOiBzdHJpbmcgPSBcIl9cIikge1xuICBpZiAoQ2FzZS5vZihzKSA9PT0gXCJjYW1lbFwiKSB7XG4gICAgcmV0dXJuIENhc2UubG93ZXIocywgc2VwKTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gcztcbiAgfVxufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0Tm9kZU1ham9yVmVyc2lvbigpOiBudW1iZXIgfCB1bmRlZmluZWQge1xuICBjb25zdCBtYXRjaCA9IHByb2Nlc3MudmVyc2lvbi5tYXRjaCgvKFxcZCspXFwuKFxcZCspXFwuKFxcZCspLyk7XG4gIGlmIChtYXRjaCkge1xuICAgIGNvbnN0IFttYWpvcl0gPSBtYXRjaC5zbGljZSgxKS5tYXAoKHgpID0+IHBhcnNlSW50KHgpKTtcbiAgICByZXR1cm4gbWFqb3I7XG4gIH1cbiAgcmV0dXJuIHVuZGVmaW5lZDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFueVNlbGVjdGVkKG9wdGlvbnM6IChib29sZWFuIHwgdW5kZWZpbmVkKVtdKTogYm9vbGVhbiB7XG4gIHJldHVybiBvcHRpb25zLnNvbWUoKG9wdCkgPT4gb3B0KTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIG11bHRpcGxlU2VsZWN0ZWQob3B0aW9uczogKGJvb2xlYW4gfCB1bmRlZmluZWQpW10pOiBib29sZWFuIHtcbiAgcmV0dXJuIG9wdGlvbnMuZmlsdGVyKChvcHQpID0+IG9wdCkubGVuZ3RoID4gMTtcbn1cblxuLyoqXG4gKiBDaGVja3MgaWYgYSBwYXRoIGlzIGEgRlMgcm9vdFxuICpcbiAqIE9wdGlvbmFsIHVzZXMgYSBwcm92aWRlZCBPUyBzcGVjaWZpYyBwYXRoIGltcGxlbWVudGF0aW9uLFxuICogZGVmYXVsdHMgdG8gdXNlIHRoZSBpbXBsZW1lbnRhdGlvbiBmb3IgdGhlIGN1cnJlbnQgT1MuXG4gKlxuICogQGludGVybmFsXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc1Jvb3QoZGlyOiBzdHJpbmcsIG9zUGF0aExpYjogdHlwZW9mIHBhdGggPSBwYXRoKTogYm9vbGVhbiB7XG4gIGNvbnN0IHBhcmVudCA9IG9zUGF0aExpYi5kaXJuYW1lKGRpcik7XG4gIHJldHVybiBwYXJlbnQgPT09IGRpcjtcbn1cblxuLyoqXG4gKiBSdW4gdXAgcHJvamVjdCB0cmVlIHRvIGZpbmQgYSBmaWxlIG9yIGRpcmVjdG9yeVxuICpcbiAqIEBwYXJhbSBsb29rRm9yIHRoZSBmaWxlIG9yIGRpcmVjdG9yeSB0byBsb29rIGZvclxuICogQHBhcmFtIGN3ZCBjdXJyZW50IHdvcmtpbmcgZGlyZWN0b3J5LCBtdXN0IGJlIGFuIGFic29sdXRlIHBhdGhcbiAqIEByZXR1cm5zIHBhdGggdG8gdGhlIGZpbGUgb3IgZGlyZWN0b3J5IHdlIGFyZSBsb29raW5nIGZvciwgdW5kZWZpbmVkIGlmIG5vdCBmb3VuZFxuICovXG5leHBvcnQgZnVuY3Rpb24gZmluZFVwKFxuICBsb29rRm9yOiBzdHJpbmcsXG4gIGN3ZDogc3RyaW5nID0gcHJvY2Vzcy5jd2QoKVxuKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgaWYgKGV4aXN0c1N5bmMocGF0aC5qb2luKGN3ZCwgbG9va0ZvcikpKSB7XG4gICAgcmV0dXJuIGN3ZDtcbiAgfVxuXG4gIGlmIChpc1Jvb3QoY3dkKSkge1xuICAgIC8vIFRoaXMgaXMgYSByb290XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuICByZXR1cm4gZmluZFVwKGxvb2tGb3IsIHBhdGguZGlybmFtZShjd2QpKTtcbn1cblxuLyoqXG4gKiBOb3JtYWxpemVzIGEgcGF0aCB0aGF0IGlzIGdvaW5nIHRvIGJlIHBlcnNpc3RlZCB0byBoYXZlIGEgY3Jvc3MgcGxhdGZvcm0gcmVwcmVzZW50YXRpb24uXG4gKlxuICogTm9ybWFsaXplZCBwYXRocyBjYW4gYmUgcGVyc2lzdGVkIGFuZCBkb2Vzbid0IG5lZWQgdG8gYmUgbW9kaWZpZWQgd2hlbiB0aGUgcGxhdGZvcm0gY2hhbmdlcy5cbiAqIGBub3JtYWxpemVQZXJzaXN0ZWRQYXRoYCB0YWtlcyBjYXJlIG9mIHBsYXRmb3JtLXNwZWNpZmljIHByb3BlcnRpZXMgbGlrZSBkaXJlY3Rvcnkgc2VwYXJhdG9yLlxuICogSXQgdXNlcyBgcGF0aC5wb3NpeC5zZXBgIHRoYXQgaXMgc3VwcG9ydGVkIGJvdGggaW4gV2luZG93cyBhbmQgVW5peCBwbGF0Zm9ybXMuXG4gKlxuICpcbiAqIEBwYXJhbSBwIHRoZSBwYXRoIHRvIGJlIG5vcm1hbGl6ZWRcbiAqIEByZXR1cm5zIHRoZSBub3JtYWxpemVkIHBhdGhcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIG5vcm1hbGl6ZVBlcnNpc3RlZFBhdGgocDogc3RyaW5nKSB7XG4gIHJldHVybiBwLnJlcGxhY2UoL1xcXFwvZywgcGF0aC5wb3NpeC5zZXApO1xufVxuIl19