"use strict";
// stolen from: https://github.com/aws/jsii/blob/main/packages/jsii-pacmak/lib/targets/version-utils.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.toReleaseVersion = exports.toPythonVersionRange = exports.toNuGetVersionRange = exports.toMavenVersionRange = exports.TargetName = void 0;
const util_1 = require("util");
const semver_1 = require("semver");
var TargetName;
(function (TargetName) {
    TargetName[TargetName["JAVA"] = 0] = "JAVA";
    TargetName[TargetName["DOTNET"] = 1] = "DOTNET";
    TargetName[TargetName["PYTHON"] = 2] = "PYTHON";
    TargetName[TargetName["GO"] = 3] = "GO";
    TargetName[TargetName["JAVASCRIPT"] = 4] = "JAVASCRIPT";
})(TargetName || (exports.TargetName = TargetName = {}));
/**
 * Converts a SemVer range expression to a Maven version range expression.
 *
 * @param semverRange the SemVer range expression to convert.
 * @param suffix      the suffix to add to versions in the range.
 *
 * @see https://cwiki.apache.org/confluence/display/MAVENOLD/Dependency+Mediation+and+Conflict+Resolution
 */
function toMavenVersionRange(semverRange, suffix) {
    return toBracketNotation(semverRange, suffix, {
        semver: false,
        target: TargetName.JAVA,
    });
}
exports.toMavenVersionRange = toMavenVersionRange;
/**
 * Converts a SemVer range expression to a NuGet version range expression.
 *
 * @param semverRange the SemVer range expression to convert.
 *
 * @see https://docs.microsoft.com/en-us/nuget/concepts/package-versioning#version-ranges-and-wildcards
 */
function toNuGetVersionRange(semverRange) {
    return toBracketNotation(semverRange, undefined, {
        semver: false,
        target: TargetName.DOTNET,
    });
}
exports.toNuGetVersionRange = toNuGetVersionRange;
/**
 * Converts a SemVer range expression to a Python setuptools compatible version
 * constraint expression.
 *
 * @param semverRange the SemVer range expression to convert.
 */
function toPythonVersionRange(semverRange) {
    const range = new semver_1.Range(semverRange);
    return range.set
        .map((set) => set
        .map((comp) => {
        const versionId = toReleaseVersion(comp.semver.raw?.replace(/-0$/, "") ?? "0.0.0", TargetName.PYTHON);
        switch (comp.operator) {
            case "":
                // With ^0.0.0, somehow we get a left entry with an empty operator and value, we'll fix this up
                return comp.value === "" ? ">=0.0.0" : `==${versionId}`;
            case "=":
                return `==${versionId}`;
            default:
                // >, >=, <, <= are all valid expressions
                return `${comp.operator}${versionId}`;
        }
    })
        .join(", "))
        .join(", ");
}
exports.toPythonVersionRange = toPythonVersionRange;
/**
 * Converts an original version number from the NPM convention to the target
 * language's convention for expressing the same. For versions that do not
 * include a prerelease identifier, this always returns the assembly version
 * unmodified.
 *
 * @param assemblyVersion the assembly version being released
 * @param target          the target language for which the version is destined
 *
 * @returns the version that should be serialized
 */
function toReleaseVersion(assemblyVersion, target) {
    const version = (0, semver_1.parse)(assemblyVersion);
    if (version == null) {
        throw new Error(`Unable to parse the provided assembly version: "${assemblyVersion}"`);
    }
    if (version.prerelease.length === 0) {
        return assemblyVersion;
    }
    switch (target) {
        case TargetName.PYTHON:
            // Python supports a limited set of identifiers... And we have a mapping table...
            // https://packaging.python.org/guides/distributing-packages-using-setuptools/#pre-release-versioning
            const [label, sequence, ...rest] = version.prerelease;
            if (rest.filter((elt) => elt !== 0).length > 0 || sequence == null) {
                throw new Error(`Unable to map prerelease identifier (in: ${assemblyVersion}) components to python: ${(0, util_1.inspect)(version.prerelease)}. The format should be 'X.Y.Z-label.sequence', where sequence is a positive integer, and label is "dev", "pre", "alpha", beta", or "rc"`);
            }
            if (!Number.isInteger(sequence)) {
                throw new Error(`Unable to map prerelease identifier (in: ${assemblyVersion}) to python, as sequence ${(0, util_1.inspect)(sequence)} is not an integer`);
            }
            const baseVersion = `${version.major}.${version.minor}.${version.patch}`;
            // See PEP 440: https://www.python.org/dev/peps/pep-0440/#pre-releases
            switch (label) {
                case "dev":
                case "pre":
                    return `${baseVersion}.dev${sequence}`; // PEP 404 see developmental release as X.Y.devN
                case "alpha":
                    return `${baseVersion}a${sequence}`; // PEP 404 see alpha release as X.YaN
                case "beta":
                    return `${baseVersion}b${sequence}`; // PEP 404 see beta release as X.YbN
                case "rc":
                    return `${baseVersion}rc${sequence}`; // PEP 404 see release candidate as X.YrcN
                default:
                    throw new Error(`Unable to map prerelease identifier (in: ${assemblyVersion}) to python, as label ${(0, util_1.inspect)(label)} is not mapped (only "dev", "pre", "alpha", "beta" and "rc" are)`);
            }
        case TargetName.DOTNET:
        case TargetName.GO:
        case TargetName.JAVA:
        case TargetName.JAVASCRIPT:
            // Not touching - the NPM version number should be usable as-is
            break;
    }
    return assemblyVersion;
}
exports.toReleaseVersion = toReleaseVersion;
/**
 * Converts a semantic version range to the kind of bracket notation used by
 * Maven and NuGet. For example, this turns `^1.2.3` into `[1.2.3,2.0.0)`.
 *
 * @param semverRange The semantic version range to be converted.
 * @param suffix A version suffix to apply to versions in the resulting expression.
 * @param semver Whether the target supports full semantic versioning (including
 *               `-0` as the lowest possible prerelease identifier)
 *
 * @returns a bracket-notation version range.
 */
function toBracketNotation(semverRange, suffix, { semver = true, target = TargetName.JAVASCRIPT, } = {}) {
    if (semverRange === "*") {
        semverRange = ">=0.0.0";
    }
    const range = new semver_1.Range(semverRange);
    if (semverRange === range.range) {
        return semverRange;
    }
    return range.set
        .map((set) => {
        if (set.length === 1) {
            const version = set[0].semver.raw;
            if (!version && range.raw === ">=0.0.0") {
                // Case where version is '*'
                return "[0.0.0,)";
            }
            switch (set[0].operator || "=") {
                // "[version]" => means exactly version
                case "=":
                    return `[${addSuffix(version)}]`;
                // "(version,]" => means greater than version
                case ">":
                    return `(${addSuffix(version)},)`;
                // "[version,]" => means greater than or equal to that version
                case ">=":
                    return `[${addSuffix(version)},)`;
                // "[,version)" => means less than version
                case "<":
                    return `(,${addSuffix(version, !semver)})`;
                // "[,version]" => means less than or equal to version
                case "<=":
                    return `(,${addSuffix(version)}]`;
            }
        }
        else if (set.length === 2) {
            const nugetRange = toBracketRange(set[0], set[1]);
            if (nugetRange) {
                return nugetRange;
            }
        }
        throw new Error(`Unsupported SemVer range set in ${semverRange}: ${set
            .map((comp) => comp.value)
            .join(", ")}`);
    })
        .join(", ");
    function toBracketRange(left, right) {
        if (left.operator.startsWith("<") && right.operator.startsWith(">")) {
            // Order isn't ideal, swap around..
            [left, right] = [right, left];
        }
        // With ^0.0.0, somehow we get a left entry with an empty operator and value, we'll fix this up
        if (left.operator === "" && left.value === "") {
            left = new semver_1.Comparator(">=0.0.0", left.options);
        }
        if (!left.operator.startsWith(">") || !right.operator.startsWith("<")) {
            // We only support ranges defined like "> (or >=) left, < (or <=) right"
            return undefined;
        }
        const leftBrace = left.operator.endsWith("=") ? "[" : "(";
        const rightBrace = right.operator.endsWith("=") ? "]" : ")";
        return `${leftBrace}${addSuffix(left.semver.raw)},${addSuffix(right.semver.raw, right.operator === "<" && !semver)}${rightBrace}`;
    }
    function addSuffix(str, trimDashZero = false) {
        if (!str) {
            return "";
        }
        if (trimDashZero) {
            str = str.replace(/-0$/, "");
        }
        return suffix ? `${str}${suffix}` : toReleaseVersion(str, target);
    }
}
//# sourceMappingURL=data:application/json;base64,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