import { Component } from "./component";
import { Project } from "./project";
import { Task } from "./task";
/**
 * Options for `Version`.
 */
export interface VersionOptions {
    /**
     * A name of a .json file to set the `version` field in after a bump.
     *
     * @example "package.json"
     */
    readonly versionInputFile: string;
    /**
     * The name of the directory into which `changelog.md` and `version.txt` files
     * are emitted.
     */
    readonly artifactsDirectory: string;
    /**
     * Custom configuration for versionrc file used by standard-release
     */
    readonly versionrcOptions?: Record<string, any>;
    /**
     * The tag prefix corresponding to this version.
     */
    readonly tagPrefix?: string;
    /**
     * Find commits that should be considered releasable
     * Used to decide if a release is required.
     *
     * @default ReleasableCommits.everyCommit()
     */
    readonly releasableCommits?: ReleasableCommits;
}
export declare class Version extends Component {
    static readonly STANDARD_VERSION = "standard-version@^9";
    readonly bumpTask: Task;
    readonly unbumpTask: Task;
    /**
     * The name of the changelog file (under `artifactsDirectory`).
     */
    readonly changelogFileName: string;
    /**
     * The name of the file that contains the version (under `artifactsDirectory`).
     */
    readonly versionFileName: string;
    /**
     * The name of the file that contains the release tag (under `artifactsDirectory`).
     */
    readonly releaseTagFileName: string;
    constructor(project: Project, options: VersionOptions);
}
/**
 * Find commits that should be considered releasable to decide if a release is required.
 */
export declare class ReleasableCommits {
    cmd: string;
    /**
     * Release every commit
     *
     * This will only not release if the most recent commit is tagged with the latest matching tag.
     *
     * @param path Consider only commits that are enough to explain how the files that match the specified paths came to be.
     * This path is relative to the current working dir of the `bump` task, i.e. to only consider commits of a subproject use `"."`.
     */
    static everyCommit(path?: string): ReleasableCommits;
    /**
     * Limit commits by their conventional commit type
     *
     * This will only release commit that match one of the provided types.
     * Commits are required to follow the conventional commit spec and will be ignored otherwise.
     *
     * @param types List of conventional commit types that should be released
     * @param path Consider only commits that are enough to explain how the files that match the specified paths came to be.
     * This path is relative to the current working dir of the `bump` task, i.e. to only consider commits of a subproject use `"."`.
     */
    static ofType(types: string[], path?: string): ReleasableCommits;
    /**
     * Release only features and fixes
     *
     * Shorthand for `ReleasableCommits.onlyOfType(['feat', 'fix'])`.
     *
     * @param path Consider only commits that are enough to explain how the files that match the specified paths came to be.
     * This path is relative to the current working dir of the `bump` task, i.e. to only consider commits of a subproject use `"."`.
     */
    static featuresAndFixes(path?: string): ReleasableCommits;
    /**
     * Use an arbitrary shell command to find releasable commits since the latest tag.
     *
     * A new release will be initiated, if the number of returned commits is greater than zero.
     * Must return a newline separate list of commits that should considered releasable.
     * `$LATEST_TAG` will be replaced with the actual latest tag for the given prefix.*
     *
     * @example "git log --oneline $LATEST_TAG..HEAD -- ."
     */
    static exec(cmd: string): ReleasableCommits;
    private constructor();
}
