"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReleasableCommits = exports.Version = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const component_1 = require("./component");
class Version extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.changelogFileName = "changelog.md";
        this.versionFileName = "version.txt";
        this.releaseTagFileName = "releasetag.txt";
        const versionInputFile = options.versionInputFile;
        // this command determines if there were any changes since the last release
        // (the top-most commit is not a bump). it is used as a condition for both
        // the `bump` and the `release` tasks.
        const changesSinceLastRelease = '! git log --oneline -1 | grep -q "chore(release):"';
        const changelogFile = path_1.posix.join(options.artifactsDirectory, this.changelogFileName);
        const bumpFile = path_1.posix.join(options.artifactsDirectory, this.versionFileName);
        const releaseTagFile = path_1.posix.join(options.artifactsDirectory, this.releaseTagFileName);
        const env = {
            OUTFILE: versionInputFile,
            CHANGELOG: changelogFile,
            BUMPFILE: bumpFile,
            RELEASETAG: releaseTagFile,
            RELEASE_TAG_PREFIX: options.tagPrefix ?? "",
            // doesn't work if custom configuration is long
            VERSIONRCOPTIONS: JSON.stringify(options.versionrcOptions),
        };
        if (options.releasableCommits) {
            env.RELEASABLE_COMMITS = options.releasableCommits.cmd;
        }
        this.bumpTask = project.addTask("bump", {
            description: "Bumps version based on latest git tag and generates a changelog entry",
            condition: changesSinceLastRelease,
            env: env,
        });
        this.bumpTask.builtin("release/bump-version");
        this.unbumpTask = project.addTask("unbump", {
            description: "Restores version to 0.0.0",
            env: env,
        });
        this.unbumpTask.builtin("release/reset-version");
        project.addGitIgnore(`/${changelogFile}`);
        project.addGitIgnore(`/${bumpFile}`);
        project.addPackageIgnore(`/${changelogFile}`);
        project.addPackageIgnore(`/${bumpFile}`);
    }
}
exports.Version = Version;
_a = JSII_RTTI_SYMBOL_1;
Version[_a] = { fqn: "projen.Version", version: "0.79.27" };
Version.STANDARD_VERSION = "standard-version@^9";
/**
 * Find commits that should be considered releasable to decide if a release is required.
 */
class ReleasableCommits {
    /**
     * Release every commit
     *
     * This will only not release if the most recent commit is tagged with the latest matching tag.
     *
     * @param path Consider only commits that are enough to explain how the files that match the specified paths came to be.
     * This path is relative to the current working dir of the `bump` task, i.e. to only consider commits of a subproject use `"."`.
     */
    static everyCommit(path) {
        const cmd = `git log --oneline $LATEST_TAG..HEAD`;
        return new ReleasableCommits(withPath(cmd, path));
    }
    /**
     * Limit commits by their conventional commit type
     *
     * This will only release commit that match one of the provided types.
     * Commits are required to follow the conventional commit spec and will be ignored otherwise.
     *
     * @param types List of conventional commit types that should be released
     * @param path Consider only commits that are enough to explain how the files that match the specified paths came to be.
     * This path is relative to the current working dir of the `bump` task, i.e. to only consider commits of a subproject use `"."`.
     */
    static ofType(types, path) {
        const allowedTypes = types.join("|");
        // @see: https://github.com/conventional-commits/parser/blob/eeefb961ebf5b9dfea0fea8b06f8ad34a1e439b9/lib/parser.js
        // -E requires this to be POSIX Extended Regular Expression, which comes with certain limitations
        // see https://en.wikibooks.org/wiki/Regular_Expressions/POSIX-Extended_Regular_Expressions for details
        const cmd = `git log --no-merges --oneline $LATEST_TAG..HEAD -E --grep '^(${allowedTypes}){1}(\\([^()[:space:]]+\\))?(!)?:[[:blank:]]+.+'`;
        return new ReleasableCommits(withPath(cmd, path));
    }
    /**
     * Release only features and fixes
     *
     * Shorthand for `ReleasableCommits.onlyOfType(['feat', 'fix'])`.
     *
     * @param path Consider only commits that are enough to explain how the files that match the specified paths came to be.
     * This path is relative to the current working dir of the `bump` task, i.e. to only consider commits of a subproject use `"."`.
     */
    static featuresAndFixes(path) {
        return ReleasableCommits.ofType(["feat", "fix"], path);
    }
    /**
     * Use an arbitrary shell command to find releasable commits since the latest tag.
     *
     * A new release will be initiated, if the number of returned commits is greater than zero.
     * Must return a newline separate list of commits that should considered releasable.
     * `$LATEST_TAG` will be replaced with the actual latest tag for the given prefix.*
     *
     * @example "git log --oneline $LATEST_TAG..HEAD -- ."
     */
    static exec(cmd) {
        return new ReleasableCommits(cmd);
    }
    constructor(cmd) {
        this.cmd = cmd;
    }
}
exports.ReleasableCommits = ReleasableCommits;
_b = JSII_RTTI_SYMBOL_1;
ReleasableCommits[_b] = { fqn: "projen.ReleasableCommits", version: "0.79.27" };
/**
 * Append a path argument to a git command if one is provided
 */
function withPath(cmd, path) {
    if (path !== undefined) {
        return `${cmd} -- ${path}`;
    }
    return cmd;
}
//# sourceMappingURL=data:application/json;base64,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