import json

from django.contrib.gis.geos import Point
from django.template import Context, Template
from django.test import TestCase

from teramap import __version__

from .app.models import Location


class TemplateTagTest(TestCase):
    def test_teramap_asset_tags(self):
        tests = (
            ("teramap_version", "{version}"),
            ("teramap_js", "/static/teramap/teramap.js?version={version}"),
            ("teramap_css", "/static/teramap/teramap.css?version={version}"),
        )
        for tag, expected in tests:
            template = Template("{% load teramap %}{% " + tag + " %}")

            self.assertEqual(template.render(Context({})), expected.format(version=__version__))

    def test_format_point_4326(self):
        p = Point(4, 52, srid=4326)

        template = Template("{% load teramap %}{{ p | format_point }}")

        self.assertEqual(
            template.render(Context({"p": p})),
            (
                '<span class="teramap-coordinates" title="WGS 84">'
                '<span class="teramap-coordinates-name text-muted">GPS</span>&nbsp;'
                '<span class="teramap-coordinates-coords">52.0000, 4.0000</span>'
                "</span>"
            ),
        )

    def test_format_point_rd(self):
        p = Point(130564, 516371, srid=28992)

        template = Template("{% load teramap %}{{ p | format_point }}")

        self.assertEqual(
            template.render(Context({"p": p})),
            (
                '<span class="teramap-coordinates" title="Amersfoort / RD New">'
                '<span class="teramap-coordinates-name text-muted">RDS</span>&nbsp;'
                '<span class="teramap-coordinates-coords">130564 516371</span>'
                "</span>"
            ),
        )

    def test_format_point_transform_28992(self):
        p = Point(4, 52, srid=4326)

        template = Template("{% load teramap %}{{ p | format_point:28992 }}")

        self.assertEqual(
            template.render(Context({"p": p})),
            (
                '<span class="teramap-coordinates" title="Amersfoort / RD New">'
                '<span class="teramap-coordinates-name text-muted">RDS</span>&nbsp;'
                '<span class="teramap-coordinates-coords">59742 446645</span>'
                "</span>"
            ),
        )

    def test_format_point_transform_31372(self):
        p = Point(4, 52, srid=4326)

        template = Template("{% load teramap %}{{ p | format_point:31370 }}")
        self.assertEqual(
            template.render(Context({"p": p})),
            (
                '<span class="teramap-coordinates" title="BD72 / Belgian Lambert 72">'
                '<span class="teramap-coordinates-name text-muted">Lambert 1972</span>&nbsp;'
                '<span class="teramap-coordinates-coords">124667 299085</span>'
                "</span>"
            ),
        )

    def test_format_point_no_srs(self):
        p = Point(4, 52)

        template = Template("{% load teramap %}{{ p | format_point }}")
        self.assertEqual(template.render(Context({"p": p})), "")

    def test_format_point_point_none(self):
        template = Template("{% load teramap %}{{ p | format_point }}")
        self.assertEqual(template.render(Context({"p": None})), "")

    def test_driving_directions(self):
        p = Point(4.5, 52.2, srid=4326)
        template = Template("{% load teramap %}{{ p | driving_directions }}")

        self.assertEqual(template.render(Context({"p": p})), "https://maps.apple.com/?daddr=52.2,4.5")
