"""
涨跌分布统计缓存管理器

实现多层缓存架构，支持内存缓存、Redis缓存和数据库缓存
"""

import hashlib
import json
import logging
import pickle
import sqlite3
import threading
import time
from collections import OrderedDict
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Tuple, Union

import pandas as pd

logger = logging.getLogger(__name__)


class CacheKeyManager:
    """缓存键管理器"""
    
    # 缓存键模板
    CACHE_KEY_TEMPLATES = {
        'price_distribution': 'price_dist:{trade_date}:{market}:{include_st}:{hash}',
        'stock_data': 'stock_daily:{trade_date}',
        'market_classification': 'market_class:{trade_date}',
        'distribution_summary': 'dist_summary:{trade_date}',
        'custom_ranges': 'custom_ranges:{trade_date}:{ranges_hash}',
        'market_breakdown': 'market_breakdown:{trade_date}:{market}',
        'statistics_metadata': 'stats_meta:{trade_date}'
    }
    
    # 缓存TTL配置 (秒)
    CACHE_TTL = {
        'price_distribution': 3600,      # 1小时
        'stock_data': 7200,              # 2小时
        'market_classification': 1800,    # 30分钟
        'distribution_summary': 3600,     # 1小时
        'custom_ranges': 1800,           # 30分钟
        'market_breakdown': 3600,        # 1小时
        'statistics_metadata': 7200      # 2小时
    }
    
    def generate_key(self, template_name: str, **params) -> str:
        """
        生成缓存键
        
        Args:
            template_name: 模板名称
            **params: 参数字典
            
        Returns:
            生成的缓存键
        """
        if template_name not in self.CACHE_KEY_TEMPLATES:
            raise ValueError(f"Unknown template: {template_name}")
        
        template = self.CACHE_KEY_TEMPLATES[template_name]
        
        # 处理特殊参数
        if 'hash' in template and 'hash' not in params:
            # 生成参数哈希
            hash_data = {k: v for k, v in params.items() if k != 'hash'}
            params['hash'] = self.generate_hash(hash_data)
        
        if 'ranges_hash' in template and 'ranges_hash' not in params:
            # 生成区间哈希
            ranges_data = params.get('ranges', {})
            params['ranges_hash'] = self.generate_hash(ranges_data)
        
        try:
            return template.format(**params)
        except KeyError as e:
            raise ValueError(f"Missing parameter for template {template_name}: {e}")
    
    def generate_hash(self, data: Any) -> str:
        """
        生成数据哈希
        
        Args:
            data: 要哈希的数据
            
        Returns:
            MD5哈希字符串
        """
        if data is None:
            return "none"
        
        # 将数据转换为JSON字符串并排序键
        json_str = json.dumps(data, sort_keys=True, ensure_ascii=False)
        return hashlib.md5(json_str.encode('utf-8')).hexdigest()[:8]
    
    def parse_key(self, cache_key: str) -> Dict[str, str]:
        """
        解析缓存键
        
        Args:
            cache_key: 缓存键
            
        Returns:
            解析出的参数字典
        """
        parts = cache_key.split(':')
        if len(parts) < 2:
            return {}
        
        # 根据键的前缀确定类型
        key_type = parts[0]
        result = {'type': key_type}
        
        # 解析不同类型的键
        if key_type == 'price_dist' and len(parts) >= 5:
            result.update({
                'trade_date': parts[1],
                'market': parts[2],
                'include_st': parts[3] == 'True',
                'hash': parts[4]
            })
        elif key_type == 'stock_daily' and len(parts) >= 2:
            result.update({
                'trade_date': parts[1]
            })
        elif key_type in ['market_class', 'dist_summary', 'stats_meta'] and len(parts) >= 2:
            result.update({
                'trade_date': parts[1]
            })
        elif key_type == 'custom_ranges' and len(parts) >= 3:
            result.update({
                'trade_date': parts[1],
                'ranges_hash': parts[2]
            })
        elif key_type == 'market_breakdown' and len(parts) >= 3:
            result.update({
                'trade_date': parts[1],
                'market': parts[2]
            })
        
        return result
    
    def get_ttl(self, template_name: str) -> int:
        """
        获取缓存TTL
        
        Args:
            template_name: 模板名称
            
        Returns:
            TTL秒数
        """
        return self.CACHE_TTL.get(template_name, 3600)


class CacheEntry:
    """缓存条目"""
    
    def __init__(self, data: Any, expire_time: Optional[datetime] = None, 
                 metadata: Optional[Dict[str, Any]] = None):
        self.data = data
        self.expire_time = expire_time
        self.created_time = datetime.now()
        self.access_time = self.created_time
        self.access_count = 0
        self.metadata = metadata or {}
    
    def is_expired(self) -> bool:
        """检查是否过期"""
        if self.expire_time is None:
            return False
        return datetime.now() > self.expire_time
    
    def update_access(self):
        """更新访问信息"""
        self.access_time = datetime.now()
        self.access_count += 1
    
    def to_dict(self) -> Dict[str, Any]:
        """转换为字典格式"""
        return {
            'data': self.data,
            'expire_time': self.expire_time.isoformat() if self.expire_time else None,
            'created_time': self.created_time.isoformat(),
            'access_time': self.access_time.isoformat(),
            'access_count': self.access_count,
            'metadata': self.metadata
        }


class MemoryCacheLayer:
    """内存缓存层"""
    
    def __init__(self, max_size: int = 1000):
        self.max_size = max_size
        self._cache: OrderedDict[str, CacheEntry] = OrderedDict()
        self._lock = threading.RLock()
        self._stats = {
            'hits': 0,
            'misses': 0,
            'sets': 0,
            'deletes': 0,
            'evictions': 0
        }
    
    async def get(self, key: str) -> Optional[Any]:
        """获取缓存数据"""
        with self._lock:
            if key not in self._cache:
                self._stats['misses'] += 1
                return None
            
            entry = self._cache[key]
            
            # 检查是否过期
            if entry.is_expired():
                del self._cache[key]
                self._stats['misses'] += 1
                return None
            
            # 更新访问信息并移到末尾（LRU策略）
            entry.update_access()
            self._cache.move_to_end(key)
            
            self._stats['hits'] += 1
            return entry.data
    
    async def set(self, key: str, data: Any, ttl: Optional[int] = None) -> bool:
        """设置缓存数据"""
        with self._lock:
            try:
                # 计算过期时间
                expire_time = None
                if ttl is not None:
                    expire_time = datetime.now() + timedelta(seconds=ttl)
                
                # 创建缓存条目
                entry = CacheEntry(data, expire_time)
                
                # 如果键已存在，更新并移到末尾
                if key in self._cache:
                    self._cache[key] = entry
                    self._cache.move_to_end(key)
                else:
                    # 检查是否需要清理空间
                    while len(self._cache) >= self.max_size:
                        # 删除最久未使用的条目
                        oldest_key = next(iter(self._cache))
                        del self._cache[oldest_key]
                        self._stats['evictions'] += 1
                    
                    self._cache[key] = entry
                
                self._stats['sets'] += 1
                return True
                
            except Exception as e:
                logger.error(f"Memory cache set failed: {e}")
                return False
    
    async def delete(self, key: str) -> bool:
        """删除缓存条目"""
        with self._lock:
            if key in self._cache:
                del self._cache[key]
                self._stats['deletes'] += 1
                return True
            return False
    
    async def clear(self) -> int:
        """清空所有缓存"""
        with self._lock:
            count = len(self._cache)
            self._cache.clear()
            return count
    
    async def clear_expired(self) -> int:
        """清理过期缓存"""
        with self._lock:
            expired_keys = []
            for key, entry in self._cache.items():
                if entry.is_expired():
                    expired_keys.append(key)
            
            for key in expired_keys:
                del self._cache[key]
            
            return len(expired_keys)
    
    def get_stats(self) -> Dict[str, Any]:
        """获取缓存统计信息"""
        with self._lock:
            total_requests = self._stats['hits'] + self._stats['misses']
            hit_rate = self._stats['hits'] / total_requests if total_requests > 0 else 0
            
            return {
                'size': len(self._cache),
                'max_size': self.max_size,
                'usage_ratio': len(self._cache) / self.max_size if self.max_size > 0 else 0,
                'hit_rate': hit_rate,
                **self._stats
            }


class DatabaseCacheLayer:
    """数据库缓存层"""
    
    def __init__(self, db_path: str):
        self.db_path = db_path
        self._lock = threading.RLock()
        self._init_database()
        self._stats = {
            'hits': 0,
            'misses': 0,
            'sets': 0,
            'deletes': 0
        }
    
    def _init_database(self):
        """初始化数据库表结构"""
        import os
        
        # 确保目录存在
        db_dir = os.path.dirname(self.db_path)
        if db_dir:
            os.makedirs(db_dir, exist_ok=True)
        
        with sqlite3.connect(self.db_path) as conn:
            conn.execute('''
                CREATE TABLE IF NOT EXISTS price_distribution_cache (
                    cache_key TEXT PRIMARY KEY,
                    data_blob BLOB NOT NULL,
                    data_type TEXT NOT NULL,
                    created_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    expire_time TIMESTAMP,
                    access_count INTEGER DEFAULT 0,
                    last_access_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    metadata TEXT
                )
            ''')
            
            # 创建索引
            conn.execute('''
                CREATE INDEX IF NOT EXISTS idx_expire_time 
                ON price_distribution_cache(expire_time)
            ''')
            
            conn.execute('''
                CREATE INDEX IF NOT EXISTS idx_data_type 
                ON price_distribution_cache(data_type)
            ''')
            
            conn.execute('''
                CREATE INDEX IF NOT EXISTS idx_last_access 
                ON price_distribution_cache(last_access_time)
            ''')
            
            conn.commit()
    
    async def get(self, key: str) -> Optional[Any]:
        """获取缓存数据"""
        with self._lock:
            try:
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    
                    # 查询数据
                    cursor.execute('''
                        SELECT data_blob, expire_time, data_type FROM price_distribution_cache 
                        WHERE cache_key = ?
                    ''', (key,))
                    
                    result = cursor.fetchone()
                    if result is None:
                        self._stats['misses'] += 1
                        return None
                    
                    data_blob, expire_time_str, data_type = result
                    
                    # 检查是否过期
                    if expire_time_str:
                        expire_time = datetime.fromisoformat(expire_time_str)
                        if datetime.now() > expire_time:
                            # 删除过期数据
                            cursor.execute('DELETE FROM price_distribution_cache WHERE cache_key = ?', (key,))
                            conn.commit()
                            self._stats['misses'] += 1
                            return None
                    
                    # 更新访问统计
                    cursor.execute('''
                        UPDATE price_distribution_cache 
                        SET access_count = access_count + 1,
                            last_access_time = CURRENT_TIMESTAMP
                        WHERE cache_key = ?
                    ''', (key,))
                    conn.commit()
                    
                    # 反序列化数据
                    data = self._deserialize_data(data_blob, data_type)
                    self._stats['hits'] += 1
                    return data
                    
            except Exception as e:
                logger.error(f"Database cache get failed: {e}")
                self._stats['misses'] += 1
                return None
    
    async def set(self, key: str, data: Any, ttl: Optional[int] = None) -> bool:
        """设置缓存数据"""
        with self._lock:
            try:
                # 序列化数据
                data_blob, data_type = self._serialize_data(data)
                
                # 计算过期时间
                expire_time = None
                if ttl is not None:
                    expire_time = datetime.now() + timedelta(seconds=ttl)
                
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    
                    # 插入或更新数据
                    cursor.execute('''
                        INSERT OR REPLACE INTO price_distribution_cache 
                        (cache_key, data_blob, data_type, created_time, expire_time, 
                         access_count, last_access_time, metadata)
                        VALUES (?, ?, ?, CURRENT_TIMESTAMP, ?, 0, CURRENT_TIMESTAMP, ?)
                    ''', (key, data_blob, data_type, 
                          expire_time.isoformat() if expire_time else None, 
                          json.dumps({})))
                    
                    conn.commit()
                    self._stats['sets'] += 1
                    return True
                    
            except Exception as e:
                logger.error(f"Database cache set failed: {e}")
                return False
    
    async def delete(self, key: str) -> bool:
        """删除缓存条目"""
        with self._lock:
            try:
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    cursor.execute('DELETE FROM price_distribution_cache WHERE cache_key = ?', (key,))
                    conn.commit()
                    deleted = cursor.rowcount > 0
                    if deleted:
                        self._stats['deletes'] += 1
                    return deleted
            except Exception as e:
                logger.error(f"Database cache delete failed: {e}")
                return False
    
    async def clear(self) -> int:
        """清空所有缓存"""
        with self._lock:
            try:
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    cursor.execute('SELECT COUNT(*) FROM price_distribution_cache')
                    count = cursor.fetchone()[0]
                    cursor.execute('DELETE FROM price_distribution_cache')
                    conn.commit()
                    return count
            except Exception as e:
                logger.error(f"Database cache clear failed: {e}")
                return 0
    
    async def clear_expired(self) -> int:
        """清理过期缓存"""
        with self._lock:
            try:
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    cursor.execute('''
                        DELETE FROM price_distribution_cache 
                        WHERE expire_time IS NOT NULL 
                        AND expire_time < CURRENT_TIMESTAMP
                    ''')
                    deleted_count = cursor.rowcount
                    conn.commit()
                    return deleted_count
            except Exception as e:
                logger.error(f"Database cache clear expired failed: {e}")
                return 0
    
    async def delete_by_pattern(self, pattern: str) -> int:
        """按模式删除缓存"""
        with self._lock:
            try:
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    cursor.execute('''
                        DELETE FROM price_distribution_cache 
                        WHERE cache_key LIKE ?
                    ''', (pattern,))
                    deleted_count = cursor.rowcount
                    conn.commit()
                    return deleted_count
            except Exception as e:
                logger.error(f"Database cache delete by pattern failed: {e}")
                return 0
    
    def _serialize_data(self, data: Any) -> Tuple[bytes, str]:
        """序列化数据"""
        if isinstance(data, pd.DataFrame):
            return pickle.dumps(data), 'dataframe'
        elif isinstance(data, dict):
            # 处理numpy类型的JSON序列化问题
            try:
                return json.dumps(data, ensure_ascii=False, default=self._json_serializer).encode('utf-8'), 'dict'
            except (TypeError, ValueError):
                # 如果JSON序列化失败，使用pickle
                return pickle.dumps(data), 'pickle'
        elif isinstance(data, (list, tuple)):
            try:
                return json.dumps(data, ensure_ascii=False, default=self._json_serializer).encode('utf-8'), 'list'
            except (TypeError, ValueError):
                # 如果JSON序列化失败，使用pickle
                return pickle.dumps(data), 'pickle'
        else:
            return pickle.dumps(data), 'pickle'
    
    def _json_serializer(self, obj):
        """JSON序列化器，处理numpy类型"""
        import numpy as np
        if isinstance(obj, np.integer):
            return int(obj)
        elif isinstance(obj, np.floating):
            return float(obj)
        elif isinstance(obj, np.ndarray):
            return obj.tolist()
        elif hasattr(obj, 'isoformat'):  # datetime objects
            return obj.isoformat()
        raise TypeError(f"Object of type {type(obj)} is not JSON serializable")
    
    def _deserialize_data(self, data_blob: bytes, data_type: str) -> Any:
        """反序列化数据"""
        if data_type == 'dataframe':
            return pickle.loads(data_blob)
        elif data_type in ['dict', 'list']:
            return json.loads(data_blob.decode('utf-8'))
        else:
            return pickle.loads(data_blob)
    
    def get_stats(self) -> Dict[str, Any]:
        """获取缓存统计信息"""
        with self._lock:
            try:
                with sqlite3.connect(self.db_path) as conn:
                    cursor = conn.cursor()
                    
                    # 总条目数
                    cursor.execute('SELECT COUNT(*) FROM price_distribution_cache')
                    total_count = cursor.fetchone()[0]
                    
                    # 过期条目数
                    cursor.execute('''
                        SELECT COUNT(*) FROM price_distribution_cache 
                        WHERE expire_time IS NOT NULL 
                        AND expire_time < CURRENT_TIMESTAMP
                    ''')
                    expired_count = cursor.fetchone()[0]
                    
                    # 数据库文件大小
                    import os
                    db_size = os.path.getsize(self.db_path) if os.path.exists(self.db_path) else 0
                    
                    # 计算命中率
                    total_requests = self._stats['hits'] + self._stats['misses']
                    hit_rate = self._stats['hits'] / total_requests if total_requests > 0 else 0
                    
                    return {
                        'total_entries': total_count,
                        'expired_entries': expired_count,
                        'valid_entries': total_count - expired_count,
                        'db_size_bytes': db_size,
                        'db_size_mb': round(db_size / (1024 * 1024), 2),
                        'hit_rate': hit_rate,
                        **self._stats
                    }
                    
            except Exception as e:
                logger.error(f"Get database cache stats failed: {e}")
                return {}


class PriceDistributionCacheManager:
    """涨跌分布统计缓存管理器"""
    
    def __init__(self, db_path: str = None, memory_cache_size: int = 1000, 
                 db_manager: Optional['DatabaseManager'] = None,
                 logger: Optional[logging.Logger] = None):
        """
        初始化缓存管理器
        
        Args:
            db_path: 数据库文件路径
            memory_cache_size: 内存缓存大小
            db_manager: 数据库管理器实例
            logger: 日志记录器
        """
        self.logger = logger or logging.getLogger(__name__)
        self.key_manager = CacheKeyManager()
        
        # 初始化缓存层
        self.memory_cache = MemoryCacheLayer(memory_cache_size)
        
        # 初始化数据库缓存层
        if db_manager:
            # 使用新的数据库缓存层（集成到主数据库）
            from .price_distribution_database_cache import PriceDistributionDatabaseCache
            self.structured_db_cache = PriceDistributionDatabaseCache(db_manager)
        else:
            self.structured_db_cache = None
        
        # 保持向后兼容的SQLite缓存
        if db_path:
            self.db_cache = DatabaseCacheLayer(db_path)
        else:
            self.db_cache = None
        
        # 缓存层级配置
        self.cache_layers = {
            'L1': 'memory',           # 内存缓存 (最快)
            'L2': 'structured_db',    # 结构化数据库缓存 (持久，结构化)
            'L3': 'database'          # SQLite缓存 (持久，向后兼容)
        }
        
        self._global_stats = {
            'total_requests': 0,
            'cache_hits': 0,
            'cache_misses': 0
        }
    
    # ========== 查询操作 ==========
    
    async def get_distribution_stats(self, trade_date: str, market: str = 'total', 
                                   include_st: bool = True, **kwargs) -> Optional[Any]:
        """
        获取分布统计缓存
        
        Args:
            trade_date: 交易日期
            market: 市场类型
            include_st: 是否包含ST股票
            **kwargs: 其他参数
            
        Returns:
            缓存的统计数据
        """
        cache_key = self.key_manager.generate_key(
            'price_distribution',
            trade_date=trade_date,
            market=market,
            include_st=include_st,
            **kwargs
        )
        
        return await self._get_from_cache(cache_key)
    
    async def get_stock_data_cache(self, trade_date: str) -> Optional[pd.DataFrame]:
        """
        获取股票数据缓存
        
        Args:
            trade_date: 交易日期
            
        Returns:
            缓存的股票数据
        """
        cache_key = self.key_manager.generate_key('stock_data', trade_date=trade_date)
        return await self._get_from_cache(cache_key)
    
    async def get_market_classification_cache(self, trade_date: str) -> Optional[Dict[str, Any]]:
        """
        获取市场分类缓存
        
        Args:
            trade_date: 交易日期
            
        Returns:
            缓存的市场分类数据
        """
        cache_key = self.key_manager.generate_key('market_classification', trade_date=trade_date)
        return await self._get_from_cache(cache_key)
    
    async def get_distribution_summary_cache(self, trade_date: str) -> Optional[Dict[str, Any]]:
        """
        获取分布摘要缓存
        
        Args:
            trade_date: 交易日期
            
        Returns:
            缓存的分布摘要
        """
        cache_key = self.key_manager.generate_key('distribution_summary', trade_date=trade_date)
        return await self._get_from_cache(cache_key)
    
    async def batch_get_cache(self, cache_keys: List[str]) -> Dict[str, Any]:
        """
        批量获取缓存
        
        Args:
            cache_keys: 缓存键列表
            
        Returns:
            缓存数据字典
        """
        results = {}
        for key in cache_keys:
            data = await self._get_from_cache(key)
            if data is not None:
                results[key] = data
        return results
    
    # ========== 存储操作 ==========
    
    async def set_distribution_stats(self, trade_date: str, stats: Any, 
                                   market: str = 'total', include_st: bool = True, 
                                   ttl: Optional[int] = None, **kwargs) -> bool:
        """
        设置分布统计缓存
        
        Args:
            trade_date: 交易日期
            stats: 统计数据
            market: 市场类型
            include_st: 是否包含ST股票
            ttl: 过期时间（秒）
            **kwargs: 其他参数
            
        Returns:
            是否设置成功
        """
        cache_key = self.key_manager.generate_key(
            'price_distribution',
            trade_date=trade_date,
            market=market,
            include_st=include_st,
            **kwargs
        )
        
        if ttl is None:
            ttl = self.key_manager.get_ttl('price_distribution')
        
        return await self._set_to_cache(cache_key, stats, ttl)
    
    async def set_stock_data_cache(self, trade_date: str, data: pd.DataFrame, 
                                 ttl: Optional[int] = None) -> bool:
        """
        设置股票数据缓存
        
        Args:
            trade_date: 交易日期
            data: 股票数据
            ttl: 过期时间（秒）
            
        Returns:
            是否设置成功
        """
        cache_key = self.key_manager.generate_key('stock_data', trade_date=trade_date)
        
        if ttl is None:
            ttl = self.key_manager.get_ttl('stock_data')
        
        return await self._set_to_cache(cache_key, data, ttl)
    
    async def set_market_classification_cache(self, trade_date: str, classification: Dict[str, Any], 
                                            ttl: Optional[int] = None) -> bool:
        """
        设置市场分类缓存
        
        Args:
            trade_date: 交易日期
            classification: 市场分类数据
            ttl: 过期时间（秒）
            
        Returns:
            是否设置成功
        """
        cache_key = self.key_manager.generate_key('market_classification', trade_date=trade_date)
        
        if ttl is None:
            ttl = self.key_manager.get_ttl('market_classification')
        
        return await self._set_to_cache(cache_key, classification, ttl)
    
    async def batch_set_cache(self, cache_data: Dict[str, Any], ttl: Optional[int] = None) -> Dict[str, bool]:
        """
        批量设置缓存
        
        Args:
            cache_data: 缓存数据字典 {key: data}
            ttl: 过期时间（秒）
            
        Returns:
            设置结果字典 {key: success}
        """
        results = {}
        for key, data in cache_data.items():
            results[key] = await self._set_to_cache(key, data, ttl)
        return results
    
    # ========== 删除操作 ==========
    
    async def delete_distribution_stats(self, trade_date: str, market: str = None, 
                                      include_st: bool = None) -> int:
        """
        删除分布统计缓存
        
        Args:
            trade_date: 交易日期
            market: 市场类型（None表示所有市场）
            include_st: 是否包含ST股票（None表示所有）
            
        Returns:
            删除的条目数
        """
        if market is None and include_st is None:
            # 删除该日期的所有分布统计缓存
            pattern = f"price_dist:{trade_date}:*"
            return await self._delete_by_pattern(pattern)
        else:
            # 删除特定的缓存条目
            cache_key = self.key_manager.generate_key(
                'price_distribution',
                trade_date=trade_date,
                market=market or 'total',
                include_st=include_st if include_st is not None else True
            )
            success = await self._delete_from_cache(cache_key)
            return 1 if success else 0
    
    async def delete_by_trade_date(self, trade_date: str) -> int:
        """
        删除指定交易日期的所有缓存
        
        Args:
            trade_date: 交易日期
            
        Returns:
            删除的条目数
        """
        pattern = f"*:{trade_date}:*"
        return await self._delete_by_pattern(pattern)
    
    async def delete_by_pattern(self, pattern: str) -> int:
        """
        按模式删除缓存
        
        Args:
            pattern: 匹配模式
            
        Returns:
            删除的条目数
        """
        return await self._delete_by_pattern(pattern)
    
    async def clear_expired_cache(self) -> int:
        """
        清理过期缓存
        
        Returns:
            清理的条目数
        """
        total_cleared = 0
        
        # 清理内存缓存
        memory_cleared = await self.memory_cache.clear_expired()
        total_cleared += memory_cleared
        
        # 清理数据库缓存
        if self.db_cache:
            db_cleared = await self.db_cache.clear_expired()
            total_cleared += db_cleared
        
        self.logger.info(f"Cleared {total_cleared} expired cache entries")
        return total_cleared
    
    async def clear_all_cache(self) -> int:
        """
        清空所有缓存
        
        Returns:
            清理的条目数
        """
        total_cleared = 0
        
        # 清空内存缓存
        memory_cleared = await self.memory_cache.clear()
        total_cleared += memory_cleared
        
        # 清空数据库缓存
        if self.db_cache:
            db_cleared = await self.db_cache.clear()
            total_cleared += db_cleared
        
        self.logger.info(f"Cleared all {total_cleared} cache entries")
        return total_cleared
    
    # ========== 管理操作 ==========
    
    async def get_cache_info(self, trade_date: str = None) -> Dict[str, Any]:
        """
        获取缓存信息
        
        Args:
            trade_date: 交易日期（None表示所有）
            
        Returns:
            缓存信息字典
        """
        info = {
            'memory_cache': self.memory_cache.get_stats(),
            'global_stats': self._global_stats.copy()
        }
        
        # 结构化数据库缓存信息
        if self.structured_db_cache:
            info['structured_database_cache'] = await self.structured_db_cache.get_cache_info()
        
        # SQLite数据库缓存信息
        if self.db_cache:
            info['database_cache'] = self.db_cache.get_stats()
        
        # 如果指定了交易日期，获取该日期的缓存信息
        if trade_date:
            info['trade_date_info'] = await self._get_trade_date_cache_info(trade_date)
        
        return info
    
    async def get_cache_statistics(self) -> Dict[str, Any]:
        """
        获取缓存统计信息
        
        Returns:
            统计信息字典
        """
        stats = {
            'memory_cache': self.memory_cache.get_stats(),
            'global_stats': self._global_stats.copy()
        }
        
        if self.db_cache:
            stats['database_cache'] = self.db_cache.get_stats()
        
        # 计算总体命中率
        total_requests = self._global_stats['total_requests']
        if total_requests > 0:
            stats['overall_hit_rate'] = self._global_stats['cache_hits'] / total_requests
        else:
            stats['overall_hit_rate'] = 0
        
        return stats
    
    async def validate_cache_consistency(self, trade_date: str) -> Dict[str, Any]:
        """
        验证缓存一致性
        
        Args:
            trade_date: 交易日期
            
        Returns:
            一致性检查结果
        """
        result = {
            'trade_date': trade_date,
            'consistent': True,
            'issues': [],
            'memory_entries': 0,
            'database_entries': 0
        }
        
        try:
            # 检查内存缓存中该日期的条目
            memory_keys = []
            with self.memory_cache._lock:
                for key in self.memory_cache._cache.keys():
                    if trade_date in key:
                        memory_keys.append(key)
            result['memory_entries'] = len(memory_keys)
            
            # 检查数据库缓存中该日期的条目
            if self.db_cache:
                with sqlite3.connect(self.db_cache.db_path) as conn:
                    cursor = conn.cursor()
                    cursor.execute('''
                        SELECT COUNT(*) FROM price_distribution_cache 
                        WHERE cache_key LIKE ?
                    ''', (f'%:{trade_date}:%',))
                    result['database_entries'] = cursor.fetchone()[0]
            
            # 检查数据一致性
            for key in memory_keys:
                memory_data = await self.memory_cache.get(key)
                if self.db_cache:
                    db_data = await self.db_cache.get(key)
                    if memory_data != db_data:
                        result['consistent'] = False
                        result['issues'].append(f"Data mismatch for key: {key}")
        
        except Exception as e:
            result['consistent'] = False
            result['issues'].append(f"Validation error: {str(e)}")
        
        return result
    
    async def refresh_cache(self, trade_date: str, force: bool = False) -> Dict[str, Any]:
        """
        刷新缓存
        
        Args:
            trade_date: 交易日期
            force: 是否强制刷新
            
        Returns:
            刷新结果
        """
        result = {
            'trade_date': trade_date,
            'refreshed': False,
            'deleted_entries': 0,
            'message': ''
        }
        
        try:
            if force:
                # 强制刷新：删除该日期的所有缓存
                deleted = await self.delete_by_trade_date(trade_date)
                result['deleted_entries'] = deleted
                result['refreshed'] = True
                result['message'] = f"Force refreshed cache for {trade_date}, deleted {deleted} entries"
            else:
                # 普通刷新：只删除过期缓存
                deleted = await self.clear_expired_cache()
                result['deleted_entries'] = deleted
                result['refreshed'] = True
                result['message'] = f"Refreshed expired cache, deleted {deleted} entries"
        
        except Exception as e:
            result['message'] = f"Refresh failed: {str(e)}"
        
        return result
    
    # ========== 内部方法 ==========
    
    async def _get_from_cache(self, cache_key: str) -> Optional[Any]:
        """从缓存获取数据"""
        self._global_stats['total_requests'] += 1
        
        # 1. 先从内存缓存获取
        data = await self.memory_cache.get(cache_key)
        if data is not None:
            self._global_stats['cache_hits'] += 1
            return data
        
        # 2. 从结构化数据库缓存获取（针对分布统计数据）
        if self.structured_db_cache and 'price_dist:' in cache_key:
            # 解析缓存键获取交易日期
            key_parts = self.key_manager.parse_key(cache_key)
            if key_parts.get('type') == 'price_dist' and 'trade_date' in key_parts:
                trade_date = key_parts['trade_date']
                data = await self.structured_db_cache.get(trade_date)
                if data is not None:
                    # 将数据同步到内存缓存
                    await self.memory_cache.set(cache_key, data, 
                                              self.key_manager.get_ttl('price_distribution'))
                    self._global_stats['cache_hits'] += 1
                    return data
        
        # 3. 从SQLite数据库缓存获取
        if self.db_cache:
            data = await self.db_cache.get(cache_key)
            if data is not None:
                # 将数据同步到内存缓存
                await self.memory_cache.set(cache_key, data, 
                                          self.key_manager.get_ttl('price_distribution'))
                self._global_stats['cache_hits'] += 1
                return data
        
        # 4. 缓存未命中
        self._global_stats['cache_misses'] += 1
        return None
    
    async def _set_to_cache(self, cache_key: str, data: Any, ttl: int) -> bool:
        """设置缓存数据"""
        success = True
        
        # 设置内存缓存
        memory_success = await self.memory_cache.set(cache_key, data, ttl)
        if not memory_success:
            success = False
        
        # 设置结构化数据库缓存（针对分布统计数据）
        if self.structured_db_cache and 'price_dist:' in cache_key:
            # 解析缓存键获取交易日期
            key_parts = self.key_manager.parse_key(cache_key)
            if (key_parts.get('type') == 'price_dist' and 'trade_date' in key_parts and 
                hasattr(data, 'trade_date')):  # 确保是PriceDistributionStats对象
                trade_date = key_parts['trade_date']
                structured_success = await self.structured_db_cache.set(trade_date, data, ttl)
                if not structured_success:
                    success = False
        
        # 设置SQLite数据库缓存
        if self.db_cache:
            db_success = await self.db_cache.set(cache_key, data, ttl)
            if not db_success:
                success = False
        
        return success
    
    async def _delete_from_cache(self, cache_key: str) -> bool:
        """从缓存删除数据"""
        memory_deleted = await self.memory_cache.delete(cache_key)
        structured_db_deleted = True
        db_deleted = True
        
        # 从结构化数据库缓存删除
        if self.structured_db_cache and 'price_dist:' in cache_key:
            key_parts = self.key_manager.parse_key(cache_key)
            if key_parts.get('type') == 'price_dist' and 'trade_date' in key_parts:
                trade_date = key_parts['trade_date']
                structured_db_deleted = await self.structured_db_cache.delete(trade_date)
        
        # 从SQLite数据库缓存删除
        if self.db_cache:
            db_deleted = await self.db_cache.delete(cache_key)
        
        return memory_deleted or structured_db_deleted or db_deleted
    
    async def _delete_by_pattern(self, pattern: str) -> int:
        """按模式删除缓存"""
        total_deleted = 0
        
        # 从内存缓存删除匹配的键
        import fnmatch
        with self.memory_cache._lock:
            keys_to_delete = []
            for key in self.memory_cache._cache.keys():
                if fnmatch.fnmatch(key, pattern):
                    keys_to_delete.append(key)
            
            for key in keys_to_delete:
                await self.memory_cache.delete(key)
                total_deleted += 1
        
        # 从结构化数据库缓存删除（针对分布统计数据）
        if self.structured_db_cache and 'price_dist:' in pattern:
            # 解析模式以提取交易日期
            # 模式格式: price_dist:{trade_date}:*
            import re
            match = re.match(r'price_dist:(\d{8}):\*', pattern)
            if match:
                trade_date = match.group(1)
                structured_deleted = await self.structured_db_cache.delete(trade_date)
                if structured_deleted:
                    total_deleted += 1
        
        # 从SQLite数据库缓存删除
        if self.db_cache:
            # 将通配符模式转换为SQL LIKE模式
            sql_pattern = pattern.replace('*', '%').replace('?', '_')
            db_deleted = await self.db_cache.delete_by_pattern(sql_pattern)
            total_deleted += db_deleted
        
        return total_deleted
    
    async def _get_trade_date_cache_info(self, trade_date: str) -> Dict[str, Any]:
        """获取指定交易日期的缓存信息"""
        info = {
            'trade_date': trade_date,
            'memory_entries': 0,
            'database_entries': 0,
            'cache_keys': []
        }
        
        # 统计内存缓存中的条目
        with self.memory_cache._lock:
            for key in self.memory_cache._cache.keys():
                if trade_date in key:
                    info['memory_entries'] += 1
                    info['cache_keys'].append(key)
        
        # 统计数据库缓存中的条目
        if self.db_cache:
            try:
                with sqlite3.connect(self.db_cache.db_path) as conn:
                    cursor = conn.cursor()
                    cursor.execute('''
                        SELECT COUNT(*), GROUP_CONCAT(cache_key) FROM price_distribution_cache 
                        WHERE cache_key LIKE ?
                    ''', (f'%:{trade_date}:%',))
                    result = cursor.fetchone()
                    if result:
                        info['database_entries'] = result[0] or 0
                        if result[1]:
                            db_keys = result[1].split(',')
                            info['cache_keys'].extend([k for k in db_keys if k not in info['cache_keys']])
            except Exception as e:
                self.logger.error(f"Failed to get database cache info: {e}")
        
        return info