"""
QuickStock客户端财务报告集成测试

测试客户端财务报告相关方法的集成功能
"""

import pytest
import asyncio
from unittest.mock import Mock, patch, AsyncMock
from datetime import datetime, timedelta

from quickstock.client import QuickStockClient
from quickstock.config import Config
from quickstock.core.errors import QuickStockError, ValidationError, FinancialDataError
from quickstock.models import FinancialReport, EarningsForecast, FlashReport


class TestClientFinancialReportsIntegration:
    """客户端财务报告集成测试类"""
    
    def setup_method(self):
        """测试前置设置"""
        # 创建测试配置
        self.test_config = Config(
            cache_enabled=True,
            log_level='ERROR',
            enable_auto_code_conversion=True
        )
        
        # 创建客户端
        self.client = QuickStockClient(self.test_config)
        
        # 创建测试数据
        self.sample_financial_report = FinancialReport(
            ts_code='000001.SZ',
            report_date='20231231',
            report_type='A',
            total_revenue=1000000.0,
            net_profit=200000.0,
            total_assets=5000000.0,
            total_liabilities=3000000.0,
            shareholders_equity=2000000.0,
            operating_cash_flow=300000.0,
            eps=2.5,
            roe=10.0
        )
        
        self.sample_earnings_forecast = EarningsForecast(
            ts_code='000001.SZ',
            forecast_date='20240315',
            forecast_period='20231231',
            forecast_type='预增',
            net_profit_min=180000.0,
            net_profit_max=220000.0,
            growth_rate_min=10.0,
            growth_rate_max=20.0,
            forecast_summary='预计净利润同比增长10%-20%'
        )
        
        self.sample_flash_report = FlashReport(
            ts_code='000001.SZ',
            report_date='20240430',
            publish_date='20240430',
            report_period='20240331',
            total_revenue=250000.0,
            net_profit=50000.0,
            revenue_growth=15.0,
            profit_growth=25.0,
            eps=0.6,
            report_summary='一季度业绩快报'
        )
    
    def test_get_financial_reports_success(self):
        """测试获取财务报告成功"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            # 创建mock服务实例
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 设置异步方法返回值
            async def mock_get_reports(*args, **kwargs):
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 执行测试
            result = self.client.get_financial_reports('000001.SZ')
            
            # 验证结果
            assert isinstance(result, list)
            assert len(result) == 1
            assert result[0]['ts_code'] == '000001.SZ'
            assert result[0]['report_type'] == 'A'
            assert result[0]['total_revenue'] == 1000000.0
    
    def test_get_financial_reports_with_parameters(self):
        """测试带参数获取财务报告"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(request):
                # 验证请求参数
                assert request.ts_code == '000001.SZ'
                assert request.start_date == '20230101'
                assert request.end_date == '20231231'
                assert request.report_type == 'A'
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 执行测试
            result = self.client.get_financial_reports(
                '000001.SZ',
                start_date='20230101',
                end_date='20231231',
                report_type='A'
            )
            
            assert len(result) == 1
    
    def test_get_financial_reports_code_conversion(self):
        """测试财务报告股票代码转换"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(request):
                # 验证代码已被标准化
                assert request.ts_code == '000001.SZ'
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 测试不同格式的代码
            test_codes = ['sz.000001', '0.000001', 'hs_000001', '000001']
            
            for code in test_codes:
                result = self.client.get_financial_reports(code)
                assert len(result) == 1
    
    def test_get_financial_reports_validation_error(self):
        """测试财务报告参数验证错误"""
        # 测试无效股票代码
        with pytest.raises(QuickStockError, match="获取财务报告失败"):
            self.client.get_financial_reports('')
        
        # 测试无效日期格式
        with pytest.raises(QuickStockError, match="获取财务报告失败"):
            self.client.get_financial_reports('000001.SZ', start_date='invalid_date')
    
    def test_get_earnings_forecast_success(self):
        """测试获取业绩预告成功"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_forecast(*args, **kwargs):
                return [self.sample_earnings_forecast]
            
            mock_service.get_earnings_forecast = mock_get_forecast
            
            # 执行测试
            result = self.client.get_earnings_forecast('000001.SZ')
            
            # 验证结果
            assert isinstance(result, list)
            assert len(result) == 1
            assert result[0]['ts_code'] == '000001.SZ'
            assert result[0]['forecast_type'] == '预增'
            assert result[0]['net_profit_min'] == 180000.0
    
    def test_get_earnings_forecast_with_parameters(self):
        """测试带参数获取业绩预告"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_forecast(request):
                assert request.ts_code == '000001.SZ'
                assert request.forecast_type == '预增'
                return [self.sample_earnings_forecast]
            
            mock_service.get_earnings_forecast = mock_get_forecast
            
            result = self.client.get_earnings_forecast(
                '000001.SZ',
                forecast_type='预增'
            )
            
            assert len(result) == 1
    
    def test_get_earnings_flash_reports_success(self):
        """测试获取业绩快报成功"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_flash_reports(*args, **kwargs):
                return [self.sample_flash_report]
            
            mock_service.get_earnings_flash_reports = mock_get_flash_reports
            
            # 执行测试
            result = self.client.get_earnings_flash_reports('000001.SZ')
            
            # 验证结果
            assert isinstance(result, list)
            assert len(result) == 1
            assert result[0]['ts_code'] == '000001.SZ'
            assert result[0]['total_revenue'] == 250000.0
            assert result[0]['profit_growth'] == 25.0
    
    def test_get_earnings_flash_reports_with_sort(self):
        """测试带排序参数获取业绩快报"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_flash_reports(request):
                assert request.sort_by == 'report_date'
                return [self.sample_flash_report]
            
            mock_service.get_earnings_flash_reports = mock_get_flash_reports
            
            result = self.client.get_earnings_flash_reports(
                '000001.SZ',
                sort_by='report_date'
            )
            
            assert len(result) == 1
    
    def test_get_batch_financial_data_success(self):
        """测试批量获取财务数据成功"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 创建批量测试数据
            batch_result = {
                '000001.SZ': {
                    'financial_reports': [self.sample_financial_report],
                    'earnings_forecast': [self.sample_earnings_forecast],
                    'flash_reports': [self.sample_flash_report]
                },
                '000002.SZ': {
                    'financial_reports': [],
                    'earnings_forecast': [],
                    'flash_reports': []
                }
            }
            
            async def mock_batch_get(**kwargs):
                assert len(kwargs['stock_codes']) == 2
                assert '000001.SZ' in kwargs['stock_codes']
                assert '000002.SZ' in kwargs['stock_codes']
                return batch_result
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            # 执行测试
            result = self.client.get_batch_financial_data(['000001.SZ', '000002.SZ'])
            
            # 验证结果
            assert isinstance(result, dict)
            assert len(result) == 2
            assert '000001.SZ' in result
            assert '000002.SZ' in result
            
            # 验证第一只股票的数据
            stock1_data = result['000001.SZ']
            assert len(stock1_data['financial_reports']) == 1
            assert len(stock1_data['earnings_forecast']) == 1
            assert len(stock1_data['flash_reports']) == 1
            
            # 验证第二只股票的数据（空数据）
            stock2_data = result['000002.SZ']
            assert len(stock2_data['financial_reports']) == 0
            assert len(stock2_data['earnings_forecast']) == 0
            assert len(stock2_data['flash_reports']) == 0
    
    def test_get_batch_financial_data_with_data_types(self):
        """测试指定数据类型的批量获取"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_batch_get(**kwargs):
                assert kwargs['data_types'] == ['financial_reports', 'earnings_forecast']
                return {
                    '000001.SZ': {
                        'financial_reports': [self.sample_financial_report],
                        'earnings_forecast': [self.sample_earnings_forecast]
                    }
                }
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            result = self.client.get_batch_financial_data(
                ['000001.SZ'],
                data_types=['financial_reports', 'earnings_forecast']
            )
            
            assert 'financial_reports' in result['000001.SZ']
            assert 'earnings_forecast' in result['000001.SZ']
            assert 'flash_reports' not in result['000001.SZ']
    
    def test_get_batch_financial_data_validation_errors(self):
        """测试批量获取参数验证错误"""
        # 测试空股票代码列表
        with pytest.raises(QuickStockError, match="股票代码列表不能为空"):
            self.client.get_batch_financial_data([])
        
        # 测试超过批量大小限制
        large_codes = [f'00000{i}.SZ' for i in range(51)]
        with pytest.raises(QuickStockError, match="批量大小超过限制"):
            self.client.get_batch_financial_data(large_codes)
        
        # 测试无效数据类型
        with pytest.raises(QuickStockError, match="无效的数据类型"):
            self.client.get_batch_financial_data(
                ['000001.SZ'],
                data_types=['invalid_type']
            )
    
    def test_batch_code_conversion(self):
        """测试批量股票代码转换"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_batch_get(**kwargs):
                # 验证所有代码都被标准化
                codes = kwargs['stock_codes']
                assert '000001.SZ' in codes
                assert '000002.SZ' in codes
                assert '600000.SH' in codes
                return {code: {'financial_reports': []} for code in codes}
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            # 使用不同格式的代码
            mixed_codes = ['sz.000001', '0.000002', 'sh.600000']
            result = self.client.get_batch_financial_data(mixed_codes)
            
            # 验证返回的键是标准化的代码
            assert '000001.SZ' in result
            assert '000002.SZ' in result
            assert '600000.SH' in result
    
    def test_service_error_handling(self):
        """测试服务层错误处理"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟服务层异常
            async def mock_error(*args, **kwargs):
                raise FinancialDataError("服务层错误")
            
            mock_service.get_financial_reports = mock_error
            
            # 验证异常被正确处理
            with pytest.raises(QuickStockError, match="获取财务报告失败"):
                self.client.get_financial_reports('000001.SZ')
    
    def test_async_method_execution(self):
        """测试异步方法执行"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 创建真实的异步方法
            async def real_async_method(*args, **kwargs):
                await asyncio.sleep(0.01)  # 模拟异步操作
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = real_async_method
            
            # 执行测试
            result = self.client.get_financial_reports('000001.SZ')
            
            # 验证异步方法被正确执行
            assert len(result) == 1
    
    def test_logging_integration(self):
        """测试日志集成"""
        # 创建带日志的客户端
        config = Config(log_level='INFO')
        client = QuickStockClient(config)
        
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(*args, **kwargs):
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 验证日志记录器被传递给服务
            result = client.get_financial_reports('000001.SZ')
            
            # 验证服务实例化时传递了日志记录器
            mock_service_class.assert_called_once()
            call_args = mock_service_class.call_args
            assert call_args[0][2] == client.logger  # 第三个参数是logger
    
    def test_cache_integration(self):
        """测试缓存集成"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(*args, **kwargs):
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 执行测试
            result = self.client.get_financial_reports('000001.SZ', force_refresh=True)
            
            # 验证缓存层被传递给服务
            mock_service_class.assert_called_once()
            call_args = mock_service_class.call_args
            assert call_args[0][1] == self.client.data_manager.cache_layer
    
    def test_parameter_forwarding(self):
        """测试参数转发"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(request):
                # 验证额外参数被正确转发
                assert request.extra_params['force_refresh'] is True
                assert request.extra_params['custom_param'] == 'test_value'
                return [self.sample_financial_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 执行测试
            result = self.client.get_financial_reports(
                '000001.SZ',
                force_refresh=True,
                custom_param='test_value'
            )
            
            assert len(result) == 1
    
    def teardown_method(self):
        """测试后清理"""
        # 清理客户端资源
        if hasattr(self.client, 'data_manager'):
            self.client.clear_cache()