"""
QuickStock客户端财务报告单元测试

测试客户端财务报告方法的参数验证和请求路由逻辑
"""

import pytest
from unittest.mock import Mock, patch

from quickstock.client import QuickStockClient
from quickstock.config import Config
from quickstock.core.errors import ValidationError, QuickStockError
from quickstock.models import FinancialReportsRequest, EarningsForecastRequest, FlashReportsRequest


class TestClientFinancialReportsUnit:
    """客户端财务报告单元测试类"""
    
    def setup_method(self):
        """测试前置设置"""
        self.test_config = Config(
            cache_enabled=True,
            log_level='ERROR',
            enable_auto_code_conversion=True
        )
        self.client = QuickStockClient(self.test_config)
    
    def test_financial_reports_request_creation(self):
        """测试财务报告请求创建"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(request):
                # 验证请求对象类型和属性
                assert isinstance(request, FinancialReportsRequest)
                assert request.data_type == 'financial_reports'
                assert request.ts_code == '000001.SZ'
                assert request.start_date == '20230101'
                assert request.end_date == '20231231'
                assert request.report_type == 'A'
                assert request.fields == ['total_revenue', 'net_profit']
                assert request.extra_params['force_refresh'] is True
                return []
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 执行测试
            self.client.get_financial_reports(
                '000001.SZ',
                start_date='20230101',
                end_date='20231231',
                report_type='A',
                fields=['total_revenue', 'net_profit'],
                force_refresh=True
            )
    
    def test_earnings_forecast_request_creation(self):
        """测试业绩预告请求创建"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_forecast(request):
                assert isinstance(request, EarningsForecastRequest)
                assert request.data_type == 'earnings_forecast'
                assert request.ts_code == '000001.SZ'
                assert request.forecast_type == '预增'
                return []
            
            mock_service.get_earnings_forecast = mock_get_forecast
            
            self.client.get_earnings_forecast(
                '000001.SZ',
                forecast_type='预增'
            )
    
    def test_flash_reports_request_creation(self):
        """测试业绩快报请求创建"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_flash_reports(request):
                assert isinstance(request, FlashReportsRequest)
                assert request.data_type == 'flash_reports'
                assert request.ts_code == '000001.SZ'
                assert request.sort_by == 'report_date'
                return []
            
            mock_service.get_earnings_flash_reports = mock_get_flash_reports
            
            self.client.get_earnings_flash_reports(
                '000001.SZ',
                sort_by='report_date'
            )
    
    def test_stock_code_validation_empty(self):
        """测试空股票代码验证"""
        with pytest.raises(QuickStockError, match="获取财务报告失败"):
            self.client.get_financial_reports('')
        
        with pytest.raises(QuickStockError, match="获取业绩预告失败"):
            self.client.get_earnings_forecast('')
        
        with pytest.raises(QuickStockError, match="获取业绩快报失败"):
            self.client.get_earnings_flash_reports('')
    
    def test_stock_code_validation_none(self):
        """测试None股票代码验证"""
        with pytest.raises(QuickStockError):
            self.client.get_financial_reports(None)
        
        with pytest.raises(QuickStockError):
            self.client.get_earnings_forecast(None)
        
        with pytest.raises(QuickStockError):
            self.client.get_earnings_flash_reports(None)
    
    def test_date_format_validation(self):
        """测试日期格式验证"""
        # 测试无效的开始日期
        with pytest.raises(QuickStockError, match="无效的日期格式"):
            self.client.get_financial_reports('000001.SZ', start_date='invalid')
        
        # 测试无效的结束日期
        with pytest.raises(QuickStockError, match="无效的日期格式"):
            self.client.get_financial_reports('000001.SZ', end_date='2023-13-45')
        
        # 测试业绩预告日期验证
        with pytest.raises(QuickStockError, match="无效的日期格式"):
            self.client.get_earnings_forecast('000001.SZ', start_date='20231301')
        
        # 测试业绩快报日期验证
        with pytest.raises(QuickStockError, match="无效的日期格式"):
            self.client.get_earnings_flash_reports('000001.SZ', end_date='abc')
    
    def test_valid_date_formats(self):
        """测试有效的日期格式"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_method(*args, **kwargs):
                return []
            
            mock_service.get_financial_reports = mock_method
            mock_service.get_earnings_forecast = mock_method
            mock_service.get_earnings_flash_reports = mock_method
            
            # 测试YYYYMMDD格式
            self.client.get_financial_reports('000001.SZ', start_date='20230101', end_date='20231231')
            
            # 测试YYYY-MM-DD格式
            self.client.get_earnings_forecast('000001.SZ', start_date='2023-01-01', end_date='2023-12-31')
            
            # 测试相同格式避免比较问题
            self.client.get_earnings_flash_reports('000001.SZ', start_date='2023-01-01', end_date='2023-12-31')
    
    def test_stock_code_normalization(self):
        """测试股票代码标准化"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(request):
                # 验证代码被标准化为000001.SZ
                assert request.ts_code == '000001.SZ'
                return []
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 测试不同格式的代码都被标准化
            test_codes = [
                'sz.000001',    # Baostock格式
                '0.000001',     # 东方财富格式
                'hs_000001',    # 同花顺格式
                '000001',       # 纯数字
                '000001.SZ'     # 标准格式
            ]
            
            for code in test_codes:
                self.client.get_financial_reports(code)
    
    def test_service_instantiation(self):
        """测试服务实例化"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_method(*args, **kwargs):
                return []
            
            mock_service.get_financial_reports = mock_method
            
            # 执行方法
            self.client.get_financial_reports('000001.SZ')
            
            # 验证服务被正确实例化
            mock_service_class.assert_called_once_with(
                self.client.data_manager,
                self.client.data_manager.cache_layer,
                self.client.logger
            )
    
    def test_batch_validation_empty_list(self):
        """测试批量方法空列表验证"""
        with pytest.raises(QuickStockError, match="股票代码列表不能为空"):
            self.client.get_batch_financial_data([])
    
    def test_batch_validation_size_limit(self):
        """测试批量方法大小限制"""
        # 创建超过50个的股票代码列表
        large_list = [f'00000{i:02d}.SZ' for i in range(51)]
        
        with pytest.raises(QuickStockError, match="批量大小超过限制"):
            self.client.get_batch_financial_data(large_list)
    
    def test_batch_validation_data_types(self):
        """测试批量方法数据类型验证"""
        # 测试无效数据类型
        with pytest.raises(QuickStockError, match="无效的数据类型"):
            self.client.get_batch_financial_data(
                ['000001.SZ'],
                data_types=['invalid_type']
            )
        
        # 测试部分无效数据类型
        with pytest.raises(QuickStockError, match="无效的数据类型"):
            self.client.get_batch_financial_data(
                ['000001.SZ'],
                data_types=['financial_reports', 'invalid_type']
            )
    
    def test_batch_valid_data_types(self):
        """测试批量方法有效数据类型"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_batch_get(**kwargs):
                # 验证数据类型参数
                expected_types = ['financial_reports', 'earnings_forecast', 'flash_reports']
                assert kwargs['data_types'] == expected_types
                return {'000001.SZ': {dt: [] for dt in expected_types}}
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            # 测试默认数据类型（所有类型）
            self.client.get_batch_financial_data(['000001.SZ'])
    
    def test_batch_code_normalization(self):
        """测试批量方法代码标准化"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_batch_get(**kwargs):
                # 验证所有代码都被标准化
                codes = kwargs['stock_codes']
                assert '000001.SZ' in codes
                assert '000002.SZ' in codes
                assert '600000.SH' in codes
                return {code: {'financial_reports': []} for code in codes}
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            # 使用不同格式的代码
            mixed_codes = ['sz.000001', '0.000002', 'sh.600000']
            self.client.get_batch_financial_data(mixed_codes)
    
    def test_batch_parameter_forwarding(self):
        """测试批量方法参数转发"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_batch_get(**kwargs):
                # 验证参数被正确转发
                assert kwargs['start_date'] == '20230101'
                assert kwargs['end_date'] == '20231231'
                assert kwargs['max_workers'] == 10
                assert kwargs['timeout'] == 120
                assert kwargs['force_refresh'] is True
                return {'000001.SZ': {'financial_reports': []}}
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            self.client.get_batch_financial_data(
                ['000001.SZ'],
                start_date='20230101',
                end_date='20231231',
                max_workers=10,
                timeout=120,
                force_refresh=True
            )
    
    def test_result_format_conversion(self):
        """测试结果格式转换"""
        from quickstock.models import FinancialReport
        
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 创建模拟的FinancialReport对象
            mock_report = FinancialReport(
                ts_code='000001.SZ',
                report_date='20231231',
                report_type='A',
                total_revenue=1000000.0,
                net_profit=200000.0,
                total_assets=5000000.0,
                total_liabilities=3000000.0,
                shareholders_equity=2000000.0,
                operating_cash_flow=300000.0,
                eps=2.5,
                roe=10.0
            )
            
            async def mock_get_reports(*args, **kwargs):
                return [mock_report]
            
            mock_service.get_financial_reports = mock_get_reports
            
            # 执行测试
            result = self.client.get_financial_reports('000001.SZ')
            
            # 验证结果格式
            assert isinstance(result, list)
            assert len(result) == 1
            assert isinstance(result[0], dict)
            assert result[0]['ts_code'] == '000001.SZ'
            assert result[0]['total_revenue'] == 1000000.0
    
    def test_client_initialization_check(self):
        """测试客户端初始化检查"""
        # 创建未初始化的客户端
        client = QuickStockClient.__new__(QuickStockClient)
        
        # 验证未初始化时抛出异常
        with pytest.raises(QuickStockError, match="客户端未正确初始化"):
            client.get_financial_reports('000001.SZ')
    
    def test_logging_messages(self):
        """测试日志消息"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_reports(*args, **kwargs):
                return []
            
            mock_service.get_financial_reports = mock_get_reports
            
            # Mock logger
            with patch.object(self.client, 'logger') as mock_logger:
                self.client.get_financial_reports('000001.SZ')
                
                # 验证成功日志被记录
                mock_logger.info.assert_called_with("获取股票000001.SZ财务报告成功，共0条记录")
    
    def test_error_logging(self):
        """测试错误日志"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_error(*args, **kwargs):
                raise Exception("测试异常")
            
            mock_service.get_financial_reports = mock_error
            
            # Mock logger
            with patch.object(self.client, 'logger') as mock_logger:
                with pytest.raises(QuickStockError):
                    self.client.get_financial_reports('000001.SZ')
                
                # 验证错误日志被记录
                mock_logger.error.assert_called_with("获取财务报告失败: 测试异常")
    
    def teardown_method(self):
        """测试后清理"""
        if hasattr(self.client, 'data_manager'):
            self.client.clear_cache()