"""
QuickStockClient 涨跌分布统计接口基础功能测试

测试客户端API接口的基本功能和参数验证
"""

import pytest
import pandas as pd
from unittest.mock import Mock, patch

from quickstock.client import QuickStockClient
from quickstock.config import Config
from quickstock.core.errors import ValidationError, QuickStockError
from quickstock.models.price_distribution_models import PriceDistributionStats


def test_client_methods_exist():
    """测试客户端方法是否存在"""
    methods_to_check = [
        'price_distribution_stats',
        'get_distribution_cache_info', 
        'clear_distribution_cache',
        'refresh_distribution_cache',
        'validate_distribution_cache',
        'get_distribution_cache_stats'
    ]
    
    for method in methods_to_check:
        assert hasattr(QuickStockClient, method), f"Method {method} should exist"


def test_price_distribution_stats_parameter_validation():
    """测试参数验证"""
    # 创建模拟客户端
    config = Mock(spec=Config)
    config.enable_auto_code_conversion = True
    config.log_code_conversions = False
    config.code_conversion_error_strategy = 'strict'
    config.log_level = 'INFO'
    config.log_file = None
    
    mock_data_manager = Mock()
    mock_data_manager.config = Mock()
    mock_data_manager.get_cache_stats.return_value = {}
    mock_data_manager.get_memory_stats.return_value = {}
    mock_data_manager.get_data_source_health.return_value = {}
    
    with patch('quickstock.client.DataManager', return_value=mock_data_manager):
        client = QuickStockClient(config)
        client.data_manager = mock_data_manager
        
        # 测试无效格式参数
        with pytest.raises(QuickStockError) as exc_info:
            client.price_distribution_stats('20240315', format='invalid')
        assert 'Invalid format' in str(exc_info.value)


def test_price_distribution_stats_basic_call():
    """测试基本调用"""
    # 创建模拟客户端
    config = Mock(spec=Config)
    config.enable_auto_code_conversion = True
    config.log_code_conversions = False
    config.code_conversion_error_strategy = 'strict'
    config.log_level = 'INFO'
    config.log_file = None
    
    mock_data_manager = Mock()
    mock_data_manager.config = Mock()
    mock_data_manager.get_cache_stats.return_value = {}
    mock_data_manager.get_memory_stats.return_value = {}
    mock_data_manager.get_data_source_health.return_value = {}
    
    # 创建示例统计数据
    sample_stats = PriceDistributionStats(
        trade_date='20240315',
        total_stocks=1000,
        positive_ranges={'0-3%': 200, '3-5%': 150},
        positive_percentages={'0-3%': 20.0, '3-5%': 15.0},
        negative_ranges={'0到-3%': 180, '-3到-5%': 120},
        negative_percentages={'0到-3%': 18.0, '-3到-5%': 12.0},
        market_breakdown={},
        processing_time=2.5,
        data_quality_score=0.95
    )
    
    with patch('quickstock.client.DataManager', return_value=mock_data_manager):
        client = QuickStockClient(config)
        client.data_manager = mock_data_manager
        
        # 模拟服务调用
        with patch('quickstock.services.price_distribution_stats_service.PriceDistributionStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_stats(request):
                return sample_stats
            
            mock_service.get_price_distribution_stats = mock_get_stats
            
            # 调用方法
            result = client.price_distribution_stats('20240315')
            
            # 验证结果
            assert isinstance(result, pd.DataFrame)
            assert not result.empty
            assert 'trade_date' in result.columns
            assert result['trade_date'].iloc[0] == '20240315'


def test_cache_management_methods():
    """测试缓存管理方法"""
    # 创建模拟客户端
    config = Mock(spec=Config)
    config.enable_auto_code_conversion = True
    config.log_code_conversions = False
    config.code_conversion_error_strategy = 'strict'
    config.log_level = 'INFO'
    config.log_file = None
    
    mock_data_manager = Mock()
    mock_data_manager.config = Mock()
    mock_data_manager.get_cache_stats.return_value = {}
    mock_data_manager.get_memory_stats.return_value = {}
    mock_data_manager.get_data_source_health.return_value = {}
    
    with patch('quickstock.client.DataManager', return_value=mock_data_manager):
        client = QuickStockClient(config)
        client.data_manager = mock_data_manager
        
        # 模拟服务调用
        with patch('quickstock.services.price_distribution_stats_service.PriceDistributionStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟缓存操作
            async def mock_get_cache_info(trade_date=None):
                return {'cache_entries': 5, 'total_size': 1024}
            
            async def mock_clear_cache(trade_date=None, pattern=None):
                return 5
            
            mock_service.get_cache_info = mock_get_cache_info
            mock_service.clear_cache = mock_clear_cache
            
            # 测试缓存信息获取
            info = client.get_distribution_cache_info()
            assert info['cache_entries'] == 5
            
            # 测试缓存清理
            cleared = client.clear_distribution_cache()
            assert cleared == 5


if __name__ == '__main__':
    pytest.main([__file__, '-v'])