"""
涨停统计服务全面测试

为LimitUpStatsService提供全面的单元测试覆盖，包括复杂业务逻辑、错误处理和性能测试
"""

import pytest
import asyncio
import pandas as pd
from datetime import datetime, timedelta
from unittest.mock import Mock, AsyncMock, patch, MagicMock

from quickstock.services.limit_up_stats_service import (
    LimitUpStatsService, LimitUpStatsError, InsufficientDataError
)
from quickstock.models import (
    LimitUpStatsRequest, LimitUpStats, StockDailyData, DataRequest
)
from quickstock.core.data_manager import DataManager
from quickstock.core.errors import DataSourceError, ValidationError


class TestLimitUpStatsServiceComprehensive:
    """涨停统计服务全面测试类"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """模拟数据管理器"""
        mock_dm = Mock(spec=DataManager)
        mock_dm.get_data = AsyncMock()
        return mock_dm
    
    @pytest.fixture
    def service(self, mock_data_manager):
        """创建服务实例"""
        return LimitUpStatsService(mock_data_manager)
    
    @pytest.fixture
    def comprehensive_stock_data(self):
        """全面的股票数据"""
        return pd.DataFrame([
            # 上海主板涨停
            {
                'ts_code': '600000.SH', 'trade_date': '20241015',
                'open': 8.0, 'high': 8.8, 'low': 8.0, 'close': 8.8,
                'pre_close': 8.0, 'change': 0.8, 'pct_chg': 10.0,
                'vol': 2000000, 'amount': 17200000.0, 'name': '浦发银行'
            },
            # 深圳主板涨停
            {
                'ts_code': '000001.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 11.0, 'low': 10.0, 'close': 11.0,
                'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                'vol': 1000000, 'amount': 10500000.0, 'name': '平安银行'
            },
            # 科创板涨停
            {
                'ts_code': '688001.SH', 'trade_date': '20241015',
                'open': 20.0, 'high': 24.0, 'low': 20.0, 'close': 24.0,
                'pre_close': 20.0, 'change': 4.0, 'pct_chg': 20.0,
                'vol': 500000, 'amount': 11000000.0, 'name': '华兴源创'
            },
            # 北证涨停
            {
                'ts_code': '430001.BJ', 'trade_date': '20241015',
                'open': 12.0, 'high': 15.6, 'low': 12.0, 'close': 15.6,
                'pre_close': 12.0, 'change': 3.6, 'pct_chg': 30.0,
                'vol': 300000, 'amount': 4380000.0, 'name': '北交所股票'
            },
            # ST股票涨停
            {
                'ts_code': '000002.SZ', 'trade_date': '20241015',
                'open': 6.3, 'high': 6.62, 'low': 6.3, 'close': 6.62,
                'pre_close': 6.3, 'change': 0.32, 'pct_chg': 5.08,
                'vol': 800000, 'amount': 5248000.0, 'name': 'ST万科'
            },
            # 创业板涨停
            {
                'ts_code': '300001.SZ', 'trade_date': '20241015',
                'open': 15.0, 'high': 16.5, 'low': 15.0, 'close': 16.5,
                'pre_close': 15.0, 'change': 1.5, 'pct_chg': 10.0,
                'vol': 600000, 'amount': 9600000.0, 'name': '特锐德'
            },
            # 非涨停股票
            {
                'ts_code': '600001.SH', 'trade_date': '20241015',
                'open': 9.0, 'high': 9.5, 'low': 8.9, 'close': 9.3,
                'pre_close': 9.0, 'change': 0.3, 'pct_chg': 3.33,
                'vol': 1500000, 'amount': 13950000.0, 'name': '邯郸钢铁'
            },
            # 跌停股票
            {
                'ts_code': '000003.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 10.0, 'low': 9.0, 'close': 9.0,
                'pre_close': 10.0, 'change': -1.0, 'pct_chg': -10.0,
                'vol': 900000, 'amount': 8550000.0, 'name': '万科A'
            }
        ])
    
    @pytest.fixture
    def comprehensive_basic_data(self):
        """全面的基础数据"""
        return pd.DataFrame([
            {'ts_code': '600000.SH', 'name': '浦发银行'},
            {'ts_code': '000001.SZ', 'name': '平安银行'},
            {'ts_code': '688001.SH', 'name': '华兴源创'},
            {'ts_code': '430001.BJ', 'name': '北交所股票'},
            {'ts_code': '000002.SZ', 'name': 'ST万科'},
            {'ts_code': '300001.SZ', 'name': '特锐德'},
            {'ts_code': '600001.SH', 'name': '邯郸钢铁'},
            {'ts_code': '000003.SZ', 'name': '万科A'}
        ])
    
    @pytest.mark.asyncio
    async def test_comprehensive_limit_up_stats(self, service, mock_data_manager, 
                                              comprehensive_stock_data, comprehensive_basic_data):
        """测试全面的涨停统计功能"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [comprehensive_stock_data, comprehensive_basic_data]
        
        # 创建请求
        request = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=True,
            market_filter=None,
            force_refresh=False,
            save_to_db=True
        )
        
        # 执行测试
        result = await service.get_daily_limit_up_stats(request)
        
        # 验证结果结构
        assert isinstance(result, LimitUpStats)
        assert result.trade_date == '20241015'
        
        # 验证统计数据
        assert result.total >= 5  # 至少5只涨停股票
        assert result.shanghai >= 1  # 至少1只上海股票
        assert result.shenzhen >= 2  # 至少2只深圳股票（包括创业板）
        assert result.star >= 1     # 至少1只科创板股票
        assert result.beijing >= 1  # 至少1只北证股票
        assert result.st >= 1       # 至少1只ST股票
        
        # 验证一致性
        assert result.total == result.shanghai + result.shenzhen + result.star + result.beijing
        assert result.total == result.st + result.non_st
        
        # 验证涨停股票列表
        assert len(result.limit_up_stocks) == result.total
        assert '600000.SH' in result.limit_up_stocks  # 浦发银行
        assert '000001.SZ' in result.limit_up_stocks  # 平安银行
        assert '688001.SH' in result.limit_up_stocks  # 华兴源创
        assert '430001.BJ' in result.limit_up_stocks  # 北交所股票
        assert '000002.SZ' in result.limit_up_stocks  # ST万科
        assert '300001.SZ' in result.limit_up_stocks  # 特锐德
        
        # 验证市场分类
        assert 'shanghai' in result.market_breakdown
        assert 'shenzhen' in result.market_breakdown
        assert 'star' in result.market_breakdown
        assert 'beijing' in result.market_breakdown
        
        # 验证非涨停股票不在列表中
        assert '600001.SH' not in result.limit_up_stocks  # 邯郸钢铁
        assert '000003.SZ' not in result.limit_up_stocks  # 万科A（跌停）
    
    @pytest.mark.asyncio
    async def test_market_filter_functionality(self, service, mock_data_manager,
                                             comprehensive_stock_data, comprehensive_basic_data):
        """测试市场过滤功能"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [comprehensive_stock_data, comprehensive_basic_data]
        
        # 测试只包含上海市场
        shanghai_request = LimitUpStatsRequest(
            trade_date='20241015',
            market_filter=['shanghai']
        )
        
        result = await service.get_daily_limit_up_stats(shanghai_request)
        
        # 验证只包含上海股票
        assert result.shanghai > 0
        assert result.shenzhen == 0
        assert result.star == 0
        assert result.beijing == 0
        
        # 验证涨停股票列表只包含上海股票
        for stock_code in result.limit_up_stocks:
            assert stock_code.endswith('.SH') and not stock_code.startswith('688')
    
    @pytest.mark.asyncio
    async def test_st_stock_filter_functionality(self, service, mock_data_manager,
                                               comprehensive_stock_data, comprehensive_basic_data):
        """测试ST股票过滤功能"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [comprehensive_stock_data, comprehensive_basic_data]
        
        # 测试不包含ST股票
        no_st_request = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=False
        )
        
        result = await service.get_daily_limit_up_stats(no_st_request)
        
        # 验证ST统计为0
        assert result.st == 0
        assert result.total == result.non_st
        
        # 验证ST股票不在列表中
        assert '000002.SZ' not in result.limit_up_stocks  # ST万科应该被过滤
    
    @pytest.mark.asyncio
    async def test_data_source_error_handling(self, service, mock_data_manager):
        """测试数据源错误处理"""
        # 模拟数据源错误
        mock_data_manager.get_data.side_effect = DataSourceError("数据源连接失败")
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 应该抛出LimitUpStatsError
        with pytest.raises(LimitUpStatsError) as exc_info:
            await service.get_daily_limit_up_stats(request)
        
        assert "数据源连接失败" in str(exc_info.value)
    
    @pytest.mark.asyncio
    async def test_insufficient_data_handling(self, service, mock_data_manager):
        """测试数据不足处理"""
        # 模拟空数据
        mock_data_manager.get_data.return_value = pd.DataFrame()
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 应该抛出InsufficientDataError
        with pytest.raises(InsufficientDataError) as exc_info:
            await service.get_daily_limit_up_stats(request)
        
        assert '20241015' in str(exc_info.value)
        assert 'No stock data available' in exc_info.value.details['missing_data_info']['reason']
    
    @pytest.mark.asyncio
    async def test_partial_data_handling(self, service, mock_data_manager):
        """测试部分数据处理"""
        # 创建部分缺失数据
        partial_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 11.0, 'low': 10.0, 'close': 11.0,
                'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                'vol': 1000000, 'amount': 10500000.0, 'name': None  # 缺失名称
            }
        ])
        
        basic_data = pd.DataFrame()  # 空的基础数据
        
        mock_data_manager.get_data.side_effect = [partial_data, basic_data]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 应该能处理部分数据
        result = await service.get_daily_limit_up_stats(request)
        
        # 验证结果
        assert isinstance(result, LimitUpStats)
        assert result.total >= 0
    
    @pytest.mark.asyncio
    async def test_complex_market_classification(self, service):
        """测试复杂的市场分类逻辑"""
        # 创建包含边界情况的股票数据
        edge_case_stocks = [
            StockDailyData(
                ts_code='688000.SH',  # 科创板边界代码
                trade_date='20241015',
                open=20.0, high=24.0, low=20.0, close=24.0,
                pre_close=20.0, change=4.0, pct_chg=20.0,
                vol=500000, amount=11000000.0, name='科创板边界'
            ),
            StockDailyData(
                ts_code='300999.SZ',  # 创业板边界代码
                trade_date='20241015',
                open=15.0, high=16.5, low=15.0, close=16.5,
                pre_close=15.0, change=1.5, pct_chg=10.0,
                vol=600000, amount=9600000.0, name='创业板边界'
            ),
            StockDailyData(
                ts_code='899999.BJ',  # 北证边界代码
                trade_date='20241015',
                open=12.0, high=15.6, low=12.0, close=15.6,
                pre_close=12.0, change=3.6, pct_chg=30.0,
                vol=300000, amount=4380000.0, name='北证边界'
            )
        ]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 执行分类
        result = await service._classify_stocks_by_market(edge_case_stocks, request)
        
        # 验证分类结果
        assert isinstance(result, dict)
        assert 'star' in result
        assert 'shenzhen' in result
        assert 'beijing' in result
        
        # 验证边界代码被正确分类
        star_codes = [stock.ts_code for stock in result['star']]
        shenzhen_codes = [stock.ts_code for stock in result['shenzhen']]
        beijing_codes = [stock.ts_code for stock in result['beijing']]
        
        assert '688000.SH' in star_codes
        assert '300999.SZ' in shenzhen_codes
        assert '899999.BJ' in beijing_codes
    
    @pytest.mark.asyncio
    async def test_statistics_aggregation_accuracy(self, service):
        """测试统计聚合的准确性"""
        # 创建已知分类的股票数据
        classified_stocks = {
            'shanghai': [Mock(ts_code='600000.SH'), Mock(ts_code='600001.SH')],
            'shenzhen': [Mock(ts_code='000001.SZ'), Mock(ts_code='300001.SZ')],
            'star': [Mock(ts_code='688001.SH')],
            'beijing': [Mock(ts_code='430001.BJ')],
            'st': [Mock(ts_code='000002.SZ')],
            'non_st': [
                Mock(ts_code='600000.SH'), Mock(ts_code='600001.SH'),
                Mock(ts_code='000001.SZ'), Mock(ts_code='300001.SZ'),
                Mock(ts_code='688001.SH'), Mock(ts_code='430001.BJ')
            ],
            'unknown': []
        }
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 执行聚合
        result = service._aggregate_statistics(classified_stocks, request)
        
        # 验证聚合结果
        assert result['total'] == 6  # 2+2+1+1
        assert result['shanghai'] == 2
        assert result['shenzhen'] == 2
        assert result['star'] == 1
        assert result['beijing'] == 1
        assert result['st'] == 1
        assert result['non_st'] == 6
        
        # 验证一致性
        assert result['total'] == result['shanghai'] + result['shenzhen'] + result['star'] + result['beijing']
        assert result['total'] == result['st'] + result['non_st']
        
        # 验证元数据
        assert 'aggregation_metadata' in result
        metadata = result['aggregation_metadata']
        assert 'consistency_check' in metadata
        assert metadata['consistency_check']['is_consistent'] is True
    
    @pytest.mark.asyncio
    async def test_performance_with_large_dataset(self, service, mock_data_manager):
        """测试大数据集性能"""
        import time
        
        # 创建大量股票数据
        large_stock_data = []
        large_basic_data = []
        
        for i in range(5000):  # 5000只股票
            stock_code = f'{i:06d}.SZ'
            large_stock_data.append({
                'ts_code': stock_code, 'trade_date': '20241015',
                'open': 10.0, 'high': 11.0, 'low': 10.0, 'close': 11.0,
                'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                'vol': 1000000, 'amount': 10500000.0, 'name': f'股票{i}'
            })
            large_basic_data.append({
                'ts_code': stock_code, 'name': f'股票{i}'
            })
        
        large_stock_df = pd.DataFrame(large_stock_data)
        large_basic_df = pd.DataFrame(large_basic_data)
        
        mock_data_manager.get_data.side_effect = [large_stock_df, large_basic_df]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 测试性能
        start_time = time.time()
        result = await service.get_daily_limit_up_stats(request)
        end_time = time.time()
        
        # 验证结果
        assert isinstance(result, LimitUpStats)
        assert result.total > 0
        
        # 性能要求：5000只股票应该在合理时间内处理完成
        processing_time = end_time - start_time
        assert processing_time < 30.0  # 30秒内完成
    
    @pytest.mark.asyncio
    async def test_concurrent_requests_handling(self, service, mock_data_manager,
                                              comprehensive_stock_data, comprehensive_basic_data):
        """测试并发请求处理"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = lambda *args: (
            comprehensive_stock_data if args[0].data_type == 'stock_daily' 
            else comprehensive_basic_data
        )
        
        # 创建多个并发请求
        requests = [
            LimitUpStatsRequest(trade_date=f'2024101{i}')
            for i in range(5)
        ]
        
        # 并发执行
        tasks = [service.get_daily_limit_up_stats(req) for req in requests]
        results = await asyncio.gather(*tasks)
        
        # 验证结果
        assert len(results) == 5
        assert all(isinstance(result, LimitUpStats) for result in results)
        
        # 验证每个结果的完整性
        for i, result in enumerate(results):
            assert result.trade_date == f'2024101{i}'
            assert result.total >= 0
    
    @pytest.mark.asyncio
    async def test_service_statistics_tracking(self, service, mock_data_manager,
                                             comprehensive_stock_data, comprehensive_basic_data):
        """测试服务统计跟踪"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [comprehensive_stock_data, comprehensive_basic_data]
        
        # 获取初始统计
        initial_stats = service.get_service_stats()
        assert initial_stats['total_requests'] == 0
        
        # 执行请求
        request = LimitUpStatsRequest(trade_date='20241015')
        await service.get_daily_limit_up_stats(request)
        
        # 获取更新后的统计
        updated_stats = service.get_service_stats()
        assert updated_stats['total_requests'] == 1
        assert updated_stats['successful_requests'] == 1
        assert updated_stats['success_rate'] == 1.0
    
    @pytest.mark.asyncio
    async def test_health_check_comprehensive(self, service, mock_data_manager):
        """测试全面的健康检查"""
        # 执行健康检查
        health_result = await service.health_check()
        
        # 验证健康检查结果
        assert 'service_status' in health_result
        assert 'components' in health_result
        assert 'last_check' in health_result
        assert 'issues' in health_result
        
        # 验证组件状态
        components = health_result['components']
        assert 'data_manager' in components
        assert 'stock_classifier' in components
        assert 'limit_up_detector' in components
        
        # 验证每个组件都有状态信息
        for component_name, component_status in components.items():
            assert 'status' in component_status
            assert 'last_check' in component_status
    
    @pytest.mark.asyncio
    async def test_data_validation_comprehensive(self, service):
        """测试全面的数据验证"""
        # 测试完整数据
        complete_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 11.0, 'low': 9.8, 'close': 11.0,
                'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                'vol': 1000000, 'amount': 10500000.0, 'name': '平安银行'
            }
        ])
        
        result = service.validate_data_completeness(complete_data)
        assert result['is_complete'] is True
        assert len(result['missing_fields']) == 0
        assert len(result['warnings']) == 0
        
        # 测试缺失字段数据
        incomplete_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 11.0, 'low': 9.8, 'close': None,
                'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                'vol': 1000000, 'amount': 10500000.0, 'name': '平安银行'
            }
        ])
        
        result = service.validate_data_completeness(incomplete_data)
        assert result['is_complete'] is False
        assert 'close' in result['missing_fields']
        
        # 测试异常数据
        abnormal_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 10.0, 'low': 10.0, 'close': 10.0,
                'pre_close': 5.0, 'change': 5.0, 'pct_chg': 100.0,  # 异常涨幅
                'vol': 1000000, 'amount': 10000000.0, 'name': '异常股票'
            }
        ])
        
        result = service.validate_data_completeness(abnormal_data)
        assert len(result['warnings']) > 0
        assert any('Unusual price change' in warning for warning in result['warnings'])
    
    @pytest.mark.asyncio
    async def test_market_insights_generation(self, service):
        """测试市场洞察生成"""
        # 创建测试统计数据
        stats = {
            'total': 150,
            'non_st': 135,
            'st': 15,
            'shanghai': 60,
            'shenzhen': 50,
            'star': 30,
            'beijing': 10,
            'limit_up_stocks': ['600000.SH'] * 150,
            'market_breakdown': {
                'shanghai': ['600000.SH'] * 60,
                'shenzhen': ['000001.SZ'] * 50,
                'star': ['688001.SH'] * 30,
                'beijing': ['430001.BJ'] * 10
            },
            'aggregation_metadata': {
                'consistency_check': {'is_consistent': True, 'issues': []},
                'final_validation': {'is_valid': True, 'issues': []}
            }
        }
        
        # 生成市场洞察
        insights = service.generate_market_insights(stats)
        
        # 验证洞察结构
        assert 'summary' in insights
        assert 'concentration_analysis' in insights
        assert 'st_analysis' in insights
        assert 'market_assessment' in insights
        assert 'key_insights' in insights
        assert 'generated_at' in insights
        
        # 验证洞察内容
        assert isinstance(insights['key_insights'], list)
        assert len(insights['key_insights']) > 0
        
        # 验证市场评估
        assessment = insights['market_assessment']
        assert 'overall_score' in assessment
        assert 'market_grade' in assessment
        assert 'market_condition' in assessment
        assert 0 <= assessment['overall_score'] <= 100
    
    @pytest.mark.asyncio
    async def test_error_recovery_mechanisms(self, service, mock_data_manager):
        """测试错误恢复机制"""
        # 模拟间歇性错误
        call_count = 0
        
        def side_effect(*args, **kwargs):
            nonlocal call_count
            call_count += 1
            if call_count == 1:
                raise DataSourceError("临时网络错误")
            else:
                return pd.DataFrame([
                    {
                        'ts_code': '000001.SZ', 'trade_date': '20241015',
                        'open': 10.0, 'high': 11.0, 'low': 10.0, 'close': 11.0,
                        'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                        'vol': 1000000, 'amount': 10500000.0, 'name': '平安银行'
                    }
                ])
        
        mock_data_manager.get_data.side_effect = side_effect
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 如果服务有重试机制，应该能够恢复
        # 这里假设服务会处理错误并提供有意义的错误信息
        with pytest.raises(LimitUpStatsError):
            await service.get_daily_limit_up_stats(request)


class TestLimitUpStatsServiceEdgeCases:
    """涨停统计服务边界情况测试"""
    
    @pytest.fixture
    def service(self):
        """创建服务实例"""
        mock_dm = Mock(spec=DataManager)
        mock_dm.get_data = AsyncMock()
        return LimitUpStatsService(mock_dm)
    
    @pytest.mark.asyncio
    async def test_empty_trading_day(self, service):
        """测试空交易日处理"""
        # 模拟没有涨停股票的交易日
        service.data_manager.get_data.side_effect = [
            pd.DataFrame([
                {
                    'ts_code': '000001.SZ', 'trade_date': '20241015',
                    'open': 10.0, 'high': 10.5, 'low': 9.5, 'close': 10.2,
                    'pre_close': 10.0, 'change': 0.2, 'pct_chg': 2.0,
                    'vol': 1000000, 'amount': 10100000.0, 'name': '平安银行'
                }
            ]),
            pd.DataFrame([{'ts_code': '000001.SZ', 'name': '平安银行'}])
        ]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        result = await service.get_daily_limit_up_stats(request)
        
        # 验证空结果
        assert result.total == 0
        assert result.shanghai == 0
        assert result.shenzhen == 0
        assert result.star == 0
        assert result.beijing == 0
        assert result.st == 0
        assert result.non_st == 0
        assert len(result.limit_up_stocks) == 0
    
    @pytest.mark.asyncio
    async def test_all_st_stocks_day(self, service):
        """测试全部ST股票涨停的交易日"""
        service.data_manager.get_data.side_effect = [
            pd.DataFrame([
                {
                    'ts_code': '000001.SZ', 'trade_date': '20241015',
                    'open': 10.0, 'high': 10.5, 'low': 10.0, 'close': 10.5,
                    'pre_close': 10.0, 'change': 0.5, 'pct_chg': 5.0,
                    'vol': 1000000, 'amount': 10250000.0, 'name': 'ST平安'
                },
                {
                    'ts_code': '000002.SZ', 'trade_date': '20241015',
                    'open': 6.0, 'high': 6.3, 'low': 6.0, 'close': 6.3,
                    'pre_close': 6.0, 'change': 0.3, 'pct_chg': 5.0,
                    'vol': 800000, 'amount': 4980000.0, 'name': '*ST万科'
                }
            ]),
            pd.DataFrame([
                {'ts_code': '000001.SZ', 'name': 'ST平安'},
                {'ts_code': '000002.SZ', 'name': '*ST万科'}
            ])
        ]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        result = await service.get_daily_limit_up_stats(request)
        
        # 验证全ST结果
        assert result.total == 2
        assert result.st == 2
        assert result.non_st == 0
        assert result.shenzhen == 2  # 都是深圳股票
    
    @pytest.mark.asyncio
    async def test_mixed_precision_prices(self, service):
        """测试混合精度价格处理"""
        service.data_manager.get_data.side_effect = [
            pd.DataFrame([
                {
                    'ts_code': '000001.SZ', 'trade_date': '20241015',
                    'open': 10.005, 'high': 11.006, 'low': 10.001, 'close': 11.006,
                    'pre_close': 10.005, 'change': 1.001, 'pct_chg': 10.005,
                    'vol': 1000000, 'amount': 10505000.0, 'name': '精度测试'
                }
            ]),
            pd.DataFrame([{'ts_code': '000001.SZ', 'name': '精度测试'}])
        ]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        result = await service.get_daily_limit_up_stats(request)
        
        # 应该能正确处理精度问题
        assert isinstance(result, LimitUpStats)
        assert result.total >= 0


if __name__ == "__main__":
    pytest.main([__file__])