"""
涨跌分布统计功能综合集成测试

本测试文件实现了涨跌分布统计功能的全面集成测试，包括：
1. 端到端测试用例，测试完整的统计流程
2. 真实数据场景测试
3. 边界情况和异常场景测试
4. 测试数据生成器
5. 与现有系统的集成验证

Requirements Coverage:
- 1.1, 1.2, 1.3, 1.4: 基本功能和错误处理
- 2.1, 2.2: 涨跌幅区间分类
- 3.1-3.7: 市场板块分类统计
- 4.1-4.4: 统计结果和数据质量
- 5.1-5.3: API接口和缓存
- 6.1, 6.2: 性能和并发
"""

import asyncio
import pytest
import pandas as pd
import numpy as np
import json
import time
import tempfile
import sqlite3
from datetime import datetime, timedelta
from unittest.mock import Mock, AsyncMock, patch, MagicMock
from typing import Dict, List, Any, Optional, Tuple
from concurrent.futures import ThreadPoolExecutor, as_completed
import threading
import random

# 导入被测试的模块
from quickstock.client import QuickStockClient
from quickstock.config import Config
from quickstock.services.price_distribution_stats_service import (
    PriceDistributionStatsService
)
from quickstock.core.price_distribution_errors import (
    PriceDistributionServiceError,
    InsufficientPriceDataError,
    MarketClassificationError
)
from quickstock.models.price_distribution_models import (
    PriceDistributionRequest,
    PriceDistributionStats,
    DistributionRange
)
from quickstock.core.data_manager import DataManager
from quickstock.core.errors import ValidationError, DataSourceError, QuickStockError
from quickstock.utils.price_distribution_analyzer import PriceDistributionAnalyzer
from quickstock.utils.distribution_calculator import DistributionCalculator
from quickstock.utils.statistics_aggregator import StatisticsAggregator


class PriceDistributionTestDataGenerator:
    """涨跌分布统计测试数据生成器"""
    
    def __init__(self, seed: int = 42):
        """初始化数据生成器"""
        np.random.seed(seed)
        random.seed(seed)
        self.seed = seed
    
    def generate_realistic_stock_data(self, 
                                    trade_date: str,
                                    total_stocks: int = 5000,
                                    market_distribution: Dict[str, float] = None) -> pd.DataFrame:
        """
        生成真实的股票数据
        
        Args:
            trade_date: 交易日期
            total_stocks: 总股票数量
            market_distribution: 市场分布比例
        """
        if market_distribution is None:
            market_distribution = {
                'shanghai_main': 0.35,    # 上海主板35%
                'shanghai_star': 0.08,    # 科创板8%
                'shenzhen_main': 0.30,    # 深圳主板30%
                'shenzhen_gem': 0.20,     # 创业板20%
                'beijing': 0.05,          # 北交所5%
                'st_stocks': 0.02         # ST股票2%
            }
        
        stocks_data = []
        stock_id = 1
        
        # 生成各市场股票
        for market, ratio in market_distribution.items():
            count = int(total_stocks * ratio)
            
            if market == 'shanghai_main':
                codes = [f'60{i:04d}.SH' for i in range(stock_id, stock_id + count)]
                names = [f'上海股票{i}' for i in range(stock_id, stock_id + count)]
            elif market == 'shanghai_star':
                codes = [f'688{i:03d}.SH' for i in range(1, count + 1)]
                names = [f'科创板{i}' for i in range(1, count + 1)]
            elif market == 'shenzhen_main':
                codes = [f'000{i:03d}.SZ' for i in range(1, count + 1)]
                names = [f'深圳股票{i}' for i in range(1, count + 1)]
            elif market == 'shenzhen_gem':
                codes = [f'300{i:03d}.SZ' for i in range(1, count + 1)]
                names = [f'创业板{i}' for i in range(1, count + 1)]
            elif market == 'beijing':
                codes = [f'8{i:05d}.BJ' for i in range(1, count + 1)]
                names = [f'北交所{i}' for i in range(1, count + 1)]
            elif market == 'st_stocks':
                codes = [f'60{i:04d}.SH' for i in range(9000, 9000 + count)]
                names = [f'*ST股票{i}' for i in range(1, count + 1)]
            
            # 生成涨跌幅数据，模拟真实市场分布
            pct_changes = self._generate_realistic_pct_changes(count, market)
            
            for i, (code, name, pct_chg) in enumerate(zip(codes, names, pct_changes)):
                stocks_data.append({
                    'ts_code': code,
                    'name': name,
                    'trade_date': trade_date,
                    'pct_chg': pct_chg,
                    'close': random.uniform(5.0, 200.0),
                    'open': random.uniform(5.0, 200.0),
                    'high': random.uniform(5.0, 200.0),
                    'low': random.uniform(5.0, 200.0),
                    'vol': random.uniform(1000, 10000000),
                    'amount': random.uniform(10000, 1000000000)
                })
            
            stock_id += count
        
        return pd.DataFrame(stocks_data)    

    def _generate_realistic_pct_changes(self, count: int, market: str) -> List[float]:
        """生成真实的涨跌幅分布"""
        pct_changes = []
        
        # 不同市场的涨跌幅特征
        if market == 'st_stocks':
            # ST股票更容易跌停
            for _ in range(count):
                if random.random() < 0.3:  # 30%概率跌停
                    pct_changes.append(-5.0)
                elif random.random() < 0.1:  # 10%概率涨停
                    pct_changes.append(5.0)
                else:
                    pct_changes.append(np.random.normal(-1.0, 2.0))
        elif market == 'shanghai_star':
            # 科创板波动更大
            for _ in range(count):
                if random.random() < 0.05:  # 5%概率大涨
                    pct_changes.append(random.uniform(15.0, 20.0))
                elif random.random() < 0.05:  # 5%概率大跌
                    pct_changes.append(random.uniform(-20.0, -15.0))
                else:
                    pct_changes.append(np.random.normal(0.0, 4.0))
        else:
            # 主板股票相对稳定
            for _ in range(count):
                if random.random() < 0.02:  # 2%概率涨停
                    pct_changes.append(10.0)
                elif random.random() < 0.02:  # 2%概率跌停
                    pct_changes.append(-10.0)
                else:
                    pct_changes.append(np.random.normal(0.0, 3.0))
        
        # 限制涨跌幅范围
        return [max(-20.0, min(20.0, pct)) for pct in pct_changes]
    
    def generate_boundary_case_data(self, trade_date: str) -> List[pd.DataFrame]:
        """生成边界情况测试数据"""
        boundary_cases = []
        
        # 1. 空数据集
        boundary_cases.append(pd.DataFrame())
        
        # 2. 单只股票
        boundary_cases.append(pd.DataFrame({
            'ts_code': ['600000.SH'],
            'name': ['测试股票'],
            'trade_date': [trade_date],
            'pct_chg': [5.0],
            'close': [10.0],
            'vol': [1000000]
        }))
        
        # 3. 所有股票涨停
        all_limit_up = pd.DataFrame({
            'ts_code': [f'60000{i}.SH' for i in range(10)],
            'name': [f'涨停股{i}' for i in range(10)],
            'trade_date': [trade_date] * 10,
            'pct_chg': [10.0] * 10,
            'close': [10.0] * 10,
            'vol': [1000000] * 10
        })
        boundary_cases.append(all_limit_up)
        
        # 4. 所有股票跌停
        all_limit_down = pd.DataFrame({
            'ts_code': [f'00000{i}.SZ' for i in range(10)],
            'name': [f'跌停股{i}' for i in range(10)],
            'trade_date': [trade_date] * 10,
            'pct_chg': [-10.0] * 10,
            'close': [10.0] * 10,
            'vol': [1000000] * 10
        })
        boundary_cases.append(all_limit_down)
        
        # 5. 包含异常值的数据
        abnormal_data = pd.DataFrame({
            'ts_code': ['600001.SH', '600002.SH', '600003.SH', '600004.SH'],
            'name': ['正常股票', '异常股票1', '异常股票2', '异常股票3'],
            'trade_date': [trade_date] * 4,
            'pct_chg': [5.0, np.nan, np.inf, -np.inf],
            'close': [10.0, 10.0, 10.0, 10.0],
            'vol': [1000000, 1000000, 1000000, 1000000]
        })
        boundary_cases.append(abnormal_data)
        
        # 6. 极端涨跌幅数据
        extreme_data = pd.DataFrame({
            'ts_code': [f'68800{i}.SH' for i in range(6)],
            'name': [f'极端股{i}' for i in range(6)],
            'trade_date': [trade_date] * 6,
            'pct_chg': [50.0, -50.0, 100.0, -100.0, 0.0, 0.01],
            'close': [10.0] * 6,
            'vol': [1000000] * 6
        })
        boundary_cases.append(extreme_data)
        
        return boundary_cases
    
    def generate_exception_scenarios(self) -> List[Dict[str, Any]]:
        """生成异常场景"""
        scenarios = []
        
        # 1. 数据源不可用
        scenarios.append({
            'name': 'data_source_unavailable',
            'description': '数据源不可用',
            'exception': DataSourceError("Data source is temporarily unavailable"),
            'expected_behavior': 'raise_exception'
        })
        
        # 2. 网络超时
        scenarios.append({
            'name': 'network_timeout',
            'description': '网络请求超时',
            'exception': TimeoutError("Request timeout after 30 seconds"),
            'expected_behavior': 'raise_exception'
        })
        
        # 3. 数据格式错误
        scenarios.append({
            'name': 'invalid_data_format',
            'description': '数据格式错误',
            'exception': ValueError("Invalid data format"),
            'expected_behavior': 'raise_exception'
        })
        
        # 4. 内存不足
        scenarios.append({
            'name': 'memory_error',
            'description': '内存不足',
            'exception': MemoryError("Not enough memory to process data"),
            'expected_behavior': 'raise_exception'
        })
        
        # 5. 数据库连接失败
        scenarios.append({
            'name': 'database_connection_error',
            'description': '数据库连接失败',
            'exception': sqlite3.OperationalError("Database connection failed"),
            'expected_behavior': 'fallback_to_memory'
        })
        
        return scenarios
    
    def generate_performance_test_data(self, 
                                     trade_date: str,
                                     stock_counts: List[int]) -> Dict[str, pd.DataFrame]:
        """生成性能测试数据"""
        performance_data = {}
        
        for count in stock_counts:
            data = self.generate_realistic_stock_data(trade_date, count)
            performance_data[f'stocks_{count}'] = data
        
        return performance_data
    
    def generate_concurrent_test_scenarios(self, 
                                         base_date: str,
                                         days: int = 10) -> List[Dict[str, Any]]:
        """生成并发测试场景"""
        scenarios = []
        
        base_dt = datetime.strptime(base_date, '%Y%m%d')
        
        for i in range(days):
            trade_date = (base_dt + timedelta(days=i)).strftime('%Y%m%d')
            
            scenarios.append({
                'trade_date': trade_date,
                'request_params': {
                    'include_st': random.choice([True, False]),
                    'market_filter': random.choice([
                        None,
                        ['shanghai'],
                        ['shenzhen'],
                        ['shanghai', 'shenzhen']
                    ]),
                    'force_refresh': random.choice([True, False])
                },
                'expected_stocks': random.randint(1000, 5000)
            })
        
        return scenarios


class TestEndToEndPriceDistributionWorkflow:
    """端到端涨跌分布统计工作流测试"""
    
    @pytest.fixture
    def test_data_generator(self):
        """测试数据生成器"""
        return PriceDistributionTestDataGenerator()
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置"""
        config = Mock()
        config.cache_enabled = True
        config.max_concurrent_requests = 10
        config.max_retries = 3
        config.cache_expire_hours = 24
        config.price_distribution_cache_db = ':memory:'
        config.price_distribution_memory_cache_size = 1000
        config.min_stock_count_for_distribution = 50
        config.max_distribution_processing_time = 120
        config.enable_fallback_classification = True
        config.parallel_processing_enabled = True
        config.log_level = 'INFO'
        return config
    
    @pytest.fixture
    def mock_data_manager(self, mock_config):
        """模拟数据管理器"""
        data_manager = Mock(spec=DataManager)
        data_manager.config = mock_config
        data_manager.get_stock_data_with_coordination = AsyncMock()
        data_manager.get_cache_stats = Mock(return_value={'hits': 0, 'misses': 0})
        data_manager.get_memory_stats = Mock(return_value={'memory_mb': 100})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        return data_manager
    
    @pytest.fixture
    def service(self, mock_data_manager):
        """创建服务实例"""
        return PriceDistributionStatsService(mock_data_manager)
    
    @pytest.fixture
    def client(self, mock_config, mock_data_manager):
        """创建客户端实例"""
        with patch('quickstock.client.DataManager', return_value=mock_data_manager):
            client = QuickStockClient(mock_config)
            client.data_manager = mock_data_manager
            return client
    
    @pytest.mark.asyncio
    async def test_complete_workflow_with_realistic_data(self, 
                                                       service, 
                                                       test_data_generator):
        """测试完整工作流程 - 真实数据场景"""
        # 生成真实的股票数据
        trade_date = '20240315'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=2000
        )
        
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 创建请求
        request = PriceDistributionRequest(trade_date=trade_date)
        
        # 执行完整流程
        start_time = time.time()
        result = await service.get_price_distribution_stats(request)
        processing_time = time.time() - start_time
        
        # 验证结果完整性
        assert isinstance(result, PriceDistributionStats)
        assert result.trade_date == trade_date
        assert result.total_stocks == len(stock_data)
        assert result.processing_time > 0
        assert processing_time < 30  # 应在30秒内完成
        
        # 验证涨跌分布数据
        assert isinstance(result.positive_ranges, dict)
        assert isinstance(result.negative_ranges, dict)
        assert len(result.positive_ranges) == 5  # 默认5个正区间
        assert len(result.negative_ranges) == 5  # 默认5个负区间
        
        # 验证数据一致性
        total_positive = sum(result.positive_ranges.values())
        total_negative = sum(result.negative_ranges.values())
        assert total_positive + total_negative == result.total_stocks
        
        # 验证百分比计算
        for range_name, percentage in result.positive_percentages.items():
            expected_pct = (result.positive_ranges[range_name] / result.total_stocks) * 100
            assert abs(percentage - expected_pct) < 0.01  # 允许小数点误差
        
        # 验证市场分类
        assert isinstance(result.market_breakdown, dict)
        assert 'total' in result.market_breakdown
        assert 'shanghai' in result.market_breakdown
        assert 'shenzhen' in result.market_breakdown
        assert 'st' in result.market_breakdown
        assert 'non_st' in result.market_breakdown
        
        # 验证数据质量分数
        assert 0.0 <= result.data_quality_score <= 1.0
        assert result.data_quality_score > 0.8  # 真实数据质量应该较高
    
    @pytest.mark.asyncio
    async def test_workflow_with_custom_ranges(self, service, test_data_generator):
        """测试自定义区间的完整工作流程"""
        # 生成测试数据
        trade_date = '20240316'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=1000
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 定义自定义区间
        custom_ranges = {
            "0-2%": (0.0, 2.0),
            "2-5%": (2.0, 5.0),
            "5-8%": (5.0, 8.0),
            ">=8%": (8.0, float('inf')),
            "0到-2%": (-2.0, 0.0),
            "-2到-5%": (-5.0, -2.0),
            "-5到-8%": (-8.0, -5.0),
            "<=-8%": (float('-inf'), -8.0)
        }
        
        # 创建自定义区间请求
        request = PriceDistributionRequest(
            trade_date=trade_date,
            distribution_ranges=custom_ranges
        )
        
        # 执行测试
        result = await service.get_price_distribution_stats(request)
        
        # 验证自定义区间被正确使用
        assert len(result.positive_ranges) == 4
        assert len(result.negative_ranges) == 4
        assert "0-2%" in result.positive_ranges
        assert ">=8%" in result.positive_ranges
        assert "0到-2%" in result.negative_ranges
        assert "<=-8%" in result.negative_ranges
    
    @pytest.mark.asyncio
    async def test_workflow_with_market_filter(self, service, test_data_generator):
        """测试市场过滤器的完整工作流程"""
        # 生成包含多个市场的数据
        trade_date = '20240317'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=1500
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 创建市场过滤请求
        request = PriceDistributionRequest(
            trade_date=trade_date,
            market_filter=['shanghai', 'shenzhen'],
            include_st=False
        )
        
        # 执行测试
        result = await service.get_price_distribution_stats(request)
        
        # 验证市场过滤效果
        assert 'shanghai' in result.market_breakdown
        assert 'shenzhen' in result.market_breakdown
        # 注意：市场过滤器主要影响数据获取，但分类器仍会对所有数据进行分类
        # 所以beijing可能仍然存在，但股票数量应该相对较少
        if 'beijing' in result.market_breakdown:
            beijing_total = result.market_breakdown['beijing']['total_stocks']
            total_stocks = result.total_stocks
            # 北交所股票应该占比较小（通常<10%）
            assert beijing_total / total_stocks < 0.1
        
        # 验证ST股票被排除
        assert result.market_breakdown['st']['total_stocks'] == 0 or 'st' not in result.market_breakdown
    
    @pytest.mark.asyncio
    async def test_workflow_with_caching(self, service, test_data_generator):
        """测试缓存机制的完整工作流程"""
        # 生成测试数据
        trade_date = '20240318'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=800
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 第一次请求（冷启动）
        request = PriceDistributionRequest(trade_date=trade_date)
        
        start_time = time.time()
        result1 = await service.get_price_distribution_stats(request)
        first_request_time = time.time() - start_time
        
        # 第二次请求（缓存命中）
        start_time = time.time()
        result2 = await service.get_price_distribution_stats(request)
        second_request_time = time.time() - start_time
        
        # 验证缓存效果
        assert result1.trade_date == result2.trade_date
        assert result1.total_stocks == result2.total_stocks
        assert second_request_time < first_request_time * 0.5  # 缓存请求应该快很多
        
        # 验证缓存统计
        stats = service.get_service_stats()
        assert stats['performance']['cache_hits'] > 0
        
        # 测试强制刷新
        request_refresh = PriceDistributionRequest(
            trade_date=trade_date,
            force_refresh=True
        )
        
        start_time = time.time()
        result3 = await service.get_price_distribution_stats(request_refresh)
        refresh_request_time = time.time() - start_time
        
        # 强制刷新应该重新计算
        assert refresh_request_time > second_request_time
        assert result3.trade_date == trade_date


class TestBoundaryAndExceptionScenarios:
    """边界情况和异常场景测试"""
    
    @pytest.fixture
    def test_data_generator(self):
        return PriceDistributionTestDataGenerator()
    
    @pytest.fixture
    def mock_config(self):
        config = Mock()
        config.cache_enabled = True
        config.min_stock_count_for_distribution = 10
        config.max_distribution_processing_time = 60
        config.enable_fallback_classification = True
        config.price_distribution_cache_db = ':memory:'
        config.price_distribution_memory_cache_size = 100
        config.max_concurrent_requests = 5
        config.max_retries = 3
        config.cache_expire_hours = 1
        config.parallel_processing_enabled = True
        return config
    
    @pytest.fixture
    def mock_data_manager(self, mock_config):
        data_manager = Mock(spec=DataManager)
        data_manager.config = mock_config
        data_manager.get_stock_data_with_coordination = AsyncMock()
        data_manager.get_cache_stats = Mock(return_value={})
        data_manager.get_memory_stats = Mock(return_value={})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        return data_manager
    
    @pytest.fixture
    def service(self, mock_data_manager):
        return PriceDistributionStatsService(mock_data_manager)
    
    @pytest.mark.asyncio
    async def test_empty_dataset_handling(self, service):
        """测试空数据集处理"""
        # 模拟返回空数据
        service.data_manager.get_stock_data_with_coordination.return_value = pd.DataFrame()
        
        request = PriceDistributionRequest(trade_date='20240320')
        
        # 应该抛出服务异常（包装了数据源错误）
        with pytest.raises(PriceDistributionServiceError) as exc_info:
            await service.get_price_distribution_stats(request)
        
        assert '20240320' in str(exc_info.value)
        # 验证这是由于空数据导致的错误
        assert 'No stock data available' in str(exc_info.value) or 'Failed to get distribution stats' in str(exc_info.value)
    
    @pytest.mark.asyncio
    async def test_single_stock_handling(self, service, test_data_generator):
        """测试单只股票处理"""
        # 生成单只股票数据
        boundary_cases = test_data_generator.generate_boundary_case_data('20240321')
        single_stock_data = boundary_cases[1]  # 单只股票数据
        
        service.data_manager.get_stock_data_with_coordination.return_value = single_stock_data
        
        request = PriceDistributionRequest(trade_date='20240321')
        result = await service.get_price_distribution_stats(request)
        
        # 验证单只股票的处理结果
        assert result.total_stocks == 1
        assert sum(result.positive_ranges.values()) + sum(result.negative_ranges.values()) == 1
        
        # 验证涨跌幅分类正确
        pct_chg = single_stock_data.iloc[0]['pct_chg']
        if pct_chg > 0:
            assert sum(result.positive_ranges.values()) == 1
            assert sum(result.negative_ranges.values()) == 0
        else:
            assert sum(result.positive_ranges.values()) == 0
            assert sum(result.negative_ranges.values()) == 1
    
    @pytest.mark.asyncio
    async def test_all_limit_up_scenario(self, service, test_data_generator):
        """测试所有股票涨停场景"""
        boundary_cases = test_data_generator.generate_boundary_case_data('20240322')
        all_limit_up_data = boundary_cases[2]  # 所有涨停数据
        
        service.data_manager.get_stock_data_with_coordination.return_value = all_limit_up_data
        
        request = PriceDistributionRequest(trade_date='20240322')
        result = await service.get_price_distribution_stats(request)
        
        # 验证所有股票都在>=10%区间
        assert result.positive_ranges.get('>=10%', 0) == len(all_limit_up_data)
        assert sum(result.negative_ranges.values()) == 0
        
        # 验证百分比计算
        assert result.positive_percentages.get('>=10%', 0) == 100.0
    
    @pytest.mark.asyncio
    async def test_all_limit_down_scenario(self, service, test_data_generator):
        """测试所有股票跌停场景"""
        boundary_cases = test_data_generator.generate_boundary_case_data('20240323')
        all_limit_down_data = boundary_cases[3]  # 所有跌停数据
        
        service.data_manager.get_stock_data_with_coordination.return_value = all_limit_down_data
        
        request = PriceDistributionRequest(trade_date='20240323')
        result = await service.get_price_distribution_stats(request)
        
        # 验证所有股票都在<=-10%区间
        assert result.negative_ranges.get('<=-10%', 0) == len(all_limit_down_data)
        assert sum(result.positive_ranges.values()) == 0
        
        # 验证百分比计算
        assert result.negative_percentages.get('<=-10%', 0) == 100.0
    
    @pytest.mark.asyncio
    async def test_abnormal_data_handling(self, service, test_data_generator):
        """测试异常数据处理"""
        boundary_cases = test_data_generator.generate_boundary_case_data('20240324')
        abnormal_data = boundary_cases[4]  # 包含异常值的数据
        
        service.data_manager.get_stock_data_with_coordination.return_value = abnormal_data
        
        request = PriceDistributionRequest(trade_date='20240324')
        result = await service.get_price_distribution_stats(request)
        
        # 验证异常数据被正确处理（过滤或修正）
        assert result.total_stocks <= len(abnormal_data)  # 可能过滤了异常数据
        assert result.data_quality_score < 1.0  # 数据质量分数应该降低
        
        # 验证统计结果仍然有效
        total_classified = sum(result.positive_ranges.values()) + sum(result.negative_ranges.values())
        assert total_classified == result.total_stocks
    
    @pytest.mark.asyncio
    async def test_extreme_values_handling(self, service, test_data_generator):
        """测试极端值处理"""
        boundary_cases = test_data_generator.generate_boundary_case_data('20240325')
        extreme_data = boundary_cases[5]  # 极端涨跌幅数据
        
        service.data_manager.get_stock_data_with_coordination.return_value = extreme_data
        
        request = PriceDistributionRequest(trade_date='20240325')
        result = await service.get_price_distribution_stats(request)
        
        # 验证极端值被正确分类
        assert result.positive_ranges.get('>=10%', 0) >= 2  # 至少有两个极端正值
        assert result.negative_ranges.get('<=-10%', 0) >= 2  # 至少有两个极端负值
        
        # 验证零值和小数值处理
        zero_and_small = [pct for pct in extreme_data['pct_chg'] if -1 <= pct <= 1]
        if zero_and_small:
            assert result.positive_ranges.get('0-3%', 0) >= len([p for p in zero_and_small if p >= 0])
            assert result.negative_ranges.get('0到-3%', 0) >= len([p for p in zero_and_small if p < 0])
    
    @pytest.mark.asyncio
    async def test_data_source_error_scenarios(self, service, test_data_generator):
        """测试数据源错误场景"""
        exception_scenarios = test_data_generator.generate_exception_scenarios()
        
        for scenario in exception_scenarios:
            # 模拟异常
            service.data_manager.get_stock_data_with_coordination.side_effect = scenario['exception']
            
            request = PriceDistributionRequest(trade_date='20240326')
            
            if scenario['expected_behavior'] == 'raise_exception':
                with pytest.raises(PriceDistributionStatsError):
                    await service.get_price_distribution_stats(request)
            elif scenario['expected_behavior'] == 'fallback_to_memory':
                # 某些错误应该有fallback机制
                try:
                    result = await service.get_price_distribution_stats(request)
                    # 如果有fallback，结果应该是有效的
                    assert isinstance(result, PriceDistributionStats)
                except PriceDistributionStatsError:
                    # 如果没有fallback，应该抛出异常
                    pass
            
            # 重置mock
            service.data_manager.get_stock_data_with_coordination.side_effect = None
    
    @pytest.mark.asyncio
    async def test_invalid_request_parameters(self, service):
        """测试无效请求参数"""
        # 测试无效日期格式
        with pytest.raises(ValidationError):
            invalid_request = PriceDistributionRequest(trade_date='invalid_date')
            service._validate_and_process_request(invalid_request)
        
        # 测试未来日期
        future_date = (datetime.now() + timedelta(days=1)).strftime('%Y%m%d')
        with pytest.raises(ValidationError):
            future_request = PriceDistributionRequest(trade_date=future_date)
            service._validate_and_process_request(future_request)
        
        # 测试无效市场过滤器
        with pytest.raises(ValidationError):
            invalid_market_request = PriceDistributionRequest(
                trade_date='20240327',
                market_filter=['invalid_market']
            )
            service._validate_and_process_request(invalid_market_request)
        
        # 测试无效自定义区间
        with pytest.raises(ValidationError):
            invalid_ranges_request = PriceDistributionRequest(
                trade_date='20240327',
                distribution_ranges={'invalid': 'range'}
            )
            service._validate_and_process_request(invalid_ranges_request)
    
    @pytest.mark.asyncio
    async def test_memory_pressure_scenarios(self, service, test_data_generator):
        """测试内存压力场景"""
        # 生成大量数据
        large_data = test_data_generator.generate_realistic_stock_data(
            trade_date='20240328',
            total_stocks=50000  # 大量股票
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = large_data
        
        request = PriceDistributionRequest(trade_date='20240328')
        
        # 监控内存使用
        import psutil
        process = psutil.Process()
        memory_before = process.memory_info().rss
        
        try:
            result = await service.get_price_distribution_stats(request)
            
            # 验证大数据集处理结果
            assert result.total_stocks == len(large_data)
            assert isinstance(result, PriceDistributionStats)
            
            # 检查内存使用是否合理
            memory_after = process.memory_info().rss
            memory_increase = memory_after - memory_before
            
            # 内存增长应该在合理范围内（这里设置为500MB限制）
            assert memory_increase < 500 * 1024 * 1024  # 500MB
            
        except MemoryError:
            # 如果内存不足，应该有适当的错误处理
            pytest.skip("Insufficient memory for large dataset test")
    
    @pytest.mark.asyncio
    async def test_timeout_scenarios(self, service, test_data_generator):
        """测试超时场景"""
        # 模拟慢速数据获取
        async def slow_data_fetch(*args, **kwargs):
            await asyncio.sleep(5)  # 模拟5秒延迟
            return test_data_generator.generate_realistic_stock_data('20240329', 1000)
        
        service.data_manager.get_stock_data_with_coordination.side_effect = slow_data_fetch
        
        # 设置短超时时间
        request = PriceDistributionRequest(
            trade_date='20240329',
            timeout=2  # 2秒超时
        )
        
        # 应该超时
        with pytest.raises(asyncio.TimeoutError):
            await asyncio.wait_for(
                service.get_price_distribution_stats(request),
                timeout=3
            )


class TestPerformanceAndConcurrency:
    """性能和并发测试"""
    
    @pytest.fixture
    def test_data_generator(self):
        return PriceDistributionTestDataGenerator()
    
    @pytest.fixture
    def performance_config(self):
        config = Mock()
        config.cache_enabled = True
        config.max_concurrent_requests = 20
        config.max_retries = 3
        config.cache_expire_hours = 24
        config.price_distribution_cache_db = ':memory:'
        config.price_distribution_memory_cache_size = 5000
        config.min_stock_count_for_distribution = 100
        config.max_distribution_processing_time = 300
        config.enable_fallback_classification = True
        config.parallel_processing_enabled = True
        return config
    
    @pytest.fixture
    def mock_data_manager(self, performance_config):
        data_manager = Mock(spec=DataManager)
        data_manager.config = performance_config
        data_manager.get_stock_data_with_coordination = AsyncMock()
        data_manager.get_cache_stats = Mock(return_value={})
        data_manager.get_memory_stats = Mock(return_value={})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        return data_manager
    
    @pytest.fixture
    def service(self, mock_data_manager):
        return PriceDistributionStatsService(mock_data_manager)
    
    @pytest.mark.asyncio
    async def test_performance_with_different_data_sizes(self, service, test_data_generator):
        """测试不同数据量的性能表现"""
        stock_counts = [100, 500, 1000, 2000, 5000]
        performance_results = {}
        
        for count in stock_counts:
            # 生成测试数据
            stock_data = test_data_generator.generate_realistic_stock_data(
                trade_date='20240330',
                total_stocks=count
            )
            
            service.data_manager.get_stock_data_with_coordination.return_value = stock_data
            
            # 测量处理时间
            request = PriceDistributionRequest(trade_date='20240330', force_refresh=True)
            
            start_time = time.time()
            result = await service.get_price_distribution_stats(request)
            processing_time = time.time() - start_time
            
            performance_results[count] = {
                'processing_time': processing_time,
                'stocks_per_second': count / processing_time if processing_time > 0 else 0,
                'memory_usage': result.processing_time,
                'data_quality': result.data_quality_score
            }
            
            # 验证结果正确性
            assert result.total_stocks == count
            assert isinstance(result, PriceDistributionStats)
        
        # 验证性能指标
        for count, metrics in performance_results.items():
            # 处理时间应该随数据量增长，但不应该是线性增长
            assert metrics['processing_time'] < 60  # 最多60秒
            assert metrics['stocks_per_second'] > 10  # 每秒至少处理10只股票
            assert metrics['data_quality'] > 0.8  # 数据质量应该保持较高
        
        # 验证性能扩展性
        small_time = performance_results[100]['processing_time']
        large_time = performance_results[5000]['processing_time']
        
        # 50倍数据量的处理时间不应该超过50倍
        assert large_time / small_time < 50
    
    @pytest.mark.asyncio
    async def test_concurrent_requests_handling(self, service, test_data_generator):
        """测试并发请求处理"""
        # 生成并发测试场景
        concurrent_scenarios = test_data_generator.generate_concurrent_test_scenarios(
            base_date='20240401',
            days=10
        )
        
        # 为每个场景准备数据
        for scenario in concurrent_scenarios:
            stock_data = test_data_generator.generate_realistic_stock_data(
                trade_date=scenario['trade_date'],
                total_stocks=scenario['expected_stocks']
            )
            
            # 使用side_effect来根据请求返回不同数据
            async def get_data_for_date(trade_date, **kwargs):
                for s in concurrent_scenarios:
                    if s['trade_date'] == trade_date:
                        return test_data_generator.generate_realistic_stock_data(
                            trade_date=s['trade_date'],
                            total_stocks=s['expected_stocks']
                        )
                return pd.DataFrame()
            
            service.data_manager.get_stock_data_with_coordination.side_effect = get_data_for_date
        
        # 创建并发请求
        tasks = []
        for scenario in concurrent_scenarios:
            request = PriceDistributionRequest(
                trade_date=scenario['trade_date'],
                **scenario['request_params']
            )
            task = service.get_price_distribution_stats(request)
            tasks.append(task)
        
        # 并发执行
        start_time = time.time()
        results = await asyncio.gather(*tasks, return_exceptions=True)
        total_time = time.time() - start_time
        
        # 验证并发处理结果
        successful_results = [r for r in results if isinstance(r, PriceDistributionStats)]
        failed_results = [r for r in results if isinstance(r, Exception)]
        
        # 至少80%的请求应该成功
        success_rate = len(successful_results) / len(results)
        assert success_rate >= 0.8
        
        # 并发处理时间应该比串行处理快
        estimated_serial_time = sum(r.processing_time for r in successful_results if hasattr(r, 'processing_time'))
        assert total_time < estimated_serial_time * 0.8  # 并发应该至少快20%
        
        # 验证每个成功结果的正确性
        for result in successful_results:
            assert isinstance(result, PriceDistributionStats)
            assert result.total_stocks > 0
            assert result.data_quality_score > 0
    
    @pytest.mark.asyncio
    async def test_cache_performance_under_load(self, service, test_data_generator):
        """测试高负载下的缓存性能"""
        # 生成测试数据
        trade_date = '20240402'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=3000
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 第一次请求建立缓存
        request = PriceDistributionRequest(trade_date=trade_date)
        await service.get_price_distribution_stats(request)
        
        # 并发缓存命中测试
        cache_hit_tasks = []
        for i in range(50):  # 50个并发缓存请求
            task = service.get_price_distribution_stats(request)
            cache_hit_tasks.append(task)
        
        start_time = time.time()
        cache_results = await asyncio.gather(*cache_hit_tasks)
        cache_time = time.time() - start_time
        
        # 验证缓存性能
        assert len(cache_results) == 50
        assert all(isinstance(r, PriceDistributionStats) for r in cache_results)
        assert cache_time < 5  # 50个缓存请求应该在5秒内完成
        
        # 验证缓存一致性
        first_result = cache_results[0]
        for result in cache_results[1:]:
            assert result.trade_date == first_result.trade_date
            assert result.total_stocks == first_result.total_stocks
        
        # 验证缓存统计
        stats = service.get_service_stats()
        assert stats['performance']['cache_hits'] >= 50
    
    @pytest.mark.asyncio
    async def test_memory_usage_optimization(self, service, test_data_generator):
        """测试内存使用优化"""
        import psutil
        process = psutil.Process()
        
        # 记录初始内存使用
        initial_memory = process.memory_info().rss
        
        # 处理多个大数据集
        for i in range(5):
            trade_date = f'2024040{3+i}'
            large_data = test_data_generator.generate_realistic_stock_data(
                trade_date=trade_date,
                total_stocks=10000
            )
            
            service.data_manager.get_stock_data_with_coordination.return_value = large_data
            
            request = PriceDistributionRequest(trade_date=trade_date)
            result = await service.get_price_distribution_stats(request)
            
            # 验证结果正确性
            assert result.total_stocks == 10000
            assert isinstance(result, PriceDistributionStats)
            
            # 检查内存使用
            current_memory = process.memory_info().rss
            memory_increase = current_memory - initial_memory
            
            # 内存增长应该控制在合理范围内
            assert memory_increase < 1024 * 1024 * 1024  # 1GB限制
        
        # 清理缓存后内存应该释放
        await service.clear_cache()
        
        # 等待垃圾回收
        import gc
        gc.collect()
        
        final_memory = process.memory_info().rss
        memory_after_cleanup = final_memory - initial_memory
        
        # 清理后内存使用应该显著降低
        assert memory_after_cleanup < memory_increase * 0.5
    
    @pytest.mark.asyncio
    async def test_error_recovery_under_load(self, service, test_data_generator):
        """测试高负载下的错误恢复"""
        # 模拟间歇性错误
        error_count = 0
        max_errors = 10
        
        async def intermittent_error_data_fetch(trade_date, **kwargs):
            nonlocal error_count
            if error_count < max_errors and random.random() < 0.3:  # 30%错误率
                error_count += 1
                raise DataSourceError(f"Simulated error {error_count}")
            
            return test_data_generator.generate_realistic_stock_data(
                trade_date=trade_date,
                total_stocks=1000
            )
        
        service.data_manager.get_stock_data_with_coordination.side_effect = intermittent_error_data_fetch
        
        # 创建大量并发请求
        tasks = []
        for i in range(50):
            trade_date = f'20240{4:02d}{10+i%20:02d}'
            request = PriceDistributionRequest(trade_date=trade_date)
            task = service.get_price_distribution_stats(request)
            tasks.append(task)
        
        # 并发执行
        results = await asyncio.gather(*tasks, return_exceptions=True)
        
        # 分析结果
        successful_results = [r for r in results if isinstance(r, PriceDistributionStats)]
        error_results = [r for r in results if isinstance(r, Exception)]
        
        # 验证错误恢复
        success_rate = len(successful_results) / len(results)
        assert success_rate >= 0.6  # 至少60%成功率（考虑到30%错误率）
        
        # 验证错误类型
        for error in error_results:
            assert isinstance(error, (PriceDistributionStatsError, DataSourceError))
        
        # 验证成功结果的质量
        for result in successful_results:
            assert isinstance(result, PriceDistributionStats)
            assert result.total_stocks > 0
            assert result.data_quality_score > 0


class TestSystemIntegration:
    """系统集成测试"""
    
    @pytest.fixture
    def test_data_generator(self):
        return PriceDistributionTestDataGenerator()
    
    @pytest.fixture
    def integration_config(self):
        """集成测试配置"""
        config = Mock()
        config.cache_enabled = True
        config.max_concurrent_requests = 10
        config.max_retries = 3
        config.cache_expire_hours = 1
        config.price_distribution_cache_db = ':memory:'
        config.price_distribution_memory_cache_size = 1000
        config.min_stock_count_for_distribution = 50
        config.max_distribution_processing_time = 120
        config.enable_fallback_classification = True
        config.parallel_processing_enabled = True
        config.log_level = 'INFO'
        config.enable_auto_code_conversion = True
        config.log_code_conversions = False
        config.code_conversion_error_strategy = 'strict'
        return config
    
    @pytest.fixture
    def mock_data_manager(self, integration_config):
        data_manager = Mock(spec=DataManager)
        data_manager.config = integration_config
        data_manager.get_stock_data_with_coordination = AsyncMock()
        data_manager.get_cache_stats = Mock(return_value={'hits': 10, 'misses': 5})
        data_manager.get_memory_stats = Mock(return_value={'memory_mb': 150})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        return data_manager
    
    @pytest.fixture
    def client(self, integration_config, mock_data_manager):
        """集成测试客户端"""
        with patch('quickstock.client.DataManager', return_value=mock_data_manager):
            client = QuickStockClient(integration_config)
            client.data_manager = mock_data_manager
            return client
    
    @pytest.fixture
    def service(self, mock_data_manager):
        return PriceDistributionStatsService(mock_data_manager)
    
    def test_client_service_integration(self, client, test_data_generator):
        """测试客户端与服务的集成"""
        # 生成测试数据
        trade_date = '20240410'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=1500
        )
        
        # 模拟服务响应
        with patch('quickstock.services.price_distribution_stats_service.PriceDistributionStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 创建预期的统计结果
            expected_stats = PriceDistributionStats(
                trade_date=trade_date,
                total_stocks=len(stock_data),
                positive_ranges={'0-3%': 300, '3-5%': 200, '5-7%': 150, '7-10%': 100, '>=10%': 50},
                positive_percentages={'0-3%': 20.0, '3-5%': 13.3, '5-7%': 10.0, '7-10%': 6.7, '>=10%': 3.3},
                negative_ranges={'0到-3%': 250, '-3到-5%': 200, '-5到-7%': 150, '-7到-10%': 100, '<=-10%': 50},
                negative_percentages={'0到-3%': 16.7, '-3到-5%': 13.3, '-5到-7%': 10.0, '-7到-10%': 6.7, '<=-10%': 3.3},
                market_breakdown={
                    'total': {'total_stocks': 1500},
                    'shanghai': {'total_stocks': 600},
                    'shenzhen': {'total_stocks': 750},
                    'beijing': {'total_stocks': 150}
                },
                processing_time=2.5,
                data_quality_score=0.95
            )
            
            async def mock_get_stats(request):
                return expected_stats
            
            mock_service.get_price_distribution_stats = mock_get_stats
            
            # 测试不同返回格式
            formats_to_test = ['dataframe', 'dict', 'json']
            
            for format_type in formats_to_test:
                result = client.price_distribution_stats(trade_date, format=format_type)
                
                if format_type == 'dataframe':
                    assert isinstance(result, pd.DataFrame)
                    assert not result.empty
                    assert 'trade_date' in result.columns
                    assert result['trade_date'].iloc[0] == trade_date
                elif format_type == 'dict':
                    assert isinstance(result, dict)
                    assert result['trade_date'] == trade_date
                    assert result['total_stocks'] == 1500
                elif format_type == 'json':
                    assert isinstance(result, str)
                    parsed = json.loads(result)
                    assert parsed['trade_date'] == trade_date
                    assert parsed['total_stocks'] == 1500
    
    def test_cache_management_integration(self, client):
        """测试缓存管理集成"""
        with patch('quickstock.services.price_distribution_stats_service.PriceDistributionStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟缓存操作
            async def mock_get_cache_info(trade_date=None):
                return {
                    'cache_entries': 15,
                    'total_size': 2048,
                    'hit_rate': 0.75,
                    'memory_usage': 150
                }
            
            async def mock_clear_cache(trade_date=None, pattern=None):
                if trade_date:
                    return 1
                elif pattern:
                    return 5
                else:
                    return 15
            
            async def mock_refresh_cache(trade_date, force=False):
                return {
                    'success': True,
                    'trade_date': trade_date,
                    'deleted_entries': 1,
                    'processing_time': 3.2,
                    'new_data_quality': 0.92
                }
            
            mock_service.get_cache_info = mock_get_cache_info
            mock_service.clear_cache = mock_clear_cache
            mock_service.refresh_cache = mock_refresh_cache
            mock_service.get_service_stats.return_value = {
                'performance': {'cache_hits': 75, 'cache_misses': 25, 'total_requests': 100}
            }
            
            # 测试缓存信息获取
            cache_info = client.get_distribution_cache_info()
            assert cache_info['cache_entries'] == 15
            assert cache_info['hit_rate'] == 0.75
            
            # 测试缓存清理
            cleared_all = client.clear_distribution_cache()
            assert cleared_all == 15
            
            cleared_specific = client.clear_distribution_cache('20240410')
            assert cleared_specific == 1
            
            cleared_pattern = client.clear_distribution_cache(pattern='2024*')
            assert cleared_pattern == 5
            
            # 测试缓存刷新
            refresh_result = client.refresh_distribution_cache('20240410')
            assert refresh_result['success'] == True
            assert refresh_result['trade_date'] == '20240410'
            
            # 测试缓存统计
            cache_stats = client.get_distribution_cache_stats()
            assert 'hit_rate' in cache_stats
            assert cache_stats['hit_rate'] == 0.75
    
    @pytest.mark.asyncio
    async def test_data_manager_integration(self, service, test_data_generator):
        """测试与数据管理器的集成"""
        # 生成测试数据
        trade_date = '20240411'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=2000
        )
        
        # 模拟数据管理器行为
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 测试数据获取集成
        request = PriceDistributionRequest(trade_date=trade_date)
        result = await service.get_price_distribution_stats(request)
        
        # 验证数据管理器被正确调用
        service.data_manager.get_stock_data_with_coordination.assert_called_once()
        call_args = service.data_manager.get_stock_data_with_coordination.call_args
        assert trade_date in str(call_args)
        
        # 验证结果正确性
        assert result.total_stocks == len(stock_data)
        assert result.trade_date == trade_date
        
        # 测试缓存集成
        service.data_manager.get_cache_stats.return_value = {'hits': 1, 'misses': 1}
        service.data_manager.get_memory_stats.return_value = {'memory_mb': 200}
        
        stats = service.get_service_stats()
        assert 'data_manager' in stats
        assert stats['data_manager']['cache_stats']['hits'] == 1
        assert stats['data_manager']['memory_stats']['memory_mb'] == 200
    
    @pytest.mark.asyncio
    async def test_stock_classifier_integration(self, service, test_data_generator):
        """测试与股票分类器的集成"""
        # 生成包含不同市场股票的数据
        trade_date = '20240412'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=1000,
            market_distribution={
                'shanghai_main': 0.4,
                'shanghai_star': 0.1,
                'shenzhen_main': 0.3,
                'shenzhen_gem': 0.15,
                'beijing': 0.03,
                'st_stocks': 0.02
            }
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 执行分类测试
        classified_data = await service._classify_stocks_by_market(stock_data)
        
        # 验证分类结果
        assert isinstance(classified_data, dict)
        assert 'total' in classified_data
        assert 'shanghai' in classified_data
        assert 'shenzhen' in classified_data
        assert 'beijing' in classified_data
        assert 'st' in classified_data
        assert 'non_st' in classified_data
        
        # 验证分类数量一致性
        total_stocks = len(classified_data['total'])
        st_stocks = len(classified_data['st'])
        non_st_stocks = len(classified_data['non_st'])
        assert st_stocks + non_st_stocks == total_stocks
        
        # 验证市场分类
        shanghai_stocks = len(classified_data['shanghai'])
        shenzhen_stocks = len(classified_data['shenzhen'])
        beijing_stocks = len(classified_data['beijing'])
        
        # 上海+深圳+北京应该等于总数（忽略其他小市场）
        major_markets_total = shanghai_stocks + shenzhen_stocks + beijing_stocks
        assert abs(major_markets_total - total_stocks) <= total_stocks * 0.1  # 允许10%误差
    
    @pytest.mark.asyncio
    async def test_analyzer_components_integration(self, service, test_data_generator):
        """测试分析器组件集成"""
        # 生成测试数据
        trade_date = '20240413'
        stock_data = test_data_generator.generate_realistic_stock_data(
            trade_date=trade_date,
            total_stocks=1500
        )
        
        service.data_manager.get_stock_data_with_coordination.return_value = stock_data
        
        # 测试分析器集成
        request = PriceDistributionRequest(trade_date=trade_date)
        result = await service.get_price_distribution_stats(request)
        
        # 验证分析器组件协作
        assert isinstance(result, PriceDistributionStats)
        
        # 验证分布计算器集成
        total_positive = sum(result.positive_ranges.values())
        total_negative = sum(result.negative_ranges.values())
        assert total_positive + total_negative == result.total_stocks
        
        # 验证统计聚合器集成
        for range_name, count in result.positive_ranges.items():
            expected_pct = (count / result.total_stocks) * 100
            actual_pct = result.positive_percentages[range_name]
            assert abs(actual_pct - expected_pct) < 0.01
        
        # 验证市场分解集成
        assert isinstance(result.market_breakdown, dict)
        for market, breakdown in result.market_breakdown.items():
            if isinstance(breakdown, dict) and 'total_stocks' in breakdown:
                market_total = breakdown['total_stocks']
                assert market_total >= 0
                assert market_total <= result.total_stocks
    
    def test_error_handling_integration(self, client, test_data_generator):
        """测试错误处理集成"""
        with patch('quickstock.services.price_distribution_stats_service.PriceDistributionStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 测试不同类型的错误
            error_scenarios = [
                (ValidationError("Invalid date format"), QuickStockError),
                (DataSourceError("Data source unavailable"), QuickStockError),
                (InsufficientDataError("Not enough data"), QuickStockError),
                (Exception("Unknown error"), QuickStockError)
            ]
            
            for original_error, expected_error_type in error_scenarios:
                async def mock_error_stats(request):
                    raise original_error
                
                mock_service.get_price_distribution_stats = mock_error_stats
                
                # 验证错误被正确转换和处理
                with pytest.raises(expected_error_type):
                    client.price_distribution_stats('20240414')
    
    @pytest.mark.asyncio
    async def test_configuration_integration(self, service):
        """测试配置集成"""
        # 验证配置被正确应用
        config_validation = await service.validate_service_configuration()
        
        assert isinstance(config_validation, dict)
        assert 'is_valid' in config_validation
        assert 'config_checks' in config_validation
        
        # 验证关键配置项
        config_checks = config_validation['config_checks']
        assert 'min_stock_count' in config_checks
        assert 'max_processing_time' in config_checks
        assert 'cache_enabled' in config_checks
        
        # 验证配置值
        assert service.config['min_stock_count'] == 50
        assert service.config['cache_enabled'] == True
        assert service.config['parallel_processing_enabled'] == True
    
    @pytest.mark.asyncio
    async def test_health_check_integration(self, service):
        """测试健康检查集成"""
        health_result = await service.health_check()
        
        # 验证健康检查结果结构
        assert isinstance(health_result, dict)
        assert 'service' in health_result
        assert 'status' in health_result
        assert 'checks' in health_result
        assert 'timestamp' in health_result
        
        # 验证服务标识
        assert health_result['service'] == 'PriceDistributionStatsService'
        
        # 验证组件检查
        checks = health_result['checks']
        expected_components = ['data_manager', 'cache_manager', 'stock_classifier', 'analyzer']
        
        for component in expected_components:
            assert component in checks
            assert 'status' in checks[component]
            assert 'details' in checks[component]
        
        # 验证整体状态
        assert health_result['status'] in ['healthy', 'unhealthy', 'degraded']
    
    def test_end_to_end_client_workflow(self, client, test_data_generator):
        """测试端到端客户端工作流"""
        # 模拟完整的客户端使用场景
        with patch('quickstock.services.price_distribution_stats_service.PriceDistributionStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 生成测试数据和预期结果
            trade_date = '20240415'
            stock_data = test_data_generator.generate_realistic_stock_data(
                trade_date=trade_date,
                total_stocks=3000
            )
            
            expected_stats = PriceDistributionStats(
                trade_date=trade_date,
                total_stocks=len(stock_data),
                positive_ranges={'0-3%': 600, '3-5%': 450, '5-7%': 300, '7-10%': 200, '>=10%': 150},
                positive_percentages={'0-3%': 20.0, '3-5%': 15.0, '5-7%': 10.0, '7-10%': 6.7, '>=10%': 5.0},
                negative_ranges={'0到-3%': 500, '-3到-5%': 400, '-5到-7%': 250, '-7到-10%': 150, '<=-10%': 100},
                negative_percentages={'0到-3%': 16.7, '-3到-5%': 13.3, '-5到-7%': 8.3, '-7到-10%': 5.0, '<=-10%': 3.3},
                market_breakdown={
                    'total': {'total_stocks': 3000},
                    'shanghai': {'total_stocks': 1200},
                    'shenzhen': {'total_stocks': 1500},
                    'beijing': {'total_stocks': 300}
                },
                processing_time=3.8,
                data_quality_score=0.94
            )
            
            async def mock_get_stats(request):
                return expected_stats
            
            async def mock_cache_operations(*args, **kwargs):
                return {'success': True, 'entries': 10}
            
            mock_service.get_price_distribution_stats = mock_get_stats
            mock_service.get_cache_info = mock_cache_operations
            mock_service.clear_cache = mock_cache_operations
            mock_service.refresh_cache = mock_cache_operations
            
            # 1. 获取基本统计
            basic_result = client.price_distribution_stats(trade_date)
            assert isinstance(basic_result, pd.DataFrame)
            assert not basic_result.empty
            
            # 2. 获取JSON格式结果
            json_result = client.price_distribution_stats(trade_date, format='json')
            parsed_json = json.loads(json_result)
            assert parsed_json['trade_date'] == trade_date
            
            # 3. 使用自定义参数
            custom_result = client.price_distribution_stats(
                trade_date,
                include_st=False,
                market_filter=['shanghai', 'shenzhen'],
                force_refresh=True
            )
            assert isinstance(custom_result, pd.DataFrame)
            
            # 4. 缓存管理操作
            cache_info = client.get_distribution_cache_info()
            assert isinstance(cache_info, dict)
            
            cleared = client.clear_distribution_cache(trade_date)
            assert isinstance(cleared, int)
            
            refreshed = client.refresh_distribution_cache(trade_date)
            assert isinstance(refreshed, dict)
            
            # 验证所有操作都成功完成
            assert True  # 如果到达这里，说明所有操作都成功了


if __name__ == '__main__':
    # 运行测试
    pytest.main([__file__, '-v', '--tb=short'])