"""
数据库操作层全面测试

为LimitUpStatsRepository提供全面的单元测试覆盖，包括CRUD操作、批量处理、事务管理等
"""

import pytest
import pytest_asyncio
import asyncio
import tempfile
import os
from datetime import datetime, timedelta
from typing import List
from unittest.mock import Mock, patch, AsyncMock

from quickstock.core.repository import LimitUpStatsRepository
from quickstock.core.database import DatabaseManager
from quickstock.models import LimitUpStats, StockDailyData
from quickstock.core.errors import DatabaseError, ValidationError


class TestLimitUpStatsRepositoryComprehensive:
    """涨停统计Repository全面测试类"""
    
    @pytest_asyncio.fixture
    async def temp_repository(self):
        """创建临时Repository实例"""
        with tempfile.NamedTemporaryFile(suffix='.db', delete=False) as tmp_file:
            db_path = tmp_file.name
        
        try:
            db_manager = DatabaseManager(db_path)
            await db_manager.initialize()
            repository = LimitUpStatsRepository(db_manager)
            yield repository
        finally:
            # 清理临时文件
            await self._cleanup_db_files(db_path)
    
    async def _cleanup_db_files(self, db_path):
        """清理数据库相关文件"""
        files_to_clean = [db_path, f"{db_path}-wal", f"{db_path}-shm"]
        for file_path in files_to_clean:
            if os.path.exists(file_path):
                try:
                    os.unlink(file_path)
                except OSError:
                    pass
    
    @pytest.fixture
    def comprehensive_limit_up_stats(self):
        """创建全面的涨停统计数据"""
        return LimitUpStats(
            trade_date='20241015',
            total=150,
            non_st=135,
            shanghai=60,
            shenzhen=50,
            star=30,
            beijing=10,
            st=15,
            limit_up_stocks=[
                '600000.SH', '600001.SH', '600002.SH',  # 上海
                '000001.SZ', '000002.SZ', '300001.SZ',  # 深圳
                '688001.SH', '688002.SH',               # 科创板
                '430001.BJ',                            # 北证
                '000003.SZ'                             # ST股票
            ],
            market_breakdown={
                'shanghai': ['600000.SH', '600001.SH', '600002.SH'],
                'shenzhen': ['000001.SZ', '000002.SZ', '300001.SZ'],
                'star': ['688001.SH', '688002.SH'],
                'beijing': ['430001.BJ'],
                'st': ['000003.SZ'],
                'non_st': ['600000.SH', '600001.SH', '600002.SH', '000001.SZ', '000002.SZ', '300001.SZ', '688001.SH', '688002.SH', '430001.BJ']
            }
        )
    
    @pytest.fixture
    def comprehensive_stock_details(self):
        """创建全面的股票详细数据"""
        return [
            # 上海股票
            StockDailyData(
                ts_code='600000.SH', trade_date='20241015',
                open=8.0, high=8.8, low=7.9, close=8.8,
                pre_close=8.0, change=0.8, pct_chg=10.0,
                vol=2000000, amount=17200000.0, name='浦发银行'
            ),
            StockDailyData(
                ts_code='600001.SH', trade_date='20241015',
                open=9.0, high=9.9, low=8.9, close=9.9,
                pre_close=9.0, change=0.9, pct_chg=10.0,
                vol=1800000, amount=17460000.0, name='邯郸钢铁'
            ),
            # 深圳股票
            StockDailyData(
                ts_code='000001.SZ', trade_date='20241015',
                open=10.0, high=11.0, low=9.8, close=11.0,
                pre_close=10.0, change=1.0, pct_chg=10.0,
                vol=1000000, amount=10500000.0, name='平安银行'
            ),
            StockDailyData(
                ts_code='300001.SZ', trade_date='20241015',
                open=15.0, high=16.5, low=14.8, close=16.5,
                pre_close=15.0, change=1.5, pct_chg=10.0,
                vol=600000, amount=9600000.0, name='特锐德'
            ),
            # 科创板股票
            StockDailyData(
                ts_code='688001.SH', trade_date='20241015',
                open=20.0, high=24.0, low=19.5, close=24.0,
                pre_close=20.0, change=4.0, pct_chg=20.0,
                vol=500000, amount=11000000.0, name='华兴源创'
            ),
            # 北证股票
            StockDailyData(
                ts_code='430001.BJ', trade_date='20241015',
                open=12.0, high=15.6, low=11.8, close=15.6,
                pre_close=12.0, change=3.6, pct_chg=30.0,
                vol=300000, amount=4380000.0, name='北交所股票'
            ),
            # ST股票
            StockDailyData(
                ts_code='000003.SZ', trade_date='20241015',
                open=6.3, high=6.62, low=6.25, close=6.62,
                pre_close=6.3, change=0.32, pct_chg=5.08,
                vol=800000, amount=5248000.0, name='ST万科'
            )
        ]
    
    @pytest.mark.asyncio
    async def test_comprehensive_save_and_retrieve(self, temp_repository, 
                                                 comprehensive_limit_up_stats, 
                                                 comprehensive_stock_details):
        """测试全面的保存和检索功能"""
        repository = temp_repository
        stats = comprehensive_limit_up_stats
        stock_details = comprehensive_stock_details
        
        # 保存完整数据
        success = await repository.save_limit_up_stats(stats, stock_details)
        assert success is True
        
        # 检索数据
        retrieved_stats = await repository.get_limit_up_stats(stats.trade_date)
        assert retrieved_stats is not None
        
        # 验证基本统计数据
        assert retrieved_stats.trade_date == stats.trade_date
        assert retrieved_stats.total == stats.total
        assert retrieved_stats.non_st == stats.non_st
        assert retrieved_stats.shanghai == stats.shanghai
        assert retrieved_stats.shenzhen == stats.shenzhen
        assert retrieved_stats.star == stats.star
        assert retrieved_stats.beijing == stats.beijing
        assert retrieved_stats.st == stats.st
        
        # 验证涨停股票列表
        assert len(retrieved_stats.limit_up_stocks) == len(stock_details)
        for stock in stock_details:
            assert stock.ts_code in retrieved_stats.limit_up_stocks
        
        # 验证市场分类
        assert 'shanghai' in retrieved_stats.market_breakdown
        assert 'shenzhen' in retrieved_stats.market_breakdown
        assert 'star' in retrieved_stats.market_breakdown
        assert 'beijing' in retrieved_stats.market_breakdown
        
        # 验证时间戳
        assert retrieved_stats.created_at is not None
        assert retrieved_stats.updated_at is not None
        assert isinstance(retrieved_stats.created_at, datetime)
        assert isinstance(retrieved_stats.updated_at, datetime)
    
    @pytest.mark.asyncio
    async def test_update_existing_stats_comprehensive(self, temp_repository, comprehensive_limit_up_stats):
        """测试更新现有统计数据的全面场景"""
        repository = temp_repository
        original_stats = comprehensive_limit_up_stats
        
        # 首次保存
        await repository.save_limit_up_stats(original_stats)
        
        # 获取原始时间戳
        original_retrieved = await repository.get_limit_up_stats(original_stats.trade_date)
        original_created_at = original_retrieved.created_at
        original_updated_at = original_retrieved.updated_at
        
        # 等待确保时间戳不同
        await asyncio.sleep(0.1)
        
        # 创建更新的统计数据
        updated_stats = LimitUpStats(
            trade_date=original_stats.trade_date,
            total=200,  # 增加总数
            non_st=180,
            shanghai=80,  # 增加上海股票
            shenzhen=70,  # 增加深圳股票
            star=40,      # 增加科创板股票
            beijing=10,   # 保持北证股票
            st=20,        # 增加ST股票
            limit_up_stocks=original_stats.limit_up_stocks + ['600003.SH', '000004.SZ'],
            market_breakdown={
                'shanghai': original_stats.market_breakdown['shanghai'] + ['600003.SH'],
                'shenzhen': original_stats.market_breakdown['shenzhen'] + ['000004.SZ'],
                'star': original_stats.market_breakdown['star'],
                'beijing': original_stats.market_breakdown['beijing'],
            }
        )
        
        # 更新数据
        success = await repository.save_limit_up_stats(updated_stats)
        assert success is True
        
        # 验证更新结果
        updated_retrieved = await repository.get_limit_up_stats(original_stats.trade_date)
        assert updated_retrieved.total == 200
        assert updated_retrieved.shanghai == 80
        assert updated_retrieved.shenzhen == 70
        assert updated_retrieved.star == 40
        assert updated_retrieved.st == 20
        
        # 验证时间戳更新
        assert updated_retrieved.created_at == original_created_at  # 创建时间不变
        assert updated_retrieved.updated_at > original_updated_at   # 更新时间改变
        
        # 验证涨停股票列表更新
        assert '600003.SH' in updated_retrieved.limit_up_stocks
        assert '000004.SZ' in updated_retrieved.limit_up_stocks
    
    @pytest.mark.asyncio
    async def test_batch_operations_comprehensive(self, temp_repository):
        """测试批量操作的全面场景"""
        repository = temp_repository
        
        # 创建多个日期的统计数据
        stats_list = []
        for i in range(10):
            date = f'2024101{i}'
            stats = LimitUpStats(
                trade_date=date,
                total=100 + i * 10,
                non_st=90 + i * 10,
                shanghai=30 + i * 2,
                shenzhen=40 + i * 3,
                star=20 + i * 2,
                beijing=10 + i * 3,
                st=10,
                limit_up_stocks=[f'00000{i}.SZ'],
                market_breakdown={'shenzhen': [f'00000{i}.SZ']}
            )
            stats_list.append(stats)
        
        # 执行批量保存
        result = await repository.batch_save_stats(stats_list)
        
        # 验证批量保存结果
        assert result['total'] == 10
        assert result['success'] == 10
        assert result['failed'] == 0
        assert len(result['errors']) == 0
        
        # 验证数据已保存
        all_dates = await repository.list_available_dates()
        assert len(all_dates) == 10
        
        # 验证每个统计数据
        for i, stats in enumerate(stats_list):
            retrieved = await repository.get_limit_up_stats(stats.trade_date)
            assert retrieved is not None
            assert retrieved.total == stats.total
            assert retrieved.shanghai == stats.shanghai
    
    @pytest.mark.asyncio
    async def test_batch_operations_with_errors(self, temp_repository):
        """测试批量操作中的错误处理"""
        repository = temp_repository
        
        # 创建包含有效和无效数据的列表
        stats_list = []
        
        # 有效数据
        for i in range(5):
            stats = LimitUpStats(
                trade_date=f'2024101{i}',
                total=100,
                non_st=90,
                shanghai=30,
                shenzhen=40,
                star=20,
                beijing=10,
                st=10,
                limit_up_stocks=[],
                market_breakdown={}
            )
            stats_list.append(stats)
        
        # 无效数据（重复日期）
        duplicate_stats = LimitUpStats(
            trade_date='20241010',  # 重复日期
            total=200,
            non_st=180,
            shanghai=60,
            shenzhen=80,
            star=40,
            beijing=20,
            st=20,
            limit_up_stocks=[],
            market_breakdown={}
        )
        stats_list.append(duplicate_stats)
        
        # 执行批量保存
        result = await repository.batch_save_stats(stats_list)
        
        # 验证结果
        assert result['total'] == 6
        assert result['success'] == 5  # 5个有效数据成功
        assert result['failed'] == 1   # 1个重复数据失败
        assert len(result['errors']) == 1
        
        # 验证错误信息
        error = result['errors'][0]
        assert '20241010' in error['trade_date']
        assert 'error' in error
    
    @pytest.mark.asyncio
    async def test_query_operations_comprehensive(self, temp_repository):
        """测试查询操作的全面场景"""
        repository = temp_repository
        
        # 创建跨月的测试数据
        test_dates = [
            '20241001', '20241005', '20241010', '20241015', '20241020',
            '20241101', '20241105', '20241110', '20241115', '20241120'
        ]
        
        for i, date in enumerate(test_dates):
            stats = LimitUpStats(
                trade_date=date,
                total=100 + i * 5,
                non_st=90 + i * 5,
                shanghai=30 + i,
                shenzhen=40 + i,
                star=20 + i,
                beijing=10 + i,
                st=10,
                limit_up_stocks=[f'00000{i}.SZ'],
                market_breakdown={'shenzhen': [f'00000{i}.SZ']}
            )
            await repository.save_limit_up_stats(stats)
        
        # 测试查询所有数据
        all_stats = await repository.query_limit_up_stats()
        assert len(all_stats) == 10
        
        # 验证按日期降序排列
        dates = [stats.trade_date for stats in all_stats]
        assert dates == sorted(dates, reverse=True)
        
        # 测试日期范围查询
        october_stats = await repository.query_limit_up_stats(
            start_date='20241001',
            end_date='20241031'
        )
        assert len(october_stats) == 5
        assert all(stats.trade_date.startswith('202410') for stats in october_stats)
        
        # 测试限制数量查询
        limited_stats = await repository.query_limit_up_stats(limit=3)
        assert len(limited_stats) == 3
        
        # 测试偏移查询
        offset_stats = await repository.query_limit_up_stats(limit=3, offset=2)
        assert len(offset_stats) == 3
        assert offset_stats[0].trade_date != all_stats[0].trade_date
        assert offset_stats[0].trade_date != all_stats[1].trade_date
        
        # 测试组合查询
        combined_stats = await repository.query_limit_up_stats(
            start_date='20241010',
            end_date='20241120',
            limit=5,
            offset=1
        )
        assert len(combined_stats) <= 5
        assert all('20241010' <= stats.trade_date <= '20241120' for stats in combined_stats)
    
    @pytest.mark.asyncio
    async def test_delete_operations_comprehensive(self, temp_repository, comprehensive_limit_up_stats, comprehensive_stock_details):
        """测试删除操作的全面场景"""
        repository = temp_repository
        
        # 保存多个日期的数据
        dates = ['20241010', '20241015', '20241020']
        for date in dates:
            stats = LimitUpStats(
                trade_date=date,
                total=100,
                non_st=90,
                shanghai=30,
                shenzhen=40,
                star=20,
                beijing=10,
                st=10,
                limit_up_stocks=['000001.SZ'],
                market_breakdown={'shenzhen': ['000001.SZ']}
            )
            await repository.save_limit_up_stats(stats, comprehensive_stock_details[:1])
        
        # 验证数据存在
        all_dates = await repository.list_available_dates()
        assert len(all_dates) == 3
        
        # 删除单个日期
        success = await repository.delete_limit_up_stats('20241015')
        assert success is True
        
        # 验证删除结果
        remaining_dates = await repository.list_available_dates()
        assert len(remaining_dates) == 2
        assert '20241015' not in remaining_dates
        
        # 验证相关的股票详细数据也被删除（级联删除）
        deleted_stats = await repository.get_limit_up_stats('20241015')
        assert deleted_stats is None
        
        # 测试批量删除
        delete_dates = ['20241010', '20241020', '20241099']  # 包含不存在的日期
        result = await repository.batch_delete_stats(delete_dates)
        
        # 验证批量删除结果
        assert result['total'] == 3
        assert result['success'] == 2  # 两个存在的日期成功删除
        assert result['failed'] == 1   # 一个不存在的日期失败
        
        # 验证所有数据已删除
        final_dates = await repository.list_available_dates()
        assert len(final_dates) == 0
    
    @pytest.mark.asyncio
    async def test_database_statistics_comprehensive(self, temp_repository):
        """测试数据库统计的全面场景"""
        repository = temp_repository
        
        # 获取初始统计
        initial_stats = await repository.get_database_stats()
        assert initial_stats['tables']['limit_up_stats'] == 0
        assert initial_stats['tables']['limit_up_stocks'] == 0
        
        # 添加不同规模的数据
        dates_and_totals = [
            ('20241001', 50),
            ('20241002', 100),
            ('20241003', 150),
            ('20241004', 200),
            ('20241005', 75)
        ]
        
        for date, total in dates_and_totals:
            stats = LimitUpStats(
                trade_date=date,
                total=total,
                non_st=total - 10,
                shanghai=total // 4,
                shenzhen=total // 3,
                star=total // 5,
                beijing=total // 10,
                st=10,
                limit_up_stocks=[f'stock_{i}.SZ' for i in range(total)],
                market_breakdown={}
            )
            await repository.save_limit_up_stats(stats)
        
        # 获取更新后的统计
        updated_stats = await repository.get_database_stats()
        
        # 验证基本统计
        assert updated_stats['tables']['limit_up_stats'] == 5
        
        # 验证日期范围
        assert 'date_range' in updated_stats
        assert updated_stats['date_range']['earliest'] == '20241001'
        assert updated_stats['date_range']['latest'] == '20241005'
        
        # 验证平均值
        assert 'averages' in updated_stats
        expected_avg_total = sum(total for _, total in dates_and_totals) / len(dates_and_totals)
        assert abs(updated_stats['averages']['total'] - expected_avg_total) < 1
        
        # 验证最大最小值
        assert 'extremes' in updated_stats
        assert updated_stats['extremes']['max_total'] == 200
        assert updated_stats['extremes']['min_total'] == 50
    
    @pytest.mark.asyncio
    async def test_transaction_handling_comprehensive(self, temp_repository):
        """测试事务处理的全面场景"""
        repository = temp_repository
        
        # 测试成功事务
        stats = LimitUpStats(
            trade_date='20241015',
            total=100,
            non_st=90,
            shanghai=30,
            shenzhen=40,
            star=20,
            beijing=10,
            st=10,
            limit_up_stocks=['000001.SZ'],
            market_breakdown={'shenzhen': ['000001.SZ']}
        )
        
        success = await repository.save_limit_up_stats(stats)
        assert success is True
        
        # 验证数据已提交
        retrieved = await repository.get_limit_up_stats('20241015')
        assert retrieved is not None
        
        # 测试事务回滚（通过模拟数据库错误）
        with patch.object(repository.db_manager, 'get_connection') as mock_get_conn:
            mock_db = AsyncMock()
            mock_db.execute.side_effect = Exception("Database error")
            mock_get_conn.return_value.__aenter__.return_value = mock_db
            
            # 尝试保存应该失败
            with pytest.raises(DatabaseError):
                await repository.save_limit_up_stats(stats)
    
    @pytest.mark.asyncio
    async def test_concurrent_operations_comprehensive(self, temp_repository):
        """测试并发操作的全面场景"""
        repository = temp_repository
        
        # 并发保存操作
        async def save_worker(worker_id):
            stats = LimitUpStats(
                trade_date=f'2024101{worker_id}',
                total=100 + worker_id,
                non_st=90 + worker_id,
                shanghai=30,
                shenzhen=40,
                star=20,
                beijing=10,
                st=10,
                limit_up_stocks=[f'stock_{worker_id}.SZ'],
                market_breakdown={'shenzhen': [f'stock_{worker_id}.SZ']}
            )
            return await repository.save_limit_up_stats(stats)
        
        # 并发查询操作
        async def query_worker():
            return await repository.list_available_dates()
        
        # 并发删除操作
        async def delete_worker(date):
            return await repository.delete_limit_up_stats(date)
        
        # 执行并发保存
        save_tasks = [save_worker(i) for i in range(10)]
        save_results = await asyncio.gather(*save_tasks)
        
        # 验证所有保存操作成功
        assert all(save_results)
        
        # 执行并发查询
        query_tasks = [query_worker() for _ in range(5)]
        query_results = await asyncio.gather(*query_tasks)
        
        # 验证查询结果一致
        assert all(len(result) == 10 for result in query_results)
        
        # 执行并发删除
        delete_dates = [f'2024101{i}' for i in range(5)]
        delete_tasks = [delete_worker(date) for date in delete_dates]
        delete_results = await asyncio.gather(*delete_tasks)
        
        # 验证删除结果
        assert all(delete_results)
        
        # 验证最终状态
        final_dates = await repository.list_available_dates()
        assert len(final_dates) == 5
    
    @pytest.mark.asyncio
    async def test_performance_with_large_dataset(self, temp_repository):
        """测试大数据集性能"""
        repository = temp_repository
        import time
        
        # 创建大量数据
        large_stats_list = []
        for i in range(1000):
            stats = LimitUpStats(
                trade_date=f'20241{i:03d}',
                total=100 + i % 100,
                non_st=90 + i % 100,
                shanghai=30,
                shenzhen=40,
                star=20,
                beijing=10,
                st=10,
                limit_up_stocks=[f'stock_{j}.SZ' for j in range(i % 10)],
                market_breakdown={}
            )
            large_stats_list.append(stats)
        
        # 测试批量保存性能
        start_time = time.time()
        result = await repository.batch_save_stats(large_stats_list)
        save_time = time.time() - start_time
        
        # 验证保存结果
        assert result['success'] == 1000
        assert result['failed'] == 0
        
        # 性能要求：1000条记录应该在合理时间内保存
        assert save_time < 30.0  # 30秒内完成
        
        # 测试查询性能
        start_time = time.time()
        all_stats = await repository.query_limit_up_stats()
        query_time = time.time() - start_time
        
        # 验证查询结果
        assert len(all_stats) == 1000
        
        # 性能要求：查询应该在合理时间内完成
        assert query_time < 5.0  # 5秒内完成
    
    @pytest.mark.asyncio
    async def test_data_integrity_validation(self, temp_repository):
        """测试数据完整性验证"""
        repository = temp_repository
        
        # 测试有效数据
        valid_stats = LimitUpStats(
            trade_date='20241015',
            total=100,
            non_st=90,
            shanghai=30,
            shenzhen=40,
            star=20,
            beijing=10,
            st=10,
            limit_up_stocks=['000001.SZ'] * 100,
            market_breakdown={'shenzhen': ['000001.SZ'] * 100}
        )
        
        success = await repository.save_limit_up_stats(valid_stats)
        assert success is True
        
        # 测试数据一致性验证
        retrieved = await repository.get_limit_up_stats('20241015')
        assert retrieved.total == retrieved.shanghai + retrieved.shenzhen + retrieved.star + retrieved.beijing
        assert retrieved.total == retrieved.st + retrieved.non_st
    
    @pytest.mark.asyncio
    async def test_error_recovery_mechanisms(self, temp_repository):
        """测试错误恢复机制"""
        repository = temp_repository
        
        # 模拟数据库连接错误
        with patch.object(repository.db_manager, 'get_connection') as mock_get_conn:
            # 第一次调用失败
            mock_get_conn.side_effect = [
                Exception("Connection failed"),
                repository.db_manager.get_connection()  # 第二次调用成功
            ]
            
            stats = LimitUpStats(
                trade_date='20241015',
                total=100,
                non_st=90,
                shanghai=30,
                shenzhen=40,
                star=20,
                beijing=10,
                st=10,
                limit_up_stocks=[],
                market_breakdown={}
            )
            
            # 第一次尝试应该失败
            with pytest.raises(DatabaseError):
                await repository.save_limit_up_stats(stats)
    
    @pytest.mark.asyncio
    async def test_repository_statistics_tracking(self, temp_repository):
        """测试Repository统计跟踪"""
        repository = temp_repository
        
        # 执行一些操作
        stats = LimitUpStats(
            trade_date='20241015',
            total=100,
            non_st=90,
            shanghai=30,
            shenzhen=40,
            star=20,
            beijing=10,
            st=10,
            limit_up_stocks=[],
            market_breakdown={}
        )
        
        await repository.save_limit_up_stats(stats)
        await repository.get_limit_up_stats('20241015')
        await repository.list_available_dates()
        
        # 获取操作统计（如果Repository实现了统计功能）
        if hasattr(repository, 'get_operation_stats'):
            operation_stats = repository.get_operation_stats()
            assert 'total_operations' in operation_stats
            assert operation_stats['total_operations'] >= 3


class TestLimitUpStatsRepositoryEdgeCases:
    """Repository边界情况测试"""
    
    @pytest_asyncio.fixture
    async def temp_repository(self):
        """创建临时Repository实例"""
        with tempfile.NamedTemporaryFile(suffix='.db', delete=False) as tmp_file:
            db_path = tmp_file.name
        
        try:
            db_manager = DatabaseManager(db_path)
            await db_manager.initialize()
            repository = LimitUpStatsRepository(db_manager)
            yield repository
        finally:
            # 清理临时文件
            files_to_clean = [db_path, f"{db_path}-wal", f"{db_path}-shm"]
            for file_path in files_to_clean:
                if os.path.exists(file_path):
                    try:
                        os.unlink(file_path)
                    except OSError:
                        pass
    
    @pytest.mark.asyncio
    async def test_empty_database_operations(self, temp_repository):
        """测试空数据库操作"""
        repository = temp_repository
        
        # 查询不存在的数据
        result = await repository.get_limit_up_stats('20241015')
        assert result is None
        
        # 查询空数据库
        all_stats = await repository.query_limit_up_stats()
        assert len(all_stats) == 0
        
        # 列出空数据库的日期
        dates = await repository.list_available_dates()
        assert len(dates) == 0
        
        # 删除不存在的数据
        success = await repository.delete_limit_up_stats('20241015')
        assert success is False
    
    @pytest.mark.asyncio
    async def test_extreme_values(self, temp_repository):
        """测试极值情况"""
        repository = temp_repository
        
        # 测试零值统计
        zero_stats = LimitUpStats(
            trade_date='20241015',
            total=0,
            non_st=0,
            shanghai=0,
            shenzhen=0,
            star=0,
            beijing=0,
            st=0,
            limit_up_stocks=[],
            market_breakdown={}
        )
        
        success = await repository.save_limit_up_stats(zero_stats)
        assert success is True
        
        retrieved = await repository.get_limit_up_stats('20241015')
        assert retrieved.total == 0
        
        # 测试大数值统计
        large_stats = LimitUpStats(
            trade_date='20241016',
            total=10000,
            non_st=9000,
            shanghai=3000,
            shenzhen=4000,
            star=2000,
            beijing=1000,
            st=1000,
            limit_up_stocks=[f'stock_{i}.SZ' for i in range(10000)],
            market_breakdown={}
        )
        
        success = await repository.save_limit_up_stats(large_stats)
        assert success is True
        
        retrieved = await repository.get_limit_up_stats('20241016')
        assert retrieved.total == 10000


if __name__ == "__main__":
    pytest.main([__file__])