"""
财务报告配置管理的单元测试
"""

import os
import json
import yaml
import tempfile
import pytest
from pathlib import Path

from quickstock.config import Config
from quickstock.core.errors import ValidationError


class TestFinancialReportsConfig:
    """财务报告配置测试"""
    
    def test_default_financial_reports_config(self):
        """测试默认财务报告配置"""
        config = Config()
        
        # 验证默认值
        assert config.financial_reports_enabled is True
        assert config.financial_reports_batch_size == 50
        assert config.financial_reports_timeout == 30
        assert config.financial_reports_retry_attempts == 3
        assert config.financial_reports_retry_delay == 1.0
        assert config.financial_reports_rate_limit == 2.0
        assert config.financial_reports_queue_size == 1000
        assert config.financial_reports_enable_metrics is True
        assert config.financial_reports_baostock_endpoint == "http://baostock.com"
        assert config.financial_reports_auth_required is False
        assert config.financial_reports_auth_token is None
        
        # 验证缓存TTL默认值
        expected_ttl = {
            'financial_reports': 86400,  # 24 hours
            'earnings_forecast': 14400,  # 4 hours
            'flash_reports': 3600        # 1 hour
        }
        assert config.financial_reports_cache_ttl == expected_ttl
        
        # 验证数据源优先级包含财务报告类型
        assert 'financial_reports' in config.data_source_priority
        assert 'earnings_forecast' in config.data_source_priority
        assert 'flash_reports' in config.data_source_priority
        assert config.data_source_priority['financial_reports'] == ['baostock']
        assert config.data_source_priority['earnings_forecast'] == ['baostock']
        assert config.data_source_priority['flash_reports'] == ['baostock']
    
    def test_financial_reports_config_validation(self):
        """测试财务报告配置验证"""
        # 测试无效的缓存TTL类型
        with pytest.raises(ValidationError, match="financial_reports_cache_ttl必须是字典类型"):
            Config(financial_reports_cache_ttl="invalid")
        
        # 测试缺少必需的缓存TTL键
        with pytest.raises(ValidationError, match="financial_reports_cache_ttl缺少必需的键"):
            Config(financial_reports_cache_ttl={'financial_reports': 86400})
        
        # 测试无效的缓存TTL值
        with pytest.raises(ValidationError, match="financial_reports_cache_ttl\\[earnings_forecast\\]必须是大于0的整数"):
            Config(financial_reports_cache_ttl={
                'financial_reports': 86400,
                'earnings_forecast': 0,
                'flash_reports': 3600
            })
        
        # 测试无效的批处理大小
        with pytest.raises(ValidationError, match="financial_reports_batch_size必须大于0"):
            Config(financial_reports_batch_size=0)
        
        # 测试无效的超时时间
        with pytest.raises(ValidationError, match="financial_reports_timeout必须大于0"):
            Config(financial_reports_timeout=-1)
        
        # 测试无效的重试次数
        with pytest.raises(ValidationError, match="financial_reports_retry_attempts不能小于0"):
            Config(financial_reports_retry_attempts=-1)
        
        # 测试无效的重试延迟
        with pytest.raises(ValidationError, match="financial_reports_retry_delay不能小于0"):
            Config(financial_reports_retry_delay=-1.0)
        
        # 测试无效的速率限制
        with pytest.raises(ValidationError, match="financial_reports_rate_limit必须大于0"):
            Config(financial_reports_rate_limit=0)
        
        # 测试无效的队列大小
        with pytest.raises(ValidationError, match="financial_reports_queue_size必须大于0"):
            Config(financial_reports_queue_size=-1)
        
        # 测试无效的baostock端点
        with pytest.raises(ValidationError, match="financial_reports_baostock_endpoint必须是非空字符串"):
            Config(financial_reports_baostock_endpoint="")
    
    def test_get_financial_reports_config(self):
        """测试获取财务报告配置"""
        config = Config()
        fr_config = config.get_financial_reports_config()
        
        # 验证返回的配置包含所有必需字段
        expected_keys = {
            'financial_reports_enabled',
            'financial_reports_cache_ttl',
            'financial_reports_batch_size',
            'financial_reports_timeout',
            'financial_reports_retry_attempts',
            'financial_reports_retry_delay',
            'financial_reports_rate_limit',
            'financial_reports_queue_size',
            'financial_reports_enable_metrics',
            'financial_reports_baostock_endpoint',
            'financial_reports_auth_required',
            'financial_reports_auth_token'
        }
        assert set(fr_config.keys()) == expected_keys
        
        # 验证缓存TTL是副本而不是引用
        fr_config['financial_reports_cache_ttl']['financial_reports'] = 999999
        assert config.financial_reports_cache_ttl['financial_reports'] == 86400
    
    def test_set_financial_reports_cache_ttl(self):
        """测试设置财务报告缓存TTL"""
        config = Config()
        
        # 测试设置有效的TTL
        config.set_financial_reports_cache_ttl('financial_reports', 7200)
        assert config.financial_reports_cache_ttl['financial_reports'] == 7200
        
        # 测试设置无效的数据类型
        with pytest.raises(ValidationError, match="data_type必须是以下之一"):
            config.set_financial_reports_cache_ttl('invalid_type', 3600)
        
        # 测试设置无效的TTL值
        with pytest.raises(ValidationError, match="ttl_seconds必须是大于0的整数"):
            config.set_financial_reports_cache_ttl('financial_reports', 0)
        
        with pytest.raises(ValidationError, match="ttl_seconds必须是大于0的整数"):
            config.set_financial_reports_cache_ttl('financial_reports', "invalid")
    
    def test_get_financial_reports_cache_ttl(self):
        """测试获取财务报告缓存TTL"""
        config = Config()
        
        # 测试获取有效的TTL
        assert config.get_financial_reports_cache_ttl('financial_reports') == 86400
        assert config.get_financial_reports_cache_ttl('earnings_forecast') == 14400
        assert config.get_financial_reports_cache_ttl('flash_reports') == 3600
        
        # 测试获取无效的数据类型
        with pytest.raises(ValidationError, match="未知的财务报告数据类型"):
            config.get_financial_reports_cache_ttl('invalid_type')
    
    def test_set_financial_reports_batch_size(self):
        """测试设置财务报告批处理大小"""
        config = Config()
        
        # 测试设置有效的批处理大小
        config.set_financial_reports_batch_size(100)
        assert config.financial_reports_batch_size == 100
        
        # 测试设置无效的批处理大小
        with pytest.raises(ValidationError, match="batch_size必须是大于0的整数"):
            config.set_financial_reports_batch_size(0)
        
        with pytest.raises(ValidationError, match="batch_size必须是大于0的整数"):
            config.set_financial_reports_batch_size("invalid")
    
    def test_set_financial_reports_rate_limit(self):
        """测试设置财务报告速率限制"""
        config = Config()
        
        # 测试设置有效的速率限制
        config.set_financial_reports_rate_limit(5.0)
        assert config.financial_reports_rate_limit == 5.0
        
        config.set_financial_reports_rate_limit(3)
        assert config.financial_reports_rate_limit == 3.0
        
        # 测试设置无效的速率限制
        with pytest.raises(ValidationError, match="rate_limit必须是大于0的数字"):
            config.set_financial_reports_rate_limit(0)
        
        with pytest.raises(ValidationError, match="rate_limit必须是大于0的数字"):
            config.set_financial_reports_rate_limit("invalid")
    
    def test_set_financial_reports_retry_config(self):
        """测试设置财务报告重试配置"""
        config = Config()
        
        # 测试设置有效的重试配置
        config.set_financial_reports_retry_config(5, 2.5)
        assert config.financial_reports_retry_attempts == 5
        assert config.financial_reports_retry_delay == 2.5
        
        # 测试设置无效的重试次数
        with pytest.raises(ValidationError, match="retry_attempts必须是非负整数"):
            config.set_financial_reports_retry_config(-1, 1.0)
        
        with pytest.raises(ValidationError, match="retry_attempts必须是非负整数"):
            config.set_financial_reports_retry_config("invalid", 1.0)
        
        # 测试设置无效的重试延迟
        with pytest.raises(ValidationError, match="retry_delay必须是非负数字"):
            config.set_financial_reports_retry_config(3, -1.0)
        
        with pytest.raises(ValidationError, match="retry_delay必须是非负数字"):
            config.set_financial_reports_retry_config(3, "invalid")
    
    def test_enable_financial_reports(self):
        """测试启用/禁用财务报告功能"""
        config = Config()
        
        # 测试禁用财务报告
        config.enable_financial_reports(False)
        assert config.financial_reports_enabled is False
        
        # 测试启用财务报告
        config.enable_financial_reports(True)
        assert config.financial_reports_enabled is True
        
        # 测试默认启用
        config.enable_financial_reports()
        assert config.financial_reports_enabled is True
    
    def test_set_financial_reports_auth(self):
        """测试设置财务报告认证"""
        config = Config()
        
        # 测试设置认证令牌
        config.set_financial_reports_auth("test_token", True)
        assert config.financial_reports_auth_token == "test_token"
        assert config.financial_reports_auth_required is True
        
        # 测试清除认证
        config.set_financial_reports_auth(None, False)
        assert config.financial_reports_auth_token is None
        assert config.financial_reports_auth_required is False
    
    def test_financial_reports_config_from_file(self):
        """测试从文件加载财务报告配置"""
        config_data = {
            "financial_reports_enabled": False,
            "financial_reports_cache_ttl": {
                "financial_reports": 7200,
                "earnings_forecast": 1800,
                "flash_reports": 900
            },
            "financial_reports_batch_size": 25,
            "financial_reports_timeout": 60,
            "financial_reports_retry_attempts": 5,
            "financial_reports_retry_delay": 2.0,
            "financial_reports_rate_limit": 1.5,
            "financial_reports_queue_size": 500,
            "financial_reports_enable_metrics": False,
            "financial_reports_baostock_endpoint": "http://custom.baostock.com",
            "financial_reports_auth_required": True,
            "financial_reports_auth_token": "test_token"
        }
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as f:
            json.dump(config_data, f)
            temp_path = f.name
        
        try:
            config = Config.from_file(temp_path)
            
            assert config.financial_reports_enabled is False
            assert config.financial_reports_cache_ttl['financial_reports'] == 7200
            assert config.financial_reports_cache_ttl['earnings_forecast'] == 1800
            assert config.financial_reports_cache_ttl['flash_reports'] == 900
            assert config.financial_reports_batch_size == 25
            assert config.financial_reports_timeout == 60
            assert config.financial_reports_retry_attempts == 5
            assert config.financial_reports_retry_delay == 2.0
            assert config.financial_reports_rate_limit == 1.5
            assert config.financial_reports_queue_size == 500
            assert config.financial_reports_enable_metrics is False
            assert config.financial_reports_baostock_endpoint == "http://custom.baostock.com"
            assert config.financial_reports_auth_required is True
            assert config.financial_reports_auth_token == "test_token"
        finally:
            os.unlink(temp_path)
    
    def test_financial_reports_config_to_file(self):
        """测试保存财务报告配置到文件"""
        config = Config()
        config.financial_reports_batch_size = 75
        config.financial_reports_rate_limit = 3.5
        config.set_financial_reports_cache_ttl('financial_reports', 10800)
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as f:
            temp_path = f.name
        
        try:
            config.to_file(temp_path, 'json')
            
            # 验证文件内容
            with open(temp_path, 'r', encoding='utf-8') as f:
                saved_config = json.load(f)
            
            assert saved_config['financial_reports_batch_size'] == 75
            assert saved_config['financial_reports_rate_limit'] == 3.5
            assert saved_config['financial_reports_cache_ttl']['financial_reports'] == 10800
        finally:
            os.unlink(temp_path)
    
    def test_financial_reports_config_update(self):
        """测试更新财务报告配置"""
        config = Config()
        
        # 测试更新有效配置
        config.update(
            financial_reports_batch_size=80,
            financial_reports_rate_limit=4.0,
            financial_reports_timeout=45
        )
        
        assert config.financial_reports_batch_size == 80
        assert config.financial_reports_rate_limit == 4.0
        assert config.financial_reports_timeout == 45
        
        # 测试更新无效配置参数
        with pytest.raises(ValidationError, match="未知的配置参数"):
            config.update(invalid_financial_param=123)
    
    def test_financial_reports_config_edge_cases(self):
        """测试财务报告配置边界情况"""
        # 测试最小有效值
        config = Config(
            financial_reports_batch_size=1,
            financial_reports_timeout=1,
            financial_reports_retry_attempts=0,
            financial_reports_retry_delay=0.0,
            financial_reports_rate_limit=0.1,
            financial_reports_queue_size=1,
            financial_reports_cache_ttl={
                'financial_reports': 1,
                'earnings_forecast': 1,
                'flash_reports': 1
            }
        )
        
        assert config.financial_reports_batch_size == 1
        assert config.financial_reports_timeout == 1
        assert config.financial_reports_retry_attempts == 0
        assert config.financial_reports_retry_delay == 0.0
        assert config.financial_reports_rate_limit == 0.1
        assert config.financial_reports_queue_size == 1
        
        # 测试大数值
        config = Config(
            financial_reports_batch_size=10000,
            financial_reports_timeout=3600,
            financial_reports_retry_attempts=100,
            financial_reports_retry_delay=60.0,
            financial_reports_rate_limit=1000.0,
            financial_reports_queue_size=100000,
            financial_reports_cache_ttl={
                'financial_reports': 604800,  # 1 week
                'earnings_forecast': 86400,   # 1 day
                'flash_reports': 3600         # 1 hour
            }
        )
        
        assert config.financial_reports_batch_size == 10000
        assert config.financial_reports_timeout == 3600
        assert config.financial_reports_retry_attempts == 100
        assert config.financial_reports_retry_delay == 60.0
        assert config.financial_reports_rate_limit == 1000.0
        assert config.financial_reports_queue_size == 100000