"""
财务报告配置集成测试

测试财务报告配置与整个系统的集成
"""

import os
import json
import yaml
import tempfile
import pytest
from pathlib import Path

from quickstock.config import Config
from quickstock.core.errors import ValidationError


class TestFinancialReportsConfigIntegration:
    """财务报告配置集成测试"""
    
    def test_config_integration_with_existing_features(self):
        """测试财务报告配置与现有功能的集成"""
        config = Config()
        
        # 验证财务报告配置不影响现有功能
        assert config.cache_enabled is True
        assert config.request_timeout == 30
        assert config.max_retries == 3
        
        # 验证财务报告配置正常工作
        assert config.financial_reports_enabled is True
        assert config.financial_reports_batch_size == 50
        
        # 验证数据源优先级包含财务报告类型
        assert 'financial_reports' in config.data_source_priority
        assert 'stock_basic' in config.data_source_priority  # 现有类型仍然存在
    
    def test_complete_config_file_loading(self):
        """测试完整配置文件加载"""
        # 创建包含所有配置的完整配置文件
        complete_config = {
            # 现有配置
            "tushare_token": "test_token",
            "cache_enabled": True,
            "cache_expire_hours": 48,
            "request_timeout": 60,
            "max_retries": 5,
            "retry_delay": 2.0,
            "log_level": "DEBUG",
            
            # 财务报告配置
            "financial_reports_enabled": True,
            "financial_reports_cache_ttl": {
                "financial_reports": 7200,
                "earnings_forecast": 1800,
                "flash_reports": 900
            },
            "financial_reports_batch_size": 25,
            "financial_reports_timeout": 45,
            "financial_reports_retry_attempts": 4,
            "financial_reports_retry_delay": 1.5,
            "financial_reports_rate_limit": 3.0,
            "financial_reports_queue_size": 2000,
            "financial_reports_enable_metrics": False,
            "financial_reports_baostock_endpoint": "http://test.baostock.com",
            "financial_reports_auth_required": True,
            "financial_reports_auth_token": "test_fr_token",
            
            # 数据源优先级（包含财务报告）
            "data_source_priority": {
                "stock_basic": ["tushare", "baostock"],
                "stock_daily": ["tushare", "eastmoney"],
                "financial_reports": ["baostock"],
                "earnings_forecast": ["baostock"],
                "flash_reports": ["baostock"]
            }
        }
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as f:
            json.dump(complete_config, f)
            temp_path = f.name
        
        try:
            config = Config.from_file(temp_path)
            
            # 验证现有配置
            assert config.tushare_token == "test_token"
            assert config.cache_enabled is True
            assert config.cache_expire_hours == 48
            assert config.request_timeout == 60
            assert config.max_retries == 5
            assert config.retry_delay == 2.0
            assert config.log_level == "DEBUG"
            
            # 验证财务报告配置
            assert config.financial_reports_enabled is True
            assert config.financial_reports_cache_ttl['financial_reports'] == 7200
            assert config.financial_reports_cache_ttl['earnings_forecast'] == 1800
            assert config.financial_reports_cache_ttl['flash_reports'] == 900
            assert config.financial_reports_batch_size == 25
            assert config.financial_reports_timeout == 45
            assert config.financial_reports_retry_attempts == 4
            assert config.financial_reports_retry_delay == 1.5
            assert config.financial_reports_rate_limit == 3.0
            assert config.financial_reports_queue_size == 2000
            assert config.financial_reports_enable_metrics is False
            assert config.financial_reports_baostock_endpoint == "http://test.baostock.com"
            assert config.financial_reports_auth_required is True
            assert config.financial_reports_auth_token == "test_fr_token"
            
            # 验证数据源优先级
            assert config.data_source_priority['stock_basic'] == ["tushare", "baostock"]
            assert config.data_source_priority['financial_reports'] == ["baostock"]
            assert config.data_source_priority['earnings_forecast'] == ["baostock"]
            assert config.data_source_priority['flash_reports'] == ["baostock"]
            
        finally:
            os.unlink(temp_path)
    
    def test_config_update_integration(self):
        """测试配置更新的集成"""
        config = Config()
        
        # 同时更新现有配置和财务报告配置
        config.update(
            cache_expire_hours=36,
            request_timeout=45,
            financial_reports_batch_size=75,
            financial_reports_timeout=60,
            financial_reports_rate_limit=2.5
        )
        
        # 验证所有配置都已更新
        assert config.cache_expire_hours == 36
        assert config.request_timeout == 45
        assert config.financial_reports_batch_size == 75
        assert config.financial_reports_timeout == 60
        assert config.financial_reports_rate_limit == 2.5
    
    def test_config_to_dict_integration(self):
        """测试配置转换为字典的集成"""
        config = Config()
        config.set_financial_reports_batch_size(100)
        config.set_financial_reports_auth("integration_token", True)
        
        config_dict = config.to_dict()
        
        # 验证字典包含所有配置
        assert 'cache_enabled' in config_dict
        assert 'financial_reports_enabled' in config_dict
        assert 'financial_reports_batch_size' in config_dict
        assert 'financial_reports_cache_ttl' in config_dict
        assert 'financial_reports_auth_token' in config_dict
        
        # 验证值正确
        assert config_dict['financial_reports_batch_size'] == 100
        assert config_dict['financial_reports_auth_token'] == "integration_token"
        assert config_dict['financial_reports_auth_required'] is True
    
    def test_config_save_and_load_cycle(self):
        """测试完整的保存和加载周期"""
        # 创建配置并设置各种值
        original_config = Config()
        original_config.update(
            cache_expire_hours=72,
            log_level="WARNING"
        )
        original_config.set_financial_reports_batch_size(80)
        original_config.set_financial_reports_rate_limit(4.5)
        original_config.set_financial_reports_cache_ttl('financial_reports', 10800)
        original_config.set_financial_reports_auth("cycle_token", True)
        
        # 保存到YAML文件
        with tempfile.NamedTemporaryFile(mode='w', suffix='.yaml', delete=False) as f:
            yaml_path = f.name
        
        # 保存到JSON文件
        with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as f:
            json_path = f.name
        
        try:
            # 保存配置
            original_config.to_file(yaml_path, 'yaml')
            original_config.to_file(json_path, 'json')
            
            # 从YAML加载
            yaml_config = Config.from_file(yaml_path)
            
            # 从JSON加载
            json_config = Config.from_file(json_path)
            
            # 验证两个加载的配置都与原始配置相同
            for loaded_config in [yaml_config, json_config]:
                assert loaded_config.cache_expire_hours == 72
                assert loaded_config.log_level == "WARNING"
                assert loaded_config.financial_reports_batch_size == 80
                assert loaded_config.financial_reports_rate_limit == 4.5
                assert loaded_config.financial_reports_cache_ttl['financial_reports'] == 10800
                assert loaded_config.financial_reports_auth_token == "cycle_token"
                assert loaded_config.financial_reports_auth_required is True
                
        finally:
            os.unlink(yaml_path)
            os.unlink(json_path)
    
    def test_config_validation_integration(self):
        """测试配置验证的集成"""
        # 测试同时违反现有配置和财务报告配置的验证
        with pytest.raises(ValidationError):
            Config(
                cache_expire_hours=-1,  # 违反现有验证
                financial_reports_batch_size=0  # 违反财务报告验证
            )
        
        # 测试只违反财务报告配置的验证
        with pytest.raises(ValidationError, match="financial_reports_rate_limit必须大于0"):
            Config(financial_reports_rate_limit=-1)
        
        # 测试只违反现有配置的验证
        with pytest.raises(ValidationError, match="max_retries不能小于0"):
            Config(max_retries=-1)
    
    def test_data_source_priority_integration(self):
        """测试数据源优先级的集成"""
        config = Config()
        
        # 验证默认数据源优先级包含财务报告类型
        assert config.get_data_source_priority('financial_reports') == ['baostock']
        assert config.get_data_source_priority('earnings_forecast') == ['baostock']
        assert config.get_data_source_priority('flash_reports') == ['baostock']
        
        # 验证现有数据源优先级仍然工作
        assert 'tushare' in config.get_data_source_priority('stock_basic')
        assert 'baostock' in config.get_data_source_priority('stock_basic')
        
        # 测试设置财务报告数据源优先级
        config.set_data_source_priority('financial_reports', ['baostock', 'custom'])
        assert config.get_data_source_priority('financial_reports') == ['baostock', 'custom']
    
    def test_config_methods_integration(self):
        """测试配置方法的集成"""
        config = Config()
        
        # 测试获取代码转换配置（现有功能）
        code_config = config.get_code_conversion_config()
        assert isinstance(code_config, dict)
        assert 'enable_auto_code_conversion' in code_config
        
        # 测试获取财务报告配置（新功能）
        fr_config = config.get_financial_reports_config()
        assert isinstance(fr_config, dict)
        assert 'financial_reports_enabled' in fr_config
        
        # 验证两个配置方法返回的都是独立的字典
        code_config['test_key'] = 'test_value'
        fr_config['test_key'] = 'test_value'
        
        # 原始配置不应该被影响
        code_config2 = config.get_code_conversion_config()
        fr_config2 = config.get_financial_reports_config()
        
        assert 'test_key' not in code_config2
        assert 'test_key' not in fr_config2
    
    def test_config_string_representation_integration(self):
        """测试配置字符串表示的集成"""
        config = Config()
        config.set_financial_reports_auth("test_token", True)
        
        # 测试字符串表示
        str_repr = str(config)
        assert "Config(" in str_repr
        assert "cache_enabled=True" in str_repr
        
        # 测试详细字符串表示
        repr_str = repr(config)
        assert "Config(" in repr_str
        
        # 验证敏感信息被隐藏
        assert "test_token" not in str_repr
        assert "test_token" not in repr_str