"""
数据源集成测试

测试DataManager和DataSourceManager对涨停统计请求的支持
"""

import pytest
import asyncio
import pandas as pd
from unittest.mock import Mock, AsyncMock, patch
from datetime import datetime, timedelta

from quickstock.core.data_manager import DataManager
from quickstock.providers.manager import DataSourceManager
from quickstock.models import LimitUpStatsRequest, DataRequest, StockDailyData
from quickstock.config import Config


class TestDataSourceIntegration:
    """数据源集成测试类"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置"""
        config = Mock(spec=Config)
        config.cache_enabled = True
        config.cache_expire_hours = 24
        config.max_concurrent_requests = 5
        config.max_retries = 3
        config.enable_baostock = True
        config.enable_eastmoney = True
        config.enable_tonghuashun = False
        config.tushare_token = None
        
        # 模拟数据源优先级配置
        config.get_data_source_priority = Mock(return_value=['baostock', 'eastmoney'])
        
        return config
    
    @pytest.fixture
    def mock_provider(self):
        """模拟数据提供者"""
        provider = Mock()
        provider.health_check = AsyncMock(return_value=True)
        
        # 模拟股票日线数据
        stock_data = pd.DataFrame({
            'ts_code': ['000001.SZ', '000002.SZ', '600000.SH'],
            'trade_date': ['20241015', '20241015', '20241015'],
            'open': [10.0, 20.0, 8.0],
            'high': [11.0, 22.0, 8.8],
            'low': [9.8, 19.8, 7.9],
            'close': [11.0, 22.0, 8.8],
            'pre_close': [10.0, 20.0, 8.0],
            'change': [1.0, 2.0, 0.8],
            'pct_chg': [10.0, 10.0, 10.0],
            'vol': [1000, 2000, 1500],
            'amount': [11000.0, 44000.0, 13200.0]
        })
        
        # 模拟股票基础信息
        basic_data = pd.DataFrame({
            'ts_code': ['000001.SZ', '000002.SZ', '600000.SH'],
            'name': ['平安银行', '万科A', '浦发银行']
        })
        
        provider.get_stock_daily = AsyncMock(return_value=stock_data)
        provider.get_stock_basic = AsyncMock(return_value=basic_data)
        
        return provider
    
    @pytest.fixture
    def data_source_manager(self, mock_config, mock_provider):
        """数据源管理器"""
        manager = DataSourceManager(mock_config)
        manager.register_provider('test_provider', mock_provider)
        return manager
    
    @pytest.fixture
    def data_manager(self, mock_config, data_source_manager):
        """数据管理器"""
        with patch('quickstock.core.data_manager.CacheLayer'), \
             patch('quickstock.core.data_manager.DataFormatter'), \
             patch('quickstock.core.data_manager.ErrorHandler'), \
             patch('quickstock.core.data_manager.RequestScheduler'):
            
            manager = DataManager(mock_config)
            manager.source_manager = data_source_manager
            
            # 模拟缓存层
            manager.cache_layer.get = AsyncMock(return_value=None)
            manager.cache_layer.set = AsyncMock()
            
            # 模拟格式化器
            manager.formatter.format_data = AsyncMock(side_effect=lambda data, data_type: data)
            
            # 模拟错误处理器
            manager.error_handler.handle_with_retry = AsyncMock(
                side_effect=lambda func, *args: func(*args)
            )
            
            return manager
    
    @pytest.mark.asyncio
    async def test_limit_up_stats_request_handling(self, data_manager):
        """测试涨停统计请求处理"""
        # 创建涨停统计请求
        request = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=True,
            force_refresh=False
        )
        
        # 模拟LimitUpStatsService
        with patch('quickstock.core.data_manager.LimitUpStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟统计结果
            from quickstock.models import LimitUpStats
            mock_stats = LimitUpStats(
                trade_date='20241015',
                total=3,
                non_st=3,
                shanghai=1,
                shenzhen=2,
                star=0,
                beijing=0,
                st=0,
                limit_up_stocks=['000001.SZ', '000002.SZ', '600000.SH'],
                market_breakdown={
                    'shanghai': ['600000.SH'],
                    'shenzhen': ['000001.SZ', '000002.SZ'],
                    'star': [],
                    'beijing': []
                }
            )
            
            mock_service.get_daily_limit_up_stats = AsyncMock(return_value=mock_stats)
            
            # 执行请求
            result = await data_manager.get_data(request)
            
            # 验证结果
            assert isinstance(result, pd.DataFrame)
            assert not result.empty
            assert result.iloc[0]['trade_date'] == '20241015'
            assert result.iloc[0]['total'] == 3
            assert result.iloc[0]['shanghai'] == 1
            assert result.iloc[0]['shenzhen'] == 2
            
            # 验证服务被正确调用
            mock_service_class.assert_called_once()
            mock_service.get_daily_limit_up_stats.assert_called_once_with(request)
    
    @pytest.mark.asyncio
    async def test_limit_up_stats_with_fallback(self, data_manager):
        """测试涨停统计的fallback机制"""
        request = LimitUpStatsRequest(
            trade_date='20241015',
            force_refresh=True
        )
        
        # 模拟第一次调用失败，第二次成功
        call_count = 0
        
        async def mock_handle_request(req):
            nonlocal call_count
            call_count += 1
            if call_count == 1:
                raise Exception("第一次调用失败")
            
            # 第二次调用成功
            return pd.DataFrame([{
                'trade_date': '20241015',
                'total': 2,
                'non_st': 2,
                'shanghai': 1,
                'shenzhen': 1,
                'star': 0,
                'beijing': 0,
                'st': 0
            }])
        
        data_manager._handle_limit_up_stats_request = mock_handle_request
        
        # 执行请求
        result = await data_manager.get_limit_up_stats_with_fallback(request)
        
        # 验证结果
        assert isinstance(result, pd.DataFrame)
        assert not result.empty
        assert result.iloc[0]['total'] == 2
        
        # 验证重试机制被调用
        assert call_count == 2
    
    @pytest.mark.asyncio
    async def test_stock_data_coordination(self, data_manager):
        """测试股票数据协调获取"""
        trade_date = '20241015'
        
        # 执行协调获取
        result = await data_manager.get_stock_data_with_coordination(trade_date)
        
        # 验证结果
        assert isinstance(result, pd.DataFrame)
        assert not result.empty
        assert len(result) == 3
        assert 'name' in result.columns
        assert all(result['trade_date'] == trade_date)
        
        # 验证数据完整性
        required_columns = ['ts_code', 'trade_date', 'open', 'high', 'low', 'close', 'name']
        for col in required_columns:
            assert col in result.columns
    
    @pytest.mark.asyncio
    async def test_data_source_health_monitoring(self, data_source_manager):
        """测试数据源健康监控"""
        # 获取健康状态
        health_status = await data_source_manager.health_check_all()
        
        # 验证结果
        assert isinstance(health_status, dict)
        assert 'test_provider' in health_status
        assert health_status['test_provider'] is True
        
        # 获取提供者统计
        stats = data_source_manager.get_provider_stats()
        assert isinstance(stats, dict)
        assert 'test_provider' in stats
    
    @pytest.mark.asyncio
    async def test_limit_up_stats_data_fetching(self, data_source_manager, mock_provider):
        """测试涨停统计数据获取"""
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 执行数据获取
        result = await data_source_manager._fetch_limit_up_stats_data(mock_provider, request)
        
        # 验证结果
        assert isinstance(result, pd.DataFrame)
        assert not result.empty
        assert len(result) == 3
        
        # 验证提供者方法被调用
        mock_provider.get_stock_daily.assert_called()
    
    @pytest.mark.asyncio
    async def test_provider_capabilities(self, data_source_manager):
        """测试提供者能力检测"""
        # 获取能力信息
        capabilities = data_source_manager.get_provider_capabilities('test_provider')
        
        # 验证结果
        assert isinstance(capabilities, dict)
        assert 'limit_up_stats' in capabilities
        assert capabilities['limit_up_stats'] is True  # 因为mock_provider支持stock_daily和stock_basic
    
    @pytest.mark.asyncio
    async def test_best_provider_selection(self, data_source_manager):
        """测试最佳提供者选择"""
        # 获取最适合涨停统计的提供者
        best_provider = data_source_manager.get_best_provider_for_limit_up_stats()
        
        # 验证结果
        assert best_provider == 'test_provider'
    
    @pytest.mark.asyncio
    async def test_circuit_breaker_mechanism(self, data_manager):
        """测试熔断器机制"""
        # 启用熔断器
        data_manager.enable_circuit_breaker(threshold=0.5, timeout=60)
        
        # 验证熔断器配置
        assert data_manager.source_manager.circuit_breaker_enabled is True
        assert data_manager.source_manager.circuit_breaker_threshold == 0.5
        assert data_manager.source_manager.circuit_breaker_timeout == 60
        
        # 禁用熔断器
        data_manager.disable_circuit_breaker()
        assert data_manager.source_manager.circuit_breaker_enabled is False
    
    @pytest.mark.asyncio
    async def test_fallback_strategy_configuration(self, data_manager):
        """测试fallback策略配置"""
        # 设置fallback策略
        data_manager.set_data_source_fallback_strategy('performance_based')
        
        # 验证策略设置
        from quickstock.providers.manager import FallbackStrategy
        assert data_manager.source_manager.fallback_strategy == FallbackStrategy.PERFORMANCE_BASED
    
    @pytest.mark.asyncio
    async def test_data_source_connectivity(self, data_manager):
        """测试数据源连通性"""
        # 测试连通性
        connectivity = await data_manager.test_data_source_connectivity()
        
        # 验证结果
        assert isinstance(connectivity, dict)
        assert 'test_provider' in connectivity
        assert connectivity['test_provider'] is True
    
    @pytest.mark.asyncio
    async def test_provider_optimization(self, data_source_manager):
        """测试提供者优化配置"""
        # 添加优化配置
        optimization_config = {
            'weight': 2.0,
            'batch_size': 50,
            'timeout': 30
        }
        
        data_source_manager.add_provider_optimization('test_provider', optimization_config)
        
        # 验证配置应用
        stats = data_source_manager.get_provider_stats('test_provider')
        assert stats['test_provider'].weight == 2.0
    
    @pytest.mark.asyncio
    async def test_error_handling_and_retry(self, data_manager):
        """测试错误处理和重试机制"""
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 模拟服务失败
        with patch('quickstock.core.data_manager.LimitUpStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            mock_service.get_daily_limit_up_stats = AsyncMock(
                side_effect=Exception("服务暂时不可用")
            )
            
            # 执行请求（应该通过错误处理器重试）
            result = await data_manager.get_data(request)
            
            # 验证返回空DataFrame（因为所有重试都失败）
            assert isinstance(result, pd.DataFrame)
            assert result.empty
    
    def test_request_validation(self):
        """测试请求验证"""
        # 测试有效请求
        valid_request = LimitUpStatsRequest(trade_date='20241015')
        assert valid_request.validate() is True
        
        # 测试无效日期格式
        with pytest.raises(ValueError):
            invalid_request = LimitUpStatsRequest(trade_date='invalid_date')
            invalid_request.validate()
        
        # 测试未来日期
        future_date = (datetime.now() + timedelta(days=1)).strftime('%Y%m%d')
        with pytest.raises(ValueError):
            future_request = LimitUpStatsRequest(trade_date=future_date)
            future_request.validate()
    
    @pytest.mark.asyncio
    async def test_concurrent_requests(self, data_manager):
        """测试并发请求处理"""
        # 创建多个请求
        requests = [
            LimitUpStatsRequest(trade_date='20241015'),
            LimitUpStatsRequest(trade_date='20241014'),
            LimitUpStatsRequest(trade_date='20241013')
        ]
        
        # 模拟服务
        with patch('quickstock.core.data_manager.LimitUpStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            async def mock_get_stats(req):
                await asyncio.sleep(0.1)  # 模拟处理时间
                from quickstock.models import LimitUpStats
                return LimitUpStats(
                    trade_date=req.trade_date,
                    total=1, non_st=1, shanghai=1, shenzhen=0,
                    star=0, beijing=0, st=0,
                    limit_up_stocks=['600000.SH'],
                    market_breakdown={'shanghai': ['600000.SH']}
                )
            
            mock_service.get_daily_limit_up_stats = mock_get_stats
            
            # 并发执行请求
            tasks = [data_manager.get_data(req) for req in requests]
            results = await asyncio.gather(*tasks)
            
            # 验证结果
            assert len(results) == 3
            for result in results:
                assert isinstance(result, pd.DataFrame)
                assert not result.empty
    
    @pytest.mark.asyncio
    async def test_cache_integration(self, data_manager):
        """测试缓存集成"""
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 第一次请求（缓存未命中）
        data_manager.cache_layer.get.return_value = None
        
        with patch('quickstock.core.data_manager.LimitUpStatsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            from quickstock.models import LimitUpStats
            mock_stats = LimitUpStats(
                trade_date='20241015', total=1, non_st=1,
                shanghai=1, shenzhen=0, star=0, beijing=0, st=0,
                limit_up_stocks=['600000.SH'],
                market_breakdown={'shanghai': ['600000.SH']}
            )
            mock_service.get_daily_limit_up_stats = AsyncMock(return_value=mock_stats)
            
            result1 = await data_manager.get_data(request)
            
            # 验证缓存被设置
            data_manager.cache_layer.set.assert_called()
            
            # 第二次请求（缓存命中）
            cached_data = pd.DataFrame([{
                'trade_date': '20241015',
                'total': 1,
                'cached': True
            }])
            data_manager.cache_layer.get.return_value = cached_data
            
            result2 = await data_manager.get_data(request)
            
            # 验证返回缓存数据
            assert 'cached' in result2.columns
            assert result2.iloc[0]['cached'] is True


if __name__ == '__main__':
    pytest.main([__file__, '-v'])