"""
财务数据错误处理集成测试

测试财务数据错误处理系统与服务层的集成
"""

import pytest
import pandas as pd
from datetime import datetime, timedelta
from unittest.mock import Mock, AsyncMock, patch
import asyncio

import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from quickstock.services.financial_reports_service import FinancialReportsService
from quickstock.models import (
    FinancialReportsRequest, EarningsForecastRequest, FlashReportsRequest
)
from quickstock.core.data_manager import DataManager
from quickstock.core.cache import CacheLayer
from quickstock.core.errors import (
    FinancialDataError, ReportNotFoundError, ForecastDataError, 
    FlashReportError, FinancialDataValidationError, ErrorHandler,
    ValidationError, NetworkError, RateLimitError
)


class TestFinancialErrorIntegration:
    """测试财务数据错误处理集成"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """创建模拟数据管理器"""
        data_manager = Mock(spec=DataManager)
        data_manager.get_data = AsyncMock()
        return data_manager
    
    @pytest.fixture
    def mock_cache_layer(self):
        """创建模拟缓存层"""
        cache_layer = Mock(spec=CacheLayer)
        cache_layer.get = AsyncMock(return_value=None)
        cache_layer.set = AsyncMock()
        return cache_layer
    
    @pytest.fixture
    def service(self, mock_data_manager, mock_cache_layer):
        """创建财务报告服务"""
        return FinancialReportsService(mock_data_manager, mock_cache_layer)
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置对象"""
        class MockConfig:
            max_retries = 3
            retry_delay = 1.0
            log_level = 'INFO'
            log_file = None
        
        return MockConfig()
    
    @pytest.fixture
    def error_handler(self, mock_config):
        """创建错误处理器"""
        return ErrorHandler(mock_config)
    
    @pytest.mark.asyncio
    async def test_financial_reports_service_report_not_found(self, service, mock_data_manager):
        """测试财务报告服务处理报告未找到错误"""
        # 设置模拟数据管理器返回空数据
        mock_data_manager.get_data.return_value = pd.DataFrame()
        
        request = FinancialReportsRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        with pytest.raises(ReportNotFoundError) as exc_info:
            await service.get_financial_reports(request)
        
        error = exc_info.value
        assert error.ts_code == "000001.SZ"
        assert error.error_code == "FINANCIAL_REPORT_NOT_FOUND"
        assert error.suggestions is not None
    
    @pytest.mark.asyncio
    async def test_earnings_forecast_service_forecast_error(self, service, mock_data_manager):
        """测试业绩预告服务处理预告数据错误"""
        # 设置模拟数据管理器返回空数据
        mock_data_manager.get_data.return_value = pd.DataFrame()
        
        request = EarningsForecastRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        with pytest.raises(ForecastDataError) as exc_info:
            await service.get_earnings_forecast(request)
        
        error = exc_info.value
        assert error.ts_code == "000001.SZ"
        assert error.error_code == "FORECAST_DATA_ERROR"
        assert "未找到业绩预告数据" in str(error)
    
    @pytest.mark.asyncio
    async def test_flash_reports_service_flash_error(self, service, mock_data_manager):
        """测试业绩快报服务处理快报错误"""
        # 设置模拟数据管理器返回空数据
        mock_data_manager.get_data.return_value = pd.DataFrame()
        
        request = FlashReportsRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        with pytest.raises(FlashReportError) as exc_info:
            await service.get_earnings_flash_reports(request)
        
        error = exc_info.value
        assert error.ts_code == "000001.SZ"
        assert error.error_code == "FLASH_REPORT_ERROR"
        assert "未找到业绩快报数据" in str(error)
    
    @pytest.mark.asyncio
    async def test_service_data_validation_error(self, service, mock_data_manager):
        """测试服务层数据验证错误"""
        # 创建有问题的数据
        problematic_data = pd.DataFrame({
            'ts_code': ['000001.SZ'],
            'report_date': ['20231231'],
            'report_type': ['A'],
            'total_revenue': ['invalid_number'],  # 无效数值
            'net_profit': [None],  # 空值
            'total_assets': [-1000],  # 负数
            'total_liabilities': [0],
            'shareholders_equity': [0],
            'operating_cash_flow': [0],
            'eps': [0],
            'roe': [0]
        })
        
        mock_data_manager.get_data.return_value = problematic_data
        
        request = FinancialReportsRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        # 应该抛出ReportNotFoundError（因为处理失败导致没有有效数据）
        with pytest.raises(ReportNotFoundError):
            await service.get_financial_reports(request)
    
    @pytest.mark.asyncio
    async def test_service_network_error_retry(self, service, mock_data_manager):
        """测试服务层网络错误重试机制"""
        # 设置模拟数据管理器前两次抛出网络错误，第三次成功
        valid_data = pd.DataFrame({
            'ts_code': ['000001.SZ'],
            'report_date': ['20231231'],
            'report_type': ['A'],
            'total_revenue': [1000000],
            'net_profit': [100000],
            'total_assets': [5000000],
            'total_liabilities': [3000000],
            'shareholders_equity': [2000000],
            'operating_cash_flow': [150000],
            'eps': [0.5],
            'roe': [0.05]
        })
        
        mock_data_manager.get_data.side_effect = [
            NetworkError("网络连接失败"),
            NetworkError("网络超时"),
            valid_data
        ]
        
        request = FinancialReportsRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        # 应该在重试后成功
        result = await service.get_financial_reports(request)
        assert len(result) == 1
        assert result[0].ts_code == "000001.SZ"
        
        # 检查重试统计
        stats = service.get_service_stats()
        assert stats['retry_attempts'] > 0
    
    @pytest.mark.asyncio
    async def test_service_rate_limit_error_retry(self, service, mock_data_manager):
        """测试服务层速率限制错误重试机制"""
        # 设置模拟数据管理器前一次抛出速率限制错误，第二次成功
        valid_data = pd.DataFrame({
            'ts_code': ['000001.SZ'],
            'forecast_date': ['20231215'],
            'forecast_period': ['20231231'],  # 修正为YYYYMMDD格式
            'forecast_type': ['预增'],
            'net_profit_min': [80000],
            'net_profit_max': [120000],
            'growth_rate_min': [20.0],
            'growth_rate_max': [50.0],
            'forecast_summary': ['业绩预增']
        })
        
        mock_data_manager.get_data.side_effect = [
            RateLimitError("API调用频率超限", details={'retry_after': 1}),
            valid_data
        ]
        
        request = EarningsForecastRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        # 应该在重试后成功
        result = await service.get_earnings_forecast(request)
        assert len(result) == 1
        assert result[0].ts_code == "000001.SZ"
    
    @pytest.mark.asyncio
    async def test_batch_processing_partial_failures(self, service, mock_data_manager):
        """测试批量处理中的部分失败"""
        # 设置不同股票返回不同结果
        def mock_get_data(request):
            if request.ts_code == "000001.SZ":
                return pd.DataFrame({
                    'ts_code': ['000001.SZ'],
                    'report_date': ['20231231'],
                    'report_type': ['A'],
                    'total_revenue': [1000000],
                    'net_profit': [100000],
                    'total_assets': [5000000],
                    'total_liabilities': [3000000],
                    'shareholders_equity': [2000000],
                    'operating_cash_flow': [150000],
                    'eps': [0.5],
                    'roe': [0.05]
                })
            elif request.ts_code == "000002.SZ":
                return pd.DataFrame()  # 空数据，会导致ReportNotFoundError
            else:
                raise NetworkError("网络错误")
        
        mock_data_manager.get_data.side_effect = mock_get_data
        
        stock_codes = ["000001.SZ", "000002.SZ", "000003.SZ"]
        
        result = await service.get_batch_financial_data(
            stock_codes=stock_codes,
            data_types=['financial_reports'],
            start_date="20231201",
            end_date="20231231"
        )
        
        # 检查结果 - 批处理会将错误包装在结果中，而不是完全失败
        assert result['total_count'] == 3
        
        # 检查是否有成功的数据（000001.SZ应该成功）
        if result['success_count'] > 0:
            assert "000001.SZ" in result['data']
            assert 'financial_reports' in result['data']["000001.SZ"]
        
        # 检查是否有失败的股票
        if result['failed_count'] > 0:
            assert len(result['failed_stocks']) > 0
    
    @pytest.mark.asyncio
    async def test_error_handler_financial_error_recovery(self, error_handler):
        """测试错误处理器的财务错误恢复功能"""
        # 测试报告未找到错误恢复
        report_error = ReportNotFoundError("000001.SZ", "20231231", "A")
        context = {"start_date": "20231201", "end_date": "20231231"}
        
        result = await error_handler.handle_financial_data_error(report_error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['report_date'] == "20231231"
        assert result['report_type'] == "A"
        assert result['data'] == []
        assert 'message' in result
    
    @pytest.mark.asyncio
    async def test_error_handler_forecast_error_recovery(self, error_handler):
        """测试错误处理器的预告错误恢复功能"""
        forecast_error = ForecastDataError("000001.SZ", "数据不可用")
        context = {"forecast_type": "annual"}
        
        result = await error_handler.handle_financial_data_error(forecast_error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['forecasts'] == []
        assert 'message' in result
        assert '暂无业绩预告数据' in result['message']
    
    @pytest.mark.asyncio
    async def test_error_handler_flash_report_error_recovery(self, error_handler):
        """测试错误处理器的快报错误恢复功能"""
        flash_error = FlashReportError("000001.SZ", "解析失败")
        context = {"report_period": "Q4"}
        
        result = await error_handler.handle_financial_data_error(flash_error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['flash_reports'] == []
        assert 'message' in result
        assert '暂无业绩快报数据' in result['message']
    
    @pytest.mark.asyncio
    async def test_error_handler_validation_error_no_recovery(self, error_handler):
        """测试错误处理器对验证错误不进行恢复"""
        validation_error = FinancialDataValidationError("eps", "invalid", "not a number")
        context = {"validation_strict": False}
        
        result = await error_handler.handle_financial_data_error(validation_error, context)
        
        # 验证错误通常不能自动恢复
        assert result is None
    
    def test_error_retry_logic_financial_errors(self, error_handler):
        """测试财务错误的重试逻辑"""
        # 验证错误不应该重试
        validation_error = FinancialDataValidationError("field", "value")
        assert not error_handler.should_retry(validation_error)
        
        # 报告未找到错误默认不重试
        report_error = ReportNotFoundError("000001.SZ", "20231231")
        assert not error_handler.should_retry(report_error)
        
        # 带临时标记的报告错误应该重试
        temp_report_error = ReportNotFoundError("000001.SZ", "20231231")
        temp_report_error.details['temporary'] = True
        assert error_handler.should_retry(temp_report_error)
        
        # 预告数据错误应该重试
        forecast_error = ForecastDataError("000001.SZ", "timeout")
        assert error_handler.should_retry(forecast_error)
        
        # 快报错误应该重试
        flash_error = FlashReportError("000001.SZ", "server error")
        assert error_handler.should_retry(flash_error)
    
    @pytest.mark.asyncio
    async def test_service_error_statistics_tracking(self, service, mock_data_manager):
        """测试服务错误统计跟踪"""
        initial_stats = service.get_service_stats()
        assert initial_stats['failed_requests'] == 0
        
        # 模拟一个失败的请求
        mock_data_manager.get_data.side_effect = ValidationError("参数错误")
        
        request = FinancialReportsRequest(
            ts_code="000001.SZ",
            start_date="20231201",
            end_date="20231231"
        )
        
        with pytest.raises((ValidationError, FinancialDataError, AttributeError)):
            await service.get_financial_reports(request)
        
        # 检查统计信息
        updated_stats = service.get_service_stats()
        assert updated_stats['failed_requests'] > initial_stats['failed_requests']
        assert updated_stats['total_requests'] > initial_stats['total_requests']
    
    def test_financial_error_hierarchy_and_inheritance(self):
        """测试财务错误层次结构和继承关系"""
        # 创建各种错误实例
        base_error = FinancialDataError("基础错误")
        report_error = ReportNotFoundError("000001.SZ")
        forecast_error = ForecastDataError("000001.SZ")
        flash_error = FlashReportError("000001.SZ")
        validation_error = FinancialDataValidationError("field", "value")
        
        # 检查继承关系
        assert isinstance(report_error, FinancialDataError)
        assert isinstance(forecast_error, FinancialDataError)
        assert isinstance(flash_error, FinancialDataError)
        assert isinstance(validation_error, FinancialDataError)
        
        # 检查ValidationError的多重继承
        from quickstock.core.errors import ValidationError
        assert isinstance(validation_error, ValidationError)
        
        # 检查所有错误都继承自Exception
        errors = [base_error, report_error, forecast_error, flash_error, validation_error]
        for error in errors:
            assert isinstance(error, Exception)
    
    def test_financial_error_serialization_and_details(self):
        """测试财务错误序列化和详细信息"""
        # 测试带详细信息的错误
        error = ReportNotFoundError("000001.SZ", "20231231", "A")
        
        # 检查错误可以被字符串化
        error_str = str(error)
        assert isinstance(error_str, str)
        assert "000001.SZ" in error_str
        assert "20231231" in error_str
        assert "A" in error_str
        
        # 检查详细信息
        assert hasattr(error, 'details')
        assert isinstance(error.details, dict)
        assert error.details['ts_code'] == "000001.SZ"
        assert error.details['report_date'] == "20231231"
        assert error.details['report_type'] == "A"
        
        # 检查建议信息
        assert hasattr(error, 'suggestions')
        assert error.suggestions is not None
        assert isinstance(error.suggestions, str)
        assert len(error.suggestions) > 0
    
    @pytest.mark.asyncio
    async def test_concurrent_error_handling(self, service, mock_data_manager):
        """测试并发错误处理"""
        # 设置不同的错误场景
        def mock_get_data(request):
            if request.ts_code == "000001.SZ":
                raise NetworkError("网络错误")
            elif request.ts_code == "000002.SZ":
                raise RateLimitError("速率限制")
            else:
                return pd.DataFrame()  # 空数据
        
        mock_data_manager.get_data.side_effect = mock_get_data
        
        # 创建多个并发请求
        requests = [
            FinancialReportsRequest(ts_code="000001.SZ", start_date="20231201", end_date="20231231"),
            FinancialReportsRequest(ts_code="000002.SZ", start_date="20231201", end_date="20231231"),
            FinancialReportsRequest(ts_code="000003.SZ", start_date="20231201", end_date="20231231")
        ]
        
        # 并发执行请求
        tasks = [service.get_financial_reports(req) for req in requests]
        results = await asyncio.gather(*tasks, return_exceptions=True)
        
        # 检查结果
        assert len(results) == 3
        
        # 所有请求都应该失败，但失败类型不同
        for result in results:
            assert isinstance(result, Exception)
        
        # 检查具体的错误类型 - 服务层会将原始错误包装为FinancialDataError
        assert isinstance(results[0], (NetworkError, FinancialDataError, AttributeError))
        assert isinstance(results[1], (RateLimitError, FinancialDataError, AttributeError))
        assert isinstance(results[2], (ReportNotFoundError, AttributeError))


class TestFinancialErrorLoggingIntegration:
    """测试财务错误日志集成"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置对象"""
        class MockConfig:
            max_retries = 3
            retry_delay = 1.0
            log_level = 'INFO'
            log_file = None
        
        return MockConfig()
    
    @pytest.fixture
    def error_handler(self, mock_config):
        """创建错误处理器"""
        return ErrorHandler(mock_config)
    
    def test_financial_error_logging(self, error_handler, caplog):
        """测试财务错误日志记录"""
        import logging
        
        # 测试不同类型的财务错误日志
        errors_and_contexts = [
            (ReportNotFoundError("000001.SZ", "20231231"), {"operation": "get_reports"}),
            (ForecastDataError("000001.SZ", "timeout"), {"operation": "get_forecast"}),
            (FlashReportError("000001.SZ", "parsing failed"), {"operation": "get_flash"}),
            (FinancialDataValidationError("eps", "invalid"), {"operation": "validate_data"})
        ]
        
        with caplog.at_level(logging.WARNING):
            for error, context in errors_and_contexts:
                error_handler.log_error(error, context)
        
        # 检查日志记录
        assert len(caplog.records) >= len(errors_and_contexts)
        
        # 检查日志内容
        log_messages = [record.message for record in caplog.records]
        assert any("ReportNotFoundError" in msg for msg in log_messages)
        assert any("ForecastDataError" in msg for msg in log_messages)
        assert any("FlashReportError" in msg for msg in log_messages)
        assert any("FinancialDataValidationError" in msg for msg in log_messages)


if __name__ == "__main__":
    pytest.main([__file__])