"""
财务数据异常测试

测试财务数据相关的异常类和错误处理
"""

import pytest
from quickstock.core.errors import (
    FinancialDataError, ReportNotFoundError, ForecastDataError, 
    FlashReportError, FinancialDataValidationError
)


class TestFinancialDataError:
    """财务数据异常基类测试"""
    
    def test_financial_data_error_creation(self):
        """测试创建财务数据异常"""
        error = FinancialDataError(
            message="财务数据异常",
            error_code="FINANCIAL_ERROR",
            details={"field": "value"},
            suggestions="请检查数据"
        )
        
        assert str(error) == "财务数据异常"
        assert error.error_code == "FINANCIAL_ERROR"
        assert error.details == {"field": "value"}
        assert error.suggestions == "请检查数据"
    
    def test_financial_data_error_inheritance(self):
        """测试财务数据异常继承关系"""
        error = FinancialDataError("测试异常")
        
        # 应该继承自DataSourceError
        from quickstock.core.errors import DataSourceError
        assert isinstance(error, DataSourceError)
        
        # 也应该继承自QuickStockError
        from quickstock.core.errors import QuickStockError
        assert isinstance(error, QuickStockError)


class TestReportNotFoundError:
    """财务报告未找到异常测试"""
    
    def test_report_not_found_error_basic(self):
        """测试基本的报告未找到异常"""
        error = ReportNotFoundError(ts_code="000001.SZ")
        
        assert "000001.SZ" in str(error)
        assert error.error_code == "FINANCIAL_REPORT_NOT_FOUND"
        assert error.ts_code == "000001.SZ"
        assert error.report_date is None
        assert error.report_type is None
        assert error.suggestions is not None
    
    def test_report_not_found_error_with_date(self):
        """测试带日期的报告未找到异常"""
        error = ReportNotFoundError(
            ts_code="000001.SZ",
            report_date="20231231"
        )
        
        assert "000001.SZ" in str(error)
        assert "20231231" in str(error)
        assert error.report_date == "20231231"
    
    def test_report_not_found_error_with_date_and_type(self):
        """测试带日期和类型的报告未找到异常"""
        error = ReportNotFoundError(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A"
        )
        
        assert "000001.SZ" in str(error)
        assert "20231231" in str(error)
        assert "A" in str(error)
        assert error.report_type == "A"
    
    def test_report_not_found_error_details(self):
        """测试报告未找到异常详细信息"""
        error = ReportNotFoundError(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A"
        )
        
        assert error.details["ts_code"] == "000001.SZ"
        assert error.details["report_date"] == "20231231"
        assert error.details["report_type"] == "A"


class TestForecastDataError:
    """业绩预告数据异常测试"""
    
    def test_forecast_data_error_basic(self):
        """测试基本的预告数据异常"""
        error = ForecastDataError(ts_code="000001.SZ")
        
        assert "000001.SZ" in str(error)
        assert error.error_code == "FORECAST_DATA_ERROR"
        assert error.ts_code == "000001.SZ"
        assert error.reason is None
        assert error.forecast_date is None
        assert error.suggestions is not None
    
    def test_forecast_data_error_with_reason(self):
        """测试带原因的预告数据异常"""
        error = ForecastDataError(
            ts_code="000001.SZ",
            reason="数据格式错误"
        )
        
        assert "000001.SZ" in str(error)
        assert "数据格式错误" in str(error)
        assert error.reason == "数据格式错误"
    
    def test_forecast_data_error_with_date(self):
        """测试带日期的预告数据异常"""
        error = ForecastDataError(
            ts_code="000001.SZ",
            reason="数据不完整",
            forecast_date="20231215"
        )
        
        assert "000001.SZ" in str(error)
        assert "数据不完整" in str(error)
        assert "20231215" in str(error)
        assert error.forecast_date == "20231215"


class TestFlashReportError:
    """业绩快报异常测试"""
    
    def test_flash_report_error_basic(self):
        """测试基本的快报异常"""
        error = FlashReportError(ts_code="000001.SZ")
        
        assert "000001.SZ" in str(error)
        assert error.error_code == "FLASH_REPORT_ERROR"
        assert error.ts_code == "000001.SZ"
        assert error.reason is None
        assert error.report_date is None
        assert error.suggestions is not None
    
    def test_flash_report_error_with_reason(self):
        """测试带原因的快报异常"""
        error = FlashReportError(
            ts_code="000001.SZ",
            reason="快报数据缺失"
        )
        
        assert "000001.SZ" in str(error)
        assert "快报数据缺失" in str(error)
        assert error.reason == "快报数据缺失"
    
    def test_flash_report_error_with_date(self):
        """测试带日期的快报异常"""
        error = FlashReportError(
            ts_code="000001.SZ",
            reason="数据解析失败",
            report_date="20240115"
        )
        
        assert "000001.SZ" in str(error)
        assert "数据解析失败" in str(error)
        assert "20240115" in str(error)
        assert error.report_date == "20240115"


class TestFinancialDataValidationError:
    """财务数据验证异常测试"""
    
    def test_financial_data_validation_error_basic(self):
        """测试基本的数据验证异常"""
        error = FinancialDataValidationError(
            field_name="total_revenue",
            field_value="invalid_value"
        )
        
        assert "total_revenue" in str(error)
        assert "invalid_value" in str(error)
        assert error.error_code == "FINANCIAL_DATA_VALIDATION_ERROR"
        assert error.field_name == "total_revenue"
        assert error.field_value == "invalid_value"
        assert error.reason is None
        assert error.suggestions is not None
    
    def test_financial_data_validation_error_with_reason(self):
        """测试带原因的数据验证异常"""
        error = FinancialDataValidationError(
            field_name="net_profit",
            field_value=-1000,
            reason="数值不能为负数"
        )
        
        assert "net_profit" in str(error)
        assert "-1000" in str(error)
        assert "数值不能为负数" in str(error)
        assert error.reason == "数值不能为负数"
    
    def test_financial_data_validation_error_inheritance(self):
        """测试数据验证异常继承关系"""
        error = FinancialDataValidationError(
            field_name="eps",
            field_value="not_a_number"
        )
        
        # 应该同时继承FinancialDataError和ValidationError
        assert isinstance(error, FinancialDataError)
        from quickstock.core.errors import ValidationError
        assert isinstance(error, ValidationError)
    
    def test_financial_data_validation_error_details(self):
        """测试数据验证异常详细信息"""
        error = FinancialDataValidationError(
            field_name="roe",
            field_value=None,
            reason="字段不能为空"
        )
        
        assert error.details["field_name"] == "roe"
        assert error.details["field_value"] is None
        assert error.details["reason"] == "字段不能为空"


class TestFinancialErrorsIntegration:
    """财务异常集成测试"""
    
    def test_error_hierarchy(self):
        """测试异常层次结构"""
        # 创建各种异常实例
        base_error = FinancialDataError("基础异常")
        report_error = ReportNotFoundError("000001.SZ")
        forecast_error = ForecastDataError("000001.SZ")
        flash_error = FlashReportError("000001.SZ")
        validation_error = FinancialDataValidationError("field", "value")
        
        # 检查继承关系
        errors = [report_error, forecast_error, flash_error, validation_error]
        for error in errors:
            assert isinstance(error, FinancialDataError)
            assert isinstance(error, Exception)
    
    def test_error_serialization(self):
        """测试异常序列化"""
        error = ReportNotFoundError(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A"
        )
        
        # 测试异常可以被字符串化
        error_str = str(error)
        assert isinstance(error_str, str)
        assert len(error_str) > 0
        
        # 测试异常详细信息
        assert hasattr(error, 'details')
        assert isinstance(error.details, dict)
        assert error.details['ts_code'] == "000001.SZ"
    
    def test_error_suggestions(self):
        """测试异常建议信息"""
        errors = [
            ReportNotFoundError("000001.SZ"),
            ForecastDataError("000001.SZ"),
            FlashReportError("000001.SZ"),
            FinancialDataValidationError("field", "value")
        ]
        
        for error in errors:
            assert hasattr(error, 'suggestions')
            assert error.suggestions is not None
            assert isinstance(error.suggestions, str)
            assert len(error.suggestions) > 0
    
    def test_error_codes(self):
        """测试异常错误代码"""
        error_code_mapping = {
            ReportNotFoundError("000001.SZ"): "FINANCIAL_REPORT_NOT_FOUND",
            ForecastDataError("000001.SZ"): "FORECAST_DATA_ERROR",
            FlashReportError("000001.SZ"): "FLASH_REPORT_ERROR",
            FinancialDataValidationError("field", "value"): "FINANCIAL_DATA_VALIDATION_ERROR"
        }
        
        for error, expected_code in error_code_mapping.items():
            assert error.error_code == expected_code


class TestFinancialErrorHandling:
    """财务数据错误处理测试"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置对象"""
        class MockConfig:
            max_retries = 3
            retry_delay = 1.0
            log_level = 'INFO'
            log_file = None
        
        return MockConfig()
    
    @pytest.fixture
    def error_handler(self, mock_config):
        """创建错误处理器"""
        from quickstock.core.errors import ErrorHandler
        return ErrorHandler(mock_config)
    
    def test_should_retry_financial_errors(self, error_handler):
        """测试财务数据错误重试逻辑"""
        # 验证错误不应该重试
        validation_error = FinancialDataValidationError("field", "value")
        assert not error_handler.should_retry(validation_error)
        
        # 报告未找到错误默认不重试
        report_error = ReportNotFoundError("000001.SZ", "20231231")
        assert not error_handler.should_retry(report_error)
        
        # 带临时标记的报告错误应该重试
        temp_report_error = ReportNotFoundError("000001.SZ", "20231231")
        temp_report_error.details['temporary'] = True
        print(f"temp_report_error.details: {temp_report_error.details}")
        print(f"should_retry result: {error_handler.should_retry(temp_report_error)}")
        assert error_handler.should_retry(temp_report_error)
        
        # 预告数据错误应该重试
        forecast_error = ForecastDataError("000001.SZ", "timeout")
        assert error_handler.should_retry(forecast_error)
        
        # 快报错误应该重试
        flash_error = FlashReportError("000001.SZ", "server error")
        assert error_handler.should_retry(flash_error)
    
    def test_should_retry_forecast_error_with_reason(self, error_handler):
        """测试带原因的预告错误重试逻辑"""
        # 包含重试关键词的错误应该重试
        retry_reasons = ['timeout', 'connection', 'server', 'temporary', 'unavailable']
        
        for reason in retry_reasons:
            error = ForecastDataError("000001.SZ", reason)
            assert error_handler.should_retry(error), f"Should retry for reason: {reason}"
        
        # 不包含重试关键词的错误也应该重试（默认行为）
        error = ForecastDataError("000001.SZ", "data format error")
        assert error_handler.should_retry(error)
    
    @pytest.mark.asyncio
    async def test_handle_financial_data_error_report_not_found(self, error_handler):
        """测试处理报告未找到错误"""
        error = ReportNotFoundError("000001.SZ", "20231231", "A")
        context = {"start_date": "20231201", "end_date": "20231231"}
        
        result = await error_handler.handle_financial_data_error(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['report_date'] == "20231231"
        assert result['report_type'] == "A"
        assert result['data'] == []
        assert 'message' in result
    
    @pytest.mark.asyncio
    async def test_handle_financial_data_error_forecast(self, error_handler):
        """测试处理预告数据错误"""
        error = ForecastDataError("000001.SZ", "no data available")
        context = {"request_type": "forecast"}
        
        result = await error_handler.handle_financial_data_error(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['forecasts'] == []
        assert 'message' in result
    
    @pytest.mark.asyncio
    async def test_handle_financial_data_error_flash_report(self, error_handler):
        """测试处理快报错误"""
        error = FlashReportError("000001.SZ", "parsing failed")
        context = {"request_type": "flash_report"}
        
        result = await error_handler.handle_financial_data_error(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['flash_reports'] == []
        assert 'message' in result
    
    @pytest.mark.asyncio
    async def test_handle_financial_data_error_validation(self, error_handler):
        """测试处理验证错误"""
        error = FinancialDataValidationError("total_revenue", "invalid", "not a number")
        context = {"field_validation": True}
        
        result = await error_handler.handle_financial_data_error(error, context)
        
        # 验证错误通常不能自动恢复
        assert result is None
    
    @pytest.mark.asyncio
    async def test_handle_financial_data_error_unknown_type(self, error_handler):
        """测试处理未知类型的财务数据错误"""
        error = FinancialDataError("unknown error", "UNKNOWN_ERROR")
        context = {"unknown": True}
        
        result = await error_handler.handle_financial_data_error(error, context)
        
        # 未知类型的错误返回None
        assert result is None


class TestFinancialErrorRecoveryStrategies:
    """财务数据错误恢复策略测试"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置对象"""
        class MockConfig:
            max_retries = 3
            retry_delay = 1.0
            log_level = 'INFO'
            log_file = None
        
        return MockConfig()
    
    @pytest.fixture
    def error_handler(self, mock_config):
        """创建错误处理器"""
        from quickstock.core.errors import ErrorHandler
        return ErrorHandler(mock_config)
    
    @pytest.mark.asyncio
    async def test_recover_from_report_not_found_basic(self, error_handler):
        """测试基本的报告未找到恢复"""
        error = ReportNotFoundError("000001.SZ", "20231231")
        context = {}
        
        result = await error_handler._recover_from_report_not_found(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['report_date'] == "20231231"
        assert isinstance(result['data'], list)
        assert len(result['data']) == 0
    
    @pytest.mark.asyncio
    async def test_recover_from_report_not_found_with_date_range(self, error_handler):
        """测试带日期范围的报告未找到恢复"""
        error = ReportNotFoundError("000001.SZ", "20231231", "A")
        context = {"start_date": "20231201", "end_date": "20231231"}
        
        result = await error_handler._recover_from_report_not_found(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['report_type'] == "A"
        assert 'message' in result
    
    @pytest.mark.asyncio
    async def test_recover_from_forecast_error(self, error_handler):
        """测试预告错误恢复"""
        error = ForecastDataError("000001.SZ", "data unavailable")
        context = {"forecast_type": "annual"}
        
        result = await error_handler._recover_from_forecast_error(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['forecasts'] == []
        assert 'message' in result
        assert '暂无业绩预告数据' in result['message']
    
    @pytest.mark.asyncio
    async def test_recover_from_flash_report_error(self, error_handler):
        """测试快报错误恢复"""
        error = FlashReportError("000001.SZ", "network timeout")
        context = {"report_period": "Q4"}
        
        result = await error_handler._recover_from_flash_report_error(error, context)
        
        assert result is not None
        assert result['ts_code'] == "000001.SZ"
        assert result['flash_reports'] == []
        assert 'message' in result
        assert '暂无业绩快报数据' in result['message']
    
    @pytest.mark.asyncio
    async def test_recover_from_validation_error(self, error_handler):
        """测试验证错误恢复"""
        error = FinancialDataValidationError("eps", "not_a_number", "invalid format")
        context = {"validation_strict": False}
        
        result = await error_handler._recover_from_validation_error(error, context)
        
        # 验证错误通常不能自动恢复
        assert result is None


class TestFinancialErrorLogging:
    """财务数据错误日志测试"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置对象"""
        class MockConfig:
            max_retries = 3
            retry_delay = 1.0
            log_level = 'INFO'
            log_file = None
        
        return MockConfig()
    
    @pytest.fixture
    def error_handler(self, mock_config):
        """创建错误处理器"""
        from quickstock.core.errors import ErrorHandler
        return ErrorHandler(mock_config)
    
    def test_log_financial_error(self, error_handler, caplog):
        """测试财务错误日志记录"""
        import logging
        
        error = ReportNotFoundError("000001.SZ", "20231231")
        context = {"operation": "get_financial_reports", "attempt": 1}
        
        with caplog.at_level(logging.WARNING):
            error_handler.log_error(error, context)
        
        # 检查日志是否包含相关信息
        assert len(caplog.records) > 0
        log_message = caplog.records[0].message
        assert "ReportNotFoundError" in log_message
        assert "000001.SZ" in log_message
    
    def test_log_financial_error_with_suggestions(self, error_handler, caplog):
        """测试带建议的财务错误日志"""
        import logging
        
        error = FinancialDataValidationError("total_revenue", -1000, "negative value")
        context = {"validation": True}
        
        with caplog.at_level(logging.WARNING):
            error_handler.log_error(error, context)
        
        assert len(caplog.records) > 0
        # 验证错误应该记录为warning级别
        assert caplog.records[0].levelno == logging.WARNING


if __name__ == "__main__":
    pytest.main([__file__])