"""
财务数据模型测试

测试财务报告、业绩预告和业绩快报数据模型的功能
"""

import pytest
from datetime import datetime
from quickstock.models import (
    FinancialReport, EarningsForecast, FlashReport,
    FinancialReportsRequest, EarningsForecastRequest, FlashReportsRequest
)
from quickstock.core.errors import ValidationError, FinancialDataValidationError


class TestFinancialReport:
    """财务报告模型测试"""
    
    def test_valid_financial_report_creation(self):
        """测试创建有效的财务报告"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=10000.0,
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=15000.0,
            eps=1.25,
            roe=5.0
        )
        
        assert report.ts_code == "000001.SZ"
        assert report.report_date == "20231231"
        assert report.report_type == "A"
        assert report.total_revenue == 100000.0
        assert report.net_profit == 10000.0
        assert report.created_at is not None
        assert report.updated_at is not None
    
    def test_financial_report_validation_invalid_stock_code(self):
        """测试无效股票代码验证"""
        with pytest.raises(ValidationError, match="Invalid stock code format"):
            FinancialReport(
                ts_code="invalid_code",
                report_date="20231231",
                report_type="A",
                total_revenue=100000.0,
                net_profit=10000.0,
                total_assets=500000.0,
                total_liabilities=300000.0,
                shareholders_equity=200000.0,
                operating_cash_flow=15000.0,
                eps=1.25,
                roe=5.0
            )
    
    def test_financial_report_validation_invalid_date(self):
        """测试无效日期验证"""
        with pytest.raises(ValidationError, match="Invalid report date format"):
            FinancialReport(
                ts_code="000001.SZ",
                report_date="2023-12-31",  # 错误格式
                report_type="A",
                total_revenue=100000.0,
                net_profit=10000.0,
                total_assets=500000.0,
                total_liabilities=300000.0,
                shareholders_equity=200000.0,
                operating_cash_flow=15000.0,
                eps=1.25,
                roe=5.0
            )
    
    def test_financial_report_validation_invalid_report_type(self):
        """测试无效报告类型验证"""
        with pytest.raises(ValidationError, match="Invalid report type"):
            FinancialReport(
                ts_code="000001.SZ",
                report_date="20231231",
                report_type="INVALID",
                total_revenue=100000.0,
                net_profit=10000.0,
                total_assets=500000.0,
                total_liabilities=300000.0,
                shareholders_equity=200000.0,
                operating_cash_flow=15000.0,
                eps=1.25,
                roe=5.0
            )
    
    def test_financial_report_validation_negative_assets(self):
        """测试负资产验证"""
        with pytest.raises(ValidationError, match="Total assets cannot be negative"):
            FinancialReport(
                ts_code="000001.SZ",
                report_date="20231231",
                report_type="A",
                total_revenue=100000.0,
                net_profit=10000.0,
                total_assets=-500000.0,  # 负数
                total_liabilities=300000.0,
                shareholders_equity=200000.0,
                operating_cash_flow=15000.0,
                eps=1.25,
                roe=5.0
            )
    
    def test_financial_report_validation_balance_sheet(self):
        """测试资产负债表平衡验证"""
        with pytest.raises(ValidationError, match="Balance sheet validation failed"):
            FinancialReport(
                ts_code="000001.SZ",
                report_date="20231231",
                report_type="A",
                total_revenue=100000.0,
                net_profit=10000.0,
                total_assets=500000.0,
                total_liabilities=300000.0,
                shareholders_equity=100000.0,  # 不平衡：500000 - 300000 != 100000
                operating_cash_flow=15000.0,
                eps=1.25,
                roe=5.0
            )
    
    def test_financial_report_to_dict(self):
        """测试转换为字典"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=10000.0,
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=15000.0,
            eps=1.25,
            roe=5.0
        )
        
        data = report.to_dict()
        assert data['ts_code'] == "000001.SZ"
        assert data['report_date'] == "20231231"
        assert data['total_revenue'] == 100000.0
        assert 'created_at' in data
        assert 'updated_at' in data
    
    def test_financial_report_from_dict(self):
        """测试从字典创建"""
        data = {
            'ts_code': "000001.SZ",
            'report_date': "20231231",
            'report_type': "A",
            'total_revenue': 100000.0,
            'net_profit': 10000.0,
            'total_assets': 500000.0,
            'total_liabilities': 300000.0,
            'shareholders_equity': 200000.0,
            'operating_cash_flow': 15000.0,
            'eps': 1.25,
            'roe': 5.0
        }
        
        report = FinancialReport.from_dict(data)
        assert report.ts_code == "000001.SZ"
        assert report.total_revenue == 100000.0
    
    def test_financial_report_formatted_date(self):
        """测试格式化日期"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=10000.0,
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=15000.0,
            eps=1.25,
            roe=5.0
        )
        
        assert report.get_formatted_date() == "2023-12-31"
    
    def test_financial_report_profit_margin(self):
        """测试净利润率计算"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=10000.0,
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=15000.0,
            eps=1.25,
            roe=5.0
        )
        
        assert report.get_profit_margin() == 10.0  # 10000/100000 * 100
    
    def test_financial_report_debt_ratio(self):
        """测试资产负债率计算"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=10000.0,
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=15000.0,
            eps=1.25,
            roe=5.0
        )
        
        assert report.get_debt_ratio() == 60.0  # 300000/500000 * 100


class TestEarningsForecast:
    """业绩预告模型测试"""
    
    def test_valid_earnings_forecast_creation(self):
        """测试创建有效的业绩预告"""
        forecast = EarningsForecast(
            ts_code="000001.SZ",
            forecast_date="20231215",
            forecast_period="20231231",
            forecast_type="预增",
            net_profit_min=8000.0,
            net_profit_max=12000.0,
            growth_rate_min=20.0,
            growth_rate_max=50.0,
            forecast_summary="预计净利润同比增长20%-50%"
        )
        
        assert forecast.ts_code == "000001.SZ"
        assert forecast.forecast_type == "预增"
        assert forecast.net_profit_min == 8000.0
        assert forecast.net_profit_max == 12000.0
        assert forecast.created_at is not None
    
    def test_earnings_forecast_validation_invalid_stock_code(self):
        """测试无效股票代码验证"""
        with pytest.raises(ValidationError, match="Invalid stock code format"):
            EarningsForecast(
                ts_code="invalid",
                forecast_date="20231215",
                forecast_period="20231231",
                forecast_type="预增",
                net_profit_min=8000.0,
                net_profit_max=12000.0,
                growth_rate_min=20.0,
                growth_rate_max=50.0,
                forecast_summary="预计净利润同比增长20%-50%"
            )
    
    def test_earnings_forecast_validation_invalid_forecast_type(self):
        """测试无效预告类型验证"""
        with pytest.raises(ValidationError, match="Invalid forecast type"):
            EarningsForecast(
                ts_code="000001.SZ",
                forecast_date="20231215",
                forecast_period="20231231",
                forecast_type="无效类型",
                net_profit_min=8000.0,
                net_profit_max=12000.0,
                growth_rate_min=20.0,
                growth_rate_max=50.0,
                forecast_summary="预计净利润同比增长20%-50%"
            )
    
    def test_earnings_forecast_validation_profit_range(self):
        """测试利润范围验证"""
        with pytest.raises(ValidationError, match="Net profit minimum cannot be greater than maximum"):
            EarningsForecast(
                ts_code="000001.SZ",
                forecast_date="20231215",
                forecast_period="20231231",
                forecast_type="预增",
                net_profit_min=12000.0,  # 最小值大于最大值
                net_profit_max=8000.0,
                growth_rate_min=20.0,
                growth_rate_max=50.0,
                forecast_summary="预计净利润同比增长20%-50%"
            )
    
    def test_earnings_forecast_validation_growth_range(self):
        """测试增长率范围验证"""
        with pytest.raises(ValidationError, match="Growth rate minimum cannot be greater than maximum"):
            EarningsForecast(
                ts_code="000001.SZ",
                forecast_date="20231215",
                forecast_period="20231231",
                forecast_type="预增",
                net_profit_min=8000.0,
                net_profit_max=12000.0,
                growth_rate_min=50.0,  # 最小值大于最大值
                growth_rate_max=20.0,
                forecast_summary="预计净利润同比增长20%-50%"
            )
    
    def test_earnings_forecast_profit_range_description(self):
        """测试利润范围描述"""
        forecast = EarningsForecast(
            ts_code="000001.SZ",
            forecast_date="20231215",
            forecast_period="20231231",
            forecast_type="预增",
            net_profit_min=8000.0,
            net_profit_max=12000.0,
            growth_rate_min=20.0,
            growth_rate_max=50.0,
            forecast_summary="预计净利润同比增长20%-50%"
        )
        
        assert forecast.get_profit_range() == "8000.0万元 ~ 12000.0万元"
        
        # 测试相等情况
        forecast.net_profit_min = forecast.net_profit_max = 10000.0
        assert forecast.get_profit_range() == "10000.0万元"
    
    def test_earnings_forecast_growth_range_description(self):
        """测试增长率范围描述"""
        forecast = EarningsForecast(
            ts_code="000001.SZ",
            forecast_date="20231215",
            forecast_period="20231231",
            forecast_type="预增",
            net_profit_min=8000.0,
            net_profit_max=12000.0,
            growth_rate_min=20.0,
            growth_rate_max=50.0,
            forecast_summary="预计净利润同比增长20%-50%"
        )
        
        assert forecast.get_growth_range() == "20.0% ~ 50.0%"


class TestFlashReport:
    """业绩快报模型测试"""
    
    def test_valid_flash_report_creation(self):
        """测试创建有效的业绩快报"""
        report = FlashReport(
            ts_code="000001.SZ",
            report_date="20240115",
            publish_date="20240120",
            report_period="20231231",
            total_revenue=95000.0,
            net_profit=9500.0,
            revenue_growth=15.0,
            profit_growth=25.0,
            eps=1.19,
            report_summary="业绩快报显示营收和净利润均实现增长"
        )
        
        assert report.ts_code == "000001.SZ"
        assert report.total_revenue == 95000.0
        assert report.revenue_growth == 15.0
        assert report.created_at is not None
    
    def test_flash_report_validation_date_logic(self):
        """测试日期逻辑验证"""
        with pytest.raises(ValidationError, match="Publish date cannot be earlier than report date"):
            FlashReport(
                ts_code="000001.SZ",
                report_date="20240120",
                publish_date="20240115",  # 发布日期早于报告日期
                report_period="20231231",
                total_revenue=95000.0,
                net_profit=9500.0,
                revenue_growth=15.0,
                profit_growth=25.0,
                eps=1.19,
                report_summary="业绩快报显示营收和净利润均实现增长"
            )
    
    def test_flash_report_profit_margin(self):
        """测试净利润率计算"""
        report = FlashReport(
            ts_code="000001.SZ",
            report_date="20240115",
            publish_date="20240120",
            report_period="20231231",
            total_revenue=95000.0,
            net_profit=9500.0,
            revenue_growth=15.0,
            profit_growth=25.0,
            eps=1.19,
            report_summary="业绩快报显示营收和净利润均实现增长"
        )
        
        assert report.get_profit_margin() == 10.0  # 9500/95000 * 100
    
    def test_flash_report_formatted_dates(self):
        """测试格式化日期"""
        report = FlashReport(
            ts_code="000001.SZ",
            report_date="20240115",
            publish_date="20240120",
            report_period="20231231",
            total_revenue=95000.0,
            net_profit=9500.0,
            revenue_growth=15.0,
            profit_growth=25.0,
            eps=1.19,
            report_summary="业绩快报显示营收和净利润均实现增长"
        )
        
        assert report.get_formatted_report_date() == "2024-01-15"
        assert report.get_formatted_publish_date() == "2024-01-20"
        assert report.get_formatted_period() == "2023-12-31"


class TestFinancialRequestModels:
    """财务数据请求模型测试"""
    
    def test_financial_reports_request(self):
        """测试财务报告请求模型"""
        request = FinancialReportsRequest(
            ts_code="000001.SZ",
            start_date="20230101",
            end_date="20231231",
            report_type="A"
        )
        
        assert request.data_type == "financial_reports"
        assert request.report_type == "A"
        assert request.ts_code == "000001.SZ"
    
    def test_financial_reports_request_invalid_report_type(self):
        """测试无效报告类型验证"""
        with pytest.raises(ValidationError, match="Invalid report type"):
            FinancialReportsRequest(
                ts_code="000001.SZ",
                report_type="INVALID"
            )
    
    def test_earnings_forecast_request(self):
        """测试业绩预告请求模型"""
        request = EarningsForecastRequest(
            ts_code="000001.SZ",
            start_date="20230101",
            end_date="20231231",
            forecast_type="预增"
        )
        
        assert request.data_type == "earnings_forecast"
        assert request.forecast_type == "预增"
    
    def test_earnings_forecast_request_invalid_type(self):
        """测试无效预告类型验证"""
        with pytest.raises(ValidationError, match="Invalid forecast type"):
            EarningsForecastRequest(
                ts_code="000001.SZ",
                forecast_type="无效类型"
            )
    
    def test_flash_reports_request(self):
        """测试业绩快报请求模型"""
        request = FlashReportsRequest(
            ts_code="000001.SZ",
            start_date="20230101",
            end_date="20231231",
            sort_by="publish_date"
        )
        
        assert request.data_type == "flash_reports"
        assert request.sort_by == "publish_date"
    
    def test_flash_reports_request_invalid_sort_field(self):
        """测试无效排序字段验证"""
        with pytest.raises(ValidationError, match="Invalid sort field"):
            FlashReportsRequest(
                ts_code="000001.SZ",
                sort_by="invalid_field"
            )


class TestFinancialDataEdgeCases:
    """财务数据边界情况测试"""
    
    def test_financial_report_zero_revenue_profit_margin(self):
        """测试零收入时的利润率计算"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=0.0,  # 零收入
            net_profit=0.0,
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=0.0,
            eps=0.0,
            roe=0.0
        )
        
        assert report.get_profit_margin() == 0.0
    
    def test_financial_report_zero_assets_debt_ratio(self):
        """测试零资产时的负债率计算"""
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=10000.0,
            total_assets=0.0,  # 零资产
            total_liabilities=0.0,
            shareholders_equity=0.0,
            operating_cash_flow=15000.0,
            eps=1.25,
            roe=5.0
        )
        
        assert report.get_debt_ratio() == 0.0
    
    def test_flash_report_zero_revenue_profit_margin(self):
        """测试业绩快报零收入时的利润率计算"""
        report = FlashReport(
            ts_code="000001.SZ",
            report_date="20240115",
            publish_date="20240120",
            report_period="20231231",
            total_revenue=0.0,  # 零收入
            net_profit=0.0,
            revenue_growth=0.0,
            profit_growth=0.0,
            eps=0.0,
            report_summary="零收入测试"
        )
        
        assert report.get_profit_margin() == 0.0
    
    def test_earnings_forecast_equal_ranges(self):
        """测试相等范围的业绩预告"""
        forecast = EarningsForecast(
            ts_code="000001.SZ",
            forecast_date="20231215",
            forecast_period="20231231",
            forecast_type="预增",
            net_profit_min=10000.0,
            net_profit_max=10000.0,  # 相等
            growth_rate_min=30.0,
            growth_rate_max=30.0,  # 相等
            forecast_summary="精确预告"
        )
        
        assert forecast.get_profit_range() == "10000.0万元"
        assert forecast.get_growth_range() == "30.0%"
    
    def test_financial_models_with_negative_values(self):
        """测试包含负值的财务数据"""
        # 测试亏损情况
        report = FinancialReport(
            ts_code="000001.SZ",
            report_date="20231231",
            report_type="A",
            total_revenue=100000.0,
            net_profit=-5000.0,  # 亏损
            total_assets=500000.0,
            total_liabilities=300000.0,
            shareholders_equity=200000.0,
            operating_cash_flow=-2000.0,  # 负现金流
            eps=-0.5,  # 负每股收益
            roe=-2.5  # 负ROE
        )
        
        assert report.get_profit_margin() == -5.0  # 负利润率
        
        # 测试负增长预告
        forecast = EarningsForecast(
            ts_code="000001.SZ",
            forecast_date="20231215",
            forecast_period="20231231",
            forecast_type="预减",
            net_profit_min=-8000.0,  # 负利润
            net_profit_max=-5000.0,
            growth_rate_min=-50.0,  # 负增长
            growth_rate_max=-20.0,
            forecast_summary="预计亏损"
        )
        
        assert forecast.get_profit_range() == "-8000.0万元 ~ -5000.0万元"
        assert forecast.get_growth_range() == "-50.0% ~ -20.0%"


if __name__ == "__main__":
    pytest.main([__file__])