"""
财务报告集成测试套件

运行所有财务报告相关的集成测试，包括：
- 综合集成测试
- 工作流集成测试
- 性能和并发测试
- 错误处理集成测试
"""

import pytest
import sys
import os
import time
from datetime import datetime
import asyncio
import logging

# 添加项目根目录到路径
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)

logger = logging.getLogger(__name__)


class FinancialReportsIntegrationTestSuite:
    """财务报告集成测试套件"""
    
    def __init__(self):
        self.test_results = {
            'total_tests': 0,
            'passed_tests': 0,
            'failed_tests': 0,
            'skipped_tests': 0,
            'test_duration': 0,
            'test_details': []
        }
    
    def run_comprehensive_tests(self):
        """运行综合集成测试"""
        logger.info("开始运行财务报告综合集成测试...")
        
        start_time = time.time()
        
        # 运行综合集成测试
        result = pytest.main([
            'test_comprehensive_financial_reports_integration.py',
            '-v',
            '--tb=short',
            '--capture=no'
        ])
        
        duration = time.time() - start_time
        
        self.test_results['test_details'].append({
            'test_suite': 'comprehensive_integration',
            'result': result,
            'duration': duration,
            'timestamp': datetime.now()
        })
        
        logger.info(f"综合集成测试完成，耗时: {duration:.2f}秒，结果: {result}")
        return result
    
    def run_workflow_tests(self):
        """运行工作流集成测试"""
        logger.info("开始运行财务报告工作流集成测试...")
        
        start_time = time.time()
        
        # 运行工作流集成测试
        result = pytest.main([
            'test_financial_reports_workflow_integration.py',
            '-v',
            '--tb=short',
            '--capture=no'
        ])
        
        duration = time.time() - start_time
        
        self.test_results['test_details'].append({
            'test_suite': 'workflow_integration',
            'result': result,
            'duration': duration,
            'timestamp': datetime.now()
        })
        
        logger.info(f"工作流集成测试完成，耗时: {duration:.2f}秒，结果: {result}")
        return result
    
    def run_existing_integration_tests(self):
        """运行现有的集成测试"""
        logger.info("开始运行现有的财务报告集成测试...")
        
        existing_tests = [
            'test_client_financial_reports_integration.py',
            'test_financial_error_integration.py',
            'test_config_financial_reports_integration.py'
        ]
        
        results = []
        
        for test_file in existing_tests:
            if os.path.exists(test_file):
                logger.info(f"运行测试文件: {test_file}")
                start_time = time.time()
                
                result = pytest.main([
                    test_file,
                    '-v',
                    '--tb=short',
                    '--capture=no'
                ])
                
                duration = time.time() - start_time
                results.append(result)
                
                self.test_results['test_details'].append({
                    'test_suite': test_file,
                    'result': result,
                    'duration': duration,
                    'timestamp': datetime.now()
                })
                
                logger.info(f"{test_file} 完成，耗时: {duration:.2f}秒，结果: {result}")
            else:
                logger.warning(f"测试文件不存在: {test_file}")
        
        return results
    
    def run_performance_tests(self):
        """运行性能测试"""
        logger.info("开始运行财务报告性能测试...")
        
        start_time = time.time()
        
        # 运行性能相关的测试
        result = pytest.main([
            'test_comprehensive_financial_reports_integration.py::TestFinancialReportsConcurrency',
            'test_financial_reports_workflow_integration.py::TestFinancialReportsWorkflowIntegration::test_performance_optimization_workflow',
            '-v',
            '--tb=short',
            '--capture=no'
        ])
        
        duration = time.time() - start_time
        
        self.test_results['test_details'].append({
            'test_suite': 'performance_tests',
            'result': result,
            'duration': duration,
            'timestamp': datetime.now()
        })
        
        logger.info(f"性能测试完成，耗时: {duration:.2f}秒，结果: {result}")
        return result
    
    def run_error_handling_tests(self):
        """运行错误处理测试"""
        logger.info("开始运行财务报告错误处理测试...")
        
        start_time = time.time()
        
        # 运行错误处理相关的测试
        result = pytest.main([
            'test_comprehensive_financial_reports_integration.py::TestFinancialReportsErrorHandling',
            'test_financial_error_integration.py',
            '-v',
            '--tb=short',
            '--capture=no'
        ])
        
        duration = time.time() - start_time
        
        self.test_results['test_details'].append({
            'test_suite': 'error_handling_tests',
            'result': result,
            'duration': duration,
            'timestamp': datetime.now()
        })
        
        logger.info(f"错误处理测试完成，耗时: {duration:.2f}秒，结果: {result}")
        return result
    
    def run_cache_integration_tests(self):
        """运行缓存集成测试"""
        logger.info("开始运行财务报告缓存集成测试...")
        
        start_time = time.time()
        
        # 运行缓存相关的测试
        result = pytest.main([
            'test_comprehensive_financial_reports_integration.py::TestFinancialReportsCacheIntegration',
            '-v',
            '--tb=short',
            '--capture=no'
        ])
        
        duration = time.time() - start_time
        
        self.test_results['test_details'].append({
            'test_suite': 'cache_integration_tests',
            'result': result,
            'duration': duration,
            'timestamp': datetime.now()
        })
        
        logger.info(f"缓存集成测试完成，耗时: {duration:.2f}秒，结果: {result}")
        return result
    
    def run_batch_processing_tests(self):
        """运行批处理测试"""
        logger.info("开始运行财务报告批处理测试...")
        
        start_time = time.time()
        
        # 运行批处理相关的测试
        result = pytest.main([
            'test_comprehensive_financial_reports_integration.py::TestFinancialReportsBatchProcessing',
            '-v',
            '--tb=short',
            '--capture=no'
        ])
        
        duration = time.time() - start_time
        
        self.test_results['test_details'].append({
            'test_suite': 'batch_processing_tests',
            'result': result,
            'duration': duration,
            'timestamp': datetime.now()
        })
        
        logger.info(f"批处理测试完成，耗时: {duration:.2f}秒，结果: {result}")
        return result
    
    def run_all_tests(self):
        """运行所有集成测试"""
        logger.info("=" * 80)
        logger.info("开始运行财务报告完整集成测试套件")
        logger.info("=" * 80)
        
        total_start_time = time.time()
        
        # 运行各类测试
        test_methods = [
            ('综合集成测试', self.run_comprehensive_tests),
            ('工作流集成测试', self.run_workflow_tests),
            ('现有集成测试', self.run_existing_integration_tests),
            ('缓存集成测试', self.run_cache_integration_tests),
            ('批处理测试', self.run_batch_processing_tests),
            ('错误处理测试', self.run_error_handling_tests),
            ('性能测试', self.run_performance_tests),
        ]
        
        all_results = []
        
        for test_name, test_method in test_methods:
            logger.info(f"\n{'='*60}")
            logger.info(f"运行 {test_name}")
            logger.info(f"{'='*60}")
            
            try:
                result = test_method()
                all_results.append(result)
                
                if result == 0:
                    logger.info(f"✅ {test_name} 全部通过")
                else:
                    logger.warning(f"❌ {test_name} 有失败的测试")
                    
            except Exception as e:
                logger.error(f"❌ {test_name} 运行时出错: {e}")
                all_results.append(-1)
        
        total_duration = time.time() - total_start_time
        self.test_results['test_duration'] = total_duration
        
        # 生成测试报告
        self.generate_test_report(all_results)
        
        return all_results
    
    def generate_test_report(self, results):
        """生成测试报告"""
        logger.info("\n" + "=" * 80)
        logger.info("财务报告集成测试套件执行报告")
        logger.info("=" * 80)
        
        total_duration = self.test_results['test_duration']
        logger.info(f"总执行时间: {total_duration:.2f}秒")
        
        # 统计结果
        passed_suites = sum(1 for r in results if r == 0)
        failed_suites = sum(1 for r in results if r != 0)
        total_suites = len(results)
        
        logger.info(f"测试套件总数: {total_suites}")
        logger.info(f"通过的套件: {passed_suites}")
        logger.info(f"失败的套件: {failed_suites}")
        logger.info(f"成功率: {passed_suites/total_suites*100:.1f}%")
        
        # 详细结果
        logger.info("\n详细结果:")
        logger.info("-" * 60)
        
        for detail in self.test_results['test_details']:
            suite_name = detail['test_suite']
            result = detail['result']
            duration = detail['duration']
            timestamp = detail['timestamp']
            
            status = "✅ PASS" if result == 0 else "❌ FAIL"
            logger.info(f"{status} {suite_name:<30} {duration:>8.2f}s {timestamp.strftime('%H:%M:%S')}")
        
        # 保存报告到文件
        self.save_test_report_to_file()
        
        logger.info("\n" + "=" * 80)
        
        if failed_suites == 0:
            logger.info("🎉 所有财务报告集成测试都通过了！")
        else:
            logger.warning(f"⚠️  有 {failed_suites} 个测试套件失败，请检查日志")
        
        logger.info("=" * 80)
    
    def save_test_report_to_file(self):
        """保存测试报告到文件"""
        report_filename = f"financial_reports_integration_test_report_{datetime.now().strftime('%Y%m%d_%H%M%S')}.txt"
        
        try:
            with open(report_filename, 'w', encoding='utf-8') as f:
                f.write("财务报告集成测试套件执行报告\n")
                f.write("=" * 80 + "\n")
                f.write(f"执行时间: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                f.write(f"总执行时间: {self.test_results['test_duration']:.2f}秒\n\n")
                
                f.write("测试套件详细结果:\n")
                f.write("-" * 60 + "\n")
                
                for detail in self.test_results['test_details']:
                    suite_name = detail['test_suite']
                    result = detail['result']
                    duration = detail['duration']
                    timestamp = detail['timestamp']
                    
                    status = "PASS" if result == 0 else "FAIL"
                    f.write(f"{status:<6} {suite_name:<35} {duration:>8.2f}s {timestamp.strftime('%H:%M:%S')}\n")
                
                f.write("\n" + "=" * 80 + "\n")
            
            logger.info(f"测试报告已保存到: {report_filename}")
            
        except Exception as e:
            logger.error(f"保存测试报告失败: {e}")


def run_quick_smoke_test():
    """运行快速冒烟测试"""
    logger.info("运行财务报告集成测试快速冒烟测试...")
    
    # 只运行关键的测试用例
    result = pytest.main([
        'test_comprehensive_financial_reports_integration.py::TestFinancialReportsEndToEndWorkflow::test_complete_financial_analysis_workflow',
        'test_financial_reports_workflow_integration.py::TestFinancialReportsWorkflowIntegration::test_complete_financial_analysis_workflow',
        '-v',
        '--tb=short'
    ])
    
    if result == 0:
        logger.info("✅ 冒烟测试通过")
    else:
        logger.error("❌ 冒烟测试失败")
    
    return result


def main():
    """主函数"""
    import argparse
    
    parser = argparse.ArgumentParser(description='财务报告集成测试套件')
    parser.add_argument('--smoke', action='store_true', help='只运行冒烟测试')
    parser.add_argument('--comprehensive', action='store_true', help='只运行综合集成测试')
    parser.add_argument('--workflow', action='store_true', help='只运行工作流测试')
    parser.add_argument('--performance', action='store_true', help='只运行性能测试')
    parser.add_argument('--error', action='store_true', help='只运行错误处理测试')
    parser.add_argument('--cache', action='store_true', help='只运行缓存测试')
    parser.add_argument('--batch', action='store_true', help='只运行批处理测试')
    
    args = parser.parse_args()
    
    suite = FinancialReportsIntegrationTestSuite()
    
    if args.smoke:
        return run_quick_smoke_test()
    elif args.comprehensive:
        return suite.run_comprehensive_tests()
    elif args.workflow:
        return suite.run_workflow_tests()
    elif args.performance:
        return suite.run_performance_tests()
    elif args.error:
        return suite.run_error_handling_tests()
    elif args.cache:
        return suite.run_cache_integration_tests()
    elif args.batch:
        return suite.run_batch_processing_tests()
    else:
        # 运行所有测试
        results = suite.run_all_tests()
        return 0 if all(r == 0 for r in results) else 1


if __name__ == "__main__":
    exit_code = main()
    sys.exit(exit_code)