"""
财务报告工作流集成测试

专门测试财务报告的完整工作流程，包括：
- 数据获取到处理的完整流程
- 多数据源协调和fallback机制
- 实时数据更新和缓存同步
- 复杂查询场景的端到端测试
"""

import pytest
import asyncio
import pandas as pd
import time
from datetime import datetime, timedelta
from unittest.mock import Mock, AsyncMock, patch, MagicMock

from quickstock.client import QuickStockClient
from quickstock.config import Config
from quickstock.services.financial_reports_service import FinancialReportsService
from quickstock.core.data_manager import DataManager
from quickstock.providers.baostock import BaostockProvider
from quickstock.models import (
    FinancialReport, EarningsForecast, FlashReport,
    FinancialReportsRequest, EarningsForecastRequest, FlashReportsRequest
)
from quickstock.core.errors import (
    FinancialDataError, ReportNotFoundError, DataSourceError, NetworkError
)


class TestFinancialReportsWorkflowIntegration:
    """财务报告工作流集成测试"""
    
    @pytest.fixture
    def workflow_config(self):
        """工作流测试配置"""
        return Config(
            cache_enabled=True,
            cache_expire_hours=24,
            enable_baostock=True,
            max_concurrent_requests=5,
            request_timeout=30,
            log_level='DEBUG'
        )
    
    @pytest.fixture
    def client_with_real_workflow(self, workflow_config):
        """创建真实工作流客户端"""
        return QuickStockClient(workflow_config)
    
    @pytest.fixture
    def comprehensive_financial_data(self):
        """综合财务数据集"""
        return {
            'financial_reports_annual': pd.DataFrame({
                'ts_code': ['000001.SZ'] * 3,
                'report_date': ['20231231', '20221231', '20211231'],
                'report_type': ['A', 'A', 'A'],
                'total_revenue': [1200000.0, 1100000.0, 1000000.0],
                'net_profit': [240000.0, 220000.0, 200000.0],
                'total_assets': [6000000.0, 5500000.0, 5000000.0],
                'total_liabilities': [3600000.0, 3300000.0, 3000000.0],
                'shareholders_equity': [2400000.0, 2200000.0, 2000000.0],
                'operating_cash_flow': [360000.0, 330000.0, 300000.0],
                'eps': [3.0, 2.75, 2.5],
                'roe': [12.0, 11.0, 10.0]
            }),
            'financial_reports_quarterly': pd.DataFrame({
                'ts_code': ['000001.SZ'] * 4,
                'report_date': ['20231231', '20230930', '20230630', '20230331'],
                'report_type': ['A', 'Q3', 'Q2', 'Q1'],
                'total_revenue': [1200000.0, 900000.0, 600000.0, 300000.0],
                'net_profit': [240000.0, 180000.0, 120000.0, 60000.0],
                'total_assets': [6000000.0, 5800000.0, 5600000.0, 5400000.0],
                'total_liabilities': [3600000.0, 3480000.0, 3360000.0, 3240000.0],
                'shareholders_equity': [2400000.0, 2320000.0, 2240000.0, 2160000.0],
                'operating_cash_flow': [360000.0, 270000.0, 180000.0, 90000.0],
                'eps': [3.0, 2.25, 1.5, 0.75],
                'roe': [12.0, 9.0, 6.0, 3.0]
            }),
            'earnings_forecast_multiple': pd.DataFrame({
                'ts_code': ['000001.SZ'] * 3,
                'forecast_date': ['20240315', '20240215', '20240115'],
                'forecast_period': ['20231231', '20231231', '20231231'],
                'forecast_type': ['预增', '预增', '预增'],
                'net_profit_min': [220000.0, 210000.0, 200000.0],
                'net_profit_max': [260000.0, 250000.0, 240000.0],
                'growth_rate_min': [15.0, 12.0, 10.0],
                'growth_rate_max': [25.0, 22.0, 20.0],
                'forecast_summary': [
                    '预计2023年净利润同比增长15%-25%',
                    '预计2023年净利润同比增长12%-22%',
                    '预计2023年净利润同比增长10%-20%'
                ]
            }),
            'flash_reports_series': pd.DataFrame({
                'ts_code': ['000001.SZ'] * 4,
                'report_date': ['20240430', '20240331', '20240229', '20240131'],
                'publish_date': ['20240430', '20240331', '20240229', '20240131'],
                'report_period': ['20240331', '20240331', '20240229', '20240131'],
                'total_revenue': [300000.0, 290000.0, 200000.0, 100000.0],
                'net_profit': [60000.0, 58000.0, 40000.0, 20000.0],
                'revenue_growth': [20.0, 18.0, 15.0, 12.0],
                'profit_growth': [30.0, 28.0, 25.0, 22.0],
                'eps': [0.75, 0.73, 0.5, 0.25],
                'report_summary': [
                    'Q1业绩快报最终版',
                    'Q1业绩快报',
                    '2月业绩快报',
                    '1月业绩快报'
                ]
            })
        }
    
    def test_complete_financial_analysis_workflow(self, client_with_real_workflow, comprehensive_financial_data):
        """测试完整的财务分析工作流程"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 设置服务方法的模拟返回
            def setup_service_mocks():
                async def mock_get_financial_reports(request):
                    if request.report_type == 'A':
                        df = comprehensive_financial_data['financial_reports_annual']
                    else:
                        df = comprehensive_financial_data['financial_reports_quarterly']
                    return [FinancialReport(**row) for _, row in df.iterrows()]
                
                async def mock_get_earnings_forecast(request):
                    df = comprehensive_financial_data['earnings_forecast_multiple']
                    return [EarningsForecast(**row) for _, row in df.iterrows()]
                
                async def mock_get_flash_reports(request):
                    df = comprehensive_financial_data['flash_reports_series']
                    return [FlashReport(**row) for _, row in df.iterrows()]
                
                mock_service.get_financial_reports = mock_get_financial_reports
                mock_service.get_earnings_forecast = mock_get_earnings_forecast
                mock_service.get_earnings_flash_reports = mock_get_flash_reports
            
            setup_service_mocks()
            
            # 步骤1: 获取年度财务报告
            annual_reports = client_with_real_workflow.get_financial_reports(
                '000001.SZ',
                start_date='20210101',
                end_date='20231231',
                report_type='A'
            )
            
            assert len(annual_reports) == 3
            assert all(report['report_type'] == 'A' for report in annual_reports)
            
            # 验证数据趋势
            revenues = [report['total_revenue'] for report in annual_reports]
            assert revenues == sorted(revenues, reverse=True)  # 收入应该是递减的（最新的在前）
            
            # 步骤2: 获取季度财务报告
            quarterly_reports = client_with_real_workflow.get_financial_reports(
                '000001.SZ',
                start_date='20230101',
                end_date='20231231'
            )
            
            assert len(quarterly_reports) == 4
            report_types = [report['report_type'] for report in quarterly_reports]
            assert 'A' in report_types and 'Q3' in report_types
            
            # 步骤3: 获取业绩预告
            forecasts = client_with_real_workflow.get_earnings_forecast(
                '000001.SZ',
                start_date='20240101',
                end_date='20240331'
            )
            
            assert len(forecasts) == 3
            assert all(forecast['forecast_type'] == '预增' for forecast in forecasts)
            
            # 验证预告数据的时间序列
            forecast_dates = [forecast['forecast_date'] for forecast in forecasts]
            assert forecast_dates == sorted(forecast_dates, reverse=True)
            
            # 步骤4: 获取业绩快报
            flash_reports = client_with_real_workflow.get_earnings_flash_reports(
                '000001.SZ',
                start_date='20240101',
                end_date='20240430'
            )
            
            assert len(flash_reports) == 4
            
            # 验证快报数据的完整性
            for report in flash_reports:
                assert report['total_revenue'] > 0
                assert report['net_profit'] > 0
                assert report['revenue_growth'] > 0
                assert report['profit_growth'] > 0
            
            # 步骤5: 综合分析 - 验证数据一致性
            latest_annual = next(r for r in annual_reports if r['report_type'] == 'A')
            latest_quarterly = next(r for r in quarterly_reports if r['report_type'] == 'A')
            
            # 年度报告和季度报告的年度数据应该一致
            assert latest_annual['total_revenue'] == latest_quarterly['total_revenue']
            assert latest_annual['net_profit'] == latest_quarterly['net_profit']
    
    def test_multi_data_source_coordination(self, client_with_real_workflow):
        """测试多数据源协调机制"""
        with patch('quickstock.core.data_manager.DataManager') as mock_dm_class:
            mock_dm = Mock()
            mock_dm_class.return_value = mock_dm
            
            # 模拟数据源协调场景
            call_count = 0
            
            async def coordinated_get_data(request):
                nonlocal call_count
                call_count += 1
                
                # 模拟第一个数据源失败，第二个成功
                if call_count == 1:
                    raise DataSourceError("Primary source failed")
                else:
                    return pd.DataFrame({
                        'ts_code': [request.ts_code],
                        'report_date': ['20231231'],
                        'report_type': ['A'],
                        'total_revenue': [1000000.0],
                        'net_profit': [200000.0],
                        'total_assets': [5000000.0],
                        'total_liabilities': [3000000.0],
                        'shareholders_equity': [2000000.0],
                        'operating_cash_flow': [300000.0],
                        'eps': [2.5],
                        'roe': [10.0]
                    })
            
            mock_dm.get_data = coordinated_get_data
            
            # 替换客户端的数据管理器
            client_with_real_workflow.data_manager = mock_dm
            
            # 执行请求
            result = client_with_real_workflow.get_financial_reports('000001.SZ')
            
            # 验证fallback机制工作
            assert len(result) == 1
            assert result[0]['ts_code'] == '000001.SZ'
            assert call_count >= 2  # 至少尝试了两次
    
    def test_real_time_data_update_workflow(self, client_with_real_workflow):
        """测试实时数据更新工作流程"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟数据更新场景
            data_versions = {
                'v1': pd.DataFrame({
                    'ts_code': ['000001.SZ'],
                    'report_date': ['20240430'],
                    'publish_date': ['20240430'],
                    'report_period': ['20240331'],
                    'total_revenue': [250000.0],
                    'net_profit': [50000.0],
                    'revenue_growth': [15.0],
                    'profit_growth': [25.0],
                    'eps': [0.6],
                    'report_summary': ['Q1业绩快报初版']
                }),
                'v2': pd.DataFrame({
                    'ts_code': ['000001.SZ'],
                    'report_date': ['20240430'],
                    'publish_date': ['20240430'],
                    'report_period': ['20240331'],
                    'total_revenue': [260000.0],  # 修正后的数据
                    'net_profit': [52000.0],      # 修正后的数据
                    'revenue_growth': [16.0],     # 修正后的数据
                    'profit_growth': [26.0],      # 修正后的数据
                    'eps': [0.65],                # 修正后的数据
                    'report_summary': ['Q1业绩快报修正版']
                })
            }
            
            current_version = 'v1'
            
            async def version_aware_get_flash_reports(request):
                df = data_versions[current_version]
                return [FlashReport(**row) for _, row in df.iterrows()]
            
            mock_service.get_earnings_flash_reports = version_aware_get_flash_reports
            
            # 第一次获取数据（初版）
            result_v1 = client_with_real_workflow.get_earnings_flash_reports('000001.SZ')
            
            assert len(result_v1) == 1
            assert result_v1[0]['total_revenue'] == 250000.0
            assert result_v1[0]['report_summary'] == 'Q1业绩快报初版'
            
            # 模拟数据更新
            current_version = 'v2'
            
            # 第二次获取数据（修正版）
            result_v2 = client_with_real_workflow.get_earnings_flash_reports(
                '000001.SZ',
                force_refresh=True  # 强制刷新缓存
            )
            
            assert len(result_v2) == 1
            assert result_v2[0]['total_revenue'] == 260000.0
            assert result_v2[0]['report_summary'] == 'Q1业绩快报修正版'
            
            # 验证数据确实更新了
            assert result_v2[0]['total_revenue'] != result_v1[0]['total_revenue']
            assert result_v2[0]['net_profit'] != result_v1[0]['net_profit']
    
    def test_complex_query_scenarios(self, client_with_real_workflow, comprehensive_financial_data):
        """测试复杂查询场景"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 场景1: 跨年度查询
            async def cross_year_query(request):
                # 根据日期范围返回不同数据
                if '2021' in request.start_date or '2021' in request.end_date:
                    # 返回历史数据
                    return [FinancialReport(
                        ts_code=request.ts_code,
                        report_date='20211231',
                        report_type='A',
                        total_revenue=800000.0,
                        net_profit=160000.0,
                        total_assets=4000000.0,
                        total_liabilities=2400000.0,
                        shareholders_equity=1600000.0,
                        operating_cash_flow=240000.0,
                        eps=2.0,
                        roe=8.0
                    )]
                else:
                    # 返回最新数据
                    df = comprehensive_financial_data['financial_reports_annual']
                    return [FinancialReport(**row) for _, row in df.iterrows()]
            
            mock_service.get_financial_reports = cross_year_query
            
            # 执行跨年度查询
            historical_data = client_with_real_workflow.get_financial_reports(
                '000001.SZ',
                start_date='20210101',
                end_date='20211231'
            )
            
            recent_data = client_with_real_workflow.get_financial_reports(
                '000001.SZ',
                start_date='20230101',
                end_date='20231231'
            )
            
            # 验证历史数据和最新数据的差异
            assert len(historical_data) == 1
            assert len(recent_data) == 3
            assert historical_data[0]['total_revenue'] < recent_data[0]['total_revenue']
            
            # 场景2: 多类型数据组合查询
            async def mock_batch_get(**kwargs):
                stock_codes = kwargs['stock_codes']
                data_types = kwargs['data_types']
                
                result = {}
                for code in stock_codes:
                    stock_data = {}
                    
                    if 'financial_reports' in data_types:
                        stock_data['financial_reports'] = [FinancialReport(
                            ts_code=code,
                            report_date='20231231',
                            report_type='A',
                            total_revenue=1000000.0,
                            net_profit=200000.0,
                            total_assets=5000000.0,
                            total_liabilities=3000000.0,
                            shareholders_equity=2000000.0,
                            operating_cash_flow=300000.0,
                            eps=2.5,
                            roe=10.0
                        ).to_dict()]
                    
                    if 'earnings_forecast' in data_types:
                        stock_data['earnings_forecast'] = [EarningsForecast(
                            ts_code=code,
                            forecast_date='20240315',
                            forecast_period='20231231',
                            forecast_type='预增',
                            net_profit_min=180000.0,
                            net_profit_max=220000.0,
                            growth_rate_min=10.0,
                            growth_rate_max=20.0,
                            forecast_summary='预计净利润同比增长10%-20%'
                        ).to_dict()]
                    
                    result[code] = stock_data
                
                return {
                    'success_count': len(stock_codes),
                    'failed_count': 0,
                    'total_count': len(stock_codes),
                    'data': result,
                    'failed_stocks': {},
                    'processing_time': 1.0
                }
            
            mock_service.get_batch_financial_data = mock_batch_get
            
            # 执行组合查询
            combo_result = client_with_real_workflow.get_batch_financial_data(
                ['000001.SZ', '000002.SZ'],
                data_types=['financial_reports', 'earnings_forecast']
            )
            
            assert combo_result['success_count'] == 2
            for code in ['000001.SZ', '000002.SZ']:
                assert code in combo_result['data']
                assert 'financial_reports' in combo_result['data'][code]
                assert 'earnings_forecast' in combo_result['data'][code]
                assert len(combo_result['data'][code]['financial_reports']) == 1
                assert len(combo_result['data'][code]['earnings_forecast']) == 1
    
    def test_data_consistency_validation_workflow(self, client_with_real_workflow):
        """测试数据一致性验证工作流程"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 创建一致性测试数据
            consistent_base_data = {
                'ts_code': '000001.SZ',
                'total_revenue': 1000000.0,
                'net_profit': 200000.0,
                'total_assets': 5000000.0,
                'total_liabilities': 3000000.0,
                'shareholders_equity': 2000000.0,
                'operating_cash_flow': 300000.0,
                'eps': 2.5,
                'roe': 10.0
            }
            
            # 模拟财务报告数据
            async def consistent_financial_reports(request):
                return [FinancialReport(
                    report_date='20231231',
                    report_type='A',
                    **consistent_base_data
                )]
            
            # 模拟业绩预告数据（应该与财务报告一致）
            async def consistent_earnings_forecast(request):
                return [EarningsForecast(
                    ts_code=consistent_base_data['ts_code'],
                    forecast_date='20240315',
                    forecast_period='20231231',
                    forecast_type='预增',
                    net_profit_min=consistent_base_data['net_profit'] * 0.9,  # 预告范围下限
                    net_profit_max=consistent_base_data['net_profit'] * 1.1,  # 预告范围上限
                    growth_rate_min=10.0,
                    growth_rate_max=20.0,
                    forecast_summary='预计净利润同比增长10%-20%'
                )]
            
            # 模拟业绩快报数据（应该与财务报告接近）
            async def consistent_flash_reports(request):
                return [FlashReport(
                    ts_code=consistent_base_data['ts_code'],
                    report_date='20240430',
                    publish_date='20240430',
                    report_period='20240331',
                    total_revenue=consistent_base_data['total_revenue'] * 0.25,  # Q1约为年度的1/4
                    net_profit=consistent_base_data['net_profit'] * 0.25,
                    revenue_growth=15.0,
                    profit_growth=25.0,
                    eps=consistent_base_data['eps'] * 0.25,
                    report_summary='Q1业绩快报'
                )]
            
            mock_service.get_financial_reports = consistent_financial_reports
            mock_service.get_earnings_forecast = consistent_earnings_forecast
            mock_service.get_earnings_flash_reports = consistent_flash_reports
            
            # 获取所有类型的数据
            financial_reports = client_with_real_workflow.get_financial_reports('000001.SZ')
            earnings_forecast = client_with_real_workflow.get_earnings_forecast('000001.SZ')
            flash_reports = client_with_real_workflow.get_earnings_flash_reports('000001.SZ')
            
            # 验证数据一致性
            assert len(financial_reports) == 1
            assert len(earnings_forecast) == 1
            assert len(flash_reports) == 1
            
            # 验证基础数据一致性
            fr = financial_reports[0]
            ef = earnings_forecast[0]
            fl = flash_reports[0]
            
            assert fr['ts_code'] == ef['ts_code'] == fl['ts_code']
            
            # 验证业绩预告与实际业绩的一致性
            actual_profit = fr['net_profit']
            forecast_min = ef['net_profit_min']
            forecast_max = ef['net_profit_max']
            
            assert forecast_min <= actual_profit <= forecast_max, \
                f"实际利润{actual_profit}不在预告范围[{forecast_min}, {forecast_max}]内"
            
            # 验证季度快报与年度数据的合理性
            annual_revenue = fr['total_revenue']
            q1_revenue = fl['total_revenue']
            
            # Q1收入应该是年度收入的合理比例（10%-40%）
            q1_ratio = q1_revenue / annual_revenue
            assert 0.1 <= q1_ratio <= 0.4, \
                f"Q1收入占年度收入比例{q1_ratio:.2%}不在合理范围内"
    
    def test_error_recovery_workflow(self, client_with_real_workflow):
        """测试错误恢复工作流程"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟错误恢复场景
            attempt_count = {'financial': 0, 'forecast': 0, 'flash': 0}
            
            async def error_recovery_financial(request):
                attempt_count['financial'] += 1
                if attempt_count['financial'] <= 2:
                    raise NetworkError("网络连接失败")
                else:
                    return [FinancialReport(
                        ts_code=request.ts_code,
                        report_date='20231231',
                        report_type='A',
                        total_revenue=1000000.0,
                        net_profit=200000.0,
                        total_assets=5000000.0,
                        total_liabilities=3000000.0,
                        shareholders_equity=2000000.0,
                        operating_cash_flow=300000.0,
                        eps=2.5,
                        roe=10.0
                    )]
            
            async def error_recovery_forecast(request):
                attempt_count['forecast'] += 1
                if attempt_count['forecast'] <= 1:
                    raise DataSourceError("数据源暂时不可用")
                else:
                    return [EarningsForecast(
                        ts_code=request.ts_code,
                        forecast_date='20240315',
                        forecast_period='20231231',
                        forecast_type='预增',
                        net_profit_min=180000.0,
                        net_profit_max=220000.0,
                        growth_rate_min=10.0,
                        growth_rate_max=20.0,
                        forecast_summary='预计净利润同比增长10%-20%'
                    )]
            
            async def error_recovery_flash(request):
                attempt_count['flash'] += 1
                if attempt_count['flash'] <= 3:
                    raise ReportNotFoundError(request.ts_code, '20240331')
                else:
                    return [FlashReport(
                        ts_code=request.ts_code,
                        report_date='20240430',
                        publish_date='20240430',
                        report_period='20240331',
                        total_revenue=250000.0,
                        net_profit=50000.0,
                        revenue_growth=15.0,
                        profit_growth=25.0,
                        eps=0.6,
                        report_summary='Q1业绩快报'
                    )]
            
            mock_service.get_financial_reports = error_recovery_financial
            mock_service.get_earnings_forecast = error_recovery_forecast
            mock_service.get_earnings_flash_reports = error_recovery_flash
            
            # 执行错误恢复测试
            start_time = time.time()
            
            # 财务报告应该在重试后成功
            financial_result = client_with_real_workflow.get_financial_reports('000001.SZ')
            assert len(financial_result) == 1
            assert attempt_count['financial'] == 3
            
            # 业绩预告应该在重试后成功
            forecast_result = client_with_real_workflow.get_earnings_forecast('000001.SZ')
            assert len(forecast_result) == 1
            assert attempt_count['forecast'] == 2
            
            # 业绩快报应该最终失败（超过重试次数）
            with pytest.raises(Exception):  # 应该抛出异常
                flash_result = client_with_real_workflow.get_earnings_flash_reports('000001.SZ')
            
            elapsed_time = time.time() - start_time
            
            # 验证重试机制增加了处理时间，但在合理范围内
            assert elapsed_time > 1.0  # 重试应该增加时间
            assert elapsed_time < 30.0  # 但不应该太长
    
    def test_performance_optimization_workflow(self, client_with_real_workflow):
        """测试性能优化工作流程"""
        with patch('quickstock.services.financial_reports_service.FinancialReportsService') as mock_service_class:
            mock_service = Mock()
            mock_service_class.return_value = mock_service
            
            # 模拟性能优化场景
            cache_hits = {'count': 0}
            data_fetches = {'count': 0}
            
            async def performance_optimized_get(request):
                data_fetches['count'] += 1
                await asyncio.sleep(0.1)  # 模拟数据获取延迟
                
                return [FinancialReport(
                    ts_code=request.ts_code,
                    report_date='20231231',
                    report_type='A',
                    total_revenue=1000000.0,
                    net_profit=200000.0,
                    total_assets=5000000.0,
                    total_liabilities=3000000.0,
                    shareholders_equity=2000000.0,
                    operating_cash_flow=300000.0,
                    eps=2.5,
                    roe=10.0
                )]
            
            mock_service.get_financial_reports = performance_optimized_get
            
            # 模拟缓存命中统计
            original_get_service_stats = mock_service.get_service_stats
            
            def mock_get_service_stats():
                return {
                    'total_requests': data_fetches['count'] + cache_hits['count'],
                    'successful_requests': data_fetches['count'] + cache_hits['count'],
                    'failed_requests': 0,
                    'cache_hits': cache_hits['count'],
                    'cache_misses': data_fetches['count'],
                    'retry_attempts': 0,
                    'total_processing_time': 0.0,
                    'last_request_time': datetime.now()
                }
            
            mock_service.get_service_stats = mock_get_service_stats
            
            # 性能测试场景1: 重复请求缓存优化
            start_time = time.time()
            
            # 第一次请求（应该从数据源获取）
            result1 = client_with_real_workflow.get_financial_reports('000001.SZ')
            first_request_time = time.time() - start_time
            
            # 模拟缓存命中
            cache_hits['count'] += 1
            
            # 第二次相同请求（应该从缓存获取）
            start_time = time.time()
            result2 = client_with_real_workflow.get_financial_reports('000001.SZ')
            second_request_time = time.time() - start_time
            
            # 验证缓存优化效果
            assert len(result1) == len(result2) == 1
            assert result1[0]['ts_code'] == result2[0]['ts_code']
            
            # 第二次请求应该更快（缓存命中）
            # 注意：由于我们在mock中模拟了缓存，实际的时间比较可能不明显
            # 但我们可以验证数据获取次数
            assert data_fetches['count'] >= 1  # 至少有一次数据获取
            
            # 性能测试场景2: 批量请求优化
            stock_codes = [f'00000{i}.SZ' for i in range(1, 6)]
            
            async def batch_optimized_get(**kwargs):
                codes = kwargs['stock_codes']
                data_fetches['count'] += len(codes)
                
                # 模拟批量处理比单个请求更高效
                await asyncio.sleep(0.05 * len(codes))  # 批量处理的总时间
                
                result = {}
                for code in codes:
                    result[code] = {
                        'financial_reports': [FinancialReport(
                            ts_code=code,
                            report_date='20231231',
                            report_type='A',
                            total_revenue=1000000.0,
                            net_profit=200000.0,
                            total_assets=5000000.0,
                            total_liabilities=3000000.0,
                            shareholders_equity=2000000.0,
                            operating_cash_flow=300000.0,
                            eps=2.5,
                            roe=10.0
                        ).to_dict()]
                    }
                
                return {
                    'success_count': len(codes),
                    'failed_count': 0,
                    'total_count': len(codes),
                    'data': result,
                    'failed_stocks': {},
                    'processing_time': 0.05 * len(codes)
                }
            
            mock_service.get_batch_financial_data = batch_optimized_get
            
            start_time = time.time()
            batch_result = client_with_real_workflow.get_batch_financial_data(stock_codes)
            batch_time = time.time() - start_time
            
            # 验证批量处理结果
            assert batch_result['success_count'] == 5
            assert len(batch_result['data']) == 5
            
            # 批量处理应该比单个请求更高效
            # 如果单独请求5只股票，每个需要0.1秒，总共0.5秒
            # 批量请求应该更快
            assert batch_time < 0.4  # 应该比单独请求快


if __name__ == "__main__":
    pytest.main([__file__, "-v"])