"""
涨停统计系统集成测试

测试完整的涨停统计工作流程，包括数据库存储、查询、错误处理和性能测试
"""

import pytest
import pytest_asyncio
import asyncio
import tempfile
import os
import pandas as pd
from datetime import datetime, timedelta
from typing import List, Dict, Any
from unittest.mock import Mock, AsyncMock, patch

from quickstock.client import QuickStockClient
from quickstock.core.data_manager import DataManager
from quickstock.core.database import DatabaseManager
from quickstock.core.repository import LimitUpStatsRepository
from quickstock.services.limit_up_stats_service import LimitUpStatsService
from quickstock.models import LimitUpStatsRequest, LimitUpStats, StockDailyData
from quickstock.core.errors import DatabaseError, DataSourceError, LimitUpStatsError


class TestLimitUpStatsIntegration:
    """涨停统计系统集成测试类"""
    
    @pytest_asyncio.fixture
    async def temp_database(self):
        """创建临时数据库"""
        with tempfile.NamedTemporaryFile(suffix='.db', delete=False) as tmp_file:
            db_path = tmp_file.name
        
        try:
            db_manager = DatabaseManager(db_path)
            await db_manager.initialize()
            yield db_manager
        finally:
            # 清理临时文件
            files_to_clean = [db_path, f"{db_path}-wal", f"{db_path}-shm"]
            for file_path in files_to_clean:
                if os.path.exists(file_path):
                    try:
                        os.unlink(file_path)
                    except OSError:
                        pass
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置"""
        config = Mock()
        config.cache_enabled = True
        config.cache_expire_hours = 24
        config.max_concurrent_requests = 5
        config.max_retries = 3
        config.database_path = None  # 使用临时数据库
        return config
    
    @pytest.fixture
    def comprehensive_stock_data(self):
        """全面的股票测试数据"""
        # 创建足够多的股票数据以满足最小要求（至少50只）
        stock_data = []
        
        # 基础涨停股票
        base_stocks = [
            # 上海主板涨停
            {
                'ts_code': '600000.SH', 'trade_date': '20241015',
                'open': 8.0, 'high': 8.8, 'low': 8.0, 'close': 8.8,
                'pre_close': 8.0, 'change': 0.8, 'pct_chg': 10.0,
                'vol': 2000000, 'amount': 17200000.0, 'name': '浦发银行'
            },
            # 深圳主板涨停
            {
                'ts_code': '000001.SZ', 'trade_date': '20241015',
                'open': 10.0, 'high': 11.0, 'low': 10.0, 'close': 11.0,
                'pre_close': 10.0, 'change': 1.0, 'pct_chg': 10.0,
                'vol': 1000000, 'amount': 10500000.0, 'name': '平安银行'
            },
            # 科创板涨停
            {
                'ts_code': '688001.SH', 'trade_date': '20241015',
                'open': 20.0, 'high': 24.0, 'low': 20.0, 'close': 24.0,
                'pre_close': 20.0, 'change': 4.0, 'pct_chg': 20.0,
                'vol': 500000, 'amount': 11000000.0, 'name': '华兴源创'
            },
            # 北证涨停
            {
                'ts_code': '430001.BJ', 'trade_date': '20241015',
                'open': 12.0, 'high': 15.6, 'low': 12.0, 'close': 15.6,
                'pre_close': 12.0, 'change': 3.6, 'pct_chg': 30.0,
                'vol': 300000, 'amount': 4380000.0, 'name': '北交所股票'
            },
            # ST股票涨停
            {
                'ts_code': '000002.SZ', 'trade_date': '20241015',
                'open': 6.3, 'high': 6.62, 'low': 6.3, 'close': 6.62,
                'pre_close': 6.3, 'change': 0.32, 'pct_chg': 5.08,
                'vol': 800000, 'amount': 5248000.0, 'name': 'ST万科'
            },
            # 创业板涨停
            {
                'ts_code': '300001.SZ', 'trade_date': '20241015',
                'open': 15.0, 'high': 16.5, 'low': 15.0, 'close': 16.5,
                'pre_close': 15.0, 'change': 1.5, 'pct_chg': 10.0,
                'vol': 600000, 'amount': 9600000.0, 'name': '特锐德'
            }
        ]
        
        stock_data.extend(base_stocks)
        
        # 添加更多股票以满足最小数量要求
        for i in range(50):
            # 生成不同类型的股票
            if i % 4 == 0:  # 上海股票
                ts_code = f'60{i+100:04d}.SH'
                market = 'shanghai'
            elif i % 4 == 1:  # 深圳股票
                ts_code = f'00{i+100:04d}.SZ'
                market = 'shenzhen'
            elif i % 4 == 2:  # 科创板
                ts_code = f'688{i+100:03d}.SH'
                market = 'star'
            else:  # 北证
                ts_code = f'43{i+100:04d}.BJ'
                market = 'beijing'
            
            base_price = 10.0 + i * 0.1
            
            # 30%的股票涨停
            if i % 3 == 0:
                if market == 'star':
                    close_price = base_price * 1.20
                    pct_chg = 20.0
                elif market == 'beijing':
                    close_price = base_price * 1.30
                    pct_chg = 30.0
                else:
                    close_price = base_price * 1.10
                    pct_chg = 10.0
                high_price = close_price
            else:
                # 非涨停股票
                close_price = base_price * (1 + (i % 10 - 5) * 0.01)
                high_price = close_price * 1.02
                pct_chg = (close_price - base_price) / base_price * 100
            
            stock_data.append({
                'ts_code': ts_code,
                'trade_date': '20241015',
                'open': base_price,
                'high': high_price,
                'low': base_price * 0.98,
                'close': close_price,
                'pre_close': base_price,
                'change': close_price - base_price,
                'pct_chg': pct_chg,
                'vol': (i + 1) * 10000,
                'amount': close_price * (i + 1) * 10000,
                'name': f'股票{i+100:03d}'
            })
        
        return pd.DataFrame(stock_data)
    
    @pytest.fixture
    def comprehensive_basic_data(self, comprehensive_stock_data):
        """全面的基础数据"""
        # 基于股票数据生成对应的基础数据
        basic_data = []
        for _, row in comprehensive_stock_data.iterrows():
            basic_data.append({
                'ts_code': row['ts_code'],
                'name': row['name']
            })
        return pd.DataFrame(basic_data)
    
    @pytest.mark.asyncio
    async def test_end_to_end_workflow(self, temp_database, mock_config, 
                                     comprehensive_stock_data, comprehensive_basic_data):
        """测试端到端工作流程"""
        # 创建完整的系统组件
        repository = LimitUpStatsRepository(temp_database)
        
        # 模拟数据管理器
        mock_data_manager = Mock(spec=DataManager)
        mock_data_manager.get_data = AsyncMock()
        mock_data_manager.get_data.side_effect = [comprehensive_stock_data, comprehensive_basic_data]
        
        # 创建服务
        service = LimitUpStatsService(mock_data_manager)
        
        # 创建请求
        request = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=True,
            save_to_db=True
        )
        
        # 执行完整工作流程
        result = await service.get_daily_limit_up_stats(request)
        
        # 验证结果
        assert isinstance(result, LimitUpStats)
        assert result.trade_date == '20241015'
        assert result.total >= 5  # 至少5只涨停股票
        
        # 验证统计一致性
        assert result.total == result.shanghai + result.shenzhen + result.star + result.beijing
        assert result.total == result.st + result.non_st
        
        # 验证涨停股票列表
        assert len(result.limit_up_stocks) == result.total
        expected_limit_up_stocks = ['600000.SH', '000001.SZ', '688001.SH', '430001.BJ', '000002.SZ', '300001.SZ']
        for stock in expected_limit_up_stocks:
            assert stock in result.limit_up_stocks
        
        # 验证非涨停股票不在列表中
        assert '600001.SH' not in result.limit_up_stocks
        
        # 验证市场分类
        assert result.shanghai >= 1  # 浦发银行
        assert result.shenzhen >= 2  # 平安银行 + 特锐德
        assert result.star >= 1      # 华兴源创
        assert result.beijing >= 1   # 北交所股票
        assert result.st >= 1        # ST万科
    
    @pytest.mark.asyncio
    async def test_database_storage_and_retrieval_workflow(self, temp_database, mock_config,
                                                         comprehensive_stock_data, comprehensive_basic_data):
        """测试数据库存储和查询的完整流程"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 模拟数据管理器
        mock_data_manager = Mock(spec=DataManager)
        mock_data_manager.get_data = AsyncMock()
        mock_data_manager.get_data.side_effect = [comprehensive_stock_data, comprehensive_basic_data]
        
        service = LimitUpStatsService(mock_data_manager)
        
        # 1. 保存数据
        request = LimitUpStatsRequest(trade_date='20241015', save_to_db=True)
        result = await service.get_daily_limit_up_stats(request)
        
        # 手动保存到数据库（模拟完整流程）
        await repository.save_limit_up_stats(result)
        
        # 2. 查询单个日期数据
        retrieved_stats = await repository.get_limit_up_stats('20241015')
        assert retrieved_stats is not None
        assert retrieved_stats.trade_date == result.trade_date
        assert retrieved_stats.total == result.total
        
        # 3. 查询所有数据
        all_stats = await repository.query_limit_up_stats()
        assert len(all_stats) == 1
        assert all_stats[0].trade_date == '20241015'
        
        # 4. 列出可用日期
        available_dates = await repository.list_available_dates()
        assert '20241015' in available_dates
        
        # 5. 获取数据库统计
        db_stats = await repository.get_database_stats()
        assert db_stats['tables']['limit_up_stats'] == 1
        
        # 6. 删除数据
        delete_success = await repository.delete_limit_up_stats('20241015')
        assert delete_success is True
        
        # 7. 验证删除结果
        deleted_stats = await repository.get_limit_up_stats('20241015')
        assert deleted_stats is None
        
        final_dates = await repository.list_available_dates()
        assert '20241015' not in final_dates
    
    @pytest.mark.asyncio
    async def test_data_integrity_workflow(self, temp_database, mock_config):
        """测试数据完整性工作流程"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 创建多个日期的测试数据
        test_dates = ['20241010', '20241011', '20241012', '20241015', '20241016']
        saved_stats = []
        
        for i, date in enumerate(test_dates):
            stats = LimitUpStats(
                trade_date=date,
                total=100 + i * 10,
                non_st=90 + i * 10,
                shanghai=30 + i * 2,
                shenzhen=40 + i * 3,
                star=20 + i * 2,
                beijing=10 + i * 3,
                st=10,
                limit_up_stocks=[f'stock_{j}.SZ' for j in range(100 + i * 10)],
                market_breakdown={
                    'shanghai': [f'600{j:03d}.SH' for j in range(30 + i * 2)],
                    'shenzhen': [f'000{j:03d}.SZ' for j in range(40 + i * 3)],
                    'star': [f'688{j:03d}.SH' for j in range(20 + i * 2)],
                    'beijing': [f'430{j:03d}.BJ' for j in range(10 + i * 3)]
                }
            )
            await repository.save_limit_up_stats(stats)
            saved_stats.append(stats)
        
        # 验证数据完整性
        for original_stats in saved_stats:
            retrieved = await repository.get_limit_up_stats(original_stats.trade_date)
            
            # 验证基本统计
            assert retrieved.total == original_stats.total
            assert retrieved.non_st == original_stats.non_st
            assert retrieved.shanghai == original_stats.shanghai
            assert retrieved.shenzhen == original_stats.shenzhen
            assert retrieved.star == original_stats.star
            assert retrieved.beijing == original_stats.beijing
            assert retrieved.st == original_stats.st
            
            # 验证一致性约束
            assert retrieved.total == retrieved.shanghai + retrieved.shenzhen + retrieved.star + retrieved.beijing
            assert retrieved.total == retrieved.st + retrieved.non_st
            
            # 验证涨停股票数量
            assert len(retrieved.limit_up_stocks) == retrieved.total
        
        # 验证查询功能
        all_stats = await repository.query_limit_up_stats()
        assert len(all_stats) == len(test_dates)
        
        # 验证日期范围查询
        range_stats = await repository.query_limit_up_stats(
            start_date='20241011',
            end_date='20241015'
        )
        expected_dates = ['20241011', '20241012', '20241015']
        retrieved_dates = [stats.trade_date for stats in range_stats]
        for date in expected_dates:
            assert date in retrieved_dates
    
    @pytest.mark.asyncio
    async def test_error_handling_and_recovery_workflow(self, temp_database, mock_config):
        """测试错误处理和恢复场景"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 模拟数据管理器
        mock_data_manager = Mock(spec=DataManager)
        mock_data_manager.get_data = AsyncMock()
        
        service = LimitUpStatsService(mock_data_manager)
        
        # 1. 测试数据源错误处理
        mock_data_manager.get_data.side_effect = DataSourceError("网络连接失败")
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        with pytest.raises(LimitUpStatsError):
            await service.get_daily_limit_up_stats(request)
        
        # 2. 测试数据不足错误处理
        mock_data_manager.get_data.side_effect = [pd.DataFrame(), pd.DataFrame()]
        
        with pytest.raises(Exception):  # 应该抛出数据不足异常
            await service.get_daily_limit_up_stats(request)
        
        # 3. 测试数据库错误恢复
        # 保存正常数据
        normal_stats = LimitUpStats(
            trade_date='20241015',
            total=100,
            non_st=90,
            shanghai=30,
            shenzhen=40,
            star=20,
            beijing=10,
            st=10,
            limit_up_stocks=['000001.SZ'] * 100,
            market_breakdown={'shenzhen': ['000001.SZ'] * 100}
        )
        
        await repository.save_limit_up_stats(normal_stats)
        
        # 验证数据已保存
        retrieved = await repository.get_limit_up_stats('20241015')
        assert retrieved is not None
        
        # 模拟数据库连接错误
        with patch.object(repository.db_manager, 'get_connection') as mock_conn:
            mock_conn.side_effect = DatabaseError("数据库连接失败")
            
            with pytest.raises(DatabaseError):
                await repository.get_limit_up_stats('20241015')
    
    @pytest.mark.asyncio
    async def test_concurrent_operations_workflow(self, temp_database, mock_config,
                                                comprehensive_stock_data, comprehensive_basic_data):
        """测试并发操作工作流程"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 模拟数据管理器
        mock_data_manager = Mock(spec=DataManager)
        mock_data_manager.get_data = AsyncMock()
        mock_data_manager.get_data.side_effect = lambda *args: (
            comprehensive_stock_data if args[0].data_type == 'stock_daily' 
            else comprehensive_basic_data
        )
        
        service = LimitUpStatsService(mock_data_manager)
        
        # 并发保存操作
        async def save_worker(date_suffix):
            request = LimitUpStatsRequest(trade_date=f'2024101{date_suffix}')
            result = await service.get_daily_limit_up_stats(request)
            await repository.save_limit_up_stats(result)
            return result
        
        # 并发查询操作
        async def query_worker():
            return await repository.list_available_dates()
        
        # 执行并发保存
        save_tasks = [save_worker(i) for i in range(5)]
        save_results = await asyncio.gather(*save_tasks)
        
        # 验证保存结果
        assert len(save_results) == 5
        assert all(isinstance(result, LimitUpStats) for result in save_results)
        
        # 执行并发查询
        query_tasks = [query_worker() for _ in range(10)]
        query_results = await asyncio.gather(*query_tasks)
        
        # 验证查询结果
        assert all(len(result) == 5 for result in query_results)
        
        # 验证数据一致性
        all_dates = await repository.list_available_dates()
        assert len(all_dates) == 5
        
        for date in all_dates:
            stats = await repository.get_limit_up_stats(date)
            assert stats is not None
            assert stats.total > 0
    
    @pytest.mark.asyncio
    async def test_performance_with_large_dataset(self, temp_database, mock_config):
        """测试大数据集性能"""
        repository = LimitUpStatsRepository(temp_database)
        import time
        
        # 创建大量测试数据
        large_stats_list = []
        for i in range(100):  # 100个交易日的数据
            stats = LimitUpStats(
                trade_date=f'20241{i:03d}',
                total=100 + i % 50,
                non_st=90 + i % 50,
                shanghai=30 + i % 10,
                shenzhen=40 + i % 15,
                star=20 + i % 8,
                beijing=10 + i % 5,
                st=10,
                limit_up_stocks=[f'stock_{j}.SZ' for j in range(100 + i % 50)],
                market_breakdown={}
            )
            large_stats_list.append(stats)
        
        # 测试批量保存性能
        start_time = time.time()
        result = await repository.batch_save_stats(large_stats_list)
        save_time = time.time() - start_time
        
        # 验证保存结果
        assert result['success'] == 100
        assert result['failed'] == 0
        
        # 性能要求：100条记录应该在合理时间内保存
        assert save_time < 10.0  # 10秒内完成
        
        # 测试查询性能
        start_time = time.time()
        all_stats = await repository.query_limit_up_stats()
        query_time = time.time() - start_time
        
        # 验证查询结果
        assert len(all_stats) == 100
        
        # 性能要求：查询应该在合理时间内完成
        assert query_time < 2.0  # 2秒内完成
        
        # 测试范围查询性能
        start_time = time.time()
        range_stats = await repository.query_limit_up_stats(
            start_date='20241020',
            end_date='20241080'
        )
        range_query_time = time.time() - start_time
        
        # 验证范围查询结果
        assert len(range_stats) > 0
        assert range_query_time < 1.0  # 1秒内完成
    
    @pytest.mark.asyncio
    async def test_database_transaction_workflow(self, temp_database):
        """测试数据库事务工作流程"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 测试成功事务
        stats1 = LimitUpStats(
            trade_date='20241015',
            total=100,
            non_st=90,
            shanghai=30,
            shenzhen=40,
            star=20,
            beijing=10,
            st=10,
            limit_up_stocks=['000001.SZ'] * 100,
            market_breakdown={'shenzhen': ['000001.SZ'] * 100}
        )
        
        # 保存应该成功
        success = await repository.save_limit_up_stats(stats1)
        assert success is True
        
        # 验证数据已提交
        retrieved = await repository.get_limit_up_stats('20241015')
        assert retrieved is not None
        assert retrieved.total == 100
        
        # 测试更新事务
        stats2 = LimitUpStats(
            trade_date='20241015',
            total=150,  # 更新总数
            non_st=135,
            shanghai=45,
            shenzhen=60,
            star=30,
            beijing=15,
            st=15,
            limit_up_stocks=['000001.SZ'] * 150,
            market_breakdown={'shenzhen': ['000001.SZ'] * 150}
        )
        
        # 更新应该成功
        success = await repository.save_limit_up_stats(stats2)
        assert success is True
        
        # 验证数据已更新
        updated = await repository.get_limit_up_stats('20241015')
        assert updated.total == 150
        assert updated.shanghai == 45
    
    @pytest.mark.asyncio
    async def test_data_validation_workflow(self, temp_database):
        """测试数据验证工作流程"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 测试有效数据
        valid_stats = LimitUpStats(
            trade_date='20241015',
            total=100,
            non_st=90,
            shanghai=30,
            shenzhen=40,
            star=20,
            beijing=10,
            st=10,
            limit_up_stocks=['000001.SZ'] * 100,
            market_breakdown={'shenzhen': ['000001.SZ'] * 100}
        )
        
        success = await repository.save_limit_up_stats(valid_stats)
        assert success is True
        
        # 验证数据一致性
        retrieved = await repository.get_limit_up_stats('20241015')
        assert retrieved.total == retrieved.shanghai + retrieved.shenzhen + retrieved.star + retrieved.beijing
        assert retrieved.total == retrieved.st + retrieved.non_st
        
        # 测试边界值
        zero_stats = LimitUpStats(
            trade_date='20241016',
            total=0,
            non_st=0,
            shanghai=0,
            shenzhen=0,
            star=0,
            beijing=0,
            st=0,
            limit_up_stocks=[],
            market_breakdown={}
        )
        
        success = await repository.save_limit_up_stats(zero_stats)
        assert success is True
        
        retrieved_zero = await repository.get_limit_up_stats('20241016')
        assert retrieved_zero.total == 0
        assert len(retrieved_zero.limit_up_stocks) == 0


class TestLimitUpStatsEndToEnd:
    """涨停统计端到端测试"""
    
    @pytest_asyncio.fixture
    async def temp_database(self):
        """创建临时数据库"""
        with tempfile.NamedTemporaryFile(suffix='.db', delete=False) as tmp_file:
            db_path = tmp_file.name
        
        try:
            db_manager = DatabaseManager(db_path)
            await db_manager.initialize()
            yield db_manager
        finally:
            # 清理临时文件
            files_to_clean = [db_path, f"{db_path}-wal", f"{db_path}-shm"]
            for file_path in files_to_clean:
                if os.path.exists(file_path):
                    try:
                        os.unlink(file_path)
                    except OSError:
                        pass
    
    @pytest.mark.asyncio
    async def test_complete_user_workflow(self, temp_database):
        """测试完整的用户工作流程"""
        # 模拟用户使用QuickStockClient的完整流程
        
        # 1. 创建客户端（模拟）
        mock_config = Mock()
        mock_config.database_path = temp_database.db_path
        
        # 2. 模拟获取涨停统计
        repository = LimitUpStatsRepository(temp_database)
        
        # 创建测试数据
        test_stats = LimitUpStats(
            trade_date='20241015',
            total=150,
            non_st=135,
            shanghai=60,
            shenzhen=50,
            star=30,
            beijing=10,
            st=15,
            limit_up_stocks=[
                '600000.SH', '600001.SH', '600002.SH',  # 上海
                '000001.SZ', '000002.SZ', '300001.SZ',  # 深圳
                '688001.SH', '688002.SH',               # 科创板
                '430001.BJ',                            # 北证
                '000003.SZ'                             # ST股票
            ],
            market_breakdown={
                'shanghai': ['600000.SH', '600001.SH', '600002.SH'],
                'shenzhen': ['000001.SZ', '000002.SZ', '300001.SZ'],
                'star': ['688001.SH', '688002.SH'],
                'beijing': ['430001.BJ'],
                'st': ['000003.SZ']
            }
        )
        
        # 3. 保存数据
        await repository.save_limit_up_stats(test_stats)
        
        # 4. 查询数据
        retrieved_stats = await repository.get_limit_up_stats('20241015')
        assert retrieved_stats is not None
        
        # 5. 验证用户关心的指标
        assert retrieved_stats.total == 150
        assert retrieved_stats.non_st == 135
        assert retrieved_stats.st == 15
        
        # 6. 验证市场分布
        assert retrieved_stats.shanghai == 60
        assert retrieved_stats.shenzhen == 50
        assert retrieved_stats.star == 30
        assert retrieved_stats.beijing == 10
        
        # 7. 查询历史数据
        all_stats = await repository.query_limit_up_stats()
        assert len(all_stats) == 1
        
        # 8. 列出可用日期
        available_dates = await repository.list_available_dates()
        assert '20241015' in available_dates
        
        # 9. 删除数据
        delete_success = await repository.delete_limit_up_stats('20241015')
        assert delete_success is True
        
        # 10. 验证删除结果
        final_stats = await repository.get_limit_up_stats('20241015')
        assert final_stats is None
    
    @pytest.mark.asyncio
    async def test_multi_day_analysis_workflow(self, temp_database):
        """测试多日分析工作流程"""
        repository = LimitUpStatsRepository(temp_database)
        
        # 创建一周的测试数据
        weekly_data = []
        base_date = datetime(2024, 10, 14)  # 周一
        
        for i in range(5):  # 工作日
            trade_date = (base_date + timedelta(days=i)).strftime('%Y%m%d')
            
            # 模拟不同的市场情况
            if i == 0:  # 周一：市场活跃
                total = 200
            elif i == 1:  # 周二：市场平稳
                total = 100
            elif i == 2:  # 周三：市场低迷
                total = 50
            elif i == 3:  # 周四：市场回暖
                total = 120
            else:  # 周五：市场强势
                total = 180
            
            stats = LimitUpStats(
                trade_date=trade_date,
                total=total,
                non_st=int(total * 0.9),
                shanghai=int(total * 0.4),
                shenzhen=int(total * 0.35),
                star=int(total * 0.15),
                beijing=int(total * 0.1),
                st=int(total * 0.1),
                limit_up_stocks=[f'stock_{j}.SZ' for j in range(total)],
                market_breakdown={}
            )
            
            await repository.save_limit_up_stats(stats)
            weekly_data.append(stats)
        
        # 分析一周的数据
        week_stats = await repository.query_limit_up_stats(
            start_date='20241014',
            end_date='20241018'
        )
        
        assert len(week_stats) == 5
        
        # 计算周统计
        total_limit_ups = sum(stats.total for stats in week_stats)
        avg_daily_limit_ups = total_limit_ups / len(week_stats)
        max_daily_limit_ups = max(stats.total for stats in week_stats)
        min_daily_limit_ups = min(stats.total for stats in week_stats)
        
        # 验证统计结果
        assert total_limit_ups == 650  # 200+100+50+120+180
        assert avg_daily_limit_ups == 130
        assert max_daily_limit_ups == 200
        assert min_daily_limit_ups == 50
        
        # 分析市场趋势
        daily_totals = [stats.total for stats in reversed(week_stats)]  # 按日期正序
        
        # 验证趋势（周一高，周三低，周五回升）
        assert daily_totals[0] == 200  # 周一
        assert daily_totals[2] == 50   # 周三
        assert daily_totals[4] == 180  # 周五


if __name__ == "__main__":
    pytest.main([__file__])