"""
涨停统计错误处理测试

测试涨停统计功能的各种错误场景和异常处理
"""

import pytest
from datetime import datetime, timedelta
from unittest.mock import Mock, patch, AsyncMock
import sqlite3

import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from quickstock.core.errors import (
    LimitUpStatsError,
    InvalidTradeDateError,
    InsufficientDataError,
    LimitUpDatabaseError,
    StockClassificationError,
    LimitUpDetectionError,
    ErrorHandler
)
from quickstock.config import Config


class TestLimitUpStatsError:
    """测试涨停统计基础异常类"""
    
    def test_basic_error_creation(self):
        """测试基础异常创建"""
        error = LimitUpStatsError("测试错误")
        assert str(error) == "测试错误"
        assert error.message == "测试错误"
        assert error.error_code is None
        assert error.details == {}
        assert error.suggestions is None
    
    def test_error_with_all_parameters(self):
        """测试带所有参数的异常创建"""
        details = {"key": "value"}
        suggestions = "这是建议"
        
        error = LimitUpStatsError(
            message="完整错误",
            error_code="TEST_ERROR",
            details=details,
            suggestions=suggestions
        )
        
        assert error.message == "完整错误"
        assert error.error_code == "TEST_ERROR"
        assert error.details == details
        assert error.suggestions == suggestions


class TestInvalidTradeDateError:
    """测试无效交易日期异常"""
    
    def test_basic_invalid_date(self):
        """测试基础无效日期异常"""
        error = InvalidTradeDateError("2024-13-01")
        
        assert "无效的交易日期 '2024-13-01'" in str(error)
        assert error.error_code == "INVALID_TRADE_DATE"
        assert error.invalid_date == "2024-13-01"
        assert error.reason is None
        assert "请检查日期格式" in error.suggestions
    
    def test_invalid_date_with_reason(self):
        """测试带原因的无效日期异常"""
        error = InvalidTradeDateError("2024-12-25", "非交易日")
        
        assert "无效的交易日期 '2024-12-25': 非交易日" in str(error)
        assert error.invalid_date == "2024-12-25"
        assert error.reason == "非交易日"
        assert error.details["reason"] == "非交易日"
    
    def test_future_date_error(self):
        """测试未来日期错误"""
        future_date = (datetime.now() + timedelta(days=30)).strftime("%Y-%m-%d")
        error = InvalidTradeDateError(future_date, "未来日期")
        
        assert future_date in str(error)
        assert error.reason == "未来日期"
    
    def test_invalid_format_error(self):
        """测试无效格式错误"""
        error = InvalidTradeDateError("invalid-date", "格式错误")
        
        assert "invalid-date" in str(error)
        assert error.reason == "格式错误"


class TestInsufficientDataError:
    """测试数据不足异常"""
    
    def test_basic_insufficient_data(self):
        """测试基础数据不足异常"""
        error = InsufficientDataError("2024-01-15")
        
        assert "日期 '2024-01-15' 的数据不足" in str(error)
        assert error.error_code == "INSUFFICIENT_DATA"
        assert error.date == "2024-01-15"
        assert error.missing_data_type is None
    
    def test_insufficient_data_with_type(self):
        """测试指定数据类型的不足异常"""
        error = InsufficientDataError("2024-01-15", "股票价格")
        
        assert "日期 '2024-01-15' 的股票价格数据不足" in str(error)
        assert error.missing_data_type == "股票价格"
    
    def test_insufficient_data_with_counts(self):
        """测试带数量信息的数据不足异常"""
        error = InsufficientDataError(
            "2024-01-15", 
            "股票数据", 
            available_count=100, 
            required_count=500
        )
        
        assert "可用: 100, 需要: 500" in str(error)
        assert error.available_count == 100
        assert error.required_count == 500
    
    def test_error_details(self):
        """测试错误详细信息"""
        error = InsufficientDataError(
            "2024-01-15",
            "价格数据",
            available_count=50,
            required_count=100
        )
        
        details = error.details
        assert details["date"] == "2024-01-15"
        assert details["missing_data_type"] == "价格数据"
        assert details["available_count"] == 50
        assert details["required_count"] == 100


class TestLimitUpDatabaseError:
    """测试涨停统计数据库异常"""
    
    def test_basic_database_error(self):
        """测试基础数据库异常"""
        error = LimitUpDatabaseError("save")
        
        assert "数据库操作 'save' 失败" in str(error)
        assert error.error_code == "DATABASE_OPERATION_FAILED"
        assert error.operation == "save"
        assert error.original_error is None
    
    def test_database_error_with_message(self):
        """测试带消息的数据库异常"""
        error = LimitUpDatabaseError("query", "连接超时")
        
        assert "数据库操作 'query' 失败: 连接超时" in str(error)
        assert error.operation == "query"
    
    def test_database_error_with_original_error(self):
        """测试带原始异常的数据库异常"""
        original = sqlite3.OperationalError("database is locked")
        error = LimitUpDatabaseError("insert", original_error=original)
        
        assert "数据库操作 'insert' 失败" in str(error)
        assert "database is locked" in str(error)
        assert error.original_error == original
        assert error.details["original_error"] == str(original)
    
    def test_database_error_suggestions(self):
        """测试数据库异常建议"""
        error = LimitUpDatabaseError("delete")
        
        assert "请检查数据库连接" in error.suggestions
        assert "磁盘空间" in error.suggestions


class TestStockClassificationError:
    """测试股票分类异常"""
    
    def test_basic_classification_error(self):
        """测试基础分类异常"""
        error = StockClassificationError("UNKNOWN.XX")
        
        assert "无法分类股票代码 'UNKNOWN.XX'" in str(error)
        assert error.error_code == "STOCK_CLASSIFICATION_FAILED"
        assert error.stock_code == "UNKNOWN.XX"
        assert error.reason is None
    
    def test_classification_error_with_reason(self):
        """测试带原因的分类异常"""
        error = StockClassificationError("123456.XX", "未知市场代码")
        
        assert "无法分类股票代码 '123456.XX': 未知市场代码" in str(error)
        assert error.reason == "未知市场代码"
        assert error.details["reason"] == "未知市场代码"
    
    def test_classification_error_suggestions(self):
        """测试分类异常建议"""
        error = StockClassificationError("INVALID")
        
        assert "请检查股票代码格式" in error.suggestions
        assert "默认分类" in error.suggestions


class TestLimitUpDetectionError:
    """测试涨停检测异常"""
    
    def test_basic_detection_error(self):
        """测试基础检测异常"""
        error = LimitUpDetectionError("000001.SZ")
        
        assert "股票 '000001.SZ' 涨停检测失败" in str(error)
        assert error.error_code == "LIMIT_UP_DETECTION_FAILED"
        assert error.stock_code == "000001.SZ"
        assert error.reason is None
        assert error.price_data is None
    
    def test_detection_error_with_reason(self):
        """测试带原因的检测异常"""
        error = LimitUpDetectionError("000001.SZ", "价格数据缺失")
        
        assert "股票 '000001.SZ' 涨停检测失败: 价格数据缺失" in str(error)
        assert error.reason == "价格数据缺失"
    
    def test_detection_error_with_price_data(self):
        """测试带价格数据的检测异常"""
        price_data = {
            "open": 10.0,
            "close": 11.0,
            "high": 11.0
        }
        
        error = LimitUpDetectionError(
            "000001.SZ", 
            "计算错误", 
            price_data=price_data
        )
        
        assert error.price_data == price_data
        assert error.details["price_data"] == price_data
    
    def test_detection_error_suggestions(self):
        """测试检测异常建议"""
        error = LimitUpDetectionError("000001.SZ")
        
        assert "请检查股票价格数据" in error.suggestions
        assert "开盘价、收盘价、最高价" in error.suggestions


class TestErrorHandlerIntegration:
    """测试错误处理器与涨停统计异常的集成"""
    
    @pytest.fixture
    def config(self):
        """创建测试配置"""
        return Config()
    
    @pytest.fixture
    def error_handler(self, config):
        """创建错误处理器"""
        return ErrorHandler(config)
    
    def test_should_retry_limit_up_errors(self, error_handler):
        """测试涨停统计异常的重试策略"""
        # 数据不足错误应该重试
        insufficient_error = InsufficientDataError("2024-01-15", "网络超时")
        assert error_handler.should_retry(insufficient_error) == False  # 默认不重试
        
        # 数据库错误应该重试
        db_error = LimitUpDatabaseError("save", "连接超时")
        assert error_handler.should_retry(db_error) == False  # 默认不重试
        
        # 无效日期错误不应该重试
        date_error = InvalidTradeDateError("invalid-date")
        assert error_handler.should_retry(date_error) == False
        
        # 分类错误不应该重试
        classification_error = StockClassificationError("UNKNOWN")
        assert error_handler.should_retry(classification_error) == False
    
    def test_create_limit_up_errors(self, error_handler):
        """测试创建涨停统计错误"""
        # 测试创建基础涨停统计错误
        error = error_handler.create_error(
            "limit_up_stats",
            "测试错误",
            error_code="TEST_ERROR",
            details={"key": "value"}
        )
        
        # 由于没有在error_classes中定义，应该返回QuickStockError
        assert isinstance(error, error_handler.create_error.__class__.__bases__[0])
    
    def test_log_limit_up_errors(self, error_handler):
        """测试记录涨停统计错误"""
        context = {
            "function": "get_limit_up_stats",
            "date": "2024-01-15"
        }
        
        # 测试记录各种涨停统计错误
        errors = [
            InvalidTradeDateError("2024-13-01"),
            InsufficientDataError("2024-01-15"),
            LimitUpDatabaseError("save"),
            StockClassificationError("UNKNOWN"),
            LimitUpDetectionError("000001.SZ")
        ]
        
        for error in errors:
            # 不应该抛出异常
            error_handler.log_error(error, context)
    
    @pytest.mark.asyncio
    async def test_handle_with_retry_limit_up_errors(self, error_handler):
        """测试涨停统计错误的重试处理"""
        
        # 模拟会抛出涨停统计异常的函数
        async def failing_function():
            raise InsufficientDataError("2024-01-15", "数据源不可用")
        
        # 应该不重试并直接抛出异常
        with pytest.raises(InsufficientDataError):
            await error_handler.handle_with_retry(failing_function)


class TestErrorScenarios:
    """测试各种错误场景"""
    
    def test_multiple_error_inheritance(self):
        """测试多重继承的错误类"""
        db_error = LimitUpDatabaseError("test")
        
        # 应该同时是DatabaseError和LimitUpStatsError的实例
        from quickstock.core.errors import DatabaseError
        assert isinstance(db_error, DatabaseError)
        assert isinstance(db_error, LimitUpStatsError)
    
    def test_error_chaining(self):
        """测试错误链"""
        original_error = sqlite3.IntegrityError("UNIQUE constraint failed")
        db_error = LimitUpDatabaseError("insert", original_error=original_error)
        
        assert db_error.original_error == original_error
        assert str(original_error) in str(db_error)
    
    def test_error_details_serialization(self):
        """测试错误详细信息的序列化"""
        price_data = {"open": 10.0, "close": 11.0}
        error = LimitUpDetectionError("000001.SZ", price_data=price_data)
        
        # 详细信息应该可以序列化
        details = error.details
        assert isinstance(details, dict)
        assert details["stock_code"] == "000001.SZ"
        assert details["price_data"] == price_data
    
    def test_user_friendly_messages(self):
        """测试用户友好的错误消息"""
        errors = [
            InvalidTradeDateError("2024-13-01", "月份无效"),
            InsufficientDataError("2024-01-15", "股票数据"),
            LimitUpDatabaseError("save", "磁盘空间不足"),
            StockClassificationError("UNKNOWN.XX", "未知交易所"),
            LimitUpDetectionError("000001.SZ", "价格数据异常")
        ]
        
        for error in errors:
            # 每个错误都应该有建议
            assert error.suggestions is not None
            assert len(error.suggestions) > 0
            
            # 错误消息应该包含关键信息
            message = str(error)
            assert len(message) > 0
            
            # 错误代码应该存在
            assert error.error_code is not None


if __name__ == "__main__":
    pytest.main([__file__])