"""
涨停统计服务测试

测试LimitUpStatsService的核心功能
"""

import pytest
import asyncio
import pandas as pd
from datetime import datetime
from unittest.mock import Mock, AsyncMock, patch, MagicMock

from quickstock.services.limit_up_stats_service import (
    LimitUpStatsService, LimitUpStatsError, InsufficientDataError
)
from quickstock.models import (
    LimitUpStatsRequest, LimitUpStats, StockDailyData, DataRequest
)
from quickstock.core.data_manager import DataManager
from quickstock.core.errors import DataSourceError


class TestLimitUpStatsService:
    """涨停统计服务测试类"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """模拟数据管理器"""
        mock_dm = Mock(spec=DataManager)
        mock_dm.get_data = AsyncMock()
        return mock_dm
    
    @pytest.fixture
    def service(self, mock_data_manager):
        """创建服务实例"""
        return LimitUpStatsService(mock_data_manager)
    
    @pytest.fixture
    def sample_stock_data(self):
        """示例股票数据"""
        return pd.DataFrame([
            {
                'ts_code': '000001.SZ',
                'trade_date': '20241015',
                'open': 10.00,
                'high': 11.00,
                'low': 10.00,
                'close': 11.00,
                'pre_close': 10.00,
                'change': 1.00,
                'pct_chg': 10.00,
                'vol': 1000000,
                'amount': 10500000.0,
                'name': '平安银行'
            },
            {
                'ts_code': '600000.SH',
                'trade_date': '20241015',
                'open': 8.00,
                'high': 8.80,
                'low': 8.00,
                'close': 8.80,
                'pre_close': 8.00,
                'change': 0.80,
                'pct_chg': 10.00,
                'vol': 2000000,
                'amount': 17200000.0,
                'name': '浦发银行'
            },
            {
                'ts_code': '688001.SH',
                'trade_date': '20241015',
                'open': 21.00,
                'high': 25.20,
                'low': 21.00,
                'close': 25.20,
                'pre_close': 21.00,
                'change': 4.20,
                'pct_chg': 20.00,
                'vol': 500000,
                'amount': 11900000.0,
                'name': '华兴源创'
            },
            {
                'ts_code': '430001.BJ',
                'trade_date': '20241015',
                'open': 12.00,
                'high': 15.60,
                'low': 12.00,
                'close': 15.60,
                'pre_close': 12.00,
                'change': 3.60,
                'pct_chg': 30.00,
                'vol': 300000,
                'amount': 4380000.0,
                'name': '北交所股票'
            },
            {
                'ts_code': '000002.SZ',
                'trade_date': '20241015',
                'open': 6.30,
                'high': 6.62,
                'low': 6.30,
                'close': 6.62,
                'pre_close': 6.30,
                'change': 0.32,
                'pct_chg': 5.08,
                'vol': 800000,
                'amount': 5248000.0,
                'name': 'ST万科'
            }
        ])
    
    @pytest.fixture
    def sample_basic_data(self):
        """示例股票基础数据"""
        return pd.DataFrame([
            {'ts_code': '000001.SZ', 'name': '平安银行'},
            {'ts_code': '600000.SH', 'name': '浦发银行'},
            {'ts_code': '688001.SH', 'name': '华兴源创'},
            {'ts_code': '430001.BJ', 'name': '北交所股票'},
            {'ts_code': '000002.SZ', 'name': 'ST万科'}
        ])
    
    @pytest.fixture
    def sample_request(self):
        """示例请求"""
        return LimitUpStatsRequest(
            trade_date='20241015',
            include_st=True,
            market_filter=None,
            force_refresh=False,
            save_to_db=True
        )
    
    @pytest.mark.asyncio
    async def test_get_daily_limit_up_stats_success(self, service, mock_data_manager, 
                                                   sample_stock_data, sample_basic_data, sample_request):
        """测试成功获取涨停统计"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [sample_stock_data, sample_basic_data]
        
        # 执行测试
        result = await service.get_daily_limit_up_stats(sample_request)
        
        # 验证结果
        assert isinstance(result, LimitUpStats)
        assert result.trade_date == '20241015'
        assert result.total > 0
        assert result.total == result.shanghai + result.shenzhen + result.star + result.beijing
        assert result.total == result.st + result.non_st
        assert len(result.limit_up_stocks) == result.total
        
        # 验证调用
        assert mock_data_manager.get_data.call_count == 2
    
    @pytest.mark.asyncio
    async def test_get_daily_limit_up_stats_no_data(self, service, mock_data_manager, sample_request):
        """测试无数据情况"""
        # 设置模拟数据为空
        mock_data_manager.get_data.return_value = pd.DataFrame()
        
        # 执行测试并验证异常
        with pytest.raises(InsufficientDataError) as exc_info:
            await service.get_daily_limit_up_stats(sample_request)
        
        assert '20241015' in str(exc_info.value)
        assert 'No stock data available' in exc_info.value.details['missing_data_info']['reason']
    
    @pytest.mark.asyncio
    async def test_get_daily_limit_up_stats_data_source_error(self, service, mock_data_manager, sample_request):
        """测试数据源错误"""
        # 设置模拟异常
        mock_data_manager.get_data.side_effect = DataSourceError("数据源连接失败")
        
        # 执行测试并验证异常
        with pytest.raises(LimitUpStatsError):
            await service.get_daily_limit_up_stats(sample_request)
    
    @pytest.mark.asyncio
    async def test_fetch_daily_stock_data(self, service, mock_data_manager, 
                                        sample_stock_data, sample_basic_data):
        """测试获取股票数据"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [sample_stock_data, sample_basic_data]
        
        # 执行测试
        result = await service._fetch_daily_stock_data('20241015')
        
        # 验证结果
        assert not result.empty
        assert len(result) == 5
        assert 'name' in result.columns
        assert all(result['name'].notna())
        
        # 验证数据请求
        calls = mock_data_manager.get_data.call_args_list
        assert len(calls) == 2
        
        # 验证第一个调用（股票日线数据）
        first_call = calls[0][0][0]
        assert first_call.data_type == 'stock_daily'
        assert first_call.start_date == '20241015'
        assert first_call.end_date == '20241015'
        
        # 验证第二个调用（股票基础信息）
        second_call = calls[1][0][0]
        assert second_call.data_type == 'stock_basic'
    
    @pytest.mark.asyncio
    async def test_fetch_daily_stock_data_no_basic_data(self, service, mock_data_manager, sample_stock_data):
        """测试无基础数据情况"""
        # 设置模拟数据：有日线数据，无基础数据
        mock_data_manager.get_data.side_effect = [sample_stock_data, pd.DataFrame()]
        
        # 执行测试
        result = await service._fetch_daily_stock_data('20241015')
        
        # 验证结果
        assert not result.empty
        assert 'name' in result.columns
        assert all(result['name'] == '未知')
    
    @pytest.mark.asyncio
    async def test_detect_limit_up_stocks(self, service, sample_request):
        """测试涨停检测"""
        # 创建测试数据
        stock_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ',
                'trade_date': '20241015',
                'open': 10.00,
                'high': 11.00,
                'low': 10.00,
                'close': 11.00,
                'pre_close': 10.00,
                'change': 1.00,
                'pct_chg': 10.00,
                'vol': 1000000,
                'amount': 10500000.0,
                'name': '平安银行'
            },
            {
                'ts_code': '600000.SH',
                'trade_date': '20241015',
                'open': 8.00,
                'high': 8.50,
                'low': 8.00,
                'close': 8.50,
                'pre_close': 8.00,
                'change': 0.50,
                'pct_chg': 6.25,
                'vol': 2000000,
                'amount': 16500000.0,
                'name': '浦发银行'
            }
        ])
        
        # 执行测试
        result = await service._detect_limit_up_stocks(stock_data, sample_request)
        
        # 验证结果
        assert isinstance(result, list)
        # 第一只股票应该被检测为涨停（收盘价=最高价=涨停价）
        # 第二只股票不应该被检测为涨停（涨幅不足10%）
        limit_up_codes = [stock.ts_code for stock in result]
        assert '000001.SZ' in limit_up_codes
        assert '600000.SH' not in limit_up_codes
    
    @pytest.mark.asyncio
    async def test_detect_limit_up_stocks_with_st_filter(self, service):
        """测试ST股票过滤"""
        # 创建包含ST股票的测试数据
        stock_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ',
                'trade_date': '20241015',
                'open': 10.00,
                'high': 11.00,
                'low': 10.00,
                'close': 11.00,
                'pre_close': 10.00,
                'change': 1.00,
                'pct_chg': 10.00,
                'vol': 1000000,
                'amount': 10500000.0,
                'name': '平安银行'
            },
            {
                'ts_code': '000002.SZ',
                'trade_date': '20241015',
                'open': 6.30,
                'high': 6.62,
                'low': 6.30,
                'close': 6.62,
                'pre_close': 6.30,
                'change': 0.32,
                'pct_chg': 5.08,
                'vol': 800000,
                'amount': 5248000.0,
                'name': 'ST万科'
            }
        ])
        
        # 测试包含ST股票
        request_with_st = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=True
        )
        result_with_st = await service._detect_limit_up_stocks(stock_data, request_with_st)
        
        # 测试不包含ST股票
        request_without_st = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=False
        )
        result_without_st = await service._detect_limit_up_stocks(stock_data, request_without_st)
        
        # 验证结果
        with_st_codes = [stock.ts_code for stock in result_with_st]
        without_st_codes = [stock.ts_code for stock in result_without_st]
        
        # 普通股票应该在两个结果中都存在
        assert '000001.SZ' in with_st_codes
        assert '000001.SZ' in without_st_codes
        
        # ST股票只应该在包含ST的结果中存在（如果它确实涨停了）
        # 注意：这里ST万科的涨幅是5.08%，符合ST股票5%的涨停标准
    
    @pytest.mark.asyncio
    async def test_classify_stocks_by_market(self, service):
        """测试股票市场分类"""
        # 创建测试数据
        limit_up_stocks = [
            StockDailyData(
                ts_code='000001.SZ',
                trade_date='20241015',
                open=10.00, high=11.00, low=10.00, close=11.00,
                pre_close=10.00, change=1.00, pct_chg=10.00,
                vol=1000000, amount=10500000.0, name='平安银行'
            ),
            StockDailyData(
                ts_code='600000.SH',
                trade_date='20241015',
                open=8.00, high=8.80, low=8.00, close=8.80,
                pre_close=8.00, change=0.80, pct_chg=10.00,
                vol=2000000, amount=17200000.0, name='浦发银行'
            ),
            StockDailyData(
                ts_code='688001.SH',
                trade_date='20241015',
                open=21.00, high=25.20, low=21.00, close=25.20,
                pre_close=21.00, change=4.20, pct_chg=20.00,
                vol=500000, amount=11900000.0, name='华兴源创'
            ),
            StockDailyData(
                ts_code='000002.SZ',
                trade_date='20241015',
                open=6.30, high=6.62, low=6.30, close=6.62,
                pre_close=6.30, change=0.32, pct_chg=5.08,
                vol=800000, amount=5248000.0, name='ST万科'
            )
        ]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 执行测试
        result = await service._classify_stocks_by_market(limit_up_stocks, request)
        
        # 验证结果
        assert isinstance(result, dict)
        assert 'shanghai' in result
        assert 'shenzhen' in result
        assert 'star' in result
        assert 'st' in result
        assert 'non_st' in result
        
        # 验证分类结果
        shanghai_codes = [stock.ts_code for stock in result['shanghai']]
        shenzhen_codes = [stock.ts_code for stock in result['shenzhen']]
        star_codes = [stock.ts_code for stock in result['star']]
        st_codes = [stock.ts_code for stock in result['st']]
        
        assert '600000.SH' in shanghai_codes
        assert '000001.SZ' in shenzhen_codes
        assert '688001.SH' in star_codes
        assert '000002.SZ' in st_codes
    
    def test_aggregate_statistics(self, service):
        """测试统计聚合"""
        # 创建分类后的股票数据
        classified_stocks = {
            'shanghai': [Mock(ts_code='600000.SH')],
            'shenzhen': [Mock(ts_code='000001.SZ'), Mock(ts_code='000002.SZ')],
            'star': [Mock(ts_code='688001.SH')],
            'beijing': [],
            'st': [Mock(ts_code='000002.SZ')],
            'non_st': [Mock(ts_code='600000.SH'), Mock(ts_code='000001.SZ'), Mock(ts_code='688001.SH')],
            'unknown': []
        }
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 执行测试
        result = service._aggregate_statistics(classified_stocks, request)
        
        # 验证结果
        assert result['total'] == 4  # 1 + 2 + 1 + 0
        assert result['shanghai'] == 1
        assert result['shenzhen'] == 2
        assert result['star'] == 1
        assert result['beijing'] == 0
        assert result['st'] == 1
        assert result['non_st'] == 3
        assert len(result['limit_up_stocks']) == 4
        assert len(result['market_breakdown']) >= 4
        
        # 验证聚合元数据
        assert 'aggregation_metadata' in result
        metadata = result['aggregation_metadata']
        assert 'consistency_check' in metadata
        assert 'final_validation' in metadata
    
    def test_validate_statistics_consistency(self, service):
        """测试统计一致性验证"""
        classified_stocks = {
            'shanghai': [Mock(ts_code='600000.SH')],
            'shenzhen': [Mock(ts_code='000001.SZ')],
            'star': [],
            'beijing': [],
            'st': [],
            'non_st': [Mock(ts_code='600000.SH'), Mock(ts_code='000001.SZ')],
            'unknown': []
        }
        
        # 测试一致的情况
        result = service._validate_statistics_consistency(2, 0, 2, classified_stocks)
        assert result['is_consistent'] is True
        assert len(result['issues']) == 0
        
        # 测试不一致的情况
        result = service._validate_statistics_consistency(3, 1, 1, classified_stocks)
        assert result['is_consistent'] is False
        assert len(result['issues']) > 0
        assert result['corrected_total'] == 2  # 应该修正为实际数量
    
    def test_build_market_breakdown(self, service):
        """测试市场分解构建"""
        classified_stocks = {
            'shanghai': [Mock(ts_code='600000.SH'), Mock(ts_code='600001.SH')],
            'shenzhen': [Mock(ts_code='000001.SZ')],
            'star': [Mock(ts_code='688001.SH')],
            'beijing': [],
            'unknown': [Mock(ts_code='INVALID.XX')]
        }
        
        result = service._build_market_breakdown(classified_stocks)
        
        # 验证结果
        assert 'all_stocks' in result
        assert 'market_breakdown' in result
        assert 'market_stats' in result
        
        # 验证股票总数
        assert result['total_unique_stocks'] == 5
        assert len(result['all_stocks']) == 5
        
        # 验证市场分解
        breakdown = result['market_breakdown']
        assert len(breakdown['shanghai']) == 2
        assert len(breakdown['shenzhen']) == 1
        assert len(breakdown['star']) == 1
        assert len(breakdown['beijing']) == 0
        assert len(breakdown['unknown']) == 1
        
        # 验证市场统计
        stats = result['market_stats']
        assert stats['shanghai']['count'] == 2
        assert stats['shanghai']['percentage'] == 40.0  # 2/5 * 100
    
    def test_validate_final_statistics(self, service):
        """测试最终统计验证"""
        # 测试正常情况
        all_stocks = ['600000.SH', '000001.SZ', '688001.SH']
        market_breakdown = {
            'shanghai': ['600000.SH'],
            'shenzhen': ['000001.SZ'],
            'star': ['688001.SH']
        }
        
        result = service._validate_final_statistics(all_stocks, 3, market_breakdown)
        assert result['is_valid'] is True
        assert len(result['issues']) == 0
        
        # 测试不一致情况
        result = service._validate_final_statistics(all_stocks, 5, market_breakdown)
        assert result['is_valid'] is False
        assert len(result['issues']) > 0
        assert result['corrected_total'] == 3
        
        # 测试重复股票
        duplicate_stocks = ['600000.SH', '000001.SZ', '600000.SH']
        result = service._validate_final_statistics(duplicate_stocks, 3, market_breakdown)
        assert result['is_valid'] is False
        assert len(result['corrected_stocks']) == 2  # 去重后
    
    def test_get_aggregation_summary(self, service):
        """测试聚合摘要"""
        stats = {
            'total': 100,
            'non_st': 80,
            'st': 20,
            'shanghai': 40,
            'shenzhen': 30,
            'star': 20,
            'beijing': 10,
            'aggregation_metadata': {
                'consistency_check': {'is_consistent': True, 'issues': []},
                'final_validation': {'is_valid': True, 'issues': []}
            }
        }
        
        summary = service.get_aggregation_summary(stats)
        
        # 验证基础统计
        assert summary['basic_stats']['total'] == 100
        assert summary['basic_stats']['non_st'] == 80
        assert summary['basic_stats']['st'] == 20
        
        # 验证百分比计算
        assert summary['percentages']['non_st_pct'] == 80.0
        assert summary['percentages']['st_pct'] == 20.0
        assert summary['percentages']['shanghai_pct'] == 40.0
        
        # 验证数据质量
        assert summary['data_quality']['consistency_passed'] is True
        assert summary['data_quality']['validation_passed'] is True
    
    def test_analyze_market_concentration(self, service):
        """测试市场集中度分析"""
        # 测试高度集中的情况
        stats = {
            'total': 100,
            'shanghai': 80,
            'shenzhen': 15,
            'star': 5,
            'beijing': 0
        }
        
        result = service.analyze_market_concentration(stats)
        
        assert result['concentration_level'] == '高度集中'
        assert result['dominant_market']['name'] == 'shanghai'
        assert result['dominant_market']['share'] == 0.8
        assert result['concentration_index'] > 0.5
        
        # 测试分散的情况
        balanced_stats = {
            'total': 100,
            'shanghai': 25,
            'shenzhen': 25,
            'star': 25,
            'beijing': 25
        }
        
        balanced_result = service.analyze_market_concentration(balanced_stats)
        assert balanced_result['concentration_level'] == '分散'
        assert balanced_result['concentration_index'] == 0.25  # 4 * (0.25)^2
    
    def test_analyze_st_stock_impact(self, service):
        """测试ST股票影响分析"""
        # 测试高ST占比
        high_st_stats = {
            'total': 100,
            'st': 40,
            'non_st': 60
        }
        
        result = service.analyze_st_stock_impact(high_st_stats)
        assert result['impact_level'] == '高影响'
        assert result['st_ratio'] == 0.4
        assert len(result['recommendations']) > 0
        
        # 测试低ST占比
        low_st_stats = {
            'total': 100,
            'st': 3,
            'non_st': 97
        }
        
        low_result = service.analyze_st_stock_impact(low_st_stats)
        assert low_result['impact_level'] == '极低影响'
        assert low_result['st_ratio'] == 0.03
    
    def test_generate_market_insights(self, service):
        """测试市场洞察生成"""
        stats = {
            'total': 150,
            'non_st': 120,
            'st': 30,
            'shanghai': 60,
            'shenzhen': 50,
            'star': 30,
            'beijing': 10,
            'limit_up_stocks': ['600000.SH'] * 150,  # 模拟数据
            'market_breakdown': {
                'shanghai': ['600000.SH'] * 60,
                'shenzhen': ['000001.SZ'] * 50,
                'star': ['688001.SH'] * 30,
                'beijing': ['430001.BJ'] * 10
            },
            'aggregation_metadata': {
                'consistency_check': {'is_consistent': True, 'issues': []},
                'final_validation': {'is_valid': True, 'issues': []}
            }
        }
        
        insights = service.generate_market_insights(stats)
        
        # 验证结构
        assert 'summary' in insights
        assert 'concentration_analysis' in insights
        assert 'st_analysis' in insights
        assert 'market_assessment' in insights
        assert 'key_insights' in insights
        assert 'generated_at' in insights
        
        # 验证内容
        assert isinstance(insights['key_insights'], list)
        assert len(insights['key_insights']) > 0
        
        # 验证市场评估
        assessment = insights['market_assessment']
        assert 'overall_score' in assessment
        assert 'market_grade' in assessment
        assert 'market_condition' in assessment
    
    def test_is_valid_stock_code(self, service):
        """测试股票代码格式验证"""
        # 有效代码
        assert service._is_valid_stock_code('000001.SZ') is True
        assert service._is_valid_stock_code('600000.SH') is True
        assert service._is_valid_stock_code('688001.SH') is True
        assert service._is_valid_stock_code('430001.BJ') is True
        
        # 无效代码
        assert service._is_valid_stock_code('INVALID') is False
        assert service._is_valid_stock_code('000001') is False
        assert service._is_valid_stock_code('000001.XX') is False
        assert service._is_valid_stock_code('') is False
    
    def test_validate_data_completeness(self, service, sample_stock_data):
        """测试数据完整性验证"""
        # 测试完整数据
        result = service.validate_data_completeness(sample_stock_data)
        assert result['is_complete'] is True
        assert result['total_stocks'] == 5
        
        # 测试空数据
        empty_result = service.validate_data_completeness(pd.DataFrame())
        assert empty_result['is_complete'] is False
        assert '股票数据为空' in empty_result['warnings']
        
        # 测试缺少字段的数据
        incomplete_data = sample_stock_data.drop(columns=['close'])
        incomplete_result = service.validate_data_completeness(incomplete_data)
        assert incomplete_result['is_complete'] is False
        assert 'close' in incomplete_result['missing_fields']
    
    def test_get_service_stats(self, service):
        """测试服务统计信息"""
        # 初始状态
        stats = service.get_service_stats()
        assert stats['total_requests'] == 0
        assert stats['success_rate'] == 0.0
        
        # 模拟一些统计数据
        service._stats['total_requests'] = 10
        service._stats['successful_requests'] = 8
        service._stats['total_processing_time'] = 20.0
        
        stats = service.get_service_stats()
        assert stats['total_requests'] == 10
        assert stats['success_rate'] == 0.8
        assert stats['average_processing_time'] == 2.5
    
    def test_reset_stats(self, service):
        """测试重置统计信息"""
        # 设置一些统计数据
        service._stats['total_requests'] = 10
        service._stats['successful_requests'] = 8
        
        # 重置
        service.reset_stats()
        
        # 验证重置结果
        stats = service.get_service_stats()
        assert stats['total_requests'] == 0
        assert stats['successful_requests'] == 0
    
    @pytest.mark.asyncio
    async def test_health_check(self, service, mock_data_manager):
        """测试健康检查"""
        # 执行健康检查
        result = await service.health_check()
        
        # 验证结果
        assert 'service_status' in result
        assert 'components' in result
        assert 'last_check' in result
        assert isinstance(result['issues'], list)
        
        # 验证组件状态
        assert 'data_manager' in result['components']
        assert 'stock_classifier' in result['components']
        assert 'limit_up_detector' in result['components']
    
    @pytest.mark.asyncio
    async def test_request_validation(self, service, mock_data_manager):
        """测试请求参数验证"""
        # 测试无效日期格式
        with pytest.raises(ValueError):
            invalid_request = LimitUpStatsRequest(
                trade_date='invalid_date'
            )
        
        # 测试未来日期
        future_date = (datetime.now().replace(year=datetime.now().year + 1)).strftime('%Y%m%d')
        with pytest.raises(ValueError):
            future_request = LimitUpStatsRequest(
                trade_date=future_date
            )
    
    @pytest.mark.asyncio
    async def test_market_filter(self, service, mock_data_manager, sample_stock_data, sample_basic_data):
        """测试市场过滤功能"""
        # 设置模拟数据
        mock_data_manager.get_data.side_effect = [sample_stock_data, sample_basic_data]
        
        # 创建只包含深圳市场的请求
        filtered_request = LimitUpStatsRequest(
            trade_date='20241015',
            market_filter=['shenzhen']
        )
        
        # 执行测试
        result = await service.get_daily_limit_up_stats(filtered_request)
        
        # 验证结果：只应该包含深圳市场的股票
        assert result.shenzhen > 0
        assert result.shanghai == 0  # 应该被过滤掉
        assert result.star == 0      # 应该被过滤掉
        assert result.beijing == 0   # 应该被过滤掉
    
    @pytest.mark.asyncio
    async def test_error_handling_in_classification(self, service):
        """测试分类过程中的错误处理"""
        # 创建包含边缘情况的股票列表（使用有效格式但可能分类困难的代码）
        edge_case_stocks = [
            StockDailyData(
                ts_code='999999.SZ',  # 边缘股票代码
                trade_date='20241015',
                open=10.00, high=11.00, low=10.00, close=11.00,
                pre_close=10.00, change=1.00, pct_chg=10.00,
                vol=1000000, amount=10500000.0, name='边缘股票'
            )
        ]
        
        request = LimitUpStatsRequest(trade_date='20241015')
        
        # 执行测试（应该不抛出异常，能够处理边缘情况）
        result = await service._classify_stocks_by_market(edge_case_stocks, request)
        
        # 验证错误处理
        assert isinstance(result, dict)
        # 股票应该被分类到某个类别中
        total_classified = sum(len(stocks) for stocks in result.values())
        assert total_classified >= len(edge_case_stocks)


class TestLimitUpStatsServiceIntegration:
    """涨停统计服务集成测试"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置"""
        config = Mock()
        config.cache_enabled = True
        config.cache_expire_hours = 24
        config.max_concurrent_requests = 5
        config.max_retries = 3
        return config
    
    @pytest.fixture
    def integration_service(self, mock_config):
        """创建集成测试服务"""
        # 创建真实的数据管理器（但使用模拟的数据源）
        data_manager = DataManager(mock_config)
        
        # 模拟数据源管理器
        mock_source_manager = Mock()
        mock_source_manager.fetch_data = AsyncMock()
        data_manager.source_manager = mock_source_manager
        
        return LimitUpStatsService(data_manager)
    
    @pytest.mark.asyncio
    async def test_end_to_end_workflow(self, integration_service):
        """测试端到端工作流程"""
        # 准备测试数据
        stock_daily_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ',
                'trade_date': '20241015',
                'open': 10.00,
                'high': 11.00,
                'low': 10.00,
                'close': 11.00,
                'pre_close': 10.00,
                'change': 1.00,
                'pct_chg': 10.00,
                'vol': 1000000,
                'amount': 10500000.0
            }
        ])
        
        stock_basic_data = pd.DataFrame([
            {'ts_code': '000001.SZ', 'name': '平安银行'}
        ])
        
        # 设置模拟数据源返回
        integration_service.data_manager.source_manager.fetch_data.side_effect = [
            stock_daily_data, stock_basic_data
        ]
        
        # 创建请求
        request = LimitUpStatsRequest(
            trade_date='20241015',
            include_st=True
        )
        
        # 执行完整流程
        result = await integration_service.get_daily_limit_up_stats(request)
        
        # 验证结果
        assert isinstance(result, LimitUpStats)
        assert result.trade_date == '20241015'
        assert result.total >= 0
        
        # 验证服务统计
        stats = integration_service.get_service_stats()
        assert stats['total_requests'] == 1
        assert stats['successful_requests'] == 1
    
    @pytest.mark.asyncio
    async def test_concurrent_requests(self, integration_service):
        """测试并发请求处理"""
        # 准备测试数据
        stock_data = pd.DataFrame([
            {
                'ts_code': '000001.SZ',
                'trade_date': '20241015',
                'open': 10.00,
                'high': 11.00,
                'low': 10.00,
                'close': 11.00,
                'pre_close': 10.00,
                'change': 1.00,
                'pct_chg': 10.00,
                'vol': 1000000,
                'amount': 10500000.0
            }
        ])
        
        basic_data = pd.DataFrame([
            {'ts_code': '000001.SZ', 'name': '平安银行'}
        ])
        
        # 设置模拟数据源
        integration_service.data_manager.source_manager.fetch_data.side_effect = [
            stock_data, basic_data,  # 第一个请求
            stock_data, basic_data,  # 第二个请求
            stock_data, basic_data   # 第三个请求
        ]
        
        # 创建多个请求
        requests = [
            LimitUpStatsRequest(trade_date='20241015'),
            LimitUpStatsRequest(trade_date='20241015'),
            LimitUpStatsRequest(trade_date='20241015')
        ]
        
        # 并发执行
        tasks = [
            integration_service.get_daily_limit_up_stats(req) 
            for req in requests
        ]
        results = await asyncio.gather(*tasks)
        
        # 验证结果
        assert len(results) == 3
        for result in results:
            assert isinstance(result, LimitUpStats)
            assert result.trade_date == '20241015'
        
        # 验证服务统计
        stats = integration_service.get_service_stats()
        assert stats['total_requests'] == 3
        assert stats['successful_requests'] == 3


if __name__ == '__main__':
    pytest.main([__file__, '-v'])