"""
财务报告性能测试

测试财务报告服务的性能优化和监控功能
"""

import pytest
import asyncio
import time
import pandas as pd
import numpy as np
from unittest.mock import Mock, AsyncMock, patch
from datetime import datetime, timedelta

import sys
from pathlib import Path
sys.path.insert(0, str(Path(__file__).parent.parent))

from quickstock.services.financial_reports_service import FinancialReportsService
from quickstock.models import FinancialReportsRequest, FinancialReport
from quickstock.core.data_manager import DataManager
from quickstock.core.cache import CacheLayer
from quickstock.utils.performance_monitor import PerformanceMonitor, AdaptiveRateLimiter
from quickstock.utils.memory_optimizer import ChunkedDataProcessor, DataFrameOptimizer, MemoryMonitor


class TestPerformanceMonitor:
    """性能监控器测试"""
    
    def test_performance_monitor_initialization(self):
        """测试性能监控器初始化"""
        monitor = PerformanceMonitor()
        
        assert monitor.max_metrics_history == 10000
        assert len(monitor.metrics_history) == 0
        assert len(monitor.operation_stats) == 0
        assert monitor.rate_limiter is not None
    
    @pytest.mark.asyncio
    async def test_measure_operation_success(self):
        """测试成功操作的性能测量"""
        monitor = PerformanceMonitor()
        
        async with monitor.measure_operation("test_operation", data_size=1000) as m:
            await asyncio.sleep(0.1)  # 模拟操作时间
        
        # 检查指标记录
        assert len(monitor.metrics_history) == 1
        metrics = monitor.metrics_history[0]
        
        assert metrics.operation == "test_operation"
        assert metrics.success is True
        assert metrics.duration >= 0.1
        assert metrics.data_size == 1000
        assert metrics.error_type is None
        
        # 检查操作统计
        stats = monitor.get_operation_stats("test_operation")
        assert "test_operation" in stats
        assert stats["test_operation"]["total_requests"] == 1
        assert stats["test_operation"]["successful_requests"] == 1
        assert stats["test_operation"]["failed_requests"] == 0
    
    @pytest.mark.asyncio
    async def test_measure_operation_failure(self):
        """测试失败操作的性能测量"""
        monitor = PerformanceMonitor()
        
        with pytest.raises(ValueError):
            async with monitor.measure_operation("test_operation") as m:
                raise ValueError("Test error")
        
        # 检查指标记录
        assert len(monitor.metrics_history) == 1
        metrics = monitor.metrics_history[0]
        
        assert metrics.operation == "test_operation"
        assert metrics.success is False
        assert metrics.error_type == "ValueError"
        
        # 检查操作统计
        stats = monitor.get_operation_stats("test_operation")
        assert stats["test_operation"]["failed_requests"] == 1
    
    def test_performance_summary(self):
        """测试性能摘要"""
        monitor = PerformanceMonitor()
        
        # 添加一些测试指标
        from quickstock.utils.performance_monitor import PerformanceMetrics
        
        # 成功操作
        monitor.record_metrics(PerformanceMetrics(
            operation="test_op",
            start_time=time.time(),
            end_time=time.time() + 1,
            duration=1.0,
            success=True,
            cache_hit=True
        ))
        
        # 失败操作
        monitor.record_metrics(PerformanceMetrics(
            operation="test_op",
            start_time=time.time(),
            end_time=time.time() + 0.5,
            duration=0.5,
            success=False,
            error_type="TestError"
        ))
        
        summary = monitor.get_performance_summary()
        
        assert summary["total_operations"] == 2
        assert summary["successful_operations"] == 1
        assert summary["failed_operations"] == 1
        assert summary["success_rate"] == 0.5
        assert summary["cache_hits"] == 1
        assert summary["cache_hit_rate"] == 0.5


class TestAdaptiveRateLimiter:
    """自适应速率限制器测试"""
    
    def test_rate_limiter_initialization(self):
        """测试速率限制器初始化"""
        limiter = AdaptiveRateLimiter(initial_rate=2.0, min_rate=0.5, max_rate=10.0)
        
        assert limiter.current_rate == 2.0
        assert limiter.min_rate == 0.5
        assert limiter.max_rate == 10.0
    
    @pytest.mark.asyncio
    async def test_rate_limiting(self):
        """测试速率限制功能"""
        limiter = AdaptiveRateLimiter(initial_rate=10.0)  # 10 requests/second
        
        start_time = time.time()
        
        # 连续请求两次
        await limiter.acquire()
        await limiter.acquire()
        
        elapsed = time.time() - start_time
        
        # 应该至少等待 0.1 秒 (1/10)
        assert elapsed >= 0.1
    
    def test_rate_adjustment_on_errors(self):
        """测试错误时的速率调整"""
        limiter = AdaptiveRateLimiter(initial_rate=5.0)
        
        # 记录多个失败响应
        for _ in range(15):
            limiter.record_response(1.0, False, "RateLimitError")
        
        # 速率应该降低
        assert limiter.current_rate < 5.0
    
    def test_rate_adjustment_on_good_performance(self):
        """测试良好性能时的速率调整"""
        limiter = AdaptiveRateLimiter(initial_rate=2.0)
        
        # 记录多个成功响应
        for _ in range(15):
            limiter.record_response(0.5, True)
        
        # 速率应该提高
        assert limiter.current_rate > 2.0
    
    def test_rate_bounds(self):
        """测试速率边界"""
        limiter = AdaptiveRateLimiter(initial_rate=5.0, min_rate=1.0, max_rate=10.0)
        
        # 记录大量错误，速率应该不低于最小值
        for _ in range(50):
            limiter.record_response(10.0, False, "Error")
        
        assert limiter.current_rate >= limiter.min_rate
        
        # 重置并记录大量成功，速率应该不高于最大值
        limiter.current_rate = 5.0
        for _ in range(50):
            limiter.record_response(0.1, True)
        
        assert limiter.current_rate <= limiter.max_rate


class TestMemoryOptimizer:
    """内存优化器测试"""
    
    def test_memory_monitor(self):
        """测试内存监控器"""
        monitor = MemoryMonitor()
        
        usage = monitor.get_memory_usage()
        
        assert usage.total_mb > 0
        assert usage.available_mb > 0
        assert usage.used_mb > 0
        assert 0 <= usage.percent <= 100
        assert usage.process_mb > 0
    
    def test_dataframe_optimizer_numeric(self):
        """测试DataFrame数值类型优化"""
        # 创建测试数据
        df = pd.DataFrame({
            'small_int': np.random.randint(0, 100, 1000),
            'large_int': np.random.randint(0, 1000000, 1000),
            'float_data': np.random.random(1000) * 100
        })
        
        original_memory = df.memory_usage(deep=True).sum()
        
        # 优化数据类型
        optimized_df = DataFrameOptimizer.optimize_dtypes(df)
        optimized_memory = optimized_df.memory_usage(deep=True).sum()
        
        # 内存使用应该减少
        assert optimized_memory <= original_memory
        
        # 数据应该保持一致
        pd.testing.assert_frame_equal(df.astype(optimized_df.dtypes), optimized_df)
    
    def test_dataframe_optimizer_categorical(self):
        """测试DataFrame分类变量优化"""
        # 创建包含重复字符串的数据
        categories = ['A', 'B', 'C', 'D', 'E']
        df = pd.DataFrame({
            'category': np.random.choice(categories, 1000),
            'text': ['text_' + str(i % 10) for i in range(1000)]
        })
        
        original_memory = df.memory_usage(deep=True).sum()
        
        # 优化数据类型
        optimized_df = DataFrameOptimizer.optimize_dtypes(df, categorical_threshold=20)
        optimized_memory = optimized_df.memory_usage(deep=True).sum()
        
        # 内存使用应该减少
        assert optimized_memory < original_memory
        
        # 分类列应该被转换为category类型
        assert optimized_df['category'].dtype.name == 'category'
    
    def test_chunked_data_processor(self):
        """测试分块数据处理器"""
        processor = ChunkedDataProcessor(chunk_size=100, memory_limit_mb=1.0)
        
        # 创建大数据集
        large_df = pd.DataFrame({
            'value': np.random.random(1000),
            'category': np.random.choice(['A', 'B', 'C'], 1000)
        })
        
        # 定义处理函数
        def process_chunk(chunk):
            return chunk.groupby('category')['value'].mean().reset_index()
        
        # 处理数据
        result = processor.process_dataframe_chunks(large_df, process_chunk)
        
        # 验证结果
        assert isinstance(result, pd.DataFrame)
        assert len(result) > 0
        assert 'category' in result.columns
        assert 'value' in result.columns


class TestFinancialReportsServicePerformance:
    """财务报告服务性能测试"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """模拟数据管理器"""
        manager = Mock(spec=DataManager)
        manager.get_data = AsyncMock()
        return manager
    
    @pytest.fixture
    def mock_cache_layer(self):
        """模拟缓存层"""
        cache = Mock(spec=CacheLayer)
        cache.get = AsyncMock(return_value=None)
        cache.set = AsyncMock()
        cache.get_stats = Mock(return_value={
            'hit_stats': {'hit_rate': 0.75, 'memory_hits': 100, 'sqlite_hits': 50, 'misses': 50}
        })
        return cache
    
    @pytest.fixture
    def service(self, mock_data_manager, mock_cache_layer):
        """创建财务报告服务实例"""
        return FinancialReportsService(mock_data_manager, mock_cache_layer)
    
    def test_service_initialization_with_performance_monitoring(self, service):
        """测试服务初始化包含性能监控"""
        assert hasattr(service, 'performance_monitor')
        assert hasattr(service, 'data_processor')
        assert isinstance(service.performance_monitor, PerformanceMonitor)
        assert isinstance(service.data_processor, ChunkedDataProcessor)
    
    @pytest.mark.asyncio
    async def test_get_financial_reports_with_performance_monitoring(self, service, mock_data_manager):
        """测试带性能监控的财务报告获取"""
        # 准备测试数据
        test_data = pd.DataFrame({
            'ts_code': ['000001.SZ'] * 4,
            'report_date': ['20231231', '20230930', '20230630', '20230331'],
            'report_type': ['A', 'Q3', 'Q2', 'Q1'],
            'total_revenue': [1000000, 750000, 500000, 250000],
            'net_profit': [100000, 75000, 50000, 25000],
            'total_assets': [5000000, 4800000, 4600000, 4400000],
            'total_liabilities': [3000000, 2900000, 2800000, 2700000],
            'shareholders_equity': [2000000, 1900000, 1800000, 1700000],
            'operating_cash_flow': [150000, 100000, 75000, 50000],
            'eps': [1.0, 0.75, 0.5, 0.25],
            'roe': [0.05, 0.04, 0.03, 0.02]
        })
        
        mock_data_manager.get_data.return_value = test_data
        
        # 创建请求
        request = FinancialReportsRequest(
            ts_code='000001.SZ',
            start_date='20230101',
            end_date='20231231'
        )
        
        # 执行请求
        result = await service.get_financial_reports(request)
        
        # 验证结果
        assert len(result) == 4
        assert all(isinstance(report, FinancialReport) for report in result)
        
        # 验证性能监控
        stats = service.get_performance_stats()
        assert 'service_stats' in stats
        assert 'performance_monitor' in stats
        assert 'cache_stats' in stats
        
        # 验证服务统计
        service_stats = stats['service_stats']
        assert service_stats['total_requests'] > 0
        assert service_stats['successful_requests'] > 0
    
    @pytest.mark.asyncio
    async def test_batch_processing_performance(self, service, mock_data_manager):
        """测试批处理性能"""
        # 准备批量测试数据
        stock_codes = [f'00000{i}.SZ' for i in range(1, 11)]  # 10只股票
        
        # 为每只股票准备数据
        def create_stock_data(ts_code):
            return pd.DataFrame({
                'ts_code': [ts_code] * 2,
                'report_date': ['20231231', '20230930'],
                'report_type': ['A', 'Q3'],
                'total_revenue': [1000000, 750000],
                'net_profit': [100000, 75000],
                'total_assets': [5000000, 4800000],
                'total_liabilities': [3000000, 2900000],
                'shareholders_equity': [2000000, 1900000],
                'operating_cash_flow': [150000, 100000],
                'eps': [1.0, 0.75],
                'roe': [0.05, 0.04]
            })
        
        # 模拟数据管理器返回不同股票的数据
        def mock_get_data(request):
            return create_stock_data(request.ts_code)
        
        mock_data_manager.get_data.side_effect = mock_get_data
        
        # 执行批量请求
        start_time = time.time()
        result = await service.get_batch_financial_data(
            stock_codes=stock_codes,
            data_types=['financial_reports'],
            start_date='20230101',
            end_date='20231231'
        )
        processing_time = time.time() - start_time
        
        # 验证结果
        assert result['success_count'] == len(stock_codes)
        assert result['failed_count'] == 0
        assert result['total_count'] == len(stock_codes)
        assert processing_time < 10.0  # 应该在10秒内完成
        
        # 验证性能统计
        stats = service.get_performance_stats()
        assert stats['service_stats']['batch_requests'] > 0
    
    def test_performance_stats_collection(self, service):
        """测试性能统计收集"""
        # 获取初始统计
        initial_stats = service.get_performance_stats()
        
        # 验证统计结构
        assert 'service_stats' in initial_stats
        assert 'performance_monitor' in initial_stats
        assert 'cache_stats' in initial_stats
        assert 'memory_processor' in initial_stats
        
        # 验证服务统计字段
        service_stats = initial_stats['service_stats']
        expected_fields = [
            'total_requests', 'successful_requests', 'failed_requests',
            'cache_hits', 'cache_misses', 'batch_requests', 'retry_attempts'
        ]
        for field in expected_fields:
            assert field in service_stats
        
        # 验证内存处理器配置
        memory_config = initial_stats['memory_processor']
        assert 'chunk_size' in memory_config
        assert 'memory_limit_mb' in memory_config
    
    def test_performance_stats_reset(self, service):
        """测试性能统计重置"""
        # 修改一些统计数据
        service._stats['total_requests'] = 100
        service._stats['successful_requests'] = 90
        
        # 重置统计
        service.reset_performance_stats()
        
        # 验证重置后的状态
        stats = service.get_performance_stats()
        service_stats = stats['service_stats']
        
        assert service_stats['total_requests'] == 0
        assert service_stats['successful_requests'] == 0


class TestPerformanceBenchmarks:
    """性能基准测试"""
    
    @pytest.mark.asyncio
    async def test_large_dataset_processing_benchmark(self):
        """大数据集处理基准测试"""
        # 创建大数据集 (10,000 行)
        large_df = pd.DataFrame({
            'ts_code': ['000001.SZ'] * 10000,
            'report_date': pd.date_range('2020-01-01', periods=10000, freq='D').strftime('%Y%m%d'),
            'total_revenue': np.random.random(10000) * 1000000,
            'net_profit': np.random.random(10000) * 100000,
            'total_assets': np.random.random(10000) * 5000000,
            'eps': np.random.random(10000) * 10,
            'roe': np.random.random(10000) * 0.2
        })
        
        # 测试数据类型优化性能
        start_time = time.time()
        optimized_df = DataFrameOptimizer.optimize_dtypes(large_df)
        optimization_time = time.time() - start_time
        
        # 验证优化效果
        original_memory = large_df.memory_usage(deep=True).sum()
        optimized_memory = optimized_df.memory_usage(deep=True).sum()
        memory_reduction = (original_memory - optimized_memory) / original_memory
        
        print(f"数据类型优化性能:")
        print(f"  处理时间: {optimization_time:.3f}s")
        print(f"  内存减少: {memory_reduction:.1%}")
        print(f"  原始内存: {original_memory / (1024*1024):.1f}MB")
        print(f"  优化后内存: {optimized_memory / (1024*1024):.1f}MB")
        
        # 性能要求
        assert optimization_time < 5.0  # 优化时间应该少于5秒
        assert memory_reduction > 0  # 应该有内存减少
    
    @pytest.mark.asyncio
    async def test_concurrent_requests_benchmark(self):
        """并发请求基准测试"""
        monitor = PerformanceMonitor()
        
        async def simulate_request(request_id):
            async with monitor.measure_operation(f"request_{request_id}"):
                # 模拟API调用延迟
                await asyncio.sleep(0.1 + np.random.random() * 0.1)
                
                # 模拟一些失败
                if np.random.random() < 0.1:  # 10% 失败率
                    raise Exception("Simulated error")
        
        # 并发执行100个请求
        start_time = time.time()
        tasks = [simulate_request(i) for i in range(100)]
        
        # 收集结果，忽略异常
        results = await asyncio.gather(*tasks, return_exceptions=True)
        total_time = time.time() - start_time
        
        # 分析结果
        successful_requests = sum(1 for r in results if not isinstance(r, Exception))
        failed_requests = len(results) - successful_requests
        
        # 获取性能统计
        stats = monitor.get_performance_summary()
        
        print(f"并发请求基准测试:")
        print(f"  总时间: {total_time:.3f}s")
        print(f"  成功请求: {successful_requests}")
        print(f"  失败请求: {failed_requests}")
        print(f"  平均响应时间: {stats['average_response_time']:.3f}s")
        print(f"  成功率: {stats['success_rate']:.1%}")
        
        # 性能要求
        assert total_time < 15.0  # 总时间应该少于15秒（考虑并发）
        assert stats['success_rate'] > 0.8  # 成功率应该大于80%
    
    def test_memory_usage_benchmark(self):
        """内存使用基准测试"""
        monitor = MemoryMonitor()
        
        # 记录初始内存使用
        initial_usage = monitor.get_memory_usage()
        
        # 创建大量数据
        large_data = []
        for i in range(1000):
            df = pd.DataFrame({
                'data': np.random.random(1000),
                'category': np.random.choice(['A', 'B', 'C'], 1000)
            })
            large_data.append(df)
        
        # 记录峰值内存使用
        peak_usage = monitor.get_memory_usage()
        
        # 清理数据
        del large_data
        import gc
        gc.collect()
        
        # 记录清理后内存使用
        final_usage = monitor.get_memory_usage()
        
        print(f"内存使用基准测试:")
        print(f"  初始内存: {initial_usage.process_mb:.1f}MB")
        print(f"  峰值内存: {peak_usage.process_mb:.1f}MB")
        print(f"  最终内存: {final_usage.process_mb:.1f}MB")
        print(f"  内存增长: {peak_usage.process_mb - initial_usage.process_mb:.1f}MB")
        print(f"  内存回收: {peak_usage.process_mb - final_usage.process_mb:.1f}MB")
        
        # 验证内存管理
        memory_growth = peak_usage.process_mb - initial_usage.process_mb
        memory_recovered = peak_usage.process_mb - final_usage.process_mb
        recovery_rate = memory_recovered / memory_growth if memory_growth > 0 else 0
        
        assert recovery_rate > 0.5  # 至少回收50%的内存


if __name__ == "__main__":
    pytest.main([__file__, "-v", "--tb=short"])