"""
性能监控集成测试

测试性能监控功能与财务报告服务的集成
"""

import pytest
import asyncio
import pandas as pd
import numpy as np
from unittest.mock import Mock, AsyncMock
import sys
from pathlib import Path

sys.path.insert(0, str(Path(__file__).parent.parent))

from quickstock.services.financial_reports_service import FinancialReportsService
from quickstock.models import FinancialReportsRequest
from quickstock.core.data_manager import DataManager
from quickstock.core.cache import CacheLayer


class TestPerformanceIntegration:
    """性能监控集成测试"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """模拟数据管理器"""
        manager = Mock(spec=DataManager)
        
        # 创建测试数据
        test_data = pd.DataFrame({
            'ts_code': ['000001.SZ'] * 4,
            'report_date': ['20231231', '20230930', '20230630', '20230331'],
            'report_type': ['A', 'Q3', 'Q2', 'Q1'],
            'total_revenue': [1000000.0, 750000.0, 500000.0, 250000.0],
            'net_profit': [100000.0, 75000.0, 50000.0, 25000.0],
            'total_assets': [5000000.0, 4800000.0, 4600000.0, 4400000.0],
            'total_liabilities': [3000000.0, 2900000.0, 2800000.0, 2700000.0],
            'shareholders_equity': [2000000.0, 1900000.0, 1800000.0, 1700000.0],
            'operating_cash_flow': [150000.0, 100000.0, 75000.0, 50000.0],
            'eps': [1.0, 0.75, 0.5, 0.25],
            'roe': [0.05, 0.04, 0.03, 0.02]
        })
        
        manager.get_data = AsyncMock(return_value=test_data)
        return manager
    
    @pytest.fixture
    def mock_cache_layer(self):
        """模拟缓存层"""
        cache = Mock(spec=CacheLayer)
        cache.get = AsyncMock(return_value=None)  # 模拟缓存未命中
        cache.set = AsyncMock()
        cache.get_stats = Mock(return_value={
            'hit_stats': {
                'memory_hits': 10,
                'sqlite_hits': 5,
                'misses': 15,
                'hit_rate': 0.5
            }
        })
        return cache
    
    @pytest.fixture
    def service(self, mock_data_manager, mock_cache_layer):
        """创建财务报告服务实例"""
        return FinancialReportsService(mock_data_manager, mock_cache_layer)
    
    @pytest.mark.asyncio
    async def test_performance_monitoring_integration(self, service):
        """测试性能监控集成"""
        # 创建请求
        request = FinancialReportsRequest(
            ts_code='000001.SZ',
            start_date='20230101',
            end_date='20231231'
        )
        
        # 执行请求
        result = await service.get_financial_reports(request)
        
        # 验证结果
        assert len(result) == 4
        
        # 验证性能统计被记录
        stats = service.get_performance_stats()
        
        # 检查统计结构
        assert 'service_stats' in stats
        assert 'performance_monitor' in stats
        assert 'cache_stats' in stats
        assert 'memory_processor' in stats
        
        # 检查服务统计
        service_stats = stats['service_stats']
        assert service_stats['total_requests'] == 1
        assert service_stats['successful_requests'] == 1
        assert service_stats['failed_requests'] == 0
        assert service_stats['cache_misses'] == 1
        
        # 检查性能监控统计
        perf_stats = stats['performance_monitor']
        assert perf_stats['total_operations'] >= 1
        assert perf_stats['successful_operations'] >= 1
        
        # 检查内存处理器配置
        memory_config = stats['memory_processor']
        assert memory_config['chunk_size'] == 1000
        assert memory_config['memory_limit_mb'] == 200.0
    
    @pytest.mark.asyncio
    async def test_multiple_requests_performance_tracking(self, service):
        """测试多个请求的性能跟踪"""
        requests = [
            FinancialReportsRequest(
                ts_code=f'00000{i}.SZ',
                start_date='20230101',
                end_date='20231231'
            )
            for i in range(1, 6)  # 5个请求
        ]
        
        # 执行多个请求
        for request in requests:
            await service.get_financial_reports(request)
        
        # 验证统计信息
        stats = service.get_performance_stats()
        service_stats = stats['service_stats']
        
        assert service_stats['total_requests'] == 5
        assert service_stats['successful_requests'] == 5
        assert service_stats['failed_requests'] == 0
        
        # 验证性能监控记录了所有操作
        perf_stats = stats['performance_monitor']
        assert perf_stats['total_operations'] >= 5
    
    @pytest.mark.asyncio
    async def test_cache_hit_performance_tracking(self, service, mock_cache_layer):
        """测试缓存命中的性能跟踪"""
        # 设置缓存返回数据（模拟缓存命中）
        cached_data = pd.DataFrame({
            'ts_code': ['000001.SZ'] * 2,
            'report_date': ['20231231', '20230930'],
            'report_type': ['A', 'Q3'],
            'total_revenue': [1000000.0, 750000.0],
            'net_profit': [100000.0, 75000.0],
            'total_assets': [5000000.0, 4800000.0],
            'total_liabilities': [3000000.0, 2900000.0],
            'shareholders_equity': [2000000.0, 1900000.0],
            'operating_cash_flow': [150000.0, 100000.0],
            'eps': [1.0, 0.75],
            'roe': [0.05, 0.04]
        })
        
        mock_cache_layer.get.return_value = cached_data
        
        # 创建请求
        request = FinancialReportsRequest(
            ts_code='000001.SZ',
            start_date='20230101',
            end_date='20231231'
        )
        
        # 执行请求
        result = await service.get_financial_reports(request)
        
        # 验证结果
        assert len(result) == 2
        
        # 验证缓存命中被记录
        stats = service.get_performance_stats()
        service_stats = stats['service_stats']
        
        assert service_stats['cache_hits'] == 1
        assert service_stats['cache_misses'] == 0
    
    def test_performance_stats_reset(self, service):
        """测试性能统计重置"""
        # 修改一些统计数据
        service._stats['total_requests'] = 10
        service._stats['successful_requests'] = 8
        service._stats['failed_requests'] = 2
        
        # 重置统计
        service.reset_performance_stats()
        
        # 验证重置
        stats = service.get_performance_stats()
        service_stats = stats['service_stats']
        
        assert service_stats['total_requests'] == 0
        assert service_stats['successful_requests'] == 0
        assert service_stats['failed_requests'] == 0
    
    @pytest.mark.asyncio
    async def test_error_handling_performance_tracking(self, service, mock_data_manager):
        """测试错误处理的性能跟踪"""
        # 设置数据管理器抛出异常
        mock_data_manager.get_data.side_effect = Exception("Test error")
        
        # 创建请求
        request = FinancialReportsRequest(
            ts_code='000001.SZ',
            start_date='20230101',
            end_date='20231231'
        )
        
        # 执行请求并期望异常
        with pytest.raises(Exception):
            await service.get_financial_reports(request)
        
        # 验证失败统计被记录
        stats = service.get_performance_stats()
        service_stats = stats['service_stats']
        
        assert service_stats['total_requests'] == 1
        assert service_stats['failed_requests'] == 1
        assert service_stats['successful_requests'] == 0


if __name__ == "__main__":
    pytest.main([__file__, "-v"])