"""
价格分布统计服务性能优化集成测试

测试价格分布统计服务与性能优化器的集成
"""

import pytest
import pytest_asyncio
import asyncio
import pandas as pd
import numpy as np
import time
import logging
from unittest.mock import Mock, patch, MagicMock
from datetime import datetime, timedelta

from quickstock.services.price_distribution_stats_service import PriceDistributionStatsService
from quickstock.core.data_manager import DataManager
from quickstock.models.price_distribution_models import PriceDistributionRequest
from quickstock.utils.price_distribution_performance import PriceDistributionPerformanceConfig


class TestPriceDistributionServicePerformanceIntegration:
    """测试价格分布统计服务性能优化集成"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """创建模拟数据管理器"""
        data_manager = Mock(spec=DataManager)
        
        # 模拟配置
        config = Mock()
        config.price_distribution_cache_db = 'test_cache.db'
        config.price_distribution_memory_cache_size = 100
        config.min_stock_count_for_distribution = 10
        config.max_distribution_processing_time = 30
        config.enable_fallback_classification = True
        config.cache_enabled = True
        config.parallel_processing_enabled = True
        config.enable_fault_tolerance = False
        
        # 性能优化配置
        config.enable_vectorization = True
        config.enable_parallel_processing = True
        config.enable_memory_optimization = True
        config.enable_performance_monitoring = False  # 避免全局状态
        config.max_workers = 2
        config.batch_size = 100
        config.chunk_size = 50
        config.use_multiprocessing = False
        config.memory_limit_mb = 512
        config.enable_dataframe_optimization = True
        config.enable_garbage_collection = True
        config.gc_frequency = 5
        config.enable_intermediate_caching = True
        config.cache_classification_results = True
        config.cache_distribution_calculations = True
        config.enable_detailed_metrics = True
        config.metrics_history_size = 1000
        config.performance_report_interval = 10
        
        data_manager.config = config
        
        # 模拟数据源健康检查
        data_manager.get_data_source_health.return_value = {'status': 'healthy'}
        
        return data_manager
    
    @pytest.fixture
    def sample_stock_data(self):
        """创建示例股票数据"""
        np.random.seed(42)
        n_stocks = 500
        
        data = {
            'ts_code': [f'{i:06d}.SZ' if i % 2 == 0 else f'{i:06d}.SH' for i in range(n_stocks)],
            'name': [f'股票{i}' if i % 10 != 0 else f'*ST股票{i}' for i in range(n_stocks)],
            'close': np.random.uniform(5, 100, n_stocks),
            'pre_close': np.random.uniform(5, 100, n_stocks),
            'open': np.random.uniform(5, 100, n_stocks),
            'high': np.random.uniform(5, 100, n_stocks),
            'low': np.random.uniform(5, 100, n_stocks),
            'volume': np.random.randint(1000, 1000000, n_stocks),
            'amount': np.random.uniform(10000, 10000000, n_stocks),
            'pct_chg': np.random.uniform(-10, 10, n_stocks)
        }
        
        return pd.DataFrame(data)
    
    @pytest_asyncio.fixture
    async def service(self, mock_data_manager):
        """创建价格分布统计服务"""
        service = PriceDistributionStatsService(mock_data_manager)
        
        # 等待异步初始化完成
        await asyncio.sleep(0.1)
        
        return service
    
    @pytest.mark.asyncio
    async def test_service_initialization_with_performance_optimization(self, service):
        """测试服务初始化包含性能优化"""
        # 验证性能优化器已初始化
        assert service.performance_optimizer is not None
        assert service.analyzer.performance_optimizer is not None
        
        # 验证配置
        config = service.performance_optimizer.config
        assert config.enable_vectorization == True
        assert config.enable_parallel_processing == True
        assert config.enable_memory_optimization == True
        assert config.batch_size == 100
        assert config.max_workers == 2
    
    @pytest.mark.asyncio
    async def test_performance_optimization_stats(self, service):
        """测试性能优化统计"""
        stats = service.get_performance_optimization_stats()
        
        assert isinstance(stats, dict)
        assert 'service_stats' in stats
        assert 'performance_optimizer_available' in stats
        assert 'optimizer_stats' in stats
        assert 'analyzer_stats' in stats
        
        assert stats['performance_optimizer_available'] == True
        assert 'config' in stats['optimizer_stats']
        assert 'performance_stats' in stats['optimizer_stats']
    
    @pytest.mark.asyncio
    async def test_configure_performance_optimization(self, service):
        """测试性能优化配置"""
        # 更新配置
        config_updates = {
            'batch_size': 200,
            'enable_vectorization': False,
            'memory_limit_mb': 1024
        }
        
        result = service.configure_performance_optimization(config_updates)
        
        assert result['success'] == True
        assert len(result['updated_configs']) == 3
        assert len(result['errors']) == 0
        
        # 验证配置已更新
        config = service.performance_optimizer.config
        assert config.batch_size == 200
        assert config.enable_vectorization == False
        assert config.memory_limit_mb == 1024
    
    @pytest.mark.asyncio
    async def test_configure_performance_optimization_invalid_param(self, service):
        """测试无效参数的性能优化配置"""
        config_updates = {
            'invalid_parameter': 'invalid_value'
        }
        
        result = service.configure_performance_optimization(config_updates)
        
        assert result['success'] == True  # 部分成功
        assert len(result['errors']) == 1
        assert 'Unknown configuration parameter' in result['errors'][0]
    
    @pytest.mark.asyncio
    async def test_benchmark_performance(self, service):
        """测试性能基准测试"""
        # 运行基准测试
        benchmark_result = await service.benchmark_performance(
            test_data_size=100, 
            iterations=2
        )
        
        assert isinstance(benchmark_result, dict)
        assert 'test_config' in benchmark_result
        assert 'results' in benchmark_result
        assert 'summary' in benchmark_result
        
        # 验证测试配置
        test_config = benchmark_result['test_config']
        assert test_config['data_size'] == 100
        assert test_config['iterations'] == 2
        
        # 验证结果
        results = benchmark_result['results']
        assert 'with_optimization' in results
        assert 'without_optimization' in results
        assert len(results['with_optimization']) == 2
        assert len(results['without_optimization']) == 2
        
        # 验证摘要
        summary = benchmark_result['summary']
        if 'performance_improvement' in summary:
            improvement = summary['performance_improvement']
            assert 'speedup_factor' in improvement
            assert 'improvement_percentage' in improvement
            assert 'time_saved_seconds' in improvement
    
    @pytest.mark.asyncio
    async def test_memory_usage_stats(self, service):
        """测试内存使用统计"""
        stats = await service.get_memory_usage_stats()
        
        assert isinstance(stats, dict)
        assert 'system_memory' in stats
        assert 'optimizer_memory' in stats
        
        # 验证系统内存信息
        system_memory = stats['system_memory']
        assert 'rss_mb' in system_memory
        assert 'vms_mb' in system_memory
        assert 'percent' in system_memory
        assert all(isinstance(v, (int, float)) for v in system_memory.values())
        
        # 验证优化器内存信息
        optimizer_memory = stats['optimizer_memory']
        assert 'rss_mb' in optimizer_memory
        assert 'vms_mb' in optimizer_memory
        assert 'percent' in optimizer_memory
    
    @pytest.mark.asyncio
    async def test_enable_disable_performance_monitoring(self, service):
        """测试启用/禁用性能监控"""
        # 启用性能监控
        result = service.enable_performance_monitoring(True)
        assert result['success'] == True
        assert result['new_state'] == True
        assert service.performance_monitor is not None
        
        # 禁用性能监控
        result = service.enable_performance_monitoring(False)
        assert result['success'] == True
        assert result['new_state'] == False
        assert service.performance_monitor is None
    
    @pytest.mark.asyncio
    async def test_performance_optimized_analysis_integration(self, service, sample_stock_data):
        """测试性能优化分析集成"""
        # 模拟数据获取
        with patch.object(service, '_fetch_stock_data', return_value=sample_stock_data):
            with patch.object(service, '_classify_stocks_by_market') as mock_classify:
                # 模拟市场分类结果
                mock_classify.return_value = {
                    'total': sample_stock_data,
                    'shanghai': sample_stock_data[sample_stock_data['ts_code'].str.endswith('.SH')],
                    'shenzhen': sample_stock_data[sample_stock_data['ts_code'].str.endswith('.SZ')],
                    'st': sample_stock_data[sample_stock_data['name'].str.contains('ST')],
                    'non_st': sample_stock_data[~sample_stock_data['name'].str.contains('ST')]
                }
                
                # 创建请求
                request = PriceDistributionRequest(
                    trade_date='20240101',
                    include_st=True,
                    force_refresh=True,
                    save_to_db=False
                )
                
                # 执行分析
                start_time = time.time()
                stats = await service.get_price_distribution_stats(request)
                processing_time = time.time() - start_time
                
                # 验证结果
                assert stats is not None
                assert stats.trade_date == '20240101'
                assert stats.total_stocks == len(sample_stock_data)
                assert stats.processing_time > 0
                assert isinstance(stats.positive_ranges, dict)
                assert isinstance(stats.negative_ranges, dict)
                assert isinstance(stats.market_breakdown, dict)
                
                # 验证性能统计更新
                perf_stats = service.get_performance_optimization_stats()
                assert perf_stats['service_stats']['total_requests'] > 0
                assert perf_stats['service_stats']['successful_requests'] > 0
                
                print(f"Performance-optimized analysis completed in {processing_time:.3f}s")
                print(f"Service stats: {perf_stats['service_stats']}")
                print(f"Optimizer stats: {perf_stats['optimizer_stats']['performance_stats']}")
    
    @pytest.mark.asyncio
    async def test_performance_comparison_with_without_optimization(self, service, sample_stock_data):
        """测试有无性能优化的对比"""
        # 模拟数据获取和分类
        with patch.object(service, '_fetch_stock_data', return_value=sample_stock_data):
            with patch.object(service, '_classify_stocks_by_market') as mock_classify:
                mock_classify.return_value = {
                    'total': sample_stock_data,
                    'shanghai': sample_stock_data[sample_stock_data['ts_code'].str.endswith('.SH')],
                    'shenzhen': sample_stock_data[sample_stock_data['ts_code'].str.endswith('.SZ')]
                }
                
                request = PriceDistributionRequest(
                    trade_date='20240101',
                    include_st=True,
                    force_refresh=True,
                    save_to_db=False
                )
                
                # 测试优化版本
                start_time = time.time()
                optimized_stats = await service.get_price_distribution_stats(request)
                optimized_time = time.time() - start_time
                
                # 临时禁用优化器测试标准版本
                original_optimizer = service.analyzer.performance_optimizer
                service.analyzer.performance_optimizer = None
                
                try:
                    start_time = time.time()
                    standard_stats = await service.get_price_distribution_stats(request)
                    standard_time = time.time() - start_time
                finally:
                    # 恢复优化器
                    service.analyzer.performance_optimizer = original_optimizer
                
                # 验证结果一致性
                assert optimized_stats.trade_date == standard_stats.trade_date
                assert optimized_stats.total_stocks == standard_stats.total_stocks
                
                # 记录性能差异
                print(f"Optimized version: {optimized_time:.3f}s")
                print(f"Standard version: {standard_time:.3f}s")
                
                if optimized_time > 0 and standard_time > 0:
                    speedup = standard_time / optimized_time
                    print(f"Speedup factor: {speedup:.2f}x")
    
    @pytest.mark.asyncio
    async def test_health_check_with_performance_components(self, service):
        """测试包含性能组件的健康检查"""
        # 模拟缓存管理器健康检查
        with patch.object(service.cache_manager, 'get_cache_info', return_value={'status': 'healthy'}):
            health_result = await service.health_check()
            
            assert health_result['status'] in ['healthy', 'unhealthy']
            assert 'checks' in health_result
            assert 'data_manager' in health_result['checks']
            assert 'cache_manager' in health_result['checks']
            assert 'stock_classifier' in health_result['checks']
            assert 'analyzer' in health_result['checks']
            
            # 验证分析器健康检查包含性能统计
            analyzer_check = health_result['checks']['analyzer']
            assert 'details' in analyzer_check
            analyzer_details = analyzer_check['details']
            assert isinstance(analyzer_details, dict)


if __name__ == "__main__":
    pytest.main([__file__, "-v"])