"""
涨跌分布统计服务集成测试

测试 PriceDistributionStatsService 的完整功能，包括数据获取、市场分类、
统计分析和缓存管理等核心业务流程
"""

import asyncio
import pytest
import pandas as pd
import numpy as np
from datetime import datetime, timedelta
from unittest.mock import Mock, AsyncMock, patch, MagicMock
import tempfile
import os

from quickstock.services.price_distribution_stats_service import (
    PriceDistributionStatsService,
    PriceDistributionStatsError,
    InsufficientDataError,
    MarketClassificationError,
    create_price_distribution_service,
    get_distribution_stats_with_fallback
)
from quickstock.models.price_distribution_models import (
    PriceDistributionRequest,
    PriceDistributionStats,
    DistributionRange
)
from quickstock.core.data_manager import DataManager
from quickstock.core.errors import ValidationError, DataSourceError


class TestPriceDistributionStatsService:
    """涨跌分布统计服务测试类"""
    
    @pytest.fixture
    def mock_config(self):
        """模拟配置对象"""
        config = Mock()
        config.cache_enabled = True
        config.max_concurrent_requests = 5
        config.max_retries = 3
        config.cache_expire_hours = 24
        config.price_distribution_cache_db = ':memory:'
        config.price_distribution_memory_cache_size = 100
        config.min_stock_count_for_distribution = 50
        config.max_distribution_processing_time = 60
        config.enable_fallback_classification = True
        config.parallel_processing_enabled = True
        return config
    
    @pytest.fixture
    def mock_data_manager(self, mock_config):
        """模拟数据管理器"""
        data_manager = Mock(spec=DataManager)
        data_manager.config = mock_config
        data_manager.get_stock_data_with_coordination = AsyncMock()
        data_manager.get_cache_stats = Mock(return_value={'hits': 10, 'misses': 5})
        data_manager.get_memory_stats = Mock(return_value={'memory_mb': 100})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        return data_manager
    
    @pytest.fixture
    def sample_stock_data(self):
        """生成样本股票数据"""
        np.random.seed(42)
        
        # 创建不同市场的股票代码
        stock_codes = [
            # 上海主板
            '600000.SH', '600001.SH', '600002.SH', '600003.SH', '600004.SH',
            # 科创板
            '688001.SH', '688002.SH', '688003.SH',
            # 深圳主板
            '000001.SZ', '000002.SZ', '000003.SZ', '000004.SZ',
            # 创业板
            '300001.SZ', '300002.SZ', '300003.SZ',
            # 北京交易所
            '430001.BJ', '430002.BJ'
        ]
        
        # 生成涨跌幅数据，覆盖不同区间
        pct_changes = [
            # 正涨幅
            1.5, 2.8, 4.2, 6.1, 8.5, 12.3, 15.7,  # 不同正区间
            # 负涨幅
            -1.2, -2.9, -4.5, -6.8, -8.2, -11.5, -13.8,  # 不同负区间
            # 边界值
            0.0, 3.0, -3.0, 5.0, -5.0
        ]
        
        # 确保有足够的数据
        while len(pct_changes) < len(stock_codes):
            pct_changes.append(np.random.uniform(-10, 10))
        
        # 生成股票名称，包含ST股票
        stock_names = []
        for i, code in enumerate(stock_codes):
            if i < 2:  # 前两只设为ST股票
                stock_names.append(f'*ST测试{i+1}')
            else:
                stock_names.append(f'测试股票{i+1}')
        
        data = pd.DataFrame({
            'ts_code': stock_codes,
            'name': stock_names,
            'trade_date': '20240115',
            'pct_chg': pct_changes[:len(stock_codes)],
            'close': np.random.uniform(10, 100, len(stock_codes)),
            'vol': np.random.uniform(1000, 100000, len(stock_codes))
        })
        
        return data
    
    @pytest.fixture
    async def service(self, mock_data_manager):
        """创建服务实例"""
        return PriceDistributionStatsService(mock_data_manager)
    
    @pytest.mark.asyncio
    async def test_service_initialization(self, service, mock_data_manager):
        """测试服务初始化"""
        assert service.data_manager == mock_data_manager
        assert service.stock_classifier is not None
        assert service.cache_manager is not None
        assert service.analyzer is not None
        assert service.config['min_stock_count'] == 50
        assert service.config['cache_enabled'] is True
    
    @pytest.mark.asyncio
    async def test_get_price_distribution_stats_success(self, service, sample_stock_data):
        """测试成功获取涨跌分布统计"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建请求
        request = PriceDistributionRequest(trade_date='20240115')
        
        # 执行测试
        result = await service.get_price_distribution_stats(request)
        
        # 验证结果
        assert isinstance(result, PriceDistributionStats)
        assert result.trade_date == '20240115'
        assert result.total_stocks > 0
        assert isinstance(result.positive_ranges, dict)
        assert isinstance(result.negative_ranges, dict)
        assert isinstance(result.market_breakdown, dict)
        
        # 验证市场分类
        assert 'total' in result.market_breakdown
        assert 'shanghai' in result.market_breakdown
        assert 'shenzhen' in result.market_breakdown
        
        # 验证数据一致性
        total_positive = sum(result.positive_ranges.values())
        total_negative = sum(result.negative_ranges.values())
        assert total_positive + total_negative == result.total_stocks
    
    @pytest.mark.asyncio
    async def test_get_price_distribution_stats_with_cache(self, service, sample_stock_data):
        """测试带缓存的统计获取"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建请求
        request = PriceDistributionRequest(trade_date='20240115')
        
        # 第一次请求
        result1 = await service.get_price_distribution_stats(request)
        
        # 第二次请求（应该命中缓存）
        result2 = await service.get_price_distribution_stats(request)
        
        # 验证结果一致
        assert result1.trade_date == result2.trade_date
        assert result1.total_stocks == result2.total_stocks
        
        # 验证缓存统计
        stats = service.get_service_stats()
        assert stats['performance']['cache_hits'] > 0
    
    @pytest.mark.asyncio
    async def test_get_price_distribution_stats_force_refresh(self, service, sample_stock_data):
        """测试强制刷新统计"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建强制刷新请求
        request = PriceDistributionRequest(trade_date='20240115', force_refresh=True)
        
        # 执行测试
        result = await service.get_price_distribution_stats(request)
        
        # 验证结果
        assert isinstance(result, PriceDistributionStats)
        assert result.trade_date == '20240115'
        
        # 验证数据管理器被调用
        service.data_manager.get_stock_data_with_coordination.assert_called()
    
    @pytest.mark.asyncio
    async def test_insufficient_data_error(self, service):
        """测试数据不足异常"""
        # 模拟返回空数据
        service.data_manager.get_stock_data_with_coordination.return_value = pd.DataFrame()
        
        # 创建请求
        request = PriceDistributionRequest(trade_date='20240115')
        
        # 执行测试并验证异常
        with pytest.raises(InsufficientDataError) as exc_info:
            await service.get_price_distribution_stats(request)
        
        assert '20240115' in str(exc_info.value)
        assert exc_info.value.service_context['available_count'] == 0
    
    @pytest.mark.asyncio
    async def test_data_source_error_handling(self, service):
        """测试数据源错误处理"""
        # 模拟数据源异常
        service.data_manager.get_stock_data_with_coordination.side_effect = DataSourceError("Data source unavailable")
        
        # 创建请求
        request = PriceDistributionRequest(trade_date='20240115')
        
        # 执行测试并验证异常
        with pytest.raises(PriceDistributionStatsError) as exc_info:
            await service.get_price_distribution_stats(request)
        
        assert 'Data source unavailable' in str(exc_info.value)
    
    @pytest.mark.asyncio
    async def test_market_classification(self, service, sample_stock_data):
        """测试市场分类功能"""
        # 执行市场分类
        classified_data = await service._classify_stocks_by_market(sample_stock_data)
        
        # 验证分类结果
        assert isinstance(classified_data, dict)
        assert 'total' in classified_data
        assert 'shanghai' in classified_data
        assert 'shenzhen' in classified_data
        assert 'st' in classified_data
        assert 'non_st' in classified_data
        
        # 验证数据一致性
        total_stocks = len(classified_data['total'])
        st_stocks = len(classified_data['st'])
        non_st_stocks = len(classified_data['non_st'])
        assert st_stocks + non_st_stocks == total_stocks
        
        # 验证ST股票识别
        assert st_stocks > 0  # 样本数据中包含ST股票
    
    @pytest.mark.asyncio
    async def test_stock_data_cleaning(self, service):
        """测试股票数据清洗"""
        # 创建包含异常数据的DataFrame
        dirty_data = pd.DataFrame({
            'ts_code': ['600000.SH', '000001.SZ', '688001.SH', '300001.SZ'],
            'pct_chg': [5.2, None, 100.0, -2.3],  # 包含空值和异常值
            'name': ['测试1', None, '测试3', '测试4']  # 包含空名称
        })
        
        # 执行清洗
        cleaned_data = service._clean_stock_data(dirty_data)
        
        # 验证清洗结果
        assert len(cleaned_data) == 2  # 移除了空值和异常值
        assert cleaned_data['pct_chg'].notna().all()  # 无空值
        assert (cleaned_data['pct_chg'].abs() <= 50).all()  # 无异常值
        assert cleaned_data['name'].notna().all()  # 名称已填充
    
    @pytest.mark.asyncio
    async def test_cache_management_operations(self, service, sample_stock_data):
        """测试缓存管理操作"""
        trade_date = '20240115'
        
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建统计数据
        request = PriceDistributionRequest(trade_date=trade_date)
        stats = await service.get_price_distribution_stats(request)
        
        # 测试获取缓存
        cached_stats = await service.get_cached_stats(trade_date)
        assert cached_stats is not None
        assert cached_stats.trade_date == trade_date
        
        # 测试删除缓存
        deleted_count = await service.delete_cached_stats(trade_date)
        assert deleted_count >= 0
        
        # 测试刷新缓存
        refresh_result = await service.refresh_cache(trade_date)
        assert refresh_result['success'] is True
        assert refresh_result['trade_date'] == trade_date
        
        # 测试获取缓存信息
        cache_info = await service.get_cache_info()
        assert isinstance(cache_info, dict)
        assert 'service_stats' in cache_info
    
    @pytest.mark.asyncio
    async def test_request_validation(self, service):
        """测试请求参数验证"""
        # 测试有效请求
        valid_request = PriceDistributionRequest(trade_date='20240115')
        validated = service._validate_and_process_request(valid_request)
        assert validated.trade_date == '20240115'
        
        # 测试无效日期格式
        with pytest.raises(ValidationError):
            invalid_request = PriceDistributionRequest(trade_date='invalid_date')
            service._validate_and_process_request(invalid_request)
        
        # 测试未来日期
        future_date = (datetime.now() + timedelta(days=1)).strftime('%Y%m%d')
        with pytest.raises(ValidationError):
            future_request = PriceDistributionRequest(trade_date=future_date)
            service._validate_and_process_request(future_request)
    
    @pytest.mark.asyncio
    async def test_custom_distribution_ranges(self, service, sample_stock_data):
        """测试自定义分布区间"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建自定义区间请求
        custom_ranges = {
            "0-2%": (0.0, 2.0),
            "2-5%": (2.0, 5.0),
            "5-10%": (5.0, 10.0),
            ">=10%": (10.0, float('inf')),
            "0到-2%": (-2.0, 0.0),
            "-2到-5%": (-5.0, -2.0),
            "-5到-10%": (-10.0, -5.0),
            "<=-10%": (float('-inf'), -10.0)
        }
        
        request = PriceDistributionRequest(
            trade_date='20240115',
            distribution_ranges=custom_ranges
        )
        
        # 执行测试
        result = await service.get_price_distribution_stats(request)
        
        # 验证自定义区间被使用
        assert isinstance(result, PriceDistributionStats)
        assert len(result.positive_ranges) == 4  # 4个正区间
        assert len(result.negative_ranges) == 4  # 4个负区间
    
    @pytest.mark.asyncio
    async def test_market_filter(self, service, sample_stock_data):
        """测试市场过滤器"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建市场过滤请求
        request = PriceDistributionRequest(
            trade_date='20240115',
            market_filter=['shanghai', 'shenzhen']
        )
        
        # 执行测试
        result = await service.get_price_distribution_stats(request)
        
        # 验证结果
        assert isinstance(result, PriceDistributionStats)
        assert 'shanghai' in result.market_breakdown
        assert 'shenzhen' in result.market_breakdown
    
    @pytest.mark.asyncio
    async def test_performance_statistics(self, service, sample_stock_data):
        """测试性能统计"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 执行多次请求
        for i in range(3):
            request = PriceDistributionRequest(trade_date=f'2024011{5+i}')
            try:
                await service.get_price_distribution_stats(request)
            except:
                pass  # 忽略错误，只关注统计
        
        # 获取性能统计
        stats = service.get_service_stats()
        
        # 验证统计信息
        assert 'performance' in stats
        assert stats['performance']['total_requests'] >= 3
        assert 'analyzer' in stats
        assert 'classifier' in stats
        assert 'config' in stats
        assert 'data_manager' in stats
    
    @pytest.mark.asyncio
    async def test_health_check(self, service):
        """测试健康检查"""
        health_result = await service.health_check()
        
        # 验证健康检查结果
        assert isinstance(health_result, dict)
        assert 'service' in health_result
        assert 'status' in health_result
        assert 'checks' in health_result
        assert health_result['service'] == 'PriceDistributionStatsService'
        
        # 验证各组件检查
        checks = health_result['checks']
        assert 'data_manager' in checks
        assert 'cache_manager' in checks
        assert 'stock_classifier' in checks
        assert 'analyzer' in checks
    
    @pytest.mark.asyncio
    async def test_configuration_validation(self, service):
        """测试配置验证"""
        validation_result = await service.validate_service_configuration()
        
        # 验证配置检查结果
        assert isinstance(validation_result, dict)
        assert 'is_valid' in validation_result
        assert 'config_checks' in validation_result
        assert 'issues' in validation_result
        assert 'warnings' in validation_result
        
        # 验证配置项检查
        config_checks = validation_result['config_checks']
        assert 'min_stock_count' in config_checks
        assert 'max_processing_time' in config_checks
        assert 'cache_enabled' in config_checks
    
    @pytest.mark.asyncio
    async def test_error_handling_and_recovery(self, service, sample_stock_data):
        """测试错误处理和恢复"""
        # 模拟间歇性错误
        call_count = 0
        
        async def mock_get_data(trade_date, timeout=30):
            nonlocal call_count
            call_count += 1
            if call_count == 1:
                raise DataSourceError("Temporary error")
            return sample_stock_data
        
        service.data_manager.get_stock_data_with_coordination.side_effect = mock_get_data
        
        # 第一次请求应该失败
        request = PriceDistributionRequest(trade_date='20240115')
        with pytest.raises(PriceDistributionStatsError):
            await service.get_price_distribution_stats(request)
        
        # 重置mock以模拟恢复
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 第二次请求应该成功
        result = await service.get_price_distribution_stats(request)
        assert isinstance(result, PriceDistributionStats)
    
    @pytest.mark.asyncio
    async def test_concurrent_requests(self, service, sample_stock_data):
        """测试并发请求处理"""
        # 模拟数据获取
        service.data_manager.get_stock_data_with_coordination.return_value = sample_stock_data
        
        # 创建多个并发请求
        requests = [
            PriceDistributionRequest(trade_date=f'2024011{5+i}')
            for i in range(5)
        ]
        
        # 并发执行
        tasks = [
            service.get_price_distribution_stats(request)
            for request in requests
        ]
        
        results = await asyncio.gather(*tasks, return_exceptions=True)
        
        # 验证结果
        successful_results = [r for r in results if isinstance(r, PriceDistributionStats)]
        assert len(successful_results) > 0  # 至少有一些成功
        
        # 验证每个成功结果
        for result in successful_results:
            assert isinstance(result, PriceDistributionStats)
            assert result.total_stocks > 0


class TestServiceUtilityFunctions:
    """测试服务工具函数"""
    
    @pytest.fixture
    def mock_data_manager(self):
        """模拟数据管理器"""
        config = Mock()
        config.cache_enabled = True
        config.price_distribution_cache_db = ':memory:'
        
        data_manager = Mock(spec=DataManager)
        data_manager.config = config
        data_manager.get_stock_data_with_coordination = AsyncMock()
        data_manager.get_cache_stats = Mock(return_value={})
        data_manager.get_memory_stats = Mock(return_value={})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        return data_manager
    
    @pytest.mark.asyncio
    async def test_create_price_distribution_service(self, mock_data_manager):
        """测试服务创建函数"""
        service = await create_price_distribution_service(mock_data_manager)
        
        assert isinstance(service, PriceDistributionStatsService)
        assert service.data_manager == mock_data_manager
    
    @pytest.mark.asyncio
    async def test_get_distribution_stats_with_fallback_success(self, mock_data_manager):
        """测试带fallback的统计获取 - 成功情况"""
        # 创建样本数据
        sample_data = pd.DataFrame({
            'ts_code': ['600000.SH', '000001.SZ'],
            'pct_chg': [5.2, -3.1],
            'name': ['测试1', '测试2']
        })
        
        mock_data_manager.get_stock_data_with_coordination.return_value = sample_data
        
        service = PriceDistributionStatsService(mock_data_manager)
        
        # 执行测试
        result = await get_distribution_stats_with_fallback(service, '20240115')
        
        # 验证结果
        assert isinstance(result, PriceDistributionStats)
        assert result.trade_date == '20240115'
    
    @pytest.mark.asyncio
    async def test_get_distribution_stats_with_fallback_failure(self, mock_data_manager):
        """测试带fallback的统计获取 - 失败情况"""
        # 模拟异常
        mock_data_manager.get_stock_data_with_coordination.side_effect = Exception("Test error")
        
        service = PriceDistributionStatsService(mock_data_manager)
        
        # 执行测试
        result = await get_distribution_stats_with_fallback(service, '20240115')
        
        # 验证fallback返回None
        assert result is None


class TestServiceIntegrationScenarios:
    """测试服务集成场景"""
    
    @pytest.fixture
    def real_like_config(self):
        """类似真实环境的配置"""
        config = Mock()
        config.cache_enabled = True
        config.max_concurrent_requests = 10
        config.max_retries = 3
        config.cache_expire_hours = 24
        config.price_distribution_cache_db = ':memory:'
        config.price_distribution_memory_cache_size = 500
        config.min_stock_count_for_distribution = 100
        config.max_distribution_processing_time = 120
        config.enable_fallback_classification = True
        config.parallel_processing_enabled = True
        return config
    
    @pytest.fixture
    def large_stock_dataset(self):
        """大型股票数据集"""
        np.random.seed(123)
        
        # 生成1000只股票的数据
        stock_count = 1000
        
        # 生成不同市场的股票代码
        codes = []
        names = []
        
        # 上海主板 (600xxx)
        for i in range(300):
            codes.append(f'60{i:04d}.SH')
            names.append(f'上海股票{i+1}' if i % 20 != 0 else f'*ST上海{i+1}')
        
        # 科创板 (688xxx)
        for i in range(100):
            codes.append(f'688{i:03d}.SH')
            names.append(f'科创股票{i+1}')
        
        # 深圳主板 (000xxx)
        for i in range(300):
            codes.append(f'000{i:03d}.SZ')
            names.append(f'深圳股票{i+1}' if i % 25 != 0 else f'ST深圳{i+1}')
        
        # 创业板 (300xxx)
        for i in range(200):
            codes.append(f'300{i:03d}.SZ')
            names.append(f'创业板{i+1}')
        
        # 北京交易所 (8xxxxx)
        for i in range(100):
            codes.append(f'8{i:05d}.BJ')
            names.append(f'北交所{i+1}')
        
        # 生成涨跌幅数据，模拟真实分布
        pct_changes = []
        for _ in range(stock_count):
            # 70%的股票在-10%到10%之间
            if np.random.random() < 0.7:
                pct_changes.append(np.random.normal(0, 3))
            # 20%的股票在-20%到20%之间
            elif np.random.random() < 0.9:
                pct_changes.append(np.random.uniform(-20, 20))
            # 10%的股票有极端涨跌幅
            else:
                pct_changes.append(np.random.choice([-10, 10]) * np.random.uniform(1, 3))
        
        # 限制涨跌幅范围
        pct_changes = np.clip(pct_changes, -20, 20)
        
        return pd.DataFrame({
            'ts_code': codes,
            'name': names,
            'trade_date': '20240115',
            'pct_chg': pct_changes,
            'close': np.random.uniform(5, 200, stock_count),
            'vol': np.random.uniform(1000, 1000000, stock_count)
        })
    
    @pytest.mark.asyncio
    async def test_large_dataset_processing(self, real_like_config, large_stock_dataset):
        """测试大数据集处理"""
        # 创建数据管理器
        data_manager = Mock(spec=DataManager)
        data_manager.config = real_like_config
        data_manager.get_stock_data_with_coordination = AsyncMock(return_value=large_stock_dataset)
        data_manager.get_cache_stats = Mock(return_value={})
        data_manager.get_memory_stats = Mock(return_value={})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        
        # 创建服务
        service = PriceDistributionStatsService(data_manager)
        
        # 执行测试
        request = PriceDistributionRequest(trade_date='20240115')
        result = await service.get_price_distribution_stats(request)
        
        # 验证结果
        assert isinstance(result, PriceDistributionStats)
        assert result.total_stocks == 1000
        assert len(result.market_breakdown) >= 5  # 至少5个市场分类
        
        # 验证处理时间合理
        assert result.processing_time < 30  # 应该在30秒内完成
        
        # 验证数据质量
        assert result.data_quality_score > 0.8  # 质量分数应该较高
    
    @pytest.mark.asyncio
    async def test_cache_performance_with_repeated_requests(self, real_like_config, large_stock_dataset):
        """测试重复请求的缓存性能"""
        # 创建数据管理器
        data_manager = Mock(spec=DataManager)
        data_manager.config = real_like_config
        data_manager.get_stock_data_with_coordination = AsyncMock(return_value=large_stock_dataset)
        data_manager.get_cache_stats = Mock(return_value={})
        data_manager.get_memory_stats = Mock(return_value={})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        
        # 创建服务
        service = PriceDistributionStatsService(data_manager)
        
        # 第一次请求（冷启动）
        request = PriceDistributionRequest(trade_date='20240115')
        start_time = time.time()
        result1 = await service.get_price_distribution_stats(request)
        first_request_time = time.time() - start_time
        
        # 第二次请求（缓存命中）
        start_time = time.time()
        result2 = await service.get_price_distribution_stats(request)
        second_request_time = time.time() - start_time
        
        # 验证缓存效果
        assert result1.trade_date == result2.trade_date
        assert result1.total_stocks == result2.total_stocks
        assert second_request_time < first_request_time * 0.5  # 缓存请求应该快很多
        
        # 验证缓存统计
        stats = service.get_service_stats()
        assert stats['performance']['cache_hits'] > 0
    
    @pytest.mark.asyncio
    async def test_service_resilience_under_stress(self, real_like_config, large_stock_dataset):
        """测试服务在压力下的弹性"""
        # 创建数据管理器，模拟间歇性错误
        data_manager = Mock(spec=DataManager)
        data_manager.config = real_like_config
        data_manager.get_cache_stats = Mock(return_value={})
        data_manager.get_memory_stats = Mock(return_value={})
        data_manager.get_data_source_health = Mock(return_value={'status': 'healthy'})
        
        # 模拟80%成功率的数据获取
        call_count = 0
        
        async def mock_get_data(trade_date, timeout=30):
            nonlocal call_count
            call_count += 1
            if call_count % 5 == 0:  # 每5次调用失败一次
                raise DataSourceError("Simulated network error")
            return large_stock_dataset
        
        data_manager.get_stock_data_with_coordination.side_effect = mock_get_data
        
        # 创建服务
        service = PriceDistributionStatsService(data_manager)
        
        # 执行多次请求
        successful_requests = 0
        failed_requests = 0
        
        for i in range(20):
            try:
                request = PriceDistributionRequest(trade_date=f'2024{1:02d}{15+i%10:02d}')
                result = await service.get_price_distribution_stats(request)
                successful_requests += 1
                assert isinstance(result, PriceDistributionStats)
            except PriceDistributionStatsError:
                failed_requests += 1
        
        # 验证弹性
        success_rate = successful_requests / (successful_requests + failed_requests)
        assert success_rate >= 0.7  # 至少70%的请求应该成功
        
        # 验证服务状态
        health_result = await service.health_check()
        assert health_result['status'] in ['healthy', 'unhealthy']  # 服务应该能报告状态


if __name__ == '__main__':
    pytest.main([__file__, '-v'])