from __future__ import annotations

from pathlib import Path

import numpy as np

from lgdo import ArrayOfEncodedEqualSizedArrays, ArrayOfEqualSizedArrays, lh5
from lgdo.compression.radware import (
    _get_hton_u16,
    _radware_sigcompress_decode,
    _radware_sigcompress_encode,
    _set_hton_u16,
    decode,
    encode,
)
from lgdo.compression.radware import _radware_sigcompress_mask as _mask

config_dir = Path(__file__).parent / "sigcompress"


def read_sigcompress_c_output(filename: str):
    enc_wf_c = np.empty(0, dtype=np.uint16)
    nsig_c = None
    shift = None
    with Path(filename).open() as f:
        nsig_c = int(f.readline())  # first number in the file
        shift = int(f.readline())  # second number in the file
        for line in f.readlines():  # then the waveform
            enc_wf_c = np.append(enc_wf_c, np.uint16(line))

    return (nsig_c, shift, enc_wf_c)


def read_sigcompress_c_output_multi(filename: str):
    enc_wf_c = []
    nsig_c = np.empty(0, dtype="uint32")
    shift = np.empty(0, dtype="int32")
    with Path(filename).open() as f:
        for line in f:
            parts = line.split()
            nsig_c = np.append(nsig_c, np.uint32(parts[0]))
            shift = np.append(shift, np.int32(parts[1]))
            enc_wf_c += [np.array([np.uint16(el) for el in parts[2:]])]

    enc_wf_c = np.array(enc_wf_c)
    return (nsig_c, shift, enc_wf_c)


_get_idx = _get_hton_u16


def _get_wf(a):
    i_1 = np.s_[..., 0::2]
    i_2 = np.s_[..., 1::2]
    return a[i_1].astype("uint16") << 8 | a[i_2]


def _to_u16(a):
    assert len(a) % 2 == 0
    out = np.empty(int(len(a) / 2), dtype="uint16")
    for i in range(int(len(a) / 2)):
        out[i] = _get_hton_u16(a, i)
    return out


def test_core_vs_original(wftable, lgnd_test_data):
    wfs = wftable.values.nda
    s = wfs.shape

    # get expected output from original C code (give shift)
    (enclen_c, shift, enc_wfs_c) = read_sigcompress_c_output_multi(
        lgnd_test_data.get_path(
            "lh5/LDQTA_r117_20200110T105115Z_cal_geds_raw-radware-sigcompressed.dat"
        )
    )

    # encode
    enc_wfs = np.zeros((*s[:-1], 2 * s[-1]), dtype="ubyte")
    enclen = np.empty(s[0], dtype="uint32")

    _radware_sigcompress_encode(wfs, enc_wfs, shift, enclen, _mask)

    assert (_get_idx(enc_wfs, 0) == s[1]).all()
    assert enc_wfs.dtype == np.ubyte

    # compare to result of original C code
    assert np.array_equal(2 * enclen_c, enclen)
    assert enc_wfs.shape[1] == 2 * enc_wfs_c.shape[1]
    assert np.array_equal(_get_wf(enc_wfs), enc_wfs_c)

    # now check if decompressed is same as the original
    dec_wf = np.empty_like(wfs)
    dec_len = np.empty(s[0], dtype="uint32")
    _radware_sigcompress_decode(enc_wfs, dec_wf, shift, dec_len, _mask)

    # check if encoding was lossless
    assert np.array_equal(dec_wf, wfs)

    # what if the pre-allocated array is of a different (compatible) type?
    dec_wf = np.empty_like(wfs, dtype="int32")
    _radware_sigcompress_decode(enc_wfs, dec_wf, shift, dec_len, _mask)

    # check if encoding was lossless
    assert np.array_equal(dec_wf, wfs)


def test_wrapper(wftable):
    wfs = wftable.values.nda
    s = wfs.shape
    shift = -32768

    enc_wfs = np.zeros((*s[:-1], 2 * s[-1]), dtype="ubyte")
    enclen = np.empty(s[0], dtype="uint32")
    _shift = np.full(s[0], shift, dtype="int32")

    _radware_sigcompress_encode(wfs, enc_wfs, _shift, enclen, _mask)

    # test if the wrapper gives the same result
    w_enc_wfs = np.zeros((*s[:-1], 2 * s[-1]), dtype="ubyte")
    w_enc_wfs, w_enclen = encode(wfs, sig_out=w_enc_wfs, shift=shift)
    assert isinstance(w_enc_wfs, np.ndarray)
    assert w_enc_wfs.dtype == np.ubyte
    assert np.array_equal(w_enclen, enclen)
    assert np.array_equal(w_enc_wfs, enc_wfs)

    # check if encoding was lossless
    w_dec_wfs, w_dec_len = decode((w_enc_wfs, w_enclen), shift=shift)
    assert isinstance(w_dec_wfs, np.ndarray)
    assert np.issubdtype(w_dec_wfs.dtype, np.integer)
    assert (w_dec_len == s[-1]).all()

    assert np.array_equal(w_dec_wfs, wfs)


def test_must_shift_wf(wftable):
    wf = wftable.values.nda[9]

    # this wf should have samples that don't fit in the int16 range
    assert (wf > np.iinfo("int16").max).any()

    shift = -32768
    enc_wf, enc_len = encode(wf, shift=shift)
    dec_wf, dec_len = decode((enc_wf, enc_len), shift=shift)
    assert np.array_equal(dec_wf, wf)


def test_must_shift_uint32_wf(wftable):
    wf = wftable.values.nda[0].astype("uint32")

    shift = -32768
    enc_wf, enc_len = encode(wf, shift=shift)
    dec_wf, dec_len = decode((enc_wf, enc_len), shift=shift)
    assert np.array_equal(dec_wf, wf)


def test_aoesa(wftable):
    shift = -32768
    enc_vov = encode(wftable.values, shift=shift)

    assert isinstance(enc_vov, ArrayOfEncodedEqualSizedArrays)
    assert enc_vov.encoded_data.dtype == np.ubyte
    assert len(wftable.values) == len(enc_vov)
    # test only first waveform
    enc_wf, enc_len = encode(wftable.values[0], shift=shift)
    assert np.array_equal(enc_vov[0], enc_wf[:enc_len])
    assert enc_vov.decoded_size.value == len(wftable.values[0])

    dec_aoesa = decode(enc_vov, shift=shift)
    assert isinstance(dec_aoesa, ArrayOfEqualSizedArrays)
    assert np.issubdtype(dec_aoesa.dtype, np.integer)

    for wf1, wf2 in zip(dec_aoesa, wftable.values):
        assert np.array_equal(wf1, wf2)

    # test using pre-allocated decoded array
    sig_out = ArrayOfEqualSizedArrays(
        shape=wftable.values.nda.shape, dtype=wftable.values.dtype
    )
    decode(enc_vov, sig_out=sig_out, shift=shift)
    assert wftable.values == sig_out


def test_performance(lgnd_test_data):
    store = lh5.LH5Store()
    obj = store.read(
        "/geds/raw/waveform",
        lgnd_test_data.get_path("lh5/LDQTA_r117_20200110T105115Z_cal_geds_raw.lh5"),
    )

    sum = 0
    for wf in obj["values"].nda:
        comp_wf, comp_len = encode(wf, shift=-32768)
        sum += comp_len / len(wf) / 2

    print(
        "number of bytes in compressed wf:",
        100 * sum / len(obj),
        "% of the original",
    )


def test_hton_u16():
    a = np.array([1, 2, 3, 4], dtype="uint16")
    b = np.empty(8, dtype="ubyte")

    for i in range(3):
        _set_hton_u16(b, i, a[i])

    for i in range(3):
        assert a[i] == _get_hton_u16(b, i)


def test_encoding_decoding_empty():
    aoesa = ArrayOfEqualSizedArrays(dims=(1, 1), nda=np.empty((0, 0)))
    enc_aoesa = encode(aoesa)
    assert enc_aoesa == ArrayOfEncodedEqualSizedArrays()
    assert decode(enc_aoesa) == aoesa


def test_special_wfs():
    # fmt: off
    wf = np.array([5, -7, -14, 17, -21, 0, -10, -2, -17, -14, 22, -5, -7, 7,
                   14, -2, 1, 0, -7, -21, -5, -15, -5, 11, 2, -24, 18, 2, -9,
                   -3, 4, 7, -14, 12, 8, -23, 9, -8, 6, -2, 3, 9, -4, -10, -20,
                   5, 9, 12, 21, 3, 95, 189, 294, 405, 487, 593, 702, 781, 896,
                   1034, 986, 1004, 1011, 996, 990, 1005, 1003, 987, 1003, 994,
                   1008, 996, 989, 995, 997, 996, 985, 1010, 1006, 995, 988,
                   993, 1013, 1005, 985, 1025, 1008, 995, 1004, 1003])

    enc_wf_exp = np.array([90, 50, 6, 65512, 29970, 43277, 33686, 7339, 37701,
                           63894, 25988, 17228, 38115, 26634, 39485, 22303,
                           10824, 389, 1942, 28181, 3601, 55396, 46512, 40,
                           40, 95, 65488, 36505, 40834, 39581, 32675, 47616,
                           16951, 8490, 16174, 8256, 10046, 9257, 13874,
                           12069, 18732, 9513, 13636, 10268, 22559, 9017,
                           12032, 0])

    enc_wf, stop = encode(wf, shift=0)
    assert np.array_equal(_to_u16(enc_wf[:stop]), enc_wf_exp)
    assert np.array_equal(decode((enc_wf, stop), shift=0)[0], wf)

    wf = np.array([107, 105, 113, 112, 105, 91, 119, 126, 110, 117, 105, 98,
                   129, 91, 112, 102, -33, 213, -54, 312, 107, 97, 107, 123,
                   114, 88, 130, 114, 103, 109, 116, 119, 98, 124, 120, 89,
                   121, 104, 118, 110, 115, 121, 108, 102, 92, 117, 121, 124,
                   133, 115, 207, 301, 406, 517, 599, 705, 814, 893, 1008,
                   1146, 1098, 1116, 1123, 1108, 1102, 1117, 1115, 1099, 1115,
                   1106, 1120, 1108, 1101, 1107, 1109, 1108, 1097, 1122, 1118,
                   1107, 1100, 1105, 1125, 1117, 1097, 1137, 1120, 1107, 1116,
                   1115])

    enc_wf_exp = np.array([90, 53, 9, 65482, 20647, 54506, 25850, 17754, 46162,
                           10963, 59781, 47685, 19612, 2754, 49174, 58634,
                           23874, 45396, 9111, 2692, 60045, 21677, 19500,
                           38344, 62842, 31064, 42068, 43988, 18884, 37549,
                           24242, 23978, 24758, 15968, 37, 40, 517, 65488,
                           33434, 40319, 41914, 66, 14113, 10815, 11808, 16423,
                           15908, 10550, 12847, 9545, 11301, 10549, 17448,
                           7256, 7971, 14639])

    enc_wf, stop = encode(wf, shift=0)
    assert np.array_equal(_to_u16(enc_wf[:stop]), enc_wf_exp)
    assert np.array_equal(decode((enc_wf, stop), shift=0)[0], wf)

    wf = np.array([-18257, -18258, -18259, -18250, -18247, -18237, -18236, -18242, -18242, -18240,
                   -18245, -18246, -18250, -18247, -18245, -18241, -18247, -18247, -18245, -18245,
                   -18245, -18246, -18246, -18241, -18240, -18237, -18234, -18241, -18245, -18249,
                   -18247, -18247, -18250, -18249, -18253, -18253, -18251, -18250, -18252, -18254,
                   -18260, -18264, -18266, -18271, -18266, -18265, -18262, -18261, -18258, -18258,
                   -18258, -18261, -18263, -18261, -18262, -18262, -18261, -18259, -18258, -18262,
                   -18265, -18265, -18269, -18269, -18274, -18271, -18274, -18271, -18269, -18268,
                   -18269, -18266, -18266, -18266, -18268, -18272, -18267, -18264, -18264, -18263,
                   -18265, -18266, -18270, -18270, -18272, -18271, -18270, -18267, -18262, -18257,
                   -18255, -18256, -18256, -18251, -18247, -18247, -18251, -18248, -18246, -18243,
                   -18234, -18230, -18233, -18239, -18246, -18249, -18254, -18256, -18253, -18252,
                   -18255, -18256, -18256, -18257, -18257, -18256, -18254, -18254, -18255, -18256,
                   -18257, -18253, -18249, -18248, -18247, -18249, -18248, -18243, -18243, -18243,
                   -18242, -18240, -18240, -18248, -18248, -18244, -18247, -18251, -18258, -18268,
                   -18267, -18262, -18256, -18258, -18263, -18274, -18272, -18270, -18270, -18269,
                   -18272, -18268, -18259, -18259, -18263, -18265, -18270, -18274, -18274, -18270,
                   -18265, -18257, -18251, -18254, -18263, -18272, -18277, -18275, -18266, -18266,
                   -18276, -18276, -18271, -18274, -18279, -18282, -18285, -18285, -18287, -18286,
                   -18282, -18277, -18271, -18268, -18261, -18252, -18250, -18255, -18262, -18260,
                   -18257, -18258, -18258, -18255, -18252, -18243, -18249, -18257, -18254, -18249,
                   -18249, -18259, -18259, -18257, -18256, -18256, -18262, -18260, -18258, -18258,
                   -18257, -18259, -18268, -18273, -18271, -18266, -18260, -18257, -18251, -18249,
                   -18250, -18255, -18256, -18257, -18256, -18259, -18262, -18262, -18258, -18259,
                   -18269, -18278, -18287, -18294, -18294, -18286, -18282, -18277, -18273, -18267,
                   -18266, -18263, -18269, -18276, -18282, -18290, -18292, -18297, -18298, -18295,
                   -18295, -18293, -18288, -18290, -18292, -18298, -18300, -18297, -18287, -18276,
                   -18275, -18282, -18285, -18287, -18289, -18302, -18320, -18327, -18330, -18330,
                   -18334, -18341, -18355, -18363, -18365, -18370, -18379, -18392, -18395, -18396,
                   -18401, -18410, -18418, -18429, -18450, -18472, -18497, -18521, -18538, -18558,
                   -18586, -18615, -18639, -18655, -18678, -18706, -18728, -18742, -18751, -18764,
                   -18775, -18784, -18799, -18814, -18826, -18833, -18839, -18842, -18846, -18846,
                   -18849, -18853, -18864, -18882, -18905, -18927, -18942, -18944, -18947, -18951,
                   -18957, -18963, -18960, -18970, -18973, -18973, -18976, -18986, -18989, -18987,
                   -18984, -18992, -18999, -19001, -19004, -19014, -19024, -19030, -19027, -19032,
                   -19040, -19048, -19050, -19047, -19050, -19055, -19061, -19063, -19067, -19070,
                   -19078, -19081, -19086, -19092, -19105, -19116, -19114, -19119, -19125, -19121,
                   -19114, -19116, -19121, -19126, -19127, -19129, -19132, -19134, -19131, -19130,
                   -19129, -19124, -19122, -19137, -19149, -19154, -19157, -19159, -19161, -19163,
                   -19163, -19166, -19164, -19157, -19153, -19157, -19160, -19159, -19158, -19155,
                   -19145, -19135, -19135, -19138, -19143, -19148, -19152, -19153, -19152, -19155,
                   -19154, -19153, -19152, -19158, -19167, -19178, -19183, -19184, -19178, -19174,
                   -19168, -19167, -19163, -19157, -19161, -19170, -19182, -19190, -19193, -19191,
                   -19187, -19184, -19186, -19189, -19191, -19188, -19180, -19174, -19172, -19174,
                   -19173, -19175, -19177, -19184, -19196, -19205, -19208, -19213, -19215, -19205,
                   -19192, -19186, -19189, -19195, -19205, -19213, -19219, -19213, -19208, -19202,
                   -19191, -19189, -19189, -19191, -19190, -19186, -19180, -19180, -19181, -19182,
                   -19181, -19187, -19189, -19195, -19205, -19217, -19219, -19219, -19211, -19213,
                   -19211, -19207, -19207, -19213, -19222, -19234, -19236, -19234, -19235, -19233,
                   -19228, -19221, -19217, -19214, -19205, -19197, -19195, -19194, -19188, -19182,
                   -19187, -19188, -19191, -19188, -19188, -19194, -19202, -19199, -19200, -19206,
                   -19207, -19207, -19209, -19210, -19217, -19223, -19223, -19217, -19219, -19219,
                   -19210, -19205, -19209, -19221, -19227, -19227, -19235, -19236, -19233, -19230,
                   -19234, -19242, -19242, -19242, -19237, -19243, -19249, -19248, -19244, -19230,
                   -19219, -19221, -19224, -19227, -19227, -19226, -19219, -19219, -19222, -19223,
                   -19224, -19218, -19219, -19226, -19224, -19230, -19239, -19245, -19247, -19243,
                   -19239, -19234, -19233, -19232, -19237, -19237, -19241, -19240, -19234, -19235,
                   -19238, -19247, -19255, -19259, -19261, -19258, -19258, -19253, -19249, -19237,
                   -19228, -19228, -19235, -19245, -19251, -19244, -19239, -19235, -19230, -19221,
                   -19215, -19213, -19215, -19217, -19217, -19221, -19222, -19217, -19215, -19214,
                   -19221, -19227, -19236, -19246, -19252, -19256, -19258, -19259, -19256, -19251,
                   -19247, -19246, -19237, -19224, -19221, -19223, -19223, -19223, -19222, -19219,
                   -19224, -19229, -19228, -19231, -19235, -19239, -19238, -19235, -19236, -19230,
                   -19220, -19213, -19214, -19222, -19221, -19212, -19209, -19213, -19213, -19218,
                   -19225, -19226, -19224, -19224, -19225, -19218, -19219, -19214, -19214, -19218,
                   -19223, -19224, -19223, -19223, -19218, -19213, -19218, -19227, -19239, -19248,
                   -19243, -19233, -19229, -19231, -19234, -19232, -19223, -19216, -19218, -19219,
                   -19217, -19214, -19218, -19214, -19215, -19218, -19230, -19231, -19226, -19220,
                   -19215, -19214, -19221, -19227, -19230, -19224, -19219, -19223, -19232, -19243,
                   -19243, -19237, -19229, -19225, -19225, -19225, -19227, -19227, -19226, -19223,
                   -19222, -19223, -19229, -19226, -19220, -19212, -19209, -19211, -19215, -19222,
                   -19222, -19222, -19219, -19215, -19208, -19209, -19209, -19209, -19213, -19219,
                   -19226, -19232, -19237, -19234, -19225, -19218, -19218, -19222, -19219, -19217,
                   -19206, -19198, -19198, -19201, -19209, -19215, -19224, -19224, -19226, -19225,
                   -19231, -19228, -19223, -19219, -19214, -19212, -19212, -19211, -19208, -19206,
                   -19206, -19210, -19209, -19208, -19204, -19195, -19191, -19189, -19192, -19194,
                   -19191, -19197, -19203, -19207, -19214, -19217, -19219, -19214, -19204, -19203,
                   -19212, -19226, -19234, -19236, -19238, -19236, -19234, -19231, -19226, -19226,
                   -19225, -19223, -19226, -19229, -19223, -19214, -19211, -19215, -19220, -19225,
                   -19228, -19232, -19230, -19234, -19235, -19234, -19228, -19232, -19236, -19236,
                   -19230, -19226, -19220, -19213, -19206, -19198, -19198, -19200, -19198, -19202,
                   -19207, -19212, -19222, -19230, -19234, -19230, -19224, -19220, -19222, -19225,
                   -19229, -19226, -19229, -19230, -19234, -19226, -19220, -19219, -19222, -19229,
                   -19235, -19240, -19243, -19237, -19228, -19223, -19221, -19219, -19218, -19219,
                   -19226, -19232, -19231, -19230, -19231, -19231, -19236, -19238, -19238, -19231,
                   -19221, -19223, -19227, -19228, -19226, -19226, -19227, -19227, -19227, -19223,
                   -19224, -19228, -19227, -19221, -19217, -19212, -19214, -19217, -19212, -19214,
                   -19218, -19218, -19226, -19235, -19239, -19241, -19241, -19240, -19242, -19240,
                   -19237, -19227, -19227, -19229, -19230, -19221, -19214, -19208, -19208, -19206,
                   -19203, -19203, -19206, -19209, -19217, -19226, -19228, -19226, -19230, -19231,
                   -19232, -19231, -19229, -19222, -19216, -19213, -19211, -19211, -19204, -19197,
                   -19202, -19209, -19215, -19219, -19219, -19214, -19204, -19203, -19204, -19214,
                   -19224, -19227, -19226, -19228, -19229, -19225, -19223, -19220, -19228, -19234,
                   -19238, -19248, -19257, -19260, -19254, -19241, -19232, -19221, -19215, -19214,
                   -19218, -19220, -19221, -19218, -19218, -19218, -19222, -19229, -19232, -19227,
                   -19219, -19210, -19202, -19193, -19190, -19186, -19188, -19191, -19196, -19199,
                   -19200, -19200, -19198, -19199, -19194, -19189, -19191, -19186, -19186, -19187,
                   -19185, -19185, -19190, -19198, -19195, -19198, -19198, -19197, -19194, -19192,
                   -19187, -19186, -19186, -19185, -19187, -19194, -19195, -19200, -19204, -19208,
                   -19214, -19214, -19211, -19209, -19206, -19202, -19199, -19205, -19212, -19208,
                   -19207, -19210, -19209, -19210, -19207, -19209, -19209, -19206, -19209, -19214])

    enc = encode(wf, shift=0)
    assert np.array_equal(decode(enc, shift=0)[0], wf)

    wf = np.array([-17947, -17943, -17940, -17936, -17931, -17933, -17930, -17923, -17929, -17932,
                   -17934, -17929, -17927, -17926, -17924, -17925, -17930, -17934, -17938, -17948,
                   -17955, -17960, -17958, -17945, -17936, -17930, -17929, -17933, -17931, -17928,
                   -17932, -17929, -17923, -17924, -17923, -17921, -17914, -17909, -17904, -17903,
                   -17911, -17912, -17915, -17923, -17923, -17923, -17923, -17923, -17925, -17926,
                   -17929, -17934, -17943, -17938, -17928, -17923, -17922, -17920, -17918, -17923,
                   -17926, -17923, -17926, -17928, -17928, -17927, -17931, -17931, -17935, -17940,
                   -17945, -17946, -17939, -17930, -17917, -17918, -17919, -17922, -17927, -17925,
                   -17925, -17925, -17925, -17931, -17928, -17924, -17930, -17929, -17929, -17929,
                   -17924, -17923, -17924, -17930, -17934, -17936, -17941, -17935, -17936, -17940,
                   -17946, -17946, -17949, -17952, -17952, -17945, -17933, -17923, -17922, -17924,
                   -17927, -17927, -17933, -17937, -17944, -17948, -17948, -17950, -17950, -17958,
                   -17960, -17961, -17961, -17962, -17956, -17947, -17942, -17934, -17930, -17925,
                   -17924, -17929, -17931, -17931, -17932, -17933, -17934, -17939, -17944, -17950,
                   -17954, -17958, -17953, -17944, -17941, -17938, -17940, -17941, -17940, -17943,
                   -17939, -17940, -17944, -17947, -17950, -17950, -17946, -17948, -17956, -17962,
                   -17968, -17972, -17971, -17966, -17960, -17964, -17964, -17959, -17951, -17938,
                   -17930, -17927, -17933, -17934, -17934, -17942, -17946, -17949, -17950, -17954,
                   -17960, -17963, -17962, -17965, -17970, -17972, -17969, -17978, -17980, -17979,
                   -17979, -17981, -17979, -17977, -17966, -17955, -17944, -17932, -17930, -17930,
                   -17927, -17931, -17932, -17938, -17940, -17942, -17943, -17952, -17948, -17949,
                   -17949, -17950, -17959, -17962, -17963, -17955, -17951, -17948, -17949, -17949,
                   -17943, -17937, -17935, -17932, -17928, -17925, -17927, -17933, -17934, -17936,
                   -17941, -17943, -17944, -17947, -17951, -17953, -17957, -17959, -17956, -17948,
                   -17941, -17928, -17917, -17913, -17910, -17911, -17907, -17918, -17922, -17927,
                   -17930, -17934, -17941, -17943, -17945, -17946, -17946, -17945, -17937, -17928,
                   -17921, -17919, -17913, -17914, -17917, -17919, -17914, -17912, -17912, -17911,
                   -17911, -17908, -17911, -17922, -17931, -17938, -17935, -17930, -17931, -17927,
                   -17924, -17917, -17913, -17910, -17904, -17901, -17903, -17905, -17904, -17903,
                   -17909, -17910, -17914, -17918, -17923, -17935, -17939, -17935, -17931, -17936,
                   -17939, -17937, -17933, -17937, -17936, -17929, -17932, -17927, -17926, -17927,
                   -17920, -17917, -17915, -17917, -17923, -17933, -17939, -17940, -17946, -17949,
                   -17954, -17955, -17956, -17956, -17956, -17966, -17974, -17974, -17977, -17987,
                   -17987, -17990, -17994, -18004, -18010, -18019, -18023, -18021, -18018, -18022,
                   -18031, -18037, -18045, -18052, -18060, -18077, -18096, -18118, -18138, -18154,
                   -18172, -18189, -18218, -18249, -18277, -18311, -18358, -18417, -18470, -18528,
                   -18594, -18654, -18707, -18759, -18800, -18842, -18872, -18900, -18924, -18937,
                   -18956, -18974, -18991, -19013, -19030, -19039, -19044, -19052, -19055, -19061,
                   -19064, -19068, -19081, -19097, -19109, -19120, -19131, -19142, -19152, -19158,
                   -19161, -19157, -19159, -19166, -19178, -19179, -19181, -19179, -19174, -19176,
                   -19170, -19171, -19178, -19189, -19196, -19198, -19196, -19201, -19206, -19216,
                   -19227, -19239, -19242, -19242, -19242, -19245, -19250, -19244, -19238, -19234,
                   -19229, -19225, -19224, -19228, -19227, -19227, -19221, -19225, -19226, -19226,
                   -19224, -19233, -19237, -19239, -19240, -19242, -19236, -19232, -19230, -19223,
                   -19220, -19215, -19213, -19214, -19211, -19206, -19202, -19198, -19205, -19208,
                   -19213, -19224, -19234, -19235, -19229, -19223, -19235, -19241, -19240, -19243,
                   -19249, -19249, -19252, -19258, -19262, -19255, -19249, -19245, -19239, -19235,
                   -19238, -19237, -19237, -19237, -19233, -19228, -19220, -19218, -19219, -19217,
                   -19218, -19222, -19223, -19228, -19228, -19224, -19226, -19235, -19231, -19228,
                   -19224, -19219, -19219, -19219, -19214, -19212, -19210, -19207, -19200, -19191,
                   -19186, -19186, -19191, -19196, -19192, -19186, -19185, -19191, -19195, -19206,
                   -19211, -19211, -19201, -19191, -19191, -19191, -19197, -19205, -19212, -19215,
                   -19218, -19222, -19223, -19223, -19222, -19219, -19218, -19211, -19206, -19207,
                   -19198, -19186, -19177, -19180, -19179, -19181, -19187, -19192, -19194, -19199,
                   -19200, -19198, -19198, -19204, -19210, -19215, -19217, -19218, -19222, -19215,
                   -19207, -19204, -19205, -19202, -19201, -19207, -19204, -19204, -19208, -19206,
                   -19202, -19196, -19192, -19192, -19191, -19193, -19193, -19203, -19213, -19218,
                   -19215, -19206, -19203, -19203, -19208, -19210, -19211, -19212, -19212, -19210,
                   -19207, -19210, -19211, -19211, -19215, -19217, -19216, -19218, -19223, -19224,
                   -19222, -19218, -19214, -19210, -19207, -19211, -19217, -19220, -19227, -19228,
                   -19229, -19226, -19217, -19213, -19203, -19195, -19191, -19193, -19198, -19207,
                   -19217, -19227, -19226, -19227, -19227, -19225, -19225, -19224, -19223, -19218,
                   -19216, -19216, -19212, -19210, -19212, -19210, -19211, -19213, -19207, -19203,
                   -19198, -19194, -19192, -19194, -19197, -19202, -19209, -19213, -19216, -19221,
                   -19217, -19210, -19208, -19209, -19216, -19216, -19218, -19213, -19207, -19203,
                   -19201, -19194, -19189, -19189, -19194, -19197, -19192, -19195, -19197, -19206,
                   -19211, -19219, -19225, -19225, -19228, -19229, -19230, -19233, -19242, -19239,
                   -19237, -19235, -19230, -19227, -19223, -19217, -19208, -19205, -19210, -19217,
                   -19219, -19214, -19205, -19199, -19196, -19195, -19197, -19200, -19207, -19220,
                   -19226, -19223, -19215, -19202, -19190, -19183, -19191, -19204, -19209, -19209,
                   -19216, -19215, -19215, -19214, -19211, -19211, -19215, -19216, -19214, -19217,
                   -19218, -19223, -19224, -19215, -19212, -19212, -19214, -19212, -19213, -19206,
                   -19207, -19210, -19213, -19217, -19214, -19202, -19196, -19195, -19193, -19193,
                   -19193, -19194, -19191, -19185, -19176, -19176, -19173, -19172, -19178, -19183,
                   -19184, -19186, -19189, -19198, -19199, -19199, -19201, -19201, -19203, -19211,
                   -19213, -19207, -19201, -19202, -19202, -19205, -19198, -19193, -19192, -19185,
                   -19185, -19195, -19195, -19187, -19179, -19180, -19178, -19177, -19184, -19190,
                   -19197, -19201, -19210, -19219, -19221, -19219, -19219, -19212, -19210, -19205,
                   -19209, -19211, -19206, -19201, -19201, -19199, -19203, -19207, -19207, -19193,
                   -19190, -19191, -19191, -19186, -19186, -19187, -19182, -19184, -19195, -19207,
                   -19208, -19211, -19216, -19218, -19224, -19234, -19227, -19223, -19222, -19219,
                   -19216, -19209, -19202, -19193, -19185, -19182, -19180, -19182, -19185, -19186,
                   -19186, -19188, -19187, -19185, -19187, -19189, -19187, -19184, -19176, -19171,
                   -19173, -19170, -19162, -19158, -19159, -19158, -19155, -19153, -19154, -19151,
                   -19151, -19156, -19164, -19169, -19173, -19177, -19183, -19192, -19195, -19197,
                   -19198, -19204, -19206, -19204, -19204, -19199, -19205, -19208, -19202, -19194,
                   -19185, -19175, -19174, -19178, -19183, -19178, -19172, -19171, -19175, -19181,
                   -19181, -19177, -19179, -19178, -19172, -19170, -19176, -19179, -19178, -19176,
                   -19172, -19166, -19165, -19170, -19177, -19176, -19181, -19186, -19187, -19186,
                   -19171, -19162, -19165, -19169, -19172, -19170, -19165, -19165, -19168, -19170,
                   -19171, -19176, -19180, -19170, -19163, -19158, -19155, -19161, -19164, -19165,
                   -19164, -19164, -19167, -19165, -19166, -19164, -19165, -19167, -19172, -19167,
                   -19160, -19157, -19150, -19156, -19150, -19147, -19148, -19143, -19143, -19146,
                   -19155, -19157, -19150, -19145, -19143, -19147, -19155, -19164, -19166, -19175,
                   -19180, -19179, -19179, -19172, -19171, -19171, -19171, -19169, -19168, -19170,
                   -19173, -19173, -19179, -19177, -19174, -19174, -19180, -19179, -19180, -19181,
                   -19188, -19201, -19204, -19201, -19196, -19191, -19194, -19194, -19192, -19183,
                   -19181, -19179, -19173, -19169, -19162, -19153, -19154, -19153, -19153, -19150,
                   -19145, -19146, -19151, -19154, -19149, -19143, -19143, -19139, -19134, -19129,
                   -19132, -19135, -19135, -19127, -19126, -19128, -19134, -19134, -19132, -19141])

    enc = encode(wf, shift=0)
    assert np.array_equal(decode(enc, shift=0)[0], wf)

    wf = np.array([14941, 14935, 14935, 14927, 14921, 14924, 14930, 14935, 14938, 14940, 14942,
                   14940, 14938, 14936, 14935, 14932, 14928, 14929, 14927, 14926, 14934, 14934,
                   14939, 14933, 14933, 14939, 14933, 14934, 14932, 14934, 14936, 14937, 14938,
                   14934, 14924, 14919, 14922, 14923, 14931, 14933, 14941, 14944, 14945, 14943,
                   14931, 14920, 14920, 14929, 14940, 14941, 14944, 14949, 14958, 14956, 14949,
                   14943, 14937, 14936, 14934, 14938, 14943, 14941, 14944, 14942, 14933, 14922,
                   14910, 14913, 14911, 14913, 14914, 14921, 14929, 14931, 14930, 14932, 14929,
                   14934, 14936, 14935, 14932, 14932, 14933, 14939, 14941, 14936, 14925, 14914,
                   14910, 14908, 14914, 14920, 14922, 14925, 14927, 14928, 14929, 14929, 14929,
                   14931, 14932, 14926, 14929, 14932, 14930, 14931, 14931, 14926, 14916, 14910,
                   14903, 14899, 14902, 14907, 14913, 14920, 14921, 14916, 14916, 14915, 14912,
                   14923, 14937, 14934, 14933, 14933, 14922, 14925, 14929, 14933, 14931, 14930,
                   14935, 14935, 14928, 14924, 14926, 14929, 14932, 14937, 14946, 14949, 14946,
                   14945, 14937, 14935, 14931, 14929, 14921, 14921, 14920, 14923, 14927, 14925,
                   14926, 14923, 14922, 14922, 14925, 14930, 14930, 14928, 14932, 14935, 14935,
                   14926, 14926, 14931, 14941, 14951, 14957, 14953, 14949, 14936, 14928, 14931,
                   14932, 14931, 14925, 14926, 14929, 14943, 14952, 14948, 14948, 14950, 14954,
                   14951, 14945, 14942, 14941, 14941, 14937, 14937, 14937, 14936, 14942, 14943,
                   14941, 14933, 14932, 14925, 14924, 14923, 14929, 14937, 14937, 14942, 14936,
                   14926, 14921, 14920, 14924, 14928, 14930, 14934, 14934, 14936, 14938, 14942,
                   14942, 14931, 14930, 14926, 14931, 14936, 14932, 14934, 14933, 14935, 14939,
                   14943, 14957, 14961, 14959, 14949, 14943, 14940, 14942, 14938, 14938, 14934,
                   14929, 14930, 14934, 14940, 14947, 14938, 14932, 14926, 14926, 14916, 14916,
                   14929, 14939, 14945, 14942, 14932, 14927, 14929, 14938, 14941, 14946, 14939,
                   14936, 14938, 14941, 14939, 14941, 14942, 14940, 14934, 14931, 14930, 14928,
                   14937, 14940, 14944, 14948, 14946, 14939, 14932, 14930, 14935, 14941, 14945,
                   14947, 14944, 14933, 14927, 14934, 14940, 14940, 14935, 14935, 14933, 14925,
                   14920, 14925, 14928, 14930, 14930, 14926, 14920, 14918, 14914, 14907, 14909,
                   14917, 14919, 14924, 14924, 14930, 14939, 14937, 14941, 14943, 14934, 14931,
                   14934, 14940, 14941, 14937, 14935, 14936, 14942, 14942, 14943, 14943, 14949,
                   14944, 14941, 14934, 14937, 14943, 14941, 14939, 14940, 14941, 14939, 14941,
                   14939, 14940, 14953, 14953, 14948, 14941, 14936, 14925, 14922, 14927, 14933,
                   14943, 14938, 14940, 14953, 14957, 14948, 14937, 14934, 14928, 14926, 14925,
                   14920, 14925, 14929, 14936, 14934, 14928, 14917, 14910, 14916, 14925, 14929,
                   14929, 14932, 14933, 14938, 14936, 14929, 14929, 14942, 14946, 14945, 14940,
                   14928, 14930, 14937, 14947, 14951, 14945, 14939, 14936, 14936, 14951, 14957,
                   14963, 14964, 14951, 14940, 14932, 14928, 14944, 14952, 14941, 14934, 14930,
                   14921, 14911, 14901, 14900, 14895, 14893, 14885, 14879, 14873, 14867, 14858,
                   14859, 14861, 14865, 14869, 14864, 14867, 14868, 14858, 14844, 14837, 14827,
                   14820, 14806, 14793, 14782, 14776, 14768, 14755, 14743, 14727, 14706, 14690,
                   14671, 14653, 14641, 14620, 14595, 14570, 14544, 14512, 14481, 14454, 14434,
                   14402, 14368, 14321, 14278, 14231, 14175, 14107, 14030, 13948, 13852, 13737,
                   13601, 13444, 13249, 13024, 12765, 12451, 12104, 11760, 11468, 11245, 11089,
                   10973, 10895, 10839, 10792, 10759, 10729, 10710, 10702, 10700, 10694, 10686,
                   10677, 10671, 10661, 10650, 10637, 10633, 10630, 10625, 10616, 10614, 10604,
                   10600, 10594, 10593, 10590, 10578, 10567, 10556, 10555, 10551, 10557, 10556,
                   10558, 10559, 10564, 10565, 10563, 10564, 10567, 10566, 10562, 10554, 10544,
                   10542, 10539, 10532, 10528, 10530, 10537, 10534, 10522, 10515, 10520, 10523,
                   10530, 10535, 10535, 10526, 10522, 10519, 10515, 10512, 10511, 10517, 10522,
                   10521, 10520, 10518, 10519, 10528, 10525, 10521, 10517, 10510, 10507, 10507,
                   10503, 10503, 10506, 10511, 10512, 10511, 10515, 10511, 10510, 10510, 10515,
                   10516, 10516, 10508, 10506, 10507, 10510, 10509, 10512, 10515, 10516, 10522,
                   10531, 10529, 10521, 10519, 10518, 10515, 10512, 10509, 10514, 10512, 10514,
                   10512, 10506, 10501, 10510, 10510, 10509, 10509, 10508, 10506, 10505, 10511,
                   10515, 10518, 10519, 10516, 10514, 10515, 10511, 10504, 10503, 10498, 10493,
                   10495, 10495, 10498, 10490, 10490, 10499, 10505, 10506, 10513, 10518, 10520,
                   10524, 10516, 10508, 10506, 10505, 10507, 10510, 10508, 10505, 10507, 10514,
                   10512, 10513, 10516, 10514, 10511, 10514, 10517, 10516, 10517, 10521, 10527,
                   10534, 10540, 10542, 10544, 10536, 10533, 10531, 10534, 10532, 10523, 10517,
                   10511, 10514, 10514, 10519, 10520, 10526, 10522, 10520, 10517, 10517, 10517,
                   10515, 10513, 10520, 10522, 10525, 10526, 10526, 10530, 10528, 10532, 10534,
                   10535, 10536, 10532, 10534, 10536, 10535, 10526, 10518, 10521, 10521, 10524,
                   10526, 10525, 10524, 10523, 10526, 10523, 10520, 10523, 10524, 10526, 10526,
                   10536, 10538, 10539, 10544, 10552, 10554, 10556, 10553, 10547, 10552, 10549,
                   10548, 10539, 10530, 10520, 10512, 10511, 10512, 10508, 10511, 10524, 10536,
                   10545, 10553, 10552, 10546, 10544, 10541, 10540, 10539, 10537, 10540, 10538,
                   10532, 10531, 10538, 10542, 10547, 10547, 10551, 10557, 10564, 10570, 10569,
                   10560, 10552, 10549, 10548, 10546, 10548, 10550, 10552, 10555, 10561, 10562,
                   10555, 10553, 10552, 10550, 10549, 10549, 10553, 10556, 10564, 10571, 10572,
                   10562, 10557, 10565, 10573, 10572, 10564, 10563, 10569, 10571, 10570, 10566,
                   10557, 10551, 10550, 10554, 10552, 10552, 10557, 10569, 10573, 10572, 10562,
                   10545, 10545, 10557, 10559, 10552, 10552, 10560, 10565, 10564, 10549, 10543,
                   10533, 10537, 10550, 10555, 10552, 10546, 10548, 10552, 10549, 10541, 10538,
                   10538, 10544, 10552, 10553, 10552, 10551, 10562, 10566, 10564, 10556, 10559,
                   10565, 10570, 10575, 10577, 10572, 10566, 10564, 10562, 10557, 10553, 10565,
                   10578, 10582, 10584, 10589, 10590, 10590, 10597, 10599, 10588, 10580, 10575,
                   10573, 10568, 10565, 10570, 10578, 10587, 10589, 10589, 10580, 10575, 10572,
                   10572, 10572, 10569, 10576, 10587, 10592, 10594, 10584, 10581, 10585, 10592,
                   10595, 10596, 10600, 10599, 10600, 10594, 10583, 10578, 10570, 10565, 10567,
                   10570, 10580, 10589, 10598, 10602, 10599, 10592, 10588, 10594, 10595, 10592,
                   10595, 10598, 10603, 10606, 10608, 10604, 10594, 10584, 10580, 10584, 10589,
                   10589, 10594, 10602, 10599, 10603, 10606, 10601, 10599, 10607, 10611, 10613,
                   10617, 10620, 10612, 10612, 10607, 10605, 10605, 10605, 10602, 10594, 10593,
                   10594, 10603, 10608, 10610, 10607, 10602, 10606, 10609, 10618, 10625, 10629,
                   10619, 10606, 10591, 10595, 10598, 10599, 10605, 10613, 10617, 10616, 10613,
                   10613, 10612, 10608, 10600, 10602, 10605, 10602, 10595, 10605, 10612, 10621,
                   10627, 10630, 10631, 10630, 10620, 10605, 10605, 10612, 10618, 10626, 10630,
                   10627, 10621, 10621, 10623, 10622, 10630, 10627, 10629, 10625, 10625, 10625,
                   10636, 10638, 10640, 10638, 10633, 10621, 10617, 10621, 10626, 10630, 10629,
                   10625, 10623, 10621, 10622, 10623, 10621, 10630, 10638, 10637, 10637, 10637,
                   10628, 10619, 10607, 10602, 10605, 10615, 10625, 10637, 10649, 10666])

    enc = encode(wf, shift=0)
    assert np.array_equal(decode(enc, shift=0)[0], wf)

    # this waveform has first derivative values that cannot be represented as int16
    wf = np.array([-17745, -17759, -17771, -17778, -17772, -17763, -17756, -17762, -17779, -17796,
                   -17802, -17799, -17786, -17770, -17756, -17746, -17743, -17744, -17750, -17742,
                   -17723, -17708, -17698, -17698, -17702, -17718, -17739, -17756, -17771, -17777,
                   -17775, -17776, -17780, -17794, -17810, -17822, -17819, -17805, -17783, -17759,
                   -17749, -17746, -17746, -17739, -17730, -17722, -17716, -17710, -17712, -17729,
                   -17747, -17757, -17769, -17778, -17779, -17783, -17784, -17800, -17817, -17831,
                   -17837, -17828, -17811, -17782, -17761, -17747, -17732, -17726, -17720, -17711,
                   -17707, -17698, -17698, -17704, -17709, -17724, -17744, -17770, -17789, -17807,
                   -17826, -17829, -17820, -17806, -17793, -17786, -17788, -17786, -17779, -17765,
                   -17741, -17727, -17710, -17701, -17703, -17715, -17733, -17741, -17747, -17752,
                   -17760, -17769, -17791, -17817, -17839, -17847, -17844, -17830, -17808, -17789,
                   -17775, -17765, -17757, -17752, -17746, -17731, -17710, -17693, -17690, -17708,
                   -17741, -17774, -17800, -17810, -17804, -17800, -17799, -17802, -17810, -17822,
                   -17831, -17830, -17814, -17785, -17747, -17712, -17698, -17700, -17708, -17717,
                   -17714, -17706, -17693, -17687, -17696, -17722, -17757, -17785, -17803, -17805,
                   -17790, -17779, -17782, -17791, -17808, -17821, -17819, -17798, -17759, -17720,
                   -17694, -17685, -17692, -17710, -17730, -17737, -17724, -17703, -17691, -17700,
                   -17725, -17757, -17789, -17799, -17797, -17783, -17768, -17767, -17778, -17798,
                   -17810, -17808, -17779, -17734, -17706, -17687, -17686, -17700, -17725, -17747,
                   -17753, -17747, -17731, -17720, -17727, -17751, -17779, -17803, -17811, -17799,
                   -17782, -17769, -17758, -17750, -17743, -17742, -17744, -17754, -17757, -17765,
                   -17767, -17765, -17767, -17781, -17790, -17789, -17791, -17801, -17808, -17810,
                   -17808, -17799, -17789, -17775, -17759, -17743, -17723, -17710, -17707, -17705,
                   -17703, -17707, -17708, -17714, -17722, -17735, -17752, -17772, -17784, -17790,
                   -17789, -17777, -17767, -17764, -17766, -17766, -17759, -17745, -17727, -17705,
                   -17689, -17678, -17679, -17693, -17713, -17730, -17742, -17746, -17749, -17757,
                   -17774, -17796, -17818, -17830, -17833, -17832, -17823, -17814, -17803, -17796,
                   -17785, -17776, -17763, -17758, -17751, -17741, -17726, -17723, -17722, -17728,
                   -17739, -17754, -17766, -17780, -17786, -17801, -17915, -18110, -18351, -18475,
                   -18291, -17838, -17287, -16843, -16707, -16771, -16967, -17200, -17370, -17480,
                   -17490, -17474, -17599, -17894, -18347, -18808, -19145, -19365, -19452, -19456,
                   -19394, -19205, -18879, -18339, -17556, -16657, 32767, 32767, 32767, 32767,
                   32767, 32767, 32767, -16639, -17071, -17542, -18118, -18788, -19515, -20223,
                   -20808, -21182, -21271, -21079, -20630, -19967, -19136, -18158, -17109, 32767,
                   32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, -16503, -17215, -17974,
                   -18781, -19641, -20483, -21191, -21662, -21814, -21624, -21096, -20302, -19333,
                   -18293, -17275, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767,
                   32767, -17107, -17911, -18704, -19461, -20143, -20693, -21049, -21147, -20966,
                   -20517, -19843, -19026, -18151, -17305, -16559, 32767, 32767, 32767, 32767,
                   32767, 32767, 32767, 32767, -16763, -17392, -18023, -18625, -19137, -19540,
                   -19813, -19993, -20154, -20272, -20338, -20131, -19527, -18575, -17427, -16388,
                   32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, -17223,
                   -18212, -19197, -20117, -20943, -21688, -22267, -22627, -22658, -22228, -21327,
                   -20049, -18551, -17072, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767,
                   32767, 32767, 32767, 32767, -18366, -20247, -21850, -23187, -24190, -24804,
                   -24955, -24602, -23758, -22482, -20925, -19211, -17460, 32767, 32767, 32767,
                   32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, -17737, -20609,
                   -22379, -23580, -24332, -24601, -24418, -23810, -22799, -21476, -19949, -18331,
                   -16734, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767,
                   -16887, -18876, -20361, -21500, -22268, -22636, -22608, -22214, -21506, -20566,
                   -19474, -18294, -17098, 32767, 32767, 32767, 32767, 32767, 32767, 32767, 32767,
                   32767, -17065, -18207, -19316, -20303, -21078, -21578, -21750, -21586, -21094,
                   -20337, -19401, -18362, -17316, 32767, 32767, 32767, 32767, 32767, 32767, 32767,
                   32767, 32767, -16577, -17620, -18672, -19635, -20442, -21054, -21419, -21513,
                   -21326, -20878, -20216, -19390, -18453, -17492, -16573, 32767, 32767, 32767,
                   32767, 32767, 32767, 32767, 32767, -16424, -17249, -18103, -18912, -19619,
                   -20163, -20500, -20622, -20531, -20256, -19817, -19254, -18608, -17923, -17254,
                   -16645, 32767, 32767, 32767, 32767, 32767, 32767, 32767, -16804, -17332, -17879,
                   -18415, -18887, -19258, -19488, -19564, -19489, -19269, -18941, -18531, -18088,
                   -17629, -17196, -16818, -16525, 32767, 32767, 32767, -16459, -16719, -17056,
                   -17432, -17824, -18202, -18546, -18831, -19047, -19157, -19153, -19037, -18806,
                   -18487, -18111, -17704, -17310, -16959, -16675, -16478, 32767, 32767, -16453,
                   -16627, -16874, -17186, -17536, -17914, -18290, -18615, -18864, -19023, -19091,
                   -19057, -18938, -18743, -18479, -18173, -17833, -17494, -17181, -16905, -16703,
                   -16583, -16563, -16630, -16771, -16969, -17208, -17476, -17761, -18042, -18302,
                   -18520, -18682, -18769, -18775, -18702, -18561, -18366, -18135, -17874, -17616,
                   -17381, -17179, -17026, -16932, -16902, -16940, -17047, -17209, -17403, -17619,
                   -17816, -18010, -18174, -18303, -18385, -18427, -18420, -18359, -18252, -18095,
                   -17906, -17710, -17526, -17368, -17248, -17174, -17155, -17182, -17246, -17343,
                   -17466, -17619, -17775, -17930, -18075, -18182, -18263, -18300, -18292, -18252,
                   -18174, -18060, -17921, -17772, -17621, -17471, -17344, -17249, -17194, -17182,
                   -17230, -17315, -17434, -17571, -17715, -17849, -17969, -18076, -18159, -18220,
                   -18252, -18245, -18199, -18112, -18003, -17887, -17765, -17651, -17543, -17445,
                   -17364, -17302, -17283, -17308, -17377, -17473, -17583, -17693, -17791, -17879,
                   -17950, -18016, -18062, -18094, -18104, -18082, -18031, -17961, -17867, -17762,
                   -17669, -17595, -17541, -17512, -17497, -17501, -17531, -17568, -17616, -17678,
                   -17746, -17820, -17892, -17939, -17969, -17972, -17955, -17924, -17890, -17856,
                   -17808, -17762, -17707, -17644, -17586, -17551, -17535, -17559, -17599, -17649,
                   -17711, -17757, -17793, -17824, -17849, -17885, -17931, -17969, -17989, -17978,
                   -17926, -17852, -17783, -17725, -17675, -17641, -17616, -17598, -17578, -17565,
                   -17566, -17581, -17619, -17669, -17734, -17795, -17853, -17885, -17902, -17904,
                   -17892, -17880, -17860, -17837, -17801, -17757, -17714, -17662, -17619, -17586,
                   -17572, -17578, -17602, -17631, -17658, -17693, -17730, -17767, -17807, -17851,
                   -17880, -17898, -17901, -17892, -17862, -17829, -17797, -17768, -17743, -17713,
                   -17684, -17654, -17628, -17608, -17609, -17633, -17675, -17721, -17764, -17796,
                   -17820, -17834, -17845, -17853, -17858, -17857, -17847, -17821, -17784, -17745,
                   -17701, -17665, -17635, -17611, -17613, -17624, -17637, -17657, -17678, -17694,
                   -17714, -17747, -17784, -17831, -17873, -17902, -17903, -17896, -17872, -17838,
                   -17803, -17776, -17754, -17735, -17719, -17693, -17664, -17641, -17634, -17647,
                   -17673, -17714, -17759, -17800, -17831, -17843, -17851, -17865, -17883, -17900,
                   -17906, -17899, -17878, -17849, -17807, -17763, -17719, -17682, -17658, -17648,
                   -17638, -17626, -17622, -17631, -17647, -17668, -17703, -17746, -17790, -17833,
                   -17866, -17880, -17883, -17875, -17863, -17852, -17831, -17812, -17786, -17748,
                   -17703, -17668, -17637, -17616, -17607, -17620, -17634, -17656, -17680, -17709,
                   -17738, -17772, -17808, -17847, -17881, -17898, -17900, -17891, -17869, -17843,
                   -17818, -17795, -17774, -17744, -17719, -17697, -17684, -17675, -17673, -17679,
                   -17698, -17722, -17751, -17765, -17779, -17802, -17823, -17842, -17849, -17846,
                   -17825, -17808, -17791, -17772, -17742, -17714, -17688, -17663, -17652, -17650,
                   -17658, -17669, -17683, -17702, -17727, -17756, -17785, -17817, -17838, -17839,
                   -17833])

    enc_wf, stop = encode(wf, shift=0)

    (nsig_c, shift, enc_wf_c) = read_sigcompress_c_output(config_dir / "special-wf-clipped.dat")
    assert shift == 0
    assert np.array_equal(_to_u16(enc_wf[:stop]), enc_wf_c)

    dec_wf, stop = decode((enc_wf, stop), shift=0)
    assert np.array_equal(dec_wf, wf)
    # fmt: on
