# Sessions

Serving supports pluggable session providers and a convenient `Session` mapping type that gives easy access to a request's session data.

## Configure in YAML

```yaml
# serving.dev.yaml
session:
  session_provider: serving.session:InMemorySessionProvider  # module:ClassName
  # Optional; your own Session subclass that customizes behavior
  session_type: serving.session:Session
  # Options passed as kwargs to the provider constructor
  config: {}
```

- `session_provider` is required; it must implement the `SessionProvider` protocol
- `session_type` is optional; defaults to `serving.session:Session`
- `config` is a free-form dict passed to your provider's constructor

When configured, Serving wires the provider and enables injection for `Session` (mapping) and `SessionParam` (key-level lookup).

## Provider Protocol

```python
from typing import Protocol

class SessionProvider(Protocol):
    async def create_session(self) -> str: ...
    async def update_session(self, token: str, values: dict[str, object]) -> None: ...
    async def get_session(self, token: str) -> dict[str, object] | None: ...
    async def invalidate_session(self, token: str) -> None: ...
```

Serving includes `InMemorySessionProvider` suitable for development and tests. For production implement your own (e.g., Redis, database) and configure it via `session.session_provider`.

Session tokens are generated by the configured `CredentialProvider` via `create_session_token()` and may be validated with `validate_session_token()`.

## Session Mapping

`Session` is a dict-like mapping bound to the current request:

```python
from serving.session import Session

# Load or create the session for the current request
sess = await Session.load_session()

# Read/write like a dict
sess["user_id"] = "u123"
sess["feature_flags"] = {"beta": True}

# Persist changes
await sess.save()

# Invalidate fully
await sess.invalidate()
```

Methods and dunder support:
- `__getitem__`, `__setitem__`, `__delitem__`, `__iter__`, `__len__`, `__contains__`
- `get(key, default=None)`, `await save()`, `await invalidate()`

The default `Session` stores the token in a cookie named `serving_session`.

## Injecting Session and SessionParam

You can inject the whole session or specific values via `SessionParam`:

```python
from serving.session import Session
from serving.injectors import SessionParam

@app.route("/whoami")
async def whoami(sess: Session) -> JSON:
    return {"user": sess.get("user_id")}

@app.route("/feature")
async def is_beta(beta: SessionParam[bool] = False) -> JSON:
    # Looks up key "beta" in the session; default used if absent
    return {"beta": beta}
```

To override the key name, use `typing.Annotated` with a string metadata specifying the session key:

```python
from typing import Annotated

@app.route("/feature")
async def is_beta(beta: Annotated[SessionParam[bool], "feature_flags.beta"] = False) -> JSON:
    ...
```

Note: if a value is present with `None`, `SessionParam` returns `None` (it distinguishes between missing and present-`None`).
