import * as iam from '../../aws-iam';
import * as logs from '../../aws-logs';
import * as s3 from '../../aws-s3';
import { IResource, Resource } from '../../core';
import { Construct } from 'constructs';
import { ISubnet, IVpc } from './vpc';
/**
 * A FlowLog
 */
export interface IFlowLog extends IResource {
    /**
     * The Id of the VPC Flow Log
     *
     * @attribute
     */
    readonly flowLogId: string;
}
/**
 * The type of VPC traffic to log
 */
export declare enum FlowLogTrafficType {
    /**
     * Only log accepts
     */
    ACCEPT = "ACCEPT",
    /**
     * Log all requests
     */
    ALL = "ALL",
    /**
     * Only log rejects
     */
    REJECT = "REJECT"
}
/**
 * The available destination types for Flow Logs
 */
export declare enum FlowLogDestinationType {
    /**
     * Send flow logs to CloudWatch Logs Group
     */
    CLOUD_WATCH_LOGS = "cloud-watch-logs",
    /**
     * Send flow logs to S3 Bucket
     */
    S3 = "s3"
}
/**
 * The type of resource to create the flow log for
 */
export declare abstract class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to
     */
    static fromSubnet(subnet: ISubnet): FlowLogResourceType;
    /**
     * The VPC to attach the Flow Log to
     */
    static fromVpc(vpc: IVpc): FlowLogResourceType;
    /**
     * The Network Interface to attach the Flow Log to
     */
    static fromNetworkInterfaceId(id: string): FlowLogResourceType;
    /**
     * The type of resource to attach a flow log to.
     */
    abstract resourceType: string;
    /**
     * The Id of the resource that the flow log should be attached to.
     */
    abstract resourceId: string;
}
/**
 * The file format for flow logs written to an S3 bucket destination
 */
export declare enum FlowLogFileFormat {
    /**
     * File will be written as plain text
     *
     * This is the default value
     */
    PLAIN_TEXT = "plain-text",
    /**
     * File will be written in parquet format
     */
    PARQUET = "parquet"
}
/**
 * Options for writing logs to a S3 destination
 */
export interface S3DestinationOptions {
    /**
     * Use Hive-compatible prefixes for flow logs
     * stored in Amazon S3
     *
     * @default false
     */
    readonly hiveCompatiblePartitions?: boolean;
    /**
     * The format for the flow log
     *
     * @default FlowLogFileFormat.PLAIN_TEXT
     */
    readonly fileFormat?: FlowLogFileFormat;
    /**
     * Partition the flow log per hour
     *
     * @default false
     */
    readonly perHourPartition?: boolean;
}
/**
 * Options for writing logs to a destination
 *
 * TODO: there are other destination options, currently they are
 * only for s3 destinations (not sure if that will change)
 */
export interface DestinationOptions extends S3DestinationOptions {
}
/**
 * The destination type for the flow log
 */
export declare abstract class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination
     */
    static toCloudWatchLogs(logGroup?: logs.ILogGroup, iamRole?: iam.IRole): FlowLogDestination;
    /**
     * Use S3 as the destination
     *
     * @param bucket optional s3 bucket to publish logs to. If one is not provided
     * a default bucket will be created
     * @param keyPrefix optional prefix within the bucket to write logs to
     * @param options additional s3 destination options
     */
    static toS3(bucket?: s3.IBucket, keyPrefix?: string, options?: S3DestinationOptions): FlowLogDestination;
    /**
     * Generates a flow log destination configuration
     */
    abstract bind(scope: Construct, flowLog: FlowLog): FlowLogDestinationConfig;
}
/**
 * Flow Log Destination configuration
 */
export interface FlowLogDestinationConfig {
    /**
     * The type of destination to publish the flow logs to.
     *
     * @default - CLOUD_WATCH_LOGS
     */
    readonly logDestinationType: FlowLogDestinationType;
    /**
     * The IAM Role that has access to publish to CloudWatch logs
     *
     * @default - default IAM role is created for you
     */
    readonly iamRole?: iam.IRole;
    /**
     * The CloudWatch Logs Log Group to publish the flow logs to
     *
     * @default - default log group is created for you
     */
    readonly logGroup?: logs.ILogGroup;
    /**
     * S3 bucket to publish the flow logs to
     *
     * @default - undefined
     */
    readonly s3Bucket?: s3.IBucket;
    /**
     * S3 bucket key prefix to publish the flow logs to
     *
     * @default - undefined
     */
    readonly keyPrefix?: string;
    /**
     * Options for writing flow logs to a supported destination
     *
     * @default - undefined
     */
    readonly destinationOptions?: DestinationOptions;
}
/**
 * Options to add a flow log to a VPC
 */
export interface FlowLogOptions {
    /**
     * The type of traffic to log. You can log traffic that the resource accepts or rejects, or all traffic.
     *
     * @default ALL
     */
    readonly trafficType?: FlowLogTrafficType;
    /**
     * Specifies the type of destination to which the flow log data is to be published.
     * Flow log data can be published to CloudWatch Logs or Amazon S3
     *
     * @default FlowLogDestinationType.toCloudWatchLogs()
     */
    readonly destination?: FlowLogDestination;
}
/**
 * Properties of a VPC Flow Log
 */
export interface FlowLogProps extends FlowLogOptions {
    /**
     * The name of the FlowLog
     *
     * It is not recommended to use an explicit name.
     *
     * @default If you don't specify a flowLogName, AWS CloudFormation generates a
     * unique physical ID and uses that ID for the group name.
     */
    readonly flowLogName?: string;
    /**
     * The type of resource for which to create the flow log
     */
    readonly resourceType: FlowLogResourceType;
}
/**
 * The base class for a Flow Log
 */
declare abstract class FlowLogBase extends Resource implements IFlowLog {
    /**
     * The Id of the VPC Flow Log
     *
     * @attribute
     */
    abstract readonly flowLogId: string;
}
/**
 * A VPC flow log.
 * @resource AWS::EC2::FlowLog
 */
export declare class FlowLog extends FlowLogBase {
    /**
     * Import a Flow Log by it's Id
     */
    static fromFlowLogId(scope: Construct, id: string, flowLogId: string): IFlowLog;
    /**
     * The Id of the VPC Flow Log
     *
     * @attribute
     */
    readonly flowLogId: string;
    /**
     * The S3 bucket to publish flow logs to
     */
    readonly bucket?: s3.IBucket;
    /**
     * S3 bucket key prefix to publish the flow logs under
     */
    readonly keyPrefix?: string;
    /**
     * The iam role used to publish logs to CloudWatch
     */
    readonly iamRole?: iam.IRole;
    /**
     * The CloudWatch Logs LogGroup to publish flow logs to
     */
    readonly logGroup?: logs.ILogGroup;
    constructor(scope: Construct, id: string, props: FlowLogProps);
}
export {};
