export interface GraphNodeProps<A> {
    readonly data?: A;
}
export declare class GraphNode<A> {
    readonly id: string;
    static of<A>(id: string, data: A): GraphNode<A>;
    readonly dependencies: GraphNode<A>[];
    readonly data?: A;
    private _parentGraph?;
    constructor(id: string, props?: GraphNodeProps<A>);
    /**
     * A graph-wide unique identifier for this node. Rendered by joining the IDs
     * of all ancestors with hyphens.
     */
    get uniqueId(): string;
    /**
     * The union of all dependencies of this node and the dependencies of all
     * parent graphs.
     */
    get allDeps(): GraphNode<A>[];
    dependOn(...dependencies: Array<GraphNode<A> | undefined>): void;
    ancestorPath(upTo: GraphNode<A>): GraphNode<A>[];
    rootPath(): GraphNode<A>[];
    get root(): GraphNode<A>;
    get parentGraph(): Graph<A> | undefined;
    /**
     * @internal
     */
    _setParentGraph(parentGraph: Graph<A>): void;
    toString(): string;
}
/**
 * A dependency set that can be constructed partially and later finished
 *
 * It doesn't matter in what order sources and targets for the dependency
 * relationship(s) get added. This class can serve as a synchronization
 * point if the order in which graph nodes get added to the graph is not
 * well-defined.
 *
 * Useful utility during graph building.
 */
export declare class DependencyBuilder<A> {
    private readonly targets;
    private readonly sources;
    dependOn(...targets: GraphNode<A>[]): this;
    dependBy(...sources: GraphNode<A>[]): this;
}
export declare class DependencyBuilders<K, A> {
    private readonly builders;
    get(key: K): DependencyBuilder<A>;
}
export interface GraphProps<A> extends GraphNodeProps<A> {
    /**
     * Initial nodes in the workflow
     */
    readonly nodes?: GraphNode<A>[];
}
export declare class Graph<A> extends GraphNode<A> {
    static of<A, B>(id: string, data: A, nodes?: GraphNode<B>[]): Graph<A | B>;
    private readonly children;
    constructor(name: string, props?: GraphProps<A>);
    get nodes(): Set<GraphNode<A>>;
    tryGetChild(name: string): GraphNode<A> | undefined;
    contains(node: GraphNode<A>): boolean;
    add(...nodes: Array<GraphNode<A>>): void;
    absorb(other: Graph<A>): void;
    /**
     * Return topologically sorted tranches of nodes at this graph level
     */
    sortedChildren(): GraphNode<A>[][];
    /**
     * Return a topologically sorted list of non-Graph nodes in the entire subgraph
     */
    sortedLeaves(): GraphNode<A>[][];
    consoleLog(indent?: number): void;
    /**
     * Return the union of all dependencies of the descendants of this graph
     */
    private deepDependencies;
    /**
     * Return all non-Graph nodes
     */
    allLeaves(): GraphNodeCollection<A>;
}
/**
 * A collection of graph nodes
 */
export declare class GraphNodeCollection<A> {
    readonly nodes: GraphNode<A>[];
    constructor(nodes: Iterable<GraphNode<A>>);
    dependOn(...dependencies: Array<GraphNode<A> | undefined>): void;
    /**
    * Returns the graph node that's shared between these nodes
    */
    commonAncestor(): GraphNode<A>;
}
export declare function isGraph<A>(x: GraphNode<A>): x is Graph<A>;
