from pathlib import Path

import numpy as np
import torch

from eir.data_load.data_preparation_modules.common import (
    process_tensor_to_length,
)
from eir.setup.input_setup_modules.setup_bytes import ComputedBytesInputInfo
from eir.setup.schemas import ByteInputDataConfig


def bytes_load_wrapper(
    data_pointer: Path | int,
    dtype: str,
) -> np.ndarray:
    assert isinstance(data_pointer, str | Path)
    bytes_data = np.fromfile(file=data_pointer, dtype=dtype)

    return bytes_data


def prepare_bytes_data(
    bytes_input_object: "ComputedBytesInputInfo",
    bytes_data: np.ndarray | torch.Tensor,
    test_mode: bool,
) -> torch.Tensor:
    """
    We use clone here to copy the original data, vs. using from_numpy
    which shares memory, causing us to modify the original data.
    """
    bio = bytes_input_object
    input_type_info = bio.input_config.input_type_info
    assert isinstance(input_type_info, ByteInputDataConfig), input_type_info

    sampling_strategy = input_type_info.sampling_strategy_if_longer
    if test_mode:
        sampling_strategy = "from_start"

    bytes_data_copy = bytes_data
    if isinstance(bytes_data, np.ndarray):
        bytes_data_copy = bytes_data.copy()
    bytes_tensor = torch.LongTensor(bytes_data_copy).detach().clone()

    padding_value = bio.vocab.get("<pad>", 0)
    cur_bytes_padded = process_tensor_to_length(
        tensor=bytes_tensor,
        max_length=bytes_input_object.computed_max_length,
        sampling_strategy_if_longer=sampling_strategy,
        padding_value=padding_value,
    )

    return cur_bytes_padded
