# UFR Design System - Enhanced Technical Specification

[![CI](https://github.com/wespiper/ufr-ds/actions/workflows/ci.yml/badge.svg)](https://github.com/wespiper/ufr-ds/actions/workflows/ci.yml)
[![PyPI](https://img.shields.io/pypi/v/ufr-ds.svg)](https://pypi.org/project/ufr-ds/)
[![npm (tokenizer)](https://img.shields.io/npm/v/%40ufr/ast-tokenizer.svg)](https://www.npmjs.com/package/@ufr/ast-tokenizer)

## Executive Summary

### What We're Building

UFR-DS is a design grammar engine that analyzes existing React design systems (e.g., MUI, custom component libraries) to discover their underlying patterns, compress redundancies, and expose a shared grammar layer that designers and developers can work with together.

Unlike traditional code generation tools, UFR-DS is not about replacing developer ownership of code. Instead, it acts as a translator between design and implementation:

-   Designers work in tokens and patterns.
-   Developers work in React code.
-   UFR-DS ensures these two worlds stay in sync via a shared grammar.

### Core Value Proposition

-   **Migration readiness**: Analyze a legacy system (like MUI usage) and output a migration map to a custom system.
-   **Compression & deduplication**: Reduce hundreds of slightly different components into clean, canonical pattern families.
-   **Design-dev bridge**: Provide a visual grammar explorer, Storybook integration, and Figma mapping so both sides share the same vocabulary.
-   **Standards enforcement**: Accessibility, performance, and testing are baked into the grammar.

## Architecture Overview

```
Input: React Codebase (TypeScript components + Design Tokens + Figma refs)
↓
Analyzer & Tokenizer (AST + Tokens)
↓
UFR Grammar Core (canonical tokens, component families, usage invariants)
↓
Translator & Collaboration Tools (Figma ↔ React mapping, MUI ↔ custom migration, Storybook, docs)
```

## Key Technical Components

### 1. Pattern Discovery Engine

-   **AST Parser**: Analyzes React component files using Babel parser
-   **Tokenizer**: Converts component structures to analyzable token sequences
-   **Grammar Induction**: Uses RePair algorithm and Minimum Description Length (MDL) optimization
-   **Pattern Classification**: Groups similar components into pattern families

### 2. Grammar Core

-   **Canonical Tokens**: Standardized representation of UI patterns
-   **Component Families**: Groups of related components (e.g., Button variants)
-   **Usage Invariants**: Rules that ensure consistency across implementations
-   **Compression Engine**: Achieves 5-10x reduction in component count

### 3. Translation Layer

-   **Component Generator**: Converts grammar patterns back to React components
-   **Migration Mapper**: Creates transition plans from legacy to optimized systems
-   **Type Preservation**: Maintains TypeScript definitions through transformation
-   **Variant System**: Handles component variations through props rather than duplication

## Implementation Architecture

```
ufr-design-system/
├── packages/
│   ├── core/                    # Grammar engine and pattern discovery
│   │   ├── src/
│   │   │   ├── engine/          # RePair + MDL implementation
│   │   │   ├── grammar/         # Grammar structures and operations
│   │   │   └── emergence/       # Pattern emergence detection
│   │
│   ├── react-analyzer/          # React-specific analysis
│   │   ├── src/
│   │   │   ├── parser/          # AST parsing and tokenization
│   │   │   ├── patterns/        # Pattern recognition
│   │   │   └── analyzer.ts      # Main analysis engine
│   │
│   ├── compiler/                # Grammar to React compiler
│   │   ├── src/
│   │   │   ├── generator/       # Component generation
│   │   │   └── optimizer/       # Code optimization
│   │
│   ├── visual-editor/           # Designer interface
│   │   └── src/                # Grammar Explorer UI
│   │
│   └── cli/                     # Command-line interface
│       └── src/commands/        # analyze, generate, dedupe commands
```

## Key Features

### 1. Pattern Discovery

Parse a React codebase into tokens and detect recurring structures using advanced grammar induction algorithms.

**Example Analysis Output:**

```
✓ Found 127 components
✓ Extracted 18 canonical patterns
✓ Identified 42 duplicates
✓ Compression ratio: 7.0x

Suggested Consolidations:
• Button, PrimaryButton, SecondaryButton → Button with variant
• IconButton, Fab → Button with size + icon
```

### 2. Deduplication Reports

CLI tool outputs duplicates, frequency, and suggested consolidations with actionable refactoring suggestions.

### 3. Migration Mapping

Analyze an existing system and output grammar + migration suggestions for transitioning from legacy systems to optimized implementations.

### 4. Design-Dev Bridge

-   **Grammar Explorer UI**: Visual interface for browsing discovered patterns
-   **Figma Plugin**: Token mapping between design and code
-   **Storybook Integration**: Auto-generated stories for pattern documentation

### 5. Standards Enforcement

-   **Accessibility Checks**: Ensure WCAG compliance in generated components
-   **Performance Constraints**: Optimize bundle size and runtime performance
-   **Testing Hooks**: Auto-generate test suites for pattern families

## Example Workflow

### Analyzing a Legacy System

```bash
ufr analyze ./node_modules/@mui/material --output analysis.json
```

**Output:**

```
✓ Found 127 components
✓ Extracted 18 canonical patterns
✓ Identified 42 duplicates
✓ Compression ratio: 7.0x

Suggested Consolidations:
• Button, PrimaryButton, SecondaryButton → Button with variant
• IconButton, Fab → Button with size + icon
```

### Generating Optimized Components

```bash
ufr generate analysis.json --output ./optimized --typescript --storybook
```

**Generated Component Example:**

```typescript
// Before: 3 separate components
import { Button, PrimaryButton, SecondaryButton } from './legacy';

// After: 1 pattern-based component
import { Button } from '@ufr/optimized';

// Usage with variants
<Button variant="primary" />
<Button variant="secondary" />
<Button variant="icon" icon={<CheckIcon />} />
```

## Success Metrics

### Technical Metrics

-   **Compression ratio**: 5-10× component reduction
-   **Type safety**: 95%+ generated code compiles without errors
-   **Performance**: 20%+ bundle size reduction through deduplication
-   **Speed**: CLI analyzes 1000+ components in <30 seconds

### Adoption Metrics

-   **Migration coverage**: % of legacy components mapped to new grammar
-   **Developer adoption**: Number of teams using grammar explorer/CLI
-   **Design-dev alignment**: Reduction in translation errors between design and code

## Roadmap

-   **Phase 1**: AST parsing + tokenization → compression reports
-   **Phase 2**: Grammar Explorer UI + Storybook integration
-   **Phase 3**: Figma plugin for token mapping
-   **Phase 4**: Migration mapping and CI integration

## Technical Requirements

### Dependencies

-   **Core**: TypeScript, React, Babel (parser/traverse/types)
-   **CLI**: Commander.js, Chalk, Ora
-   **Analysis**: RePair algorithm implementation, MDL optimization
-   **Generation**: TypeScript Compiler API, Prettier

### Performance Targets

-   Parse 1000+ components in <30 seconds

## Quick Start (Prototype CLI)

- Analyze a React project directory using the vendored Emergence Engine:

  python -m ufr_ds.cli analyze ./examples/material-ui-subset --pretty

- Include emergence analysis (entropy trajectory + events):

  python -m ufr_ds.cli analyze ./examples/material-ui-subset --emergence --pretty

- Generate canonical components from discovered families (writes TSX stubs):

  python -m ufr_ds.cli generate ./examples/material-ui-subset --out ./generated --pretty

What this does:

- Tokenizes JSX/TSX/JS files for tags, props, and imports (regex-based).
- Runs RePair + MDL scoring to induce a grammar and compute compression.
- Reports discovered rules and top tokens; optional emergence events.
 - Generates family-based component skeletons with a `variant` prop.
-   Generate optimized library in <10 seconds
-   Visual editor loads in <2 seconds
-   Support codebases up to 10,000 components

## Integration Examples

### With Existing Workflow

```typescript
// Current: Multiple similar components
import { PrimaryButton, SecondaryButton, IconButton } from './components';

// After UFR Analysis: Single flexible component
import { Button } from '@ufr/optimized';

// Automatic migration suggestions
<Button variant="primary" />    // was PrimaryButton
<Button variant="secondary" />  // was SecondaryButton
<Button variant="icon" />       // was IconButton
```

### With Design Systems

```typescript
// Integration with design tokens
const Button = ({ variant, size, ...props }) => {
    return (
        <StyledButton
            className={buttonVariants({ variant, size })}
            {...props}
        />
    );
};

// Auto-generated from discovered patterns
const buttonVariants = cva("base-button-styles", {
    variants: {
        variant: {
            primary: "bg-primary text-white",
            secondary: "bg-secondary text-gray",
        },
        size: {
            small: "px-2 py-1 text-sm",
            large: "px-4 py-2 text-lg",
        },
    },
});
```

## Philosophy

The best design system is not the biggest—it's the smallest set of reusable patterns that both designers and developers can own together. UFR-DS exists to discover, enforce, and evolve that shared grammar.

**Core Principles:**

-   **Developer Ownership**: Code remains in developer control, UFR-DS provides translation
-   **Pattern-First**: Focus on discovering existing patterns rather than imposing new ones
-   **Compression Over Creation**: Reduce complexity rather than add new abstractions
-   **Shared Vocabulary**: Bridge design and development through common grammar
-   **Evidence-Based**: All recommendations backed by actual usage analysis

## Getting Started

### Installation

```bash
npm install -g @ufr/cli
# or
npx @ufr/cli analyze ./src/components
```

### Quick Analysis

```bash
# Analyze your components
ufr analyze ./src/components --output analysis.json

# Review suggestions
ufr report analysis.json

# Generate optimized components
ufr generate analysis.json --output ./optimized
```

### Integration

```bash
# Add to existing project
npm install @ufr/core @ufr/react-analyzer

# Use programmatically
import { ReactPatternAnalyzer } from '@ufr/react-analyzer';
const analyzer = new ReactPatternAnalyzer();
const results = await analyzer.analyzeCodbase('./src');
```

---

**Next Steps:** Start with a pilot analysis of your existing component library to see compression opportunities, then gradually adopt the generated patterns while maintaining your existing development workflow.
