# This file is part of monday-client.
#
# Copyright (C) 2024 Leet Cyber Security <https://leetcybersecurity.com/>
#
# monday-client is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# monday-client is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with monday-client. If not, see <https://www.gnu.org/licenses/>.

"""
Monday.com API asset type definitions and structures.

This module contains dataclasses that represent Monday.com asset objects,
including files, images, and other uploaded content with their metadata.
"""

from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING, Any

if TYPE_CHECKING:
    from monday.types.user import User


@dataclass
class Asset:
    """
    Represents a Monday.com asset (file) with its metadata and URLs.

    This dataclass maps to the Monday.com API asset object structure, containing
    fields like name, file size, URLs, and upload information.

    See Also:
        https://developer.monday.com/api-reference/reference/assets#fields

    """

    created_at: str = ''
    """The asset's creation date. Returned as ``YYYY-MM-DDTHH:MM:SS``"""

    file_extension: str = ''
    """The asset's extension"""

    file_size: int = 0
    """The asset's size in bytes"""

    id: str = ''
    """The asset's unique identifier"""

    name: str = ''
    """The asset's name"""

    original_geometry: str = ''
    """The asset's original geometry"""

    public_url: str = ''
    """The asset's public URL (valid for 1 hour). Accessing this link will allow users without a monday.com user profile to see the file directly while the link is valid."""

    uploaded_by: User | None = None
    """The user who uploaded the asset. This field will not return anything if the asset is a duplicate of something generated by a system."""

    url: str = ''
    """The asset's URL. This will only be available to users who have access to the file as part of your account. If the asset is stored on a private or shareable board, it will also need to be part of the board in question."""

    url_thumbnail: str = ''
    """The URL to view the asset in thumbnail mode. Only available for images."""

    def to_dict(self) -> dict[str, Any]:
        """Convert to dictionary for API requests."""
        result = {}

        if self.created_at:
            result['created_at'] = self.created_at
        if self.file_extension:
            result['file_extension'] = self.file_extension
        if self.file_size:
            result['file_size'] = self.file_size
        if self.id:
            result['id'] = self.id
        if self.name:
            result['name'] = self.name
        if self.original_geometry:
            result['original_geometry'] = self.original_geometry
        if self.public_url:
            result['public_url'] = self.public_url
        if self.uploaded_by:
            result['uploaded_by'] = (
                self.uploaded_by.to_dict()
                if hasattr(self.uploaded_by, 'to_dict')
                else self.uploaded_by
            )
        if self.url:
            result['url'] = self.url
        if self.url_thumbnail:
            result['url_thumbnail'] = self.url_thumbnail

        return result

    @classmethod
    def from_dict(cls, data: dict[str, Any]) -> Asset:
        """Create from dictionary."""
        from monday.types.user import User  # noqa: PLC0415

        return cls(
            created_at=str(data.get('created_at', '')),
            file_extension=str(data.get('file_extension', '')),
            file_size=int(data.get('file_size', 0)),
            id=str(data.get('id', '')),
            name=str(data.get('name', '')),
            original_geometry=str(data.get('original_geometry', '')),
            public_url=str(data.get('public_url', '')),
            uploaded_by=User.from_dict(data['uploaded_by'])
            if data.get('uploaded_by')
            else None,
            url=str(data.get('url', '')),
            url_thumbnail=str(data.get('url_thumbnail', '')),
        )
