from __future__ import annotations

from pathlib import Path
import logging

from trendify.api.base.data_product import ProductGenerator
from trendify.api.generator.data_product_collection import DataProductCollection
from trendify.api.base.helpers import DATA_PRODUCTS_FNAME_DEFAULT

__all__ = ["DataProductGenerator"]

logger = logging.getLogger(__name__)


class DataProductGenerator:
    """
    A wrapper for saving the data products generated by a user defined function

    Args:
        processor (ProductGenerator): A callable that receives a working directory
            and returns a list of data products.
    """

    def __init__(self, processor: ProductGenerator):
        self._processor = processor

    def process_and_save(
        self, workdir: Path, data_products_fname: str = DATA_PRODUCTS_FNAME_DEFAULT
    ):
        """
        Runs the user-defined processor method stored at instantiation.

        Saves the returned products to a JSON file in the same directory.

        Args:
            workdir (Path): working directory on which to run the processor method.
            data_products_fname (str): Name of data products file
        """

        logger.info(f"Processing {workdir = } with {self._processor = }")
        collection = DataProductCollection.from_iterable(self._processor(workdir))
        if collection.elements:
            workdir.mkdir(exist_ok=True, parents=True)
            workdir.joinpath(data_products_fname).write_text(
                collection.model_dump_json()
            )
