from __future__ import annotations

import argparse
import asyncio
import contextlib
import logging
import sys
from argparse import ArgumentParser, Namespace
from collections.abc import Callable, Sequence
from importlib.resources import files
from pathlib import Path
from typing import Any, Literal

from rich.table import Table

import WebtoonScraper
from WebtoonScraper import __version__
from WebtoonScraper.base import console, logger, platforms
from WebtoonScraper.exceptions import PlatformError, URLError
from WebtoonScraper.scrapers import EpisodeRange, Scraper


class LazyVersionAction(argparse._VersionAction):
    version: Callable[[], str] | str | None

    def __call__(
        self,
        parser: ArgumentParser,
        namespace: Namespace,
        values: str | Sequence[Any] | None,
        option_string: str | None = None,
    ) -> None:
        if callable(self.version):
            self.version = self.version()
        return super().__call__(parser, namespace, values, option_string)


def _version_info() -> str:
    def check_dependency():
        ALL_DEPENDENCIES = {
            "lezhin_comics": "download Lezhin Comics (partially)",
        }
        installed = set()

        # fmt: off  # 맥락상 import와 installed가 서로 붙어있는 편이 어울림
        with contextlib.suppress(Exception):
            from PIL import Image  # noqa

            installed.add("lezhin_comics")
        # fmt: on

        missing_dependencies = ALL_DEPENDENCIES.keys() - installed
        match len(missing_dependencies):
            case 0:
                return "✅ All extra dependencies are installed!"

            case 1:
                missing = missing_dependencies.pop()
                return (
                    f"⚠️  Extra dependency '{missing}' is not installed. "
                    f"You won't be able to {ALL_DEPENDENCIES[missing]}.\n"
                    "Download a missing dependency via `pip install -U WebtoonScraper[full]`"
                )

            case _:
                SEP = "', '"
                return (
                    f"⚠️  Extra dependencies '{SEP.join(missing_dependencies)}' are not installed.\n"
                    f"You won't be able to {', '.join(ALL_DEPENDENCIES[missing] for missing in missing_dependencies)}.\n"
                    "Download missing dependencies via `pip install -U WebtoonScraper[full]`"
                )

    return f"WebtoonScraper {__version__} of Python {sys.version} at {str(files(WebtoonScraper))}"


def _add_version(parser: ArgumentParser):
    parser.add_argument(
        "--version",
        action="version",
        version=_version_info,  # type: ignore
    )


def _parse_options(value: str):
    try:
        key, value = value.split("=")
        return key.strip(), value.strip()
    except ValueError:
        return value.strip(), None


def _parse_excluding(string: str) -> tuple[str, ...]:
    return tuple(value.strip() for value in string.split(",") if value.strip())


parser = argparse.ArgumentParser(
    prog="WebtoonScraper",
    formatter_class=argparse.RawTextHelpFormatter,
)
parser.register("action", "version", LazyVersionAction)
parser.add_argument("--mock", action="store_true", help="Print argument parsing result and exit. Exist for debug or practice purpose")
parser.add_argument(
    "-v",
    "--verbose",
    action="store_true",
    help="Set logger level to DEBUG and show detailed error",
)
parser.add_argument(
    "--format-error",
    action="store_true",
    help="Format exceptions nicely",
)
parser.add_argument("--no-progress-bar", action="store_true", help="Use log instead progress bar to display status")
# 이유는 정말 모르겠지만 subparser specifier 뒤에 이 옵션을 위치시켜야*만* 적용됨. 이유는 불명...
parser.add_argument(
    "-N",
    "--thread-number",
    type=int,
    help="Set concurrent thread number. You can also use `THREAD_NUMBER` to set thread numbers to use.",
)
subparsers = parser.add_subparsers(title="Commands")

# download subparser
download_subparser = subparsers.add_parser("download", help="Download webtoons")
download_subparser.set_defaults(subparser_name="download")
download_subparser.add_argument(
    "webtoon_ids",
    help="URL or webtoon ID. You can provide multiple URLs or webtoon IDs",
    nargs="+",
)
download_subparser.add_argument(
    "-p",
    "--platform",
    type=lambda x: str(x).lower(),
    choices=("url", *platforms),
    metavar="PLATFORM",
    default="url",
    help=f"Webtoon platform to download. Only specify when you want to use webtoon id rather than url. Supported platforms: {', '.join(platforms)}",
)
download_subparser.add_argument("--cookie")
download_subparser.add_argument(
    "-r",
    "--range",
    type=EpisodeRange.from_string,
    help="Episode number range you want to download.",
)
download_subparser.add_argument(
    "-d",
    "--base-directory",
    type=Path,
    default=Path.cwd(),
    help="Where 'webtoon directory' is stored",
)
download_subparser.add_argument("--list-episodes", action="store_true", help="List all episodes")
download_subparser.add_argument(
    "-O",
    "--option",
    type=_parse_options,
    help="An additional option for scraper",
    metavar='OPTION_NAME="OPTION_VALUE"',
    action="append",
)
download_subparser.add_argument(
    "--existing-episode",
    choices=["skip", "raise", "download_again", "hard_check"],
    default="skip",
    help="Determine what to do when episode directory already exists",
)
download_subparser.add_argument(
    "--excluding",
    type=_parse_excluding,
    default=Scraper.information_to_exclude,
    help="Exclude specific information from information.json. Defaults to `extra/,credentials/`.",
)
download_subparser.add_argument(
    "--webtoon-dir-name",
    help="Customize webtoon directory name.",
)
download_subparser.add_argument(
    "--episode-dir-name",
    help="Customize episode directory name.",
)
download_subparser.add_argument(
    "--skip-status",
    help="Previous episode status to skip. Specify multiple values separated by comma.",
    type=lambda value: [stripped for item in value.split(",") if (stripped := item.strip())],
    default=(),
)
# 기본적으로 WebtoonScraper는 다운로드에 실패하더라도 원칙적으로는 오류를 발생시키지 않아야 함.
# 오류가 발생한다는 건 기본적으로 스크래퍼가 잘못되었거나, 웹툰 플랫폼이 변경되었거나, 기타 오류가 발생했음을 의미함.
# 따라서 이를 무시하고 계속 다운로드를 진행하는 것을 기본값으로 설정하는 것은 더 깊게 고민해봐야 할 문제임.
download_subparser.add_argument(
    "--suppress-error-on-batch",
    help="Ignore errors on batch download.",
    action="store_true",
)


def _register(platform_name: str, scraper=None):
    if scraper is None:
        return lambda scraper: _register(platform_name, scraper)

    platforms[platform_name] = scraper
    return scraper


def instantiate(webtoon_platform: str, webtoon_id: str) -> Scraper:
    """웹툰 플랫폼 코드와 웹툰 ID로부터 스크레퍼를 인스턴스화하여 반환합니다. cookie, bearer 등의 추가적인 설정이 필요할 수도 있습니다."""

    Scraper: type[Scraper] | None = platforms.get(webtoon_platform.lower())  # type: ignore
    if Scraper is None:
        raise ValueError(f"Invalid webtoon platform: {webtoon_platform}")
    return Scraper._from_string(webtoon_id)


def instantiate_from_url(webtoon_url: str) -> Scraper:
    """웹툰 URL로부터 자동으로 알맞은 스크래퍼를 인스턴스화합니다. cookie, bearer 등의 추가적인 설정이 필요할 수 있습니다."""

    for PlatformClass in platforms.values():
        try:
            platform = PlatformClass.from_url(webtoon_url)
        except URLError:
            continue
        return platform
    raise PlatformError(f"Platform not detected: {webtoon_url}")


def setup_instance(
    webtoon_id_or_url: str,
    webtoon_platform: str | Literal["url"],
    *,
    existing_episode_policy: Literal["skip", "raise", "download_again", "hard_check"] = "skip",
    cookie: str | None = None,
    download_directory: str | Path | None = None,
    options: dict[str, str] | None = None,
) -> Scraper:
    """여러 설정으로부터 적절한 스크래퍼 인스턴스를 반환합니다. CLI 사용을 위해 디자인되었습니다."""

    # 스크래퍼 불러오기
    if webtoon_platform == "url" or "." in webtoon_id_or_url:  # URL인지 확인
        scraper = instantiate_from_url(webtoon_id_or_url)
    else:
        scraper = instantiate(webtoon_platform, webtoon_id_or_url)

    # 부가 정보 불러오기
    if cookie:
        scraper.cookie = cookie
    if options:
        scraper._apply_options(options)

    # attribute 형식 설정 설정
    if download_directory:
        scraper.base_directory = download_directory
    scraper.existing_episode_policy = existing_episode_policy

    return scraper


async def parse_download(args: argparse.Namespace) -> None:
    for webtoon_id in args.webtoon_ids:
        try:
            scraper = setup_instance(
                webtoon_id,
                args.platform,
                cookie=args.cookie,
                download_directory=args.base_directory,
                options=dict(args.option or {}),
                existing_episode_policy=args.existing_episode,
            )

            if args.list_episodes:
                await scraper.fetch_all()
                table = Table(show_header=True, header_style="bold blue", box=None)
                table.add_column("Episode number [dim](ID)[/dim]", width=12)
                table.add_column("Episode Title", style="bold")
                for i, (episode_id, episode_title) in enumerate(zip(scraper.episode_ids, scraper.episode_titles, strict=True), 1):
                    table.add_row(
                        f"[red][bold]{i:04d}[/bold][/red] [dim]({episode_id})[/dim]",
                        str(episode_title),
                    )
                console.print(table)
                return

            if args.no_progress_bar:
                scraper.use_progress_bar = False

            if args.webtoon_dir_name:
                scraper._webtoon_directory_format = args.webtoon_dir_name
            if args.episode_dir_name:
                scraper._episode_directory_format = args.episode_dir_name

            if hasattr(scraper, "thread_number"):
                scraper.thread_number = args.thread_number  # type: ignore

            scraper.information_to_exclude = args.excluding
            scraper.previous_status_to_skip = args.previous_status_to_skip
            await scraper.async_download_webtoon(args.range)
        except Exception as exc:
            if args.suppress_error_on_batch:
                logger.error(f"Error occurred while downloading {webtoon_id}", exc_info=exc)
                continue
            else:
                raise


def main(argv=None, *, propagate_keyboard_interrupt: bool = False) -> Literal[0, 1]:
    if propagate_keyboard_interrupt:
        return asyncio.run(async_main(argv))
    else:
        try:
            return asyncio.run(async_main(argv))
        except KeyboardInterrupt:
            logger.error("Aborted")
            return 1


async def async_main(argv=None) -> Literal[0, 1]:
    """모든 CLI 명령어를 처리하는 함수입니다.

    Arguments:
        argv: 커맨드라인 명령어입니다. None이라면 자동으로 인자를 인식합니다.

    Returns:
        정상적으로 프로그램이 종료했다면 0을, 비정상적으로 종료되었다면 1을 반환합니다.

    Raises:
        이 함수는 KeyboardInterrupt를 제외한 어떠한 오류도 발생시키지 않습니다.
        그 대신 성공했을 때는 0을, 실패했을 때에는 1을 반환합니다.
    """
    # 다른 곳에서 이미 version 커맨드를 추가했다면 따로 추가하지 않음
    with contextlib.suppress(argparse.ArgumentError):
        _add_version(parser)
    args = parser.parse_args(argv)  # 주어진 argv가 None이면 sys.argv[1:]을 기본값으로 삼음

    # 어떠한 command도 입력하지 않았을 경우 도움말을 표시함.
    if not hasattr(args, "subparser_name"):
        args = parser.parse_args(["--help"])

    # --mock 인자가 포함된 경우 실제 다운로드까지 가지 않고 표현된 인자를 보여주고 종료.
    if args.mock:
        print("Arguments:", str(args).removeprefix("Namespace(").removesuffix(")"))
        return 0

    if args.verbose:
        logger.setLevel(logging.DEBUG)

    if not args.format_error:
        match args.subparser_name:
            case "download":
                await parse_download(args)
            case unknown_subparser:
                raise NotImplementedError(f"{unknown_subparser} is not a valid command.")
        return 0
    else:
        try:
            match args.subparser_name:
                case "download":
                    await parse_download(args)
                case unknown_subparser:
                    raise NotImplementedError(f"{unknown_subparser} is not a valid command.")
        except KeyboardInterrupt:
            logger.error("Aborted")
            return 1
        except SystemExit as exc:
            return exc.code  # type: ignore
        except BaseException:
            console.print_exception()
            return 1
        else:
            return 0


if __name__ == "__main__":
    sys.exit(main())
