# BRS-KB

### 社区 XSS 知识库

**开源安全知识**

[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Python 3.8+](https://img.shields.io/badge/python-3.8+-blue.svg)](https://www.python.org/downloads/)
[![Version](https://img.shields.io/badge/version-1.1.0-green.svg)](https://github.com/EPTLLC/BRS-KB)
[![Code Size](https://img.shields.io/badge/code-16.5k%20lines-brightgreen.svg)]()
[![Contexts](https://img.shields.io/badge/contexts-27-orange.svg)]()
[![Tests](https://img.shields.io/badge/tests-passing-success.svg)]()

全面的社区驱动的跨站脚本（XSS）漏洞知识库

[功能](#-功能) • [安装](#-安装) • [使用](#-使用) • [API](#-api-参考) • [示例](#-示例) • [贡献](#-贡献)

---

## 为什么选择 BRS-KB？

| 功能 | 描述 |
|---------|-------------|
| **27 上下文** | 涵盖经典和现代 XSS 漏洞类型 |
| **详细信息** | 攻击向量、绕过技术、防御策略 |
| **简单 API** | Python 库，易于集成 |
| **零依赖** | 纯 Python 3.8+ |
| **SIEM 兼容** | CVSS 分数、CWE/OWASP 映射、严重性级别 |
| **开源** | MIT 许可，欢迎社区贡献 |
| **生产就绪** | 用于安全扫描器和工具 |

## 安装

```bash
pip install brs-kb
```

**从源码安装：**
```bash
git clone https://github.com/EPTLLC/BRS-KB.git
cd BRS-KB
pip install -e .
```

**要求：** Python 3.8+ • 无外部依赖

## 快速开始

```python
from brs_kb import get_vulnerability_details, list_contexts

# 获取 XSS 上下文的详细信息
details = get_vulnerability_details('html_content')

print(details['title'])        # Cross-Site Scripting (XSS) in HTML Content
print(details['severity'])     # critical
print(details['cvss_score'])   # 8.8
print(details['cwe'])          # ['CWE-79']
print(details['owasp'])        # ['A03:2021']

# 列出所有可用上下文
contexts = list_contexts()
# ['css_context', 'default', 'dom_xss', 'html_attribute', ...]
```

## 可用上下文

<details>
<summary><b>27 XSS 漏洞上下文</b>（点击展开）</summary>

### 核心 HTML 上下文
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `html_content` | HTML 内容中的 XSS | 398 | 严重 |
| `html_attribute` | HTML 属性中的 XSS | 529 | 严重 |
| `html_comment` | HTML 注释中的 XSS | 68 | 中等 |

### JavaScript 上下文
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `javascript_context` | 直接 JavaScript 注入 | 636 | 严重 |
| `js_string` | JavaScript 字符串注入 | 619 | 严重 |
| `js_object` | JavaScript 对象注入 | 619 | 高 |

### 样式和标记
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `css_context` | CSS 注入和样式属性 | 675 | 高 |
| `svg_context` | 基于 SVG 的 XSS 向量 | 288 | 高 |
| `markdown_context` | Markdown 渲染 XSS | 101 | 中等 |

### 数据格式
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `json_value` | JSON 上下文 XSS | 72 | 中等 |
| `xml_content` | XML/XHTML XSS 向量 | 81 | 高 |

### 高级向量
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `url_context` | 基于 URL/协议的 XSS | 545 | 高 |
| `dom_xss` | 基于 DOM 的 XSS（客户端） | 350 | 高 |
| `template_injection` | 客户端模板注入 | 107 | 严重 |
| `postmessage_xss` | PostMessage API 漏洞 | 125 | 高 |
| `wasm_context` | WebAssembly 上下文 XSS | 110 | 中等 |

### 现代 Web 技术（新增）
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `websocket_xss` | WebSocket 实时 XSS | 407 | 高 |
| `service_worker_xss` | Service Worker 注入 | 398 | 高 |
| `webrtc_xss` | WebRTC P2P 通信 XSS | 420 | 高 |
| `indexeddb_xss` | IndexedDB 存储 XSS | 378 | 中等 |
| `webgl_xss` | WebGL 着色器注入 | 395 | 中等 |
| `shadow_dom_xss` | Shadow DOM 封装绕过 | 385 | 高 |
| `custom_elements_xss` | 自定义元素 XSS | 390 | 高 |
| `http2_push_xss` | HTTP/2 服务器推送 XSS | 375 | 中等 |
| `graphql_xss` | GraphQL API 注入 | 390 | 高 |
| `iframe_sandbox_xss` | iframe 沙箱绕过 | 380 | 中等 |

### 后备
| 上下文 | 描述 | 行数 | 严重性 |
|---------|-------------|-------|----------|
| `default` | 通用 XSS 信息 | 156 | - |

</details>

## 功能

### 元数据结构

每个上下文包含安全元数据：

```python
{
    # 核心信息
    "title": "Cross-Site Scripting (XSS) in HTML Content",
    "description": "详细漏洞解释...",
    "attack_vector": "真实攻击技术...",
    "remediation": "可操作安全措施...",

    # 安全元数据
    "severity": "critical",           # low | medium | high | critical
    "cvss_score": 8.8,                # CVSS 3.1 基础分数
    "cvss_vector": "CVSS:3.1/...",    # 完整 CVSS 向量字符串
    "reliability": "certain",         # tentative | firm | certain
    "cwe": ["CWE-79"],                # CWE 标识符
    "owasp": ["A03:2021"],            # OWASP Top 10 映射
    "tags": ["xss", "html", "reflected"]  # 分类标签
}
```

### 反向映射系统

将 payloads 映射到上下文和防御：

```python
from brs_kb.reverse_map import find_contexts_for_payload, get_defenses_for_context

# Payload → Context 映射
info = find_contexts_for_payload("<script>alert(1)</script>")
# → {'contexts': ['html_content', 'html_comment', 'svg_context'],
#    'severity': 'critical',
#    'defenses': ['html_encoding', 'csp', 'sanitization']}

# Context → Defense 映射
defenses = get_defenses_for_context('html_content')
# → [{'defense': 'html_encoding', 'priority': 1, 'required': True},
#     {'defense': 'csp', 'priority': 1, 'required': True}, ...]
```

## CLI 工具

BRS-KB 包含用于安全研究和测试的综合命令行界面：

```bash
# 安装包
pip install brs-kb

# 显示所有可用命令
brs-kb --help

# 显示系统信息
brs-kb info

# 列出所有 XSS 上下文
brs-kb list-contexts

# 获取上下文的详细信息
brs-kb get-context websocket_xss

# 分析 payload
brs-kb analyze-payload "<script>alert(1)</script>"

# 在数据库中搜索 payloads
brs-kb search-payloads websocket --limit 5

# 测试 payload 有效性
brs-kb test-payload "<script>alert(1)</script>" html_content

# 生成综合报告
brs-kb generate-report

# 验证数据库完整性
brs-kb validate

# 导出数据
brs-kb export contexts --format json --output contexts.json
```

**可用命令：**
- `info` - 显示系统信息和统计
- `list-contexts` - 列出所有可用 XSS 上下文及其严重性
- `get-context <name>` - 获取漏洞的详细信息
- `analyze-payload <payload>` - 使用反向映射分析 payload
- `search-payloads <query>` - 使用相关性评分搜索 payload 数据库
- `test-payload <payload> <context>` - 测试 payload 在上下文中的有效性
- `generate-report` - 生成综合系统分析
- `validate` - 验证 payload 数据库完整性
- `export <type> --format <format>` - 导出数据（payloads、contexts、reports）

## 安全扫描器插件

BRS-KB 包含用于流行安全测试工具的插件：

### 🛡️ Burp Suite 插件
- 代理期间的实时 XSS payload 分析
- 拦截请求的自动上下文检测
- 与 27 个 XSS 上下文的集成
- 专业安全团队界面

**安装：** 将 `plugins/burp_suite/BRSKBExtension.java` 复制到 Burp 扩展

### ⚡ OWASP ZAP 集成
- 使用 BRS-KB intelligence 的自动化 XSS 扫描
- 上下文感知 payload 注入
- WAF 绕过技术检测
- 专业安全工作流支持

**安装：** 在 ZAP 脚本中加载 `plugins/owasp_zap/brs_kb_zap.py`

### 🎯 Nuclei 模板
- 200+ 分类的 XSS payloads
- 上下文特定测试（27 个 XSS 上下文）
- WAF 绕过技术检测
- 现代 Web 技术测试

**安装：** 将模板复制到 Nuclei 模板目录

查看 [plugins/README.md](plugins/README.md) 了解详细的安装和使用说明。

## SIEM 集成

BRS-KB 与企业 SIEM 系统集成以实现实时监控：

#### 🔍 Splunk 集成
- 实时 XSS 漏洞数据摄取
- XSS 上下文分析的自定义仪表板
- 关键漏洞的警报规则
- 历史趋势分析

**安装：** 将 `siem_connectors/splunk/brs_kb_app.tar.gz` 复制到 Splunk apps 目录

#### 📊 Elasticsearch 集成
- BRS-KB 数据的 Logstash/Beats 集成
- XSS 分析的 Kibana 仪表板
- 机器学习异常检测
- Elasticsearch Watcher 警报

**安装：** 从 `siem_connectors/elastic/` 部署 Logstash 配置

#### 📋 Graylog 集成
- 实时日志摄取的 GELF 集成
- 自定义仪表板和窗口小部件
- 警报规则和通知
- XSS 事件的流处理

**安装：** 从 `siem_connectors/graylog/` 导入内容包

查看 [siem_connectors/README.md](siem_connectors/README.md) 了解详细的安装和使用说明。

## CI/CD 管道

BRS-KB 包含用于自动化测试和部署的综合 CI/CD 配置：

### 🔄 GitLab CI (`.gitlab-ci.yml`)
- 多 Python 版本测试（3.8-3.12）
- 代码质量检查和安全扫描
- 包构建和 PyPI 部署
- 性能测试和覆盖率报告

### 🔄 GitLab CI (`.gitlab-ci.yml`) - 高级配置
- 跨 Python 版本的并行测试
- 包构建和部署
- 文档部署（GitLab Pages）
- 性能和安全测试

### 🔄 Jenkins 管道 (`Jenkinsfile`)
- 具有并行执行的声明式管道
- 工件管理和部署
- 通知集成和报告
- 企业级管道管理

### ⚙️ 设置脚本 (`scripts/setup_cicd.py`)
自动 CI/CD 管道设置和配置。

**快速设置：**
```bash
python3 scripts/setup_cicd.py
```

查看 [DEVELOPMENT_PLAN.md](DEVELOPMENT_PLAN.md) 了解详细的 CI/CD 文档。

## 示例

查看 [examples/](examples/) 目录了解集成示例：

| 示例 | 描述 |
|---------|-------------|
| [`basic_usage.py`](examples/basic_usage.py) | 基本 API 使用和功能 |
| [`scanner_integration.py`](examples/scanner_integration.py) | 集成到安全扫描器 |
| [`siem_integration.py`](examples/siem_integration.py) | SIEM/SOC 威胁情报 |
| [`reverse_mapping.py`](examples/reverse_mapping.py) | 增强的反向映射与 ML-ready 功能 |
| [`payload_database.py`](examples/payload_database.py) | 200+ payload 数据库与测试 API |
| [`cli_demo.py`](examples/cli_demo.py) | 命令行界面演示 |
| [`plugin_demo.py`](examples/plugin_demo.py) | 安全扫描器插件集成 |
| [`siem_integration.py`](examples/siem_integration.py) | SIEM 系统集成展示 |
| [`cicd_demo.py`](examples/cicd_demo.py) | CI/CD 管道演示 |
| [`multilanguage_demo.py`](examples/multilanguage_demo.py) | 多语言支持演示 |
| [`integrated_demo.py`](examples/integrated_demo.py) | 完整系统集成展示 |

**运行示例：**
```bash
# Python 示例
python3 examples/basic_usage.py
python3 examples/scanner_integration.py
python3 examples/cli_demo.py
python3 examples/plugin_demo.py
python3 examples/integrated_demo.py

# CLI 命令
brs-kb info                    # 系统信息
brs-kb list-contexts           # 所有 XSS 上下文
brs-kb analyze-payload "<script>alert(1)</script>"  # payload 分析
brs-kb search-payloads websocket  # 搜索 payloads
brs-kb generate-report         # 综合报告

# 安全扫描器集成
nuclei -t plugins/nuclei/templates/brs-kb-xss.yaml -u https://target.com

# SIEM 集成
python3 siem_connectors/splunk/brs_kb_splunk_connector.py --api-key YOUR_KEY --splunk-url https://splunk.company.com:8088

# CI/CD 管道
python3 scripts/setup_cicd.py

# 多语言支持
brs-kb language ru
brs-kb language --list
```

## API 参考

### 核心函数

#### `get_vulnerability_details(context: str) -> Dict[str, Any]`
获取漏洞上下文的详细信息。

```python
details = get_vulnerability_details('html_content')
```

#### `list_contexts() -> List[str]`
获取所有可用上下文的列表。

```python
contexts = list_contexts()  # ['css_context', 'default', 'dom_xss', ...]
```

#### `get_kb_info() -> Dict[str, Any]`
获取知识库信息（版本、构建、上下文数量）。

```python
info = get_kb_info()
print(f"Version: {info['version']}, Total contexts: {info['total_contexts']}")
```

#### `get_kb_version() -> str`
获取版本字符串。

```python
version = get_kb_version()  # "1.1.0"
```

### 增强反向映射函数

导入自 `brs_kb.reverse_map`：

#### `find_contexts_for_payload(payload: str) -> Dict`
使用模式匹配和置信度评分进行高级 payload 分析。

#### `predict_contexts_ml_ready(payload: str) -> Dict`
用于未来机器学习集成的 ML-ready 分析与特征提取。

#### `get_defenses_for_context(context: str) -> List[Dict]`
获取上下文的推荐防御措施与增强元数据和实现细节。

#### `get_defense_info(defense: str) -> Dict`
获取防御机制的综合信息，包括绕过难度和标签。

#### `analyze_payload_with_patterns(payload: str) -> List[Tuple]`
针对模式数据库分析 payload，返回带有置信度评分的匹配。

#### `get_reverse_map_info() -> Dict`
获取反向映射系统信息，包括版本、能力和统计。

#### `reverse_lookup(query_type: str, query: str) -> Dict`
通用查找函数，支持 payload、context、defense 和 pattern 查询。

### Payload 数据库函数

#### `get_payloads_by_context(context: str) -> List[Dict]`
获取特定上下文中有效的所有 payloads。

#### `get_payloads_by_severity(severity: str) -> List[Dict]`
按严重性级别获取所有 payloads。

#### `search_payloads(query: str) -> List[Dict]`
使用相关性评分的 payloads 搜索。

#### `test_payload_in_context(payload: str, context: str) -> Dict`
测试 payload 在特定上下文中的有效性。

#### `get_database_info() -> Dict`
获取 payload 数据库统计和信息。

### CLI 工具函数

#### `get_cli() -> BRSKBCLI`
获取 CLI 实例用于编程使用。

**CLI 命令：**
- `brs-kb info` - 系统信息
- `brs-kb list-contexts` - 列出所有 XSS 上下文
- `brs-kb get-context <name>` - 上下文细节
- `brs-kb analyze-payload <payload>` - payload 分析
- `brs-kb search-payloads <query>` - 搜索 payloads
- `brs-kb test-payload <payload> <context>` - 测试有效性
- `brs-kb generate-report` - 综合报告
- `brs-kb validate` - 数据库验证
- `brs-kb export <type>` - 导出数据

## 贡献

欢迎安全社区的贡献。

### 贡献方式

- 添加新的 XSS 上下文
- 使用新绕过更新现有上下文
- 改进文档
- 报告问题或过时信息
- 分享真实示例

**快速开始：**
```bash
git clone https://github.com/EPTLLC/BRS-KB.git
cd BRS-KB
git checkout -b feature/new-context
# 进行更改
pytest tests/ -v
git commit -m "Add: New context for WebSocket XSS"
git push origin feature/new-context
# 打开 Pull Request
```

查看 [CONTRIBUTING.md](CONTRIBUTING.md) 了解详细指南。

## 项目结构

```
BRS-KB/
├── brs_kb/                    # 主包
│   ├── __init__.py            # 核心 API
│   ├── schema.json            # JSON Schema 验证
│   ├── reverse_map.py         # 反向映射系统
│   ├── i18n.py                # 国际化系统
│   ├── cli.py                 # 命令行接口
│   ├── payload_testing.py     # Payload 测试框架
│   ├── payloads_db.py         # Payload 数据库
│   └── contexts/              # 27 个漏洞模块
│       ├── html_content.py
│       ├── javascript_context.py
│       ├── websocket_xss.py
│       └── ...
├── examples/                  # 集成示例
├── tests/                     # 测试套件 (pytest)
├── docs/                      # 多语言文档
├── i18n/locales/              # 翻译文件
├── plugins/                   # 安全扫描器插件
├── siem_connectors/           # SIEM 系统集成
├── web_ui/                    # React 基础的 Web 界面
├── LICENSE                    # MIT 许可证
├── CONTRIBUTING.md            # 贡献指南
├── CHANGELOG.md               # 版本历史
└── README.md                  # 本文件
```

## 测试

```bash
# 运行所有测试
pytest tests/ -v

# 使用覆盖率运行（需要 pytest-cov）
pytest tests/ -v --cov=brs_kb

# 运行特定测试
pytest tests/test_basic.py -v
```

## 统计

| 指标 | 值 |
|--------|-------|
| 总行数 | ~16,500+ |
| 上下文模块 | 27 |
| Payload 数据库 | 200+ |
| 反向映射模式 | 29 |
| 支持的上下文 | 27 |
| 平均模块大小 | 418 行 |
| 测试覆盖 | 33 测试 |
| CLI 命令 | 9 命令 |
| 扫描器插件 | 3 平台 |
| SIEM 集成 | 3 系统 |
| CI/CD 管道 | GitLab CI, Jenkins |
| 部署脚本 | ✅ |
| Docker 支持 | ✅ |
| Kubernetes 支持 | ✅ |
| 监控 | ✅ |
| 多语言支持 | ✅ |
| 俄语本地化 | ✅ |
| 中文本地化 | ✅ |
| 西班牙语本地化 | ✅ |
| 外部依赖 | 0 |
| Python 版本 | 3.8+ |
| 代码质量 | 生产就绪 |
| ML 就绪 | ✅ |
| 置信度评分 | ✅ |
| 现代 XSS 支持 | ✅ |
| WebSocket XSS | ✅ |
| Service Worker XSS | ✅ |
| WebRTC XSS | ✅ |
| GraphQL XSS | ✅ |
| Shadow DOM XSS | ✅ |
| 自定义元素 XSS | ✅ |
| Payload 测试 API | ✅ |
| WAF 绕过检测 | ✅ |
| CLI 工具 | ✅ |
| 导出功能 | ✅ |
| 安全扫描器插件 | ✅ |
| Burp Suite 集成 | ✅ |
| OWASP ZAP 集成 | ✅ |
| Nuclei 模板 | ✅ |
| SIEM 连接器 | ✅ |
| Splunk 集成 | ✅ |
| Elasticsearch 集成 | ✅ |
| Graylog 集成 | ✅ |
| CI/CD 自动化 | ✅ |
| GitLab CI | ✅ |
| Jenkins 管道 | ✅ |
| 多语言支持 | ✅ |
| 俄语本地化 | ✅ |
| 中文本地化 | ✅ |
| 西班牙语本地化 | ✅ |

## 许可

**MIT 许可** - 可自由用于任何项目（商业或非商业）

```
版权所有 (c) 2025 EasyProTech LLC / Brabus

特此免费授予获得本软件副本和相关文档文件（“软件”）的人不受限制地处理
软件的权限，包括但不限于使用、复制、修改、合并、发布、分发、再许可和/或销售
软件副本的权限...
```

查看 [LICENSE](LICENSE) 了解完整文本。

## 项目信息

| | |
|---|---|
| **项目** | BRS-KB (BRS XSS Knowledge Base) |
| **公司** | EasyProTech LLC |
| **网站** | [www.easypro.tech](https://www.easypro.tech) |
| **开发者** | Brabus |
| **联系** | [https://t.me/easyprotech](https://t.me/easyprotech) |
| **仓库** | [https://github.com/EPTLLC/BRS-KB](https://github.com/EPTLLC/BRS-KB) |
| **许可** | MIT |
| **状态** | Production-Ready |
| **版本** | 1.1.0 |

## 相关项目

- **[BRS-XSS](https://github.com/EPTLLC/brs-xss)** - Advanced XSS Scanner (使用 BRS-KB)

## 支持政策

**无官方支持**

这是一个社区驱动的项目。虽然我们欢迎贡献：
- 使用 GitHub Issues 报告错误
- 使用 Pull Requests 贡献
- 无 SLA 或保证响应时间

这个项目由社区维护。

## 致谢

- 为知识做出贡献的安全研究人员
- 为支持提供帮助的开源社区
- 报告问题和改进的所有人

---

<div align="center">

**开源 XSS 知识库**

*MIT 许可 • Python 3.8+ • 零依赖*

[GitHub 星标](https://github.com/EPTLLC/BRS-KB) • [报告错误](https://github.com/EPTLLC/BRS-KB/issues) • [请求功能](https://github.com/EPTLLC/BRS-KB/issues)

</div>
