#!/usr/bin/env python3
"""
Unified DFA (Detrended Fluctuation Analysis) Estimator for Long-Range Dependence Analysis.

This module implements the DFA estimator with automatic optimization framework
selection (JAX, Numba, NumPy) for the best performance on the available hardware.
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy import stats
from typing import Dict, Any, Optional, Union, Tuple, List
import warnings

from lrdbenchmark.analysis.backend_utils import select_backend, JAX_AVAILABLE, NUMBA_AVAILABLE

# Import optimization frameworks
if JAX_AVAILABLE:
    import jax
    import jax.numpy as jnp
    from jax import jit, vmap
if NUMBA_AVAILABLE:
    import numba
    from numba import jit as numba_jit, prange
else:
    # Create a dummy decorator when numba is not available
    def numba_jit(*args, **kwargs):
        def decorator(func):
            return func
        return decorator
    prange = range  # Dummy prange

# Import base estimator
try:
    from lrdbenchmark.analysis.base_estimator import BaseEstimator
except ImportError:
    from lrdbenchmark.analysis.base_estimator import BaseEstimator


@numba_jit(nopython=True, parallel=True)
def _calculate_fluctuation_numba(data: np.ndarray, scales: np.ndarray, order: int) -> np.ndarray:
    """Numba-jitted fluctuation calculation."""
    n = len(data)
    fluctuation_values = np.zeros(len(scales))

    for i in prange(len(scales)):
        scale = scales[i]
        n_segments = n // scale
        if n_segments == 0:
            fluctuation_values[i] = np.nan
            continue

        segment_fluctuations = np.zeros(n_segments)
        for j in range(n_segments):
            start_idx = j * scale
            end_idx = start_idx + scale
            segment_data = data[start_idx:end_idx]

            x = np.arange(scale)
            if order == 0:
                detrended = segment_data - np.mean(segment_data)
            else:
                coeffs = np.polyfit(x, segment_data, order)
                trend = np.polyval(coeffs, x)
                detrended = segment_data - trend
            
            segment_fluctuations[j] = np.sqrt(np.mean(detrended**2))
        
        fluctuation_values[i] = np.mean(segment_fluctuations)
        
    return fluctuation_values


if JAX_AVAILABLE:
    from functools import partial

    @partial(jit, static_argnums=(1, 2))
    def _dfa_fluctuation_jax(cumsum: jnp.ndarray, scale: int, order: int) -> jnp.ndarray:
        n_segments = cumsum.shape[0] // scale
        if n_segments == 0:
            return jnp.nan

        trimmed = cumsum[: n_segments * scale]
        segments = trimmed.reshape((n_segments, scale))
        x = jnp.arange(scale, dtype=cumsum.dtype)

        if order == 0:
            detrended = segments - jnp.mean(segments, axis=1, keepdims=True)
            variances = jnp.mean(detrended**2, axis=1)
            return jnp.sqrt(jnp.mean(variances))

        X = jnp.vander(x, N=order + 1)
        XtX = X.T @ X
        XtX_inv = jnp.linalg.inv(XtX)
        projector = XtX_inv @ X.T

        def segment_variance(segment):
            coeffs = projector @ segment
            trend = X @ coeffs
            detrended = segment - trend
            return jnp.mean(detrended**2)

        variances = vmap(segment_variance)(segments)
        return jnp.sqrt(jnp.mean(variances))

class DFAEstimator(BaseEstimator):
    """
    Unified DFA (Detrended Fluctuation Analysis) Estimator for Long-Range Dependence Analysis.

    DFA analyzes the root-mean-square fluctuation of detrended time series data
    to estimate the Hurst parameter, which characterizes long-range dependence.

    Features:
    - Automatic optimization framework selection (JAX, Numba, NumPy)
    - GPU acceleration with JAX when available
    - JIT compilation with Numba for CPU optimization
    - Graceful fallbacks when optimization frameworks fail

    Parameters
    ----------
    min_scale : int, optional (default=10)
        Minimum scale for analysis
    max_scale : int, optional (default=None)
        Maximum scale for analysis. If None, uses data length / 4
    num_scales : int, optional (default=10)
        Number of scales to test
    order : int, optional (default=1)
        Order of polynomial for detrending
    use_optimization : str, optional (default='auto')
        Optimization framework to use: 'auto', 'jax', 'numba', 'numpy'
    """

    def __init__(
        self,
        min_scale: int = 10,
        max_scale: Optional[int] = None,
        num_scales: int = 10,
        order: int = 1,
        use_optimization: str = "auto",
    ):
        super().__init__()
        
        # Estimator parameters
        self.parameters = {
            "min_scale": min_scale,
            "max_scale": max_scale,
            "num_scales": num_scales,
            "order": order,
        }
        
        # Optimization framework
        self.optimization_framework = select_backend(use_optimization)
        
        # Results storage
        self.results = {}
        
        # Validation
        self._validate_parameters()

    def _validate_parameters(self) -> None:
        """Validate estimator parameters."""
        if self.parameters["min_scale"] < 4:
            raise ValueError("min_scale must be at least 4")
        if self.parameters["max_scale"] is not None:
            if self.parameters["max_scale"] <= self.parameters["min_scale"]:
                raise ValueError("max_scale must be greater than min_scale")
        if self.parameters["num_scales"] < 3:
            raise ValueError("num_scales must be at least 3")
        if self.parameters["order"] < 0:
            raise ValueError("order must be non-negative")

    def estimate(self, data: Union[np.ndarray, list]) -> Dict[str, Any]:
        """
        Estimate the Hurst parameter using DFA with automatic optimization.

        Parameters
        ----------
        data : array-like
            Input time series data

        Returns
        -------
        dict
            Dictionary containing estimation results including:
            - hurst_parameter: Estimated Hurst parameter
            - r_squared: R-squared value of the fit
            - scales: Scales used in the analysis
            - fluctuation_values: Fluctuation values for each scale
            - log_scales: Log of scales
            - log_fluctuations: Log of fluctuation values
        """
        data = np.asarray(data)
        n = len(data)

        if n < 100:
            warnings.warn("Data length is small, results may be unreliable")

        # Select optimal method based on data size and framework
        backend = self.optimization_framework
        if backend == "jax":
            try:
                return self._estimate_jax(data)
            except Exception as e:
                warnings.warn(f"JAX implementation failed: {e}, falling back to NumPy")
                return self._estimate_numpy(data)
        elif backend == "numba":
            try:
                return self._estimate_numba(data)
            except Exception as e:
                warnings.warn(f"Numba implementation failed: {e}, falling back to NumPy")
                return self._estimate_numpy(data)
        else: # numpy
            return self._estimate_numpy(data)

    def _estimate_numba(self, data: np.ndarray) -> Dict[str, Any]:
        """Numba-optimized implementation of DFA estimation."""
        n = len(data)
        cumsum = np.cumsum(data - np.mean(data))

        if self.parameters["max_scale"] is None:
            self.parameters["max_scale"] = n // 4

        scales = np.logspace(
            np.log10(self.parameters["min_scale"]),
            np.log10(self.parameters["max_scale"]),
            self.parameters["num_scales"],
            dtype=int
        )
        scales = np.unique(scales)
        scales = scales[scales <= n // 2]

        if len(scales) < 3:
            raise ValueError("Insufficient valid scales for analysis")

        fluctuation_values = _calculate_fluctuation_numba(cumsum, scales, self.parameters["order"])
        
        valid_mask = (fluctuation_values > 0) & ~np.isnan(fluctuation_values)
        if np.sum(valid_mask) < 3:
            raise ValueError("Insufficient valid fluctuation values for analysis")

        valid_scales = scales[valid_mask]
        valid_fluctuations = fluctuation_values[valid_mask]

        log_scales = np.log(valid_scales)
        log_fluctuations = np.log(valid_fluctuations)

        slope, intercept, r_value, p_value, std_err = stats.linregress(
            log_scales, log_fluctuations
        )
        r_squared = r_value**2
        hurst_parameter = slope

        self.results = {
            "hurst_parameter": float(hurst_parameter),
            "r_squared": float(r_squared),
            "slope": float(slope),
            "intercept": float(intercept),
            "p_value": float(p_value),
            "std_error": float(std_err),
            "scales": valid_scales.tolist(),
            "fluctuation_values": valid_fluctuations.tolist(),
            "log_scales": log_scales.tolist(),
            "log_fluctuations": log_fluctuations.tolist(),
            "method": "numba",
            "optimization_framework": self.optimization_framework,
        }
        
        return self.results

    def _estimate_jax(self, data: np.ndarray) -> Dict[str, Any]:
        """JAX-optimized implementation of DFA estimation."""
        n = len(data)
        data_jax = jnp.array(data)
        cumsum = jnp.cumsum(data_jax - jnp.mean(data_jax))

        if self.parameters["max_scale"] is None:
            self.parameters["max_scale"] = n // 4

        scales = np.logspace(
            np.log10(self.parameters["min_scale"]),
            np.log10(self.parameters["max_scale"]),
            self.parameters["num_scales"],
            dtype=int
        )
        scales = np.unique(scales)
        scales = scales[scales <= n // 2]

        if len(scales) < 3:
            raise ValueError("Insufficient valid scales for analysis")

        fluctuation_values = []
        for scale in scales:
            fluct = _dfa_fluctuation_jax(cumsum, int(scale), int(self.parameters["order"]))
            fluctuation_values.append(fluct)
        fluctuation_values = jnp.asarray(fluctuation_values, dtype=jnp.float64)
        
        valid_mask = (fluctuation_values > 0) & ~jnp.isnan(fluctuation_values)
        if jnp.sum(valid_mask) < 3:
            raise ValueError("Insufficient valid fluctuation values for analysis")

        valid_scales = scales[valid_mask]
        valid_fluctuations = fluctuation_values[valid_mask]

        log_scales = jnp.log(valid_scales)
        log_fluctuations = jnp.log(valid_fluctuations)

        # JAX-based linear regression
        A = jnp.vstack([log_scales, jnp.ones(len(log_scales))]).T
        slope, intercept = jnp.linalg.lstsq(A, log_fluctuations, rcond=None)[0]

        r_squared = 1 - jnp.sum((log_fluctuations - (slope * log_scales + intercept))**2) / jnp.sum((log_fluctuations - jnp.mean(log_fluctuations))**2)
        hurst_parameter = slope

        self.results = {
            "hurst_parameter": float(hurst_parameter),
            "r_squared": float(r_squared),
            "slope": float(slope),
            "intercept": float(intercept),
            "p_value": None,
            "std_error": None,
            "scales": valid_scales.tolist(),
            "fluctuation_values": valid_fluctuations.tolist(),
            "log_scales": log_scales.tolist(),
            "log_fluctuations": log_fluctuations.tolist(),
            "method": "jax",
            "optimization_framework": self.optimization_framework,
        }
        
        return self.results

    def _estimate_numpy(self, data: np.ndarray) -> Dict[str, Any]:
        """NumPy implementation of DFA estimation."""
        n = len(data)
        
        # CRITICAL FIX: DFA works on the CUMULATIVE SUM of the data, not the original data
        # This was the main issue causing low Hurst parameter estimates
        cumsum = np.cumsum(data - np.mean(data))
        
        # Set max scale if not provided
        if self.parameters["max_scale"] is None:
            self.parameters["max_scale"] = n // 4
        
        # Generate scales
        scales = np.logspace(
            np.log10(self.parameters["min_scale"]),
            np.log10(self.parameters["max_scale"]),
            self.parameters["num_scales"],
            dtype=int
        )
        
        # Ensure scales are unique and valid
        scales = np.unique(scales)
        scales = scales[scales <= n // 2]
        
        if len(scales) < 3:
            raise ValueError("Insufficient valid scales for analysis")
        
        # Calculate fluctuation values for each scale using cumulative sum
        fluctuation_values = []
        for scale in scales:
            fluct_val = self._calculate_fluctuation_numpy(cumsum, scale)
            fluctuation_values.append(fluct_val)
        
        fluctuation_values = np.array(fluctuation_values)
        
        # Filter out invalid values
        valid_mask = (fluctuation_values > 0) & ~np.isnan(fluctuation_values)
        if np.sum(valid_mask) < 3:
            raise ValueError("Insufficient valid fluctuation values for analysis")
        
        valid_scales = scales[valid_mask]
        valid_fluctuations = fluctuation_values[valid_mask]
        
        # Log-log regression
        log_scales = np.log(valid_scales)
        log_fluctuations = np.log(valid_fluctuations)
        
        # Linear regression
        slope, intercept, r_value, p_value, std_err = stats.linregress(
            log_scales, log_fluctuations
        )
        
        # Calculate R-squared
        r_squared = r_value**2
        
        # Hurst parameter is the slope
        hurst_parameter = slope
        
        # Store results
        self.results = {
            "hurst_parameter": float(hurst_parameter),
            "r_squared": float(r_squared),
            "slope": float(slope),
            "intercept": float(intercept),
            "p_value": float(p_value),
            "std_error": float(std_err),
            "scales": valid_scales.tolist(),
            "fluctuation_values": valid_fluctuations.tolist(),
            "log_scales": log_scales.tolist(),
            "log_fluctuations": log_fluctuations.tolist(),
            "method": "numpy",
            "optimization_framework": self.optimization_framework,
        }
        
        return self.results

    def _calculate_fluctuation_numpy(self, data: np.ndarray, scale: int) -> float:
        """Calculate fluctuation value for a given scale using NumPy."""
        n = len(data)
        n_segments = n // scale
        
        if n_segments == 0:
            return np.nan
        
        fluctuation_values = []
        
        for i in range(n_segments):
            start_idx = i * scale
            end_idx = start_idx + scale
            segment_data = data[start_idx:end_idx]
            
            # Detrend the segment
            x = np.arange(scale)
            if self.parameters["order"] == 0:
                # Remove mean
                detrended = segment_data - np.mean(segment_data)
            else:
                # Polynomial detrending
                coeffs = np.polyfit(x, segment_data, self.parameters["order"])
                trend = np.polyval(coeffs, x)
                detrended = segment_data - trend
            
            # Calculate RMS fluctuation
            fluctuation = np.sqrt(np.mean(detrended**2))
            fluctuation_values.append(fluctuation)
        
        if len(fluctuation_values) == 0:
            return np.nan
        
        return np.mean(fluctuation_values)

    def get_optimization_info(self) -> Dict[str, Any]:
        """Get information about available optimizations and current selection."""
        return {
            "current_framework": self.optimization_framework,
            "jax_available": JAX_AVAILABLE,
            "numba_available": NUMBA_AVAILABLE,
            "recommended_framework": self._get_recommended_framework()
        }

    def _get_recommended_framework(self) -> str:
        """Get the recommended optimization framework."""
        if JAX_AVAILABLE:
            return "jax"  # Best for GPU acceleration
        elif NUMBA_AVAILABLE:
            return "numba"  # Good for CPU optimization
        else:
            return "numpy"  # Fallback

    def plot_analysis(self, figsize: Tuple[int, int] = (12, 8), save_path: Optional[str] = None) -> None:
        """Plot the DFA analysis results."""
        if not self.results:
            raise ValueError("No results available. Run estimate() first.")

        fig, axes = plt.subplots(2, 2, figsize=figsize)
        fig.suptitle('DFA Analysis Results', fontsize=16)

        # Plot 1: Log-log relationship
        ax1 = axes[0, 0]
        x = self.results["log_scales"]
        y = self.results["log_fluctuations"]

        ax1.scatter(x, y, s=60, alpha=0.7, label="Data points")

        # Plot fitted line
        slope = self.results["slope"]
        intercept = self.results["intercept"]
        x_fit = np.linspace(min(x), max(x), 100)
        y_fit = slope * x_fit + intercept
        ax1.plot(x_fit, y_fit, "r--", label=f"Linear fit (slope={slope:.3f})")

        ax1.set_xlabel("log(Scale)")
        ax1.set_ylabel("log(Fluctuation)")
        ax1.set_title("DFA Scaling")
        ax1.legend()
        ax1.grid(True, alpha=0.3)

        # Plot 2: Fluctuation vs Scale (log-log)
        ax2 = axes[0, 1]
        scales = self.results["scales"]
        fluctuations = self.results["fluctuation_values"]
        
        ax2.scatter(scales, fluctuations, s=60, alpha=0.7)
        ax2.set_xscale("log")
        ax2.set_yscale("log")
        ax2.set_xlabel("Scale")
        ax2.set_ylabel("Fluctuation")
        ax2.set_title("Fluctuation vs Scale (log-log)")
        ax2.grid(True, which="both", ls=":", alpha=0.3)

        # Plot 3: Hurst parameter estimate
        ax3 = axes[1, 0]
        hurst = self.results["hurst_parameter"]
        
        ax3.bar(["Hurst Parameter"], [hurst], alpha=0.7, color='skyblue')
        ax3.axhline(y=0.5, color='red', linestyle='--', alpha=0.7, label='H=0.5 (no memory)')
        ax3.set_ylabel("Hurst Parameter")
        ax3.set_title(f"Hurst Parameter Estimate: {hurst:.3f}")
        ax3.legend()
        ax3.grid(True, alpha=0.3)

        # Plot 4: R-squared
        ax4 = axes[1, 1]
        r_squared = self.results["r_squared"]
        
        ax4.bar(["R²"], [r_squared], alpha=0.7, color='lightgreen')
        ax4.set_ylabel("R²")
        ax4.set_title(f"Goodness of Fit: R² = {r_squared:.3f}")
        ax4.set_ylim(0, 1)
        ax4.grid(True, alpha=0.3)

        plt.tight_layout()

        if save_path:
            plt.savefig(save_path, dpi=300, bbox_inches="tight")
        plt.show()

    def get_method_recommendation(self, n: int) -> Dict[str, Any]:
        """Get method recommendation for a given data size."""
        if n < 100:
            return {
                "recommended_method": "numpy",
                "reasoning": f"Data size n={n} is too small for optimization benefits",
                "method_details": {
                    "description": "NumPy implementation",
                    "best_for": "Small datasets (n < 100)",
                    "complexity": "O(n²)",
                    "memory": "O(n)",
                    "accuracy": "Medium"
                }
            }
        elif n < 1000:
            return {
                "recommended_method": "numba",
                "reasoning": f"Data size n={n} benefits from JIT compilation",
                "method_details": {
                    "description": "Numba JIT-compiled implementation",
                    "best_for": "Medium datasets (100 ≤ n < 1000)",
                    "complexity": "O(n²)",
                    "memory": "O(n)",
                    "accuracy": "High"
                }
            }
        else:
            return {
                "recommended_method": "jax",
                "reasoning": f"Data size n={n} benefits from GPU acceleration",
                "method_details": {
                    "description": "JAX GPU-accelerated implementation",
                    "best_for": "Large datasets (n ≥ 1000)",
                    "complexity": "O(n²)",
                    "memory": "O(n)",
                    "accuracy": "High"
                }
            }
